//
// $Id: DataInStream.h,v 1.15 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(DATAINSTREAM_OL_GUARD)
#define DATAINSTREAM_OL_GUARD

#include <ObjectiveLib/ConnectedInStream.h>

#if !defined(OL_NO_OPENSTEP)
@class NSData;
#endif

/**
 * @class OLDataInStream DataInStream.h ObjectiveLib/DataInStream.h
 *
 * An in stream that is connected to a memory buffer. This is a concrete connected stream
 * that reads all of its data from a memory buffer.
 *
 * @sa OLDataOutStream
 *
 * @ingroup Streams
 */
@interface OLDataInStream : OLConnectedInStream
{
@protected
    /**
     * The memory buffer from which data is read
     */
    uint8_t*    bytes;

    /**
     * The number of bytes in the buffer
     */
    unsigned    byteCount;

    /**
     * The current position from which to read
     */
    unsigned    position;
}

/**
 * Create and return a new stream. The contents of the given buffer are copied
 * and the data are then used as the source of this connected stream.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param buffer the bytes to use as the data source
 * @param length the number of bytes
 * @return a reference to the newly created stream
 */
+ (id) streamWithBytes: (const uint8_t*)buffer count: (unsigned)length;

#if !defined(OL_NO_OPENSTEP)
/**
 * Create and return a new stream. The given data object is used to initialize the stream.
 * The returned stream is autoreleased before being returned.
 *
 * @param data the data from which the stream will read
 * @return a reference to the newly created stream
 */
+ (id) streamWithDataObject: (NSData*)data;
#endif

/**
 * @name Initializers and Deallocators
 */
/* @{ */
#if !defined(OL_NO_OPENSTEP)
/**
 * Initialize the stream. The stream copies the data from the given data object.
 *
 * @param data the data object used to initialize the stream
 * @return a reference to this stream
 */
- (id) initWithDataObject: (NSData*)data;
#endif

/**
 * Initialize the stream. The contents of @a buffer are copied into the stream's buffer.
 *
 * @param buffer the bytes from which the stream should read
 * @param length the number of bytes
 * @return a reference to this stream
 */
- (id) initWithBytes: (const uint8_t*)buffer count: (unsigned)length;

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Read a byte from the stream. This optimized version of reading a byte merely
 * copies a byte from the memory buffer unless there is no more data in the buffer.
 *
 * @exception OLEndOfStreamException if the end of the stream has been reached
 * @return the byte read
 */
- (uint8_t) readByte;

/**
 * Read a sequence of bytes from the stream. Bytes are copied from the buffer
 * unless the end of the buffer had already been reached before this message
 * was sent. In case there is no more data in the buffer the message returns
 * UINT_MAX.
 *
 * @param buffer the address to which the bytes should be read
 * @param max the maximum number bytes that should be read
 * @return the number of bytes read or UINT_MAX if no bytes could be read
 */
- (unsigned) readBytes: (uint8_t*)buffer count: (unsigned)max;

@end

#endif
