//
// $Id: FileOutStream.h,v 1.19 2007/04/15 23:32:57 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(FILEOUTSTREAM_OL_GUARD)
#define FILEOUTSTREAM_OL_GUARD

#include <ObjectiveLib/ConnectedOutStream.h>
#include <ObjectiveLib/Types.h>

/**
 * @class OLFileOutStream FileOutStream.h ObjectiveLib/FileOutStream.h
 *
 * A connected stream that sends its data to a file. This is a concrete connected stream
 * that is connected to a file. All data written will be sent to the file.
 *
 * @sa OLFileInStream
 *
 * @ingroup Streams
 */
@interface OLFileOutStream : OLConnectedOutStream
{
@protected
    /**
     * The file descriptor of the file to which data is sent
     */
    ol_fd fd;
}

/**
 * Create and return a new file out stream. A new stream is created with the
 * given path. If a file already exists at the indicated path, then that file will
 * be deleted and a new one will be created in its place.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if the path cannot be converted to a file name or
 * if a file cannot be opened with the given file name
 * @param path the path 
 * @return a reference to theh newly created stream
 */
+ (id) streamWithPath: (const char*)path;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The stream will be created with the given file descriptor and
 * all new data will be appended to the given file. The stream will take ownership of
 * the file descriptor and will assume that the stream is responsible for closing the
 * file descriptor.
 *
 * @param fileDesc the file descriptor of the file to which to write
 * @return a reference to this stream
 */
- (id) initWithFileDescriptor: (ol_fd)fileDesc;

/**
 * Initialize the stream. The stream will be created with the given file name. If a file
 * already exists at the given path, then that file will be erased and a new file will
 * written in its place.
 *
 * @exception OLInputOutputException if a file cannot be opened with the given file name
 * @param path the path to the file
 * @return a reference to this stream
 */
- (id) initWithPath: (const char*)path;

/**
 * Initialize the stream. The stream will be created with the given file name. If a file
 * already exists at the given path, then that file will either be truncated or appended to
 * depending on the value of @a append.
 *
 * @exception OLInputOutputException if a file cannot be opened with the given file name
 * @param path the path to the file
 * @param append YES if data should be appended to an existing file, NO if an existing file
 * should be truncated
 * @return a reference to this stream
 */
- (id) initWithPath: (const char*)path append: (BOOL)append;

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;

/**
 * Return the file descriptor associated with the file stream.
 *
 * @note The file stream always owns the file descriptor, so one should never close a
 * file descriptor that is already associated with a file stream.
 *
 * @return the file descriptor
 */
- (ol_fd) fileDescriptor;
- (void) flush;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Write a sequence of bytes to the stream. As many bytes up to the value of @a count
 * will be written to the file, and the actual number written will be returned.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param bytes the bytes to write
 * @param count the number of bytes to write
 * @return the number of bytes written
 */
- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)count;

@end

#endif
