/*
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2006 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2015)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/

package GoGrinder.sgf;

import java.awt.Point;
import java.util.*;

import GoGrinder.*;
import GoGrinder.command.*;

/**
 *
 * @author  tkington
 * @author  Ruediger Klehn
 *
 * This class corresponds to gametree in the SGF spec
 *
 */
public abstract class Node implements TagListener {
    static Random rand = new Random();
    private String parserMsg;
    protected Node parent;
    protected ArrayList ch;
    protected ArrayList cmds;
    protected HashMap props;
        
    protected int size;
    protected String comment;
    protected ArrayList marks;
    public static int currMainBranchSZ = 0;
    public static boolean szIsSet = false; // the node is processed from first to last char, so when SZ
                                           // is needed before it has been read out, this becomes chaotic
    
    protected Node(Node parent) {
        this.parent = parent;
        ch = new ArrayList();
        cmds = new ArrayList();
        props = new HashMap();
        marks = new ArrayList();
        
        //szIsSet = this.szIsSet;
    }
    
    protected void parse(String s) throws SGFParseException { // gets a node without "(;" / ")"
      // shouldn't this be named parseSize? - and called only once for a mainbranch?
 //d.b.g(size, "Node.parse");
    	cmds.clear();
    	props.clear();
    	marks.clear();
    	comment = null;
    	size = 0;
      SGFParser.parseTags(s, this);
// d.b.g(szIsSet);
      if (!szIsSet){// 
        String sz = getProperty("SZ");
//d.b.g(sz , "in Node, nach parse in sgfparser");
        // its enough to find this once per main branch - did we already find it in the root node?
        if(sz != null){
            if (sz.indexOf(":") != -1){ // SZ[3:9] = 3x9-board; MultiGo allows this - and some other programs also
              parserMsg = "Still only quadratic boards possible! (found \":\" in the size value: > " + sz + " < )";
              SGFLog.errorType = 3; // this is already catched in SGFParser, ~line 177, but, who knows...
              throw new SGFParseException(parserMsg, s);
            }
            size = Integer.parseInt(sz);
            if (size > 19){
              parserMsg = "Board size too big: " + size 
                        + " - at the moment not bigger than 19x19 (and quadratic)!";
              SGFLog.errorType = 3; // this is already catched in SGFParser, ~line 189, but, who knows...
              throw new SGFParseException(parserMsg, s);
            }
        }
        else {
          size = 19; // this is because of 19 is default size, if SZ[##] is not given (SGF Standard) d.b.g(currMainBranchSZ);
        }
        currMainBranchSZ = size;
        szIsSet = true;
// d.b.g(size, "Node.parse weiter hinne");
      }
      else{size = currMainBranchSZ;}
      if(SGFParser.VALIDATING){ // and node is root
          SGFParser.sgfValiSZ = size; 
      }
      else if(!GS.getEnableWGFEditor()){SGFParser.sgfSZ = size; }
      else {SGFParser.wgfSZ = size; }
// d.b.g(SGFParser.wgfSZ, "Node.parse, hinne");
	}

	public ArrayList getPropertyList(String tag) {
        ArrayList l = (ArrayList)props.get(tag);
        if(l == null) {
            l = new ArrayList();
            props.put(tag, l);
        }
        // DEBUG BY T.K.:
        /*else { //could have been less bytes: if("XS YG YB YW ...".indexOf(tag) == -1 && !tag.equals("W") ...)... and then 
            if(!tag.equals("XS") && !tag.equals("YG") && !tag.equals("B") && //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            !tag.equals("W") && !tag.equals("YB") && !tag.equals("YW") && //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            !tag.equals("LB") && !tag.equals("XB") && !tag.equals("XW") && //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            !tag.equals("LS") && !tag.equals("LN") && !tag.equals("TW") && //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            !tag.equals("TB") && !tag.equals("XG") && !tag.equals("LR")) //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                printProp(tag); //System.out.println("Mult: " + tag); // DEBUG BY T.K.
        }*/
        
        return l;
    }
    
    public void addTag(String tag, ArrayList propVals) throws SGFParseException {
        // first get size and store!! 
        if(tag.equals("C")) { //$NON-NLS-1$
            comment = (String)propVals.get(0); // extra handling of backslashes must be done in text frames
        }
        else if(!NodeMark.create(marks, tag, propVals)) { // we still don't know the board size
            Command cmd = Command.createCommand(tag, propVals);
            if(cmd != null)
                cmds.add(cmd);
        }
        
        getPropertyList(tag).addAll(propVals);
    }
    
    public void execute(Board board) {
        board.undoTempCommands();
        
        board.setMarks(marks);        
        CompositeCommand c = new CompositeCommand(cmds);
        board.executeCommand(c);
    }
    
    public void updateState(Board b) {
    	b.setMarks(marks);
    }
    
    public boolean hasAddCommand(int x, int y, int color) {
    	for (Iterator iter = cmds.iterator(); iter.hasNext();) {
			Object o = iter.next();
			if(o instanceof AddCommand) {
				AddCommand c = (AddCommand)o;
				if(c.getPlayer() == color &&
						c.hasPoint(x, y))
					return true;
			}
		}
    	
    	return false;
    }
    
    public void removeAddCommand(int x, int y, int color) {
    	for (Iterator iter = cmds.iterator(); iter.hasNext();) {
			Object o = iter.next();
			if(o instanceof AddCommand) {
				AddCommand c = (AddCommand)o;
				if(c.getPlayer() == color &&
						c.removePoint(x, y))
					return;
			}
		}
    }
    
    public void addAddCommand(int x, int y, int color) {
    	AddCommand c = null;
    	for (Iterator iter = cmds.iterator(); iter.hasNext();) {
			Object o = iter.next();
			if(o instanceof AddCommand) {
				AddCommand a = (AddCommand)o;
				if(a.getPlayer() == color) {
					c = a;
					break;
				}
			}
		}
    	
    	if(c == null) {
    		c = new AddCommand(color);
    		cmds.add(c);
    	}
    	
    	c.addPoint(new Point(x, y));
    }
    
    public boolean hasCommand(Command c) {
    	return cmds.contains(c);
    }
    
    public void addCommand(Command c) {
    	cmds.add(c);
    }
    
    public boolean removeCommand(Command c) { 
    	return cmds.remove(c);
    }
    
    public String getProperty(String p) {
        ArrayList l = (ArrayList)props.get(p);
        if(l == null)
            return null;
        
        return (String)l.get(0);
    }
    
    public void addChild(Node n) {
        ch.add(n);
    }
    
    public void validatePoints(int boardSize) throws SGFParseException {
        for(int i = 0; i < marks.size(); i++) {
            NodeMark m = (NodeMark)marks.get(i);
            m.validatePoints(boardSize);
        }
    
        for(int i = 0; i < cmds.size(); i++) {
            Command c = (Command)cmds.get(i);
            c.validatePoints(boardSize);
        }
    
        for(int i = 0; i < ch.size(); i++) {
            Node n = (Node)ch.get(i);
            n.validatePoints(boardSize);
        }
    }
    
    public Node getFirstChild() {
        if(ch.isEmpty())
            return null;
        return (Node)ch.get(0);
    }
    
    public void printProp(String prop) {
        final String NL = Main.NEW_LINE;
        System.out.println(prop); // WHAT IS THIS?? AHH - a debugging routine! (by T.K.)
        ArrayList p = (ArrayList)props.get(prop);
        if(p == null) {
            System.out.println(":null"); //$NON-NLS-1$ 
            parserMsg = "Common Exception (cannot yet say, what is wrong) in: "
                       + new Exception().getStackTrace()[0] + NL + " (compare to sgf-log.txt)";
            new SGFParseException(new Exception().fillInStackTrace(), parserMsg);
            d.b.g(parserMsg);
            return;
        }
        
        for(int i = 0; i < p.size(); i++) {
            System.out.println("[" + p.get(i) + "]"); //$NON-NLS-1$ //$NON-NLS-2$
        }
    }
    
    public SimpleMark getSimpleMark(int x, int y, int type) {
        for(int i = 0; i < marks.size(); i++) {
            Object o = marks.get(i);
            if(o instanceof SimpleMark) {
            	SimpleMark m = (SimpleMark)o;
	            Point p = m.getPoint();
	            if(p.x == x && p.y == y && m.getType() == type) {
	                return m;
	            }
            }
        }
        return null;
    }
    
	protected SimpleMark getMarkAt(Point p, boolean ignoreStoneMarks) {
	    for(int j = 0; j < marks.size(); j++) {
	        NodeMark m = (NodeMark)marks.get(j);
	        if(m instanceof SimpleMark) {
	        	SimpleMark s = (SimpleMark)m;
	        	if(p.equals(s.getPoint()) && (s.getType() > 3 || !ignoreStoneMarks))
	        		return (SimpleMark)m;
	        }
	    }
	    return null;
	}
    
    public String getNextLabel() {
    	for(int j = -1; j < 26; j++) {
    		String prefix = ""; //$NON-NLS-1$
    		if(j != -1) {
    			char c = (char)('A' + j);
    			prefix += c;
    		}
    		
	    	for(int i = 0; i < 26; i++) {
	    		char c = (char)('A' + i);
	    		String l = prefix + c;
	    		
	    		boolean found = false;
	    		for (Iterator iter = marks.iterator(); iter.hasNext();) {
					Object o = iter.next();
					if(o instanceof NodeLabel) {
						NodeLabel label = (NodeLabel)o;
						if(l.equals(label.getText())) {
							found = true;
							break;
						}
					}
				}
	    		
	    		if(!found)
	    			return l;
	    	}
    	}
    	
    	return "You must be joking"; //$NON-NLS-1$
    }
    
    public String getNextNumberLabel() {
    	for(int i = 1; i < 500; i++) {
    		String l = String.valueOf(i);
    		
    		boolean found = false;
    		for (Iterator iter = marks.iterator(); iter.hasNext();) {
				Object o = iter.next();
				if(o instanceof NodeLabel) {
					NodeLabel label = (NodeLabel)o;
					if(l.equals(label.getText())) {
						found = true;
						break;
					}
				}
			}
    		
    		if(!found)
    			return l;
    	}
    	
    	return null;
    }
    
    public boolean hasStoneMarkAt(int x, int y) { // this seems to be connected to the WGF part
        for(int i = 0; i < marks.size(); i++) {
            Object o = marks.get(i);
            if(o instanceof SimpleMark) {
            	SimpleMark m = (SimpleMark)o;
	            Point p = m.getPoint();
	            if(p.x == x && p.y == y) {
	            	switch(m.getType()) {
	            	case NodeMark.FAKEB:
	            	case NodeMark.FAKEW:
	            	case NodeMark.GREYSTONE:
	            	case NodeMark.LOCALB:
	            	case NodeMark.LOCALW:
	            		return true;
	            	}
	            }
            }
        }
        
        return false;    	
    }
    
    public void addMark(NodeMark m) { // this seems to be connected to the WGF part
    	marks.add(m);
    	if(m instanceof SimpleMark) {
    		SimpleMark s = (SimpleMark)m;
    		if(s.getType() < 4)
    			Collections.sort(marks);
    	}
    }
    
    public static String removeEscChar(String fromThisStr){
      if (fromThisStr == null)
        return "";
      String returnStr = "";
      for (int i=0;i<fromThisStr.length();i++){
        char c = fromThisStr.charAt(i);
        if (!(new Character(c)).equals('\\')){
          returnStr += c;
        }
        else {
          i++;
          returnStr += fromThisStr.charAt(i);
        }
      }
      return returnStr;
    }
    
    public void removeMark(NodeMark m) { marks.remove(m); }
    
    public Node getParent() { return parent; }
    public void setParent(Node n) { parent = n; }
    public String getComment() { return removeEscChar(comment); } // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
    public void setComment(String c) { comment = c; }
    public int getSize() { return size; }
}
