//
// $Id: Map.h,v 1.25 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(MAP_OL_GUARD)
#define MAP_OL_GUARD

#include <ObjectiveLib/Iterator.h>
#include <ObjectiveLib/Functional.h>

/**
 * @internal The internal red-black tree data structure
 */
@class OLTreeMap;

/**
 * @class OLMap Map.h ObjectiveLib/Map.h
 *
 * A collection of key-value pairs. A map is a sorted associative container of elements.
 * Keys are associated with values, and the keys can be used to look up and gain
 * access to values. Each key is unique. If a key-value pair is inserted into the map
 * and the key already exists, then the insertion will fail.
 * Maps are sorted by definition and the sorting order is determined by a
 * comparison function that is passed at initialization. If no comparison function is
 * specified, then OLLess is used, thus sorting the map in ascending order.
 *
 * The object to which a map's iterators point is an instance of OLPair. The first
 * value of the pair is the key, and the second value of the pair is the key's value.
 *
 * @sa OLAssociativeIterator
 *
 * @ingroup Containers
 */
@interface OLMap :
#if defined(OL_NO_OPENSTEP)
    Object <OLInserter, OLStreamable>
#else
    NSObject <OLInserter, OLStreamable, NSCopying, NSCoding>
#endif
{
@protected
    /**
     * The red-black tree that provides the underlying data structure
     */
    OLTreeMap* tree;
}

/**
 * Create and return a new map. An instance of OLLess is
 * used to compare keys for sorting.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return a new map
 */
+ (id) map;

/**
 * Create and return a new map. The map is initialized
 * with the contents of the range <tt>[first, last)</tt>. An instance of OLLess
 * is used to compare keys for sorting.
 *
 * @pre Each element in the range <tt>[first, last)</tt> must be an instance of
 * OLPair. The first element of the pair is the key and the second element is the
 * key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param first the first in the range of elements to insert
 * @param last one beyond the last in the range of elements to insert
 * @return a new map
 */
+ (id) mapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last;

/**
 * Create and return a new map. The map is empty and the
 * comparison function @a comp is used to sort keys.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param comp the comparison function used to sort keys
 * @return a new map
 */
+ (id) mapWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp;

/**
 * Create and return a new map. The map is initialized
 * with the contents of @a right.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param right the map to copy
 * @return a new map
 */
+ (id) mapWithMap: (OLMap*)right;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the map. The map is empty and OLLess is used as the comparison function.
 *
 * @return a reference to this map
 */
- (id) init;

/**
 * Initialize the map. All elements in the range <tt>[first, last)</tt> are inserted
 * into the map. OLLess is used as the comparison function.
 * If any duplicate keys exist in the range <tt>[first, last)</tt>,
 * then only the first instance is inserted.
 *
 * @pre Each element in the range <tt>[first, last)</tt> must be an instance of
 * OLPair. The first element of the pair is the key and the second element is the
 * key's value.
 *
 * @param first the first in the range of elements to insert
 * @param last one beyond the last in the range of elements to insert
 * @return a reference to this map
 */
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last;

/**
 * Initialize the map. All elements in the range <tt>[first, last)</tt> are inserted
 * into the map. The comparison function @a comp is used.
 *
 * @pre Each element in the range <tt>[first, last)</tt> must be an instance of
 * OLPair. The first element of the pair is the key and the second element is the
 * key's value.
 *
 * @param first the first in the range of elements to insert
 * @param last one beyond the last in the range of elements to insert
 * @param comp the comparison function used to sort keys
 * @return a reference to this map
 */
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last compare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp;

#if !defined(OL_NO_OPENSTEP)
/**
 * Initialize the map. This initializer creates a new map
 * from an archive and returns it.
 *
 * @post The map returned will be identical to the map
 * saved to the archive using the #encodeWithCoder: message.
 *
 * @param decoder the coder which will decode the archived map
 * @return a reference to this map
 */
- (id) initWithCoder: (NSCoder*)decoder;
#endif

/**
 * Initialize the map. The map is empty and the comparison function @a comp is used
 * to sort keys.
 *
 * @param comp the comparison function used to sort keys
 * @return a reference to this map
 */
- (id) initWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp;

/**
 * Initialize the map. The comparison function and all elements are
 * copied from @a right into this map.
 *
 * @param right the map with which to initialize this one
 * @return a reference to this map
 */
- (id) initWithMap: (OLMap*)right;

- (id) initWithObjectInStream: (OLObjectInStream*)stream;

/**
 * Finalize the map and deallocate any allocated memory.
 */
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
/* @} */

/**
 * Assign a value to a key. If the key exists in the map, then @a value is assigned to
 * the key and the old value is dropped. If the key does not exist in the map, then
 * the key-value pair is inserted.
 *
 * @param key the key to which to assign
 * @param value the value to assign
 */
- (void) assignKey: (id)key value: (id)value;

/**
 * Return an iterator that points to the first element in the map. The object to which
 * the iterator points is an instance of OLPair. The first element of the pair is the
 * key and the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return the first iterator for the map
 */
- (OLAssociativeIterator*) begin;

/**
 * Remove all elements.
 */
- (void) clear;

/**
 * Compare this map to another object. If the other object is of type OLMap, each
 * of the contained objects is compared to the corresponding object in @a other by
 * calling the @c compare: method.
 *
 * @param other the object with which to compare this one
 * @return a value greater than, equal to, or less than zero accoringly as this object
 * is greater than, equal to, or less than @a other
 */
- (int) compare: (id)other;
#if defined(OL_NO_OPENSTEP)
/**
 * Make a copy of this map.
 *
 * @return the copy
 */
- (id) copy;
#else
/**
 * Make a copy of this map allocating memory from the given zone.
 *
 * @param zone the zone from which to allocate memory
 * @return the copy
 */
- (id) copyWithZone: (NSZone*)zone;
#endif

/**
 * Return the number of instances of @a key in the map.
 *
 * @param key the key for which to search
 * @return the number of values that exist for @a key
 */
- (unsigned) count: (id)key;

/**
 * Return whether the map is empty.
 *
 * @return YES if the map is empty, NO otherwise
 */
- (BOOL) empty;

#if !defined(OL_NO_OPENSTEP)
/**
 * Encode the map. The map is saved to an archive using @a encoder. The map
 * will be retrieved from the archive using the initializer #initWithCoder:.
 *
 * @param encoder the coder which will save the bit set to the archive
 */
- (void) encodeWithCoder: (NSCoder*)encoder;
#endif

/**
 * Return an iterator that points to one position beyond the last element. The object to which
 * the iterator points is an instance of OLPair. The first element of the pair is the
 * key and the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return an iterator one position beyond the last element in the map
 */
- (OLAssociativeIterator*) end;

/**
 * Return a range whose elements are equal to @a key. The returned pair contains two instances
 * of OLAssociativeIterator delimiting the range of elements. The objects to which
 * the iterators point are instances of OLPair. The first element of the pair is the
 * key and the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param key the value for which to search
 * @return a pair of OLAssociativeIterator instances that define the range of elements
 * in the map equal to @a key
 */
- (OLPair*) equalRange: (id)key;

/**
 * Remove the element designated by @a where.
 *
 * @pre @a where must point to an element in this map.
 *
 * @param where an iterator designating the element to remove
 */
- (void) erase: (OLAssociativeIterator*)where;

/**
 * Remove a range of elements. All elements in the range <tt>[first, last)</tt> will
 * be removed from the map.
 *
 * @pre @a first and @a last must refer to elements in this map.
 *
 * @param first the first in the range of elements to remove
 * @param last one position beyond the last in the range of elements to remove
 */
- (void) eraseFrom: (OLAssociativeIterator*)first to: (OLAssociativeIterator*)last;

/**
 * Erase all instances of @a key. The map is searched for @a key, all instances are removed
 * from the map and the number removed is returned.
 *
 * @param key the key to remove from the map
 * @return the number of elements removed
 */
- (unsigned) eraseKey: (id)key;

/**
 * Find an element. The element @a key is searched for in the map and an iterator to its
 * location is returned. If the element does not exist in the map, then the returned
 * iterator will be equal to the iterator returned by #end.
 *
 * The object to which the iterator points is an instance of OLPair. The first element
 * of the pair is the key and the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param key the key for which to search
 * @return an iterator pointing to the location of @a key, or an iterator equal to that
 * returned by #end if @a key does not exist
 */
- (OLAssociativeIterator*) find: (id)key;

/**
 * Insert a key-value pair into the map. An attempt is made to insert @a keyValue into the map,
 * and an instance of OLPair is returned indicating the state of the insertion. The first
 * element in the returned pair is an instance of OLAssociativeIterator indicating the
 * position of @a keyValue in the map, and the second element of the returned pair is
 * an object that responds to the message @c boolValue. The message @c boolValue will
 * return YES if the insertion was successful, or NO if not.
 *
 * @pre The first element of @a keyValue must be the key and the second element must
 * be the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param keyValue the key-value pair to insert
 * @return an instance of OLPair indicating the position of @a keyVal in the map and
 * whether the insertion succeeded or failed
 */
- (id) insert: (OLPair*)keyValue;

/**
 * Insert a key-value pair into the map. The @a keyVal is inserted and the position indicated by
 * @a where is used as a hint about where in the map the @a keyVal should be placed. There
 * is no guarantee that the @a keyVal will ultimately placed anywhere close to @a where.
 * Note that the insertion may fail if the key is already in the map. An iterator is
 * returned that points to @a object. The object to which the iterator points is an instance
 * of OLPair. The first element of the pair is the key and the second element of the
 * pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param where a hint as to where in the map the @a keyVal should be located
 * @param keyVal the key-value pair to insert
 * @return the iterator indicating where @a keyVal is now located in the map
 */
- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (OLPair*)keyVal;

/**
 * Insert a range of key-value pairs into the map. An attempt is made to insert all pairs
 * in the range <tt>[first, last)</tt>, however there is no guarantee that any of
 * the elements in the range will actually be inserted if they already exist in the
 * map.
 *
 * @pre All elements in the range <tt>[first, last)</tt> must be instances of OLPair
 * where the first element of the pair is the key and the second element is the key's
 * value.
 *
 * @param first the first in the range of pairs to insert
 * @param last one position beyond the last in the range of pairs to insert
 */
- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last;

/**
 * Insert a key-value pair. This message is simply a wrapper for #insert:, provided
 * as a convenience. It elminiates the need to create an instance of OLPair before
 * performing insertion. As with #insert: the message returns an instance of OLPair
 * indicating the state of the insertion. The first element of the pair returned
 * is an instance of OLAssociativeIterator indicating the position of @a key in the
 * map, and the second element of the pair is an object that responds to the
 * message @c boolValue. The message @c boolValue will return YES if the insertion
 * succeeded, or NO if it did not.
 *
 * @param key the key to insert
 * @param val the value of the key
 * @return an instance of OLPair indicating the position of @a key in the map and
 * whether the insertion succeeded or failed
 */
- (id) insertKey: (id)key value: (id)val;

/**
 * Test whether another map is equal to this one. Two maps are considered equal if they
 * contain the same number of objects and the objects are in the same order and each
 * object is equal to the corresponding object in the other map.
 *
 * @param object the object to test
 * @return YES if @a object is equal to this map, NO otherwise
 */
- (BOOL) isEqual: (id)object;

/**
 * Return the comparison function for keys. This function ultimately determines the
 * sorting order of the map.
 *
 * @return the @ref Functors "function object" used to compare keys
 */
- (OLStreamableFunctor<OLBoolBinaryFunction>*) keyComp;

/**
 * Find the lower bound of a given @a key. The lower bound is the first position in
 * the map at which @a key can be inserted without disturbing the sorting order. An
 * iterator pointing to the lower bound is returned. The object to which the iterator
 * points is an instance of OLPair. The first element of the pair is the key and
 * the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param key the key for which to find the lower bound
 * @return an iterator indicating the lower bound
 */
- (OLAssociativeIterator*) lowerBound: (id)key;

/**
 * Return the maxiumum number of objects that can be stored in a map. This limit is
 * theoretical, meaning that there is no guarantee that you can insert this many
 * objects into any given map. The memory conditions of the run-time system play an
 * important role.
 *
 * @return the maximum number of objects for a map
 */
- (unsigned) maxSize;

/**
 * Return a reverse iterator pointing to the end of the sequence. Advancing the returned
 * iterator will move backwards through the sequence to the point indicated by the
 * iterator returned by #rend. The object to which the iterator points is an instance
 * of OLPair. The first element of the pair is the key and the second element of the
 * pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return a reverse iterator for the end of the sequence
 */
- (OLReverseBidiIterator*) rbegin;

/**
 * Return a reverse iterator pointing to the beginning of the sequence. This iterator
 * indicates one position beyond the last position that may be referenced by a
 * reverse iterator. The object to which the iterator points is an instance
 * of OLPair. The first element of the pair is the key and the second element of the
 * pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return a reverse iterator for the beginning of the sequence
 */
- (OLReverseBidiIterator*) rend;

/**
 * Return the number of elements in the map.
 *
 * @return the number of elements
 */
- (unsigned) size;

/**
 * Swap this map with another one. All elements in @a right will be placed into
 * this map, and all elements in this map will be placed in @a right.
 *
 * @param right the map with which to swap this one
 */
- (void) swap: (OLMap*)right;

/**
 * Find the upper bound of a given @a key. The upper bound is the last position in
 * the map at which @a key can be inserted without disturbing the sorting order. An
 * iterator pointing to the upper bound is returned. The object to which the
 * iterator points is an instance of OLPair. The first element of the pair is the
 * key and the second element of the pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param key the key for which to find the upper bound
 * @return an iterator indicating the upper bound
 */
- (OLAssociativeIterator*) upperBound: (id)key;

/**
 * Return the function used to compare values. This returns the same function as
 * the message #keyComp.
 *
 * @return the @ref Functors "function object" for comparing values
 */
- (OLStreamableFunctor<OLBoolBinaryFunction>*) valueComp;

/**
 * Find the value of a given key. The map is searched for @a key, and if it is found
 * its value is returned. Otherwise, the message returns @c nil.
 *
 * @param key the key for which to search
 * @return the key's value or @c nil if the key does not exist in the map
 */
- (id) valueForKey: (id)key;

- (void) writeSelfToStream: (OLObjectOutStream*)stream;

@end

/**
 * @class OLMultiMap Map.h ObjectiveLib/Map.h
 *
 * A map that allows many values to be associated with equivilent keys. Multimap is identical to
 * map expect that as many instances of any given key may be inserted as desired.
 *
 * @sa OLMap, OLAssociativeIterator
 *
 * @ingroup Containers
 */
@interface OLMultiMap : OLMap
{
}


/**
 * Create and return a new map. An instance of OLLess is
 * used to compare keys for sorting.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return a new map
 */
+ (id) multiMap;

/**
 * Create and return a new map. The map is initialized
 * with the contents of the range <tt>[first, last)</tt>. An instance of OLLess
 * is used to compare keys for sorting.
 *
 * @pre Each element in the range <tt>[first, last)</tt> must be an instance of
 * OLPair. The first element of the pair is the key and the second element is the
 * key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param first the first in the range of elements to insert
 * @param last one beyond the last in the range of elements to insert
 * @return a new map
 */
+ (id) multiMapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last;

/**
 * Create and return a new map. The map is empty and the
 * comparison function @a comp is used to sort keys.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param comp the comparison function used to sort keys
 * @return a new map
 */
+ (id) multiMapWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp;

/**
 * Create and return a new map. The map is initialized
 * with the contents of @a right.
 *
 * @note The map @a right may be either an instance of OLMap or an instance of
 * OLMultiMap.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param right the map to copy
 * @return a new map
 */
+ (id) multiMapWithMap: (OLMap*)right;

/**
 * Assign a value to a key. This message always inserts the key-value pair
 * <tt>(key, value)</tt>.
 *
 * @param key the key to insert
 * @param value the key's value
 */
- (void) assignKey: (id)key value: (id)value;

/**
 * Insert a key-value pair into the map. The pair is inserted and an instance of
 * OLAssociativeIterator is returned indicating the position of @a keyValue
 * in the set. The object to which the iterator points is an instance
 * of OLPair. The first element of the pair is the key and the second element of the
 * pair is the key's value.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @param keyValue the element to insert
 * @return an instance of OLAssociativeIterator pointing to the newly inserted object
 */
- (id) insert: (OLPair*)keyValue;
- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (OLPair*)keyVal;
- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last;

/**
 * Test whether another map is equal to this one. Two maps are considered equal if they
 * contain the same number of objects and the and the value of each key is equal to the
 * value of the corresponding key in the other map.
 *
 * @note When two maps each contain
 * a range of equal keys, the values that correspond to those keys do not have to appear
 * in the same order. They just have to exist. 
 *
 * @param object the object to test
 * @return YES if @a object is equal to this map, NO otherwise
 */
- (BOOL) isEqual: (id)object;

/**
 * Find the value of a given key. Since a multimap can have any number of values
 * associated with equivilent keys, this message does nothing. The value @c nil
 * is @b always returned.
 *
 * @param key the key for which to search
 * @return @c nil
 */
- (id) valueForKey: (id)key;

@end

#endif
