/**
 * @file libcomprex/archive_io.h Archive I/O Functions
 * 
 * $Id: archive_io.h,v 1.7 2003/01/01 06:22:35 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_ARCHIVE_IO_H_
#define _LIBCOMPREX_ARCHIVE_IO_H_

#include <libcomprex/archive.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Opens an archive for reading or writing.
 *
 * @param path The path to the archive.
 * 
 * @return The archive structure, or NULL if the file couldn't be
 *         opened.
 *
 * @see cxCloseArchive()
 * @see cxOpenArchiveBuffer()
 * @see cxOpenArchiveStream()
 * @see cxSaveArchive()
 */
CxArchive *cxOpenArchive(const char *path);

/**
 * Opens an archive from an existing file handle (CxFile structure).
 *
 * @param file The file handle.
 * @param mode The access mode.
 *
 * @return The archive, or @c NULL if it's unable to open.
 *
 * @see cxOpenBuffer()
 * @see cxOpenFile()
 * @see cxOpenStream()
 * @see cxClose()
 */
CxArchive *cxOpenArchiveFromHandle(CxFile *file, CxAccessMode mode);

/**
 * Opens an archive from an existing file stream for reading or writing.
 *
 * @param fp The existing file pointer stream.
 * 
 * @return The archive structure, or NULL if an archive could not be found.
 *
 * @see cxCloseArchive()
 * @see cxOpenArchive()
 * @see cxOpenArchiveBuffer()
 * @see cxSaveArchive()
 */
CxArchive *cxOpenArchiveStream(FILE *fp);

/**
 * Opens an archive from a buffer for reading or writing.
 *
 * @param buffer The buffer to read from or write to.
 * @param size   The size of the buffer.
 * 
 * @return The archive structure, or NULL if an archive could not be found.
 *
 * @see cxCloseArchive()
 * @see cxOpenArchive()
 * @see cxOpenArchiveStream()
 * @see cxSaveArchive()
 */
CxArchive *cxOpenArchiveBuffer(char *buffer, size_t size);

/**
 * Closes an archive.
 *
 * If changes were made to this archive, they will not be saved.
 * To save them, use cxSaveArchive().
 *
 * @param archive The archive structure.
 *
 * @see cxOpenArchive()
 * @see cxOpenArchiveBuffer()
 * @see cxOpenArchiveStream()
 * @see cxSaveArchive()
 */
void cxCloseArchive(CxArchive *archive);

/**
 * Saves an archive.
 *
 * This will not close the archive. To close the archive, use
 * cxCloseArchive().
 *
 * @param archive  The archive structure.
 * @param filename The destination filename.
 *
 * @return The status of the operation.
 *
 * @see cxCloseArchive()
 * @see cxOpenArchive()
 * @see cxOpenArchiveBuffer()
 * @see cxOpenArchiveStream()
 */
CxStatus cxSaveArchive(CxArchive *archive, const char *filename);

/**
 * Extracts the specified archive's files to directories off of the current
 * directory.
 *
 * @param archive  The archive to extract.
 *
 * @return The status of the operation.
 */
CxStatus cxExtractArchive(CxArchive *archive);

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_ARCHIVE_IO_H_ */

