/* -*- Mode: C; c-file-style: "gnu" -*-
   jniint.h -- Java Native Interface internal header file.
   Created: Chris Toshok <toshok@hungry.com>, 26-Jul-1997
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _hungry_jniint_h
#define _hungry_jniint_h

#ifdef __cplusplus
extern "C" {
#endif

#include "jni.h"
#include "classpath.h"
#include "class-repository.h"
#include "native-threads.h"
#include "interp.h"
#include "gc.h"

#define JNIFUNC(x) _hungryJNI_##x

  /* verify modes -- since sun in their infinite wisdom used plain integers
     and didn't spec a more intuitive enumerated set of constants. */
  enum {
    VERIFY_NONE,
    VERIFY_REMOTE, /* verify only code loaded from a remote source */
    VERIFY_ALL
  };

  /*
  ** we wrap the sun defined types for the JavaVM and JNIEnv so that we can
  ** keep data along with the vtables.  While this may cause binary compatibility
  ** problems, it allows us to have more than one VM per process (which sun can't
  ** even do at this time.)
  */
  typedef struct HungryJavaVM HungryJavaVM;
  typedef struct HungryJNIEnv HungryJNIEnv;

  struct HungryJavaVM {
    /* the sun vtable */
    JavaVM _vm;

    /* the version of JNI used for this JavaVM */
    jint			_version;

    /* queue pointers */
    HungryJavaVM* 	next;
    HungryJavaVM* 	prev;

    HMonitor		_mon;

    /* if the version is JNI_VERSION_1_1 */
    JDK1_1InitArgs	_args;

    /* a hook for a function that redirects all VM messages. */
    jint (*vfprintf)(FILE *fp, const char *format, va_list args);
    void (*exit)(jint code);	/* a VM exit hook. */
    void (*abort)(void);		/* a VM abort hook. */

    const char		*_classpath;
    ClasspathEntry	*_cp_entries;
    int			_num_cp_entries;

    class_list		**_class_repository;

    jboolean		_finalizers_on_exit;
    HungryJNIEnv		*_envs;
    HungryJNIEnv		*_initial_env;

    JGC_struct		*_gc;

#define VERBOSE_JNI	0x01
#define VERBOSE_CLASS	0x02
#define VERBOSE_GC	0x04
#define VERBOSE_METHOD  0x08
    jint			_verbose_flags;

    ClazzFile		*_string_cf;
    ClazzFile		*_object_cf;
    ClazzFile		*_class_cf;
  };

  typedef struct _JIT JIT;

  struct HungryJNIEnv {
    /* the sun vtable  */
    JNIEnv _env;

    /* queue pointers */
    HungryJNIEnv *next;
    HungryJNIEnv *prev;

    /* back pointer to the vm */
    HungryJavaVM*		_vm;

    /* back pointer to our thread. */
    HThread		_thr;

    /* pointer to our java thread info (used by the interpreter loop */
    JThreadInfo*		_java_info;

#if 0
    /* the local references for this thread.  These will not be collected
       until the native method that created them returns.  once the method
       returns they will all be collected. */
    GC_obj		*_local_refs;
#endif

    /* our current exception. */
    jthrowable		_exception;

    /* the jit state. */
    JIT			*_jit;
  };

  extern jclass sig_to_jclass(JNIEnv *env, Signature *sig);

  extern HungryJNIEnv* JNIFUNC(AllocHungryJNIEnv)(void);
  extern void JNIFUNC(DeallocHungryJNIEnv)(HungryJNIEnv*);
  extern HungryJavaVM* JNIFUNC(AllocHungryJavaVM)(void);
  extern void JNIFUNC(DeallocHungryJavaVM)(HungryJavaVM*);

  extern jint JNIFUNC(GetVersion)(JNIEnv *env);

  extern jclass JNIFUNC(DefineClass)(JNIEnv *env, jobject loader, const jbyte* buf, jsize bufLen);
  extern jclass JNIFUNC(FindClass)(JNIEnv *env, const char *name);
  extern jclass JNIFUNC(GetSuperclass)(JNIEnv *env, jclass clazz);
  extern jboolean JNIFUNC(IsAssignableFrom)(JNIEnv *env, jclass clazz1, jclass clazz2);

  extern jint JNIFUNC(Throw)(JNIEnv *env, jthrowable obj);
  extern jint JNIFUNC(ThrowNew)(JNIEnv *env, jclass clazz, const char *message);
  extern jthrowable JNIFUNC(ExceptionOccurred)(JNIEnv *env);
  extern void JNIFUNC(ExceptionDescribe)(JNIEnv *env);
  extern void JNIFUNC(ExceptionClear)(JNIEnv *env);
  extern void JNIFUNC(FatalError)(JNIEnv *env, const char *msg);

  extern jobject JNIFUNC(NewGlobalRef)(JNIEnv *env, jobject obj);
  extern void JNIFUNC(DeleteGlobalRef)(JNIEnv *env, jobject globalRef);
  extern void JNIFUNC(DeleteLocalRef)(JNIEnv *env, jobject localRef);

  extern jobject JNIFUNC(AllocObject)(JNIEnv *env, jclass clazz);
  extern jobject JNIFUNC(NewObject)(JNIEnv *env, jclass clazz,
				    jmethodID methodID, ...);
  extern jobject JNIFUNC(NewObjectA)(JNIEnv *env, jclass clazz,
				     jmethodID methodID, jvalue *args);
  extern jobject JNIFUNC(NewObjectV)(JNIEnv *env, jclass clazz,
				     jmethodID methodID, va_list args);
  extern jclass JNIFUNC(GetObjectClass)(JNIEnv *env, jobject obj);
  extern jboolean JNIFUNC(IsInstanceOf)(JNIEnv *env, jobject obj, jclass clazz);
  extern jboolean JNIFUNC(IsSameObject)(JNIEnv *env, jobject ref, jobject ref2);

  extern jfieldID JNIFUNC(GetFieldID)(JNIEnv *env, jclass clazz,
				      const char *name, const char *sig);
  extern jboolean JNIFUNC(GetBooleanField)(JNIEnv *env, jobject obj,
					   jfieldID fieldID);
  extern jbyte JNIFUNC(GetByteField)(JNIEnv *env, jobject obj,
				     jfieldID fieldID);
  extern jchar JNIFUNC(GetCharField)(JNIEnv *env, jobject obj,
				     jfieldID fieldID);
  extern jshort JNIFUNC(GetShortField)(JNIEnv *env, jobject obj,
				       jfieldID fieldID);
  extern jint JNIFUNC(GetIntField)(JNIEnv *env, jobject obj,
				   jfieldID fieldID);
  extern jlong JNIFUNC(GetLongField)(JNIEnv *env, jobject obj,
				     jfieldID fieldID);
  extern jfloat JNIFUNC(GetFloatField)(JNIEnv *env, jobject obj,
				       jfieldID fieldID);
  extern jdouble JNIFUNC(GetDoubleField)(JNIEnv *env, jobject obj,
					 jfieldID fieldID);
  extern jobject JNIFUNC(GetObjectField)(JNIEnv *env, jobject obj,
					 jfieldID fieldID);

  extern void JNIFUNC(SetBooleanField)(JNIEnv *env, jobject obj,
				       jfieldID fieldID, jboolean value);
  extern void JNIFUNC(SetByteField)(JNIEnv *env, jobject obj,
				    jfieldID fieldID, jbyte value);
  extern void JNIFUNC(SetCharField)(JNIEnv *env, jobject obj,
				    jfieldID fieldID, jchar value);
  extern void JNIFUNC(SetShortField)(JNIEnv *env, jobject obj,
				     jfieldID fieldID, jshort value);
  extern void JNIFUNC(SetIntField)(JNIEnv *env, jobject obj,
				   jfieldID fieldID, jint value);
  extern void JNIFUNC(SetLongField)(JNIEnv *env, jobject obj,
				    jfieldID fieldID, jlong value);
  extern void JNIFUNC(SetFloatField)(JNIEnv *env, jobject obj,
				     jfieldID fieldID, jfloat value);
  extern void JNIFUNC(SetDoubleField)(JNIEnv *env, jobject obj,
				      jfieldID fieldID, jdouble value);
  extern void JNIFUNC(SetObjectField)(JNIEnv *env, jobject obj,
				      jfieldID fieldID, jobject value);

  extern jmethodID JNIFUNC(GetMethodID)(JNIEnv *env, jclass clazz,
					const char *name, const char *sig);
  extern jboolean JNIFUNC(CallBooleanMethod)(JNIEnv *env, jobject obj,
					     jmethodID methodID, ...);
  extern jboolean JNIFUNC(CallBooleanMethodA)(JNIEnv *env, jobject obj,
					      jmethodID methodID, jvalue *args);
  extern jboolean JNIFUNC(CallBooleanMethodV)(JNIEnv *env, jobject obj,
					      jmethodID methodID, va_list args);
  extern jbyte JNIFUNC(CallByteMethod)(JNIEnv *env, jobject obj,
				       jmethodID methodID, ...);
  extern jbyte JNIFUNC(CallByteMethodA)(JNIEnv *env, jobject obj,
					jmethodID methodID, jvalue *args);
  extern jbyte JNIFUNC(CallByteMethodV)(JNIEnv *env, jobject obj,
					jmethodID methodID, va_list args);
  extern jchar JNIFUNC(CallCharMethod)(JNIEnv *env, jobject obj,
				       jmethodID methodID, ...);
  extern jchar JNIFUNC(CallCharMethodA)(JNIEnv *env, jobject obj,
					jmethodID methodID, jvalue *args);
  extern jchar JNIFUNC(CallCharMethodV)(JNIEnv *env, jobject obj,
					jmethodID methodID, va_list args);
  extern jshort JNIFUNC(CallShortMethod)(JNIEnv *env, jobject obj,
					 jmethodID methodID, ...);
  extern jshort JNIFUNC(CallShortMethodA)(JNIEnv *env, jobject obj,
					  jmethodID methodID, jvalue *args);
  extern jshort JNIFUNC(CallShortMethodV)(JNIEnv *env, jobject obj,
					  jmethodID methodID, va_list args);
  extern jint JNIFUNC(CallIntMethod)(JNIEnv *env, jobject obj,
				     jmethodID methodID, ...);
  extern jint JNIFUNC(CallIntMethodA)(JNIEnv *env, jobject obj,
				      jmethodID methodID, jvalue *args);
  extern jint JNIFUNC(CallIntMethodV)(JNIEnv *env, jobject obj,
				      jmethodID methodID, va_list args);
  extern jlong JNIFUNC(CallLongMethod)(JNIEnv *env, jobject obj,
				       jmethodID methodID, ...);
  extern jlong JNIFUNC(CallLongMethodA)(JNIEnv *env, jobject obj,
					jmethodID methodID, jvalue *args);
  extern jlong JNIFUNC(CallLongMethodV)(JNIEnv *env, jobject obj,
					jmethodID methodID, va_list args);
  extern jfloat JNIFUNC(CallFloatMethod)(JNIEnv *env, jobject obj,
					 jmethodID methodID, ...);
  extern jfloat JNIFUNC(CallFloatMethodA)(JNIEnv *env, jobject obj,
					  jmethodID methodID, jvalue *args);
  extern jfloat JNIFUNC(CallFloatMethodV)(JNIEnv *env, jobject obj,
					  jmethodID methodID, va_list args);
  extern jdouble JNIFUNC(CallDoubleMethod)(JNIEnv *env, jobject obj,
					   jmethodID methodID, ...);
  extern jdouble JNIFUNC(CallDoubleMethodA)(JNIEnv *env, jobject obj,
					    jmethodID methodID, jvalue *args);
  extern jdouble JNIFUNC(CallDoubleMethodV)(JNIEnv *env, jobject obj,
					    jmethodID methodID, va_list args);
  extern void JNIFUNC(CallVoidMethod)(JNIEnv *env, jobject obj,
				      jmethodID methodID, ...);
  extern void JNIFUNC(CallVoidMethodA)(JNIEnv *env, jobject obj,
				       jmethodID methodID, jvalue *args);
  extern void JNIFUNC(CallVoidMethodV)(JNIEnv *env, jobject obj,
				       jmethodID methodID, va_list args);
  extern jobject JNIFUNC(CallObjectMethod)(JNIEnv *env, jobject obj,
					   jmethodID methodID, ...);
  extern jobject JNIFUNC(CallObjectMethodA)(JNIEnv *env, jobject obj,
					    jmethodID methodID, jvalue *args);
  extern jobject JNIFUNC(CallObjectMethodV)(JNIEnv *env, jobject obj,
					    jmethodID methodID, va_list args);

  extern jboolean JNIFUNC(CallNonvirtualBooleanMethod)(JNIEnv *env, jobject obj,
						       jmethodID methodID, ...);
  extern jboolean JNIFUNC(CallNonvirtualBooleanMethodA)(JNIEnv *env, jobject obj,
							jmethodID methodID, jvalue *args);
  extern jboolean JNIFUNC(CallNonvirtualBooleanMethodV)(JNIEnv *env, jobject obj,
							jmethodID methodID, va_list args);
  extern jbyte JNIFUNC(CallNonvirtualByteMethod)(JNIEnv *env, jobject obj,
						 jmethodID methodID, ...);
  extern jbyte JNIFUNC(CallNonvirtualByteMethodA)(JNIEnv *env, jobject obj,
						  jmethodID methodID, jvalue *args);
  extern jbyte JNIFUNC(CallNonvirtualByteMethodV)(JNIEnv *env, jobject obj,
						  jmethodID methodID, va_list args);
  extern jchar JNIFUNC(CallNonvirtualCharMethod)(JNIEnv *env, jobject obj,
						 jmethodID methodID, ...);
  extern jchar JNIFUNC(CallNonvirtualCharMethodA)(JNIEnv *env, jobject obj,
						  jmethodID methodID, jvalue *args);
  extern jchar JNIFUNC(CallNonvirtualCharMethodV)(JNIEnv *env, jobject obj,
						  jmethodID methodID, va_list args);
  extern jshort JNIFUNC(CallNonvirtualShortMethod)(JNIEnv *env, jobject obj,
						   jmethodID methodID, ...);
  extern jshort JNIFUNC(CallNonvirtualShortMethodA)(JNIEnv *env, jobject obj,
						    jmethodID methodID, jvalue *args);
  extern jshort JNIFUNC(CallNonvirtualShortMethodV)(JNIEnv *env, jobject obj,
						    jmethodID methodID, va_list args);
  extern jint JNIFUNC(CallNonvirtualIntMethod)(JNIEnv *env, jobject obj,
					       jmethodID methodID, ...);
  extern jint JNIFUNC(CallNonvirtualIntMethodA)(JNIEnv *env, jobject obj,
						jmethodID methodID, jvalue *args);
  extern jint JNIFUNC(CallNonvirtualIntMethodV)(JNIEnv *env, jobject obj,
						jmethodID methodID, va_list args);
  extern jlong JNIFUNC(CallNonvirtualLongMethod)(JNIEnv *env, jobject obj,
						 jmethodID methodID, ...);
  extern jlong JNIFUNC(CallNonvirtualLongMethodA)(JNIEnv *env, jobject obj,
						  jmethodID methodID, jvalue *args);
  extern jlong JNIFUNC(CallNonvirtualLongMethodV)(JNIEnv *env, jobject obj,
						  jmethodID methodID, va_list args);
  extern jfloat JNIFUNC(CallNonvirtualFloatMethod)(JNIEnv *env, jobject obj,
						   jmethodID methodID, ...);
  extern jfloat JNIFUNC(CallNonvirtualFloatMethodA)(JNIEnv *env, jobject obj,
						    jmethodID methodID, jvalue *args);
  extern jfloat JNIFUNC(CallNonvirtualFloatMethodV)(JNIEnv *env, jobject obj,
						    jmethodID methodID, va_list args);
  extern jdouble JNIFUNC(CallNonvirtualDoubleMethod)(JNIEnv *env, jobject obj,
						     jmethodID methodID, ...);
  extern jdouble JNIFUNC(CallNonvirtualDoubleMethodA)(JNIEnv *env, jobject obj,
						      jmethodID methodID, jvalue *args);
  extern jdouble JNIFUNC(CallNonvirtualDoubleMethodV)(JNIEnv *env, jobject obj,
						      jmethodID methodID, va_list args);
  extern void JNIFUNC(CallNonvirtualVoidMethod)(JNIEnv *env, jobject obj,
						jmethodID methodID, ...);
  extern void JNIFUNC(CallNonvirtualVoidMethodA)(JNIEnv *env, jobject obj,
						 jmethodID methodID, jvalue *args);
  extern void JNIFUNC(CallNonvirtualVoidMethodV)(JNIEnv *env, jobject obj,
						 jmethodID methodID, va_list args);
  extern jobject JNIFUNC(CallNonvirtualObjectMethod)(JNIEnv *env, jobject obj,
						     jmethodID methodID, ...);
  extern jobject JNIFUNC(CallNonvirtualObjectMethodA)(JNIEnv *env, jobject obj,
						      jmethodID methodID, jvalue *args);
  extern jobject JNIFUNC(CallNonvirtualObjectMethodV)(JNIEnv *env, jobject obj,
						      jmethodID methodID, va_list args);

  extern jmethodID JNIFUNC(GetStaticMethodID)(JNIEnv *env, jclass clazz,
					      const char *name, const char *sig);
  extern jboolean JNIFUNC(CallStaticBooleanMethod)(JNIEnv *env, jclass clazz,
						   jmethodID methodID, ...);
  extern jboolean JNIFUNC(CallStaticBooleanMethodA)(JNIEnv *env, jclass clazz,
						    jmethodID methodID, jvalue *args);
  extern jboolean JNIFUNC(CallStaticBooleanMethodV)(JNIEnv *env, jclass clazz,
						    jmethodID methodID, va_list args);
  extern jbyte JNIFUNC(CallStaticByteMethod)(JNIEnv *env, jclass clazz,
					     jmethodID methodID, ...);
  extern jbyte JNIFUNC(CallStaticByteMethodA)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, jvalue *args);
  extern jbyte JNIFUNC(CallStaticByteMethodV)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, va_list args);
  extern jchar JNIFUNC(CallStaticCharMethod)(JNIEnv *env, jclass clazz,
					     jmethodID methodID, ...);
  extern jchar JNIFUNC(CallStaticCharMethodA)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, jvalue *args);
  extern jchar JNIFUNC(CallStaticCharMethodV)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, va_list args);
  extern jshort JNIFUNC(CallStaticShortMethod)(JNIEnv *env, jclass clazz,
					       jmethodID methodID, ...);
  extern jshort JNIFUNC(CallStaticShortMethodA)(JNIEnv *env, jclass clazz,
						jmethodID methodID, jvalue *args);
  extern jshort JNIFUNC(CallStaticShortMethodV)(JNIEnv *env, jclass clazz,
						jmethodID methodID, va_list args);
  extern jint JNIFUNC(CallStaticIntMethod)(JNIEnv *env, jclass clazz,
					   jmethodID methodID, ...);
  extern jint JNIFUNC(CallStaticIntMethodA)(JNIEnv *env, jclass clazz,
					    jmethodID methodID, jvalue *args);
  extern jint JNIFUNC(CallStaticIntMethodV)(JNIEnv *env, jclass clazz,
					    jmethodID methodID, va_list args);
  extern jlong JNIFUNC(CallStaticLongMethod)(JNIEnv *env, jclass clazz,
					     jmethodID methodID, ...);
  extern jlong JNIFUNC(CallStaticLongMethodA)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, jvalue *args);
  extern jlong JNIFUNC(CallStaticLongMethodV)(JNIEnv *env, jclass clazz,
					      jmethodID methodID, va_list args);
  extern jfloat JNIFUNC(CallStaticFloatMethod)(JNIEnv *env, jclass clazz,
					       jmethodID methodID, ...);
  extern jfloat JNIFUNC(CallStaticFloatMethodA)(JNIEnv *env, jclass clazz,
						jmethodID methodID, jvalue *args);
  extern jfloat JNIFUNC(CallStaticFloatMethodV)(JNIEnv *env, jclass clazz,
						jmethodID methodID, va_list args);
  extern jdouble JNIFUNC(CallStaticDoubleMethod)(JNIEnv *env, jclass clazz,
						 jmethodID methodID, ...);
  extern jdouble JNIFUNC(CallStaticDoubleMethodA)(JNIEnv *env, jclass clazz,
						  jmethodID methodID, jvalue *args);
  extern jdouble JNIFUNC(CallStaticDoubleMethodV)(JNIEnv *env, jclass clazz,
						  jmethodID methodID, va_list args);
  extern void JNIFUNC(CallStaticVoidMethod)(JNIEnv *env, jclass clazz,
					    jmethodID methodID, ...);
  extern void JNIFUNC(CallStaticVoidMethodA)(JNIEnv *env, jclass clazz,
					     jmethodID methodID, jvalue *args);
  extern void JNIFUNC(CallStaticVoidMethodV)(JNIEnv *env, jclass clazz,
					     jmethodID methodID, va_list args);
  extern jobject JNIFUNC(CallStaticObjectMethod)(JNIEnv *env, jclass clazz,
						 jmethodID methodID, ...);
  extern jobject JNIFUNC(CallStaticObjectMethodA)(JNIEnv *env, jclass clazz,
						  jmethodID methodID, jvalue *args);
  extern jobject JNIFUNC(CallStaticObjectMethodV)(JNIEnv *env, jclass clazz,
						  jmethodID methodID, va_list args);


  extern jfieldID JNIFUNC(GetStaticFieldID)(JNIEnv *env, jclass clazz,
					    const char *name, const char *sig);
  extern jboolean JNIFUNC(GetStaticBooleanField)(JNIEnv *env, jclass clazz,
						 jfieldID fieldID);
  extern jbyte JNIFUNC(GetStaticByteField)(JNIEnv *env, jclass clazz,
					   jfieldID fieldID);
  extern jchar JNIFUNC(GetStaticCharField)(JNIEnv *env, jclass clazz,
					   jfieldID fieldID);
  extern jshort JNIFUNC(GetStaticShortField)(JNIEnv *env, jclass clazz,
					     jfieldID fieldID);
  extern jint JNIFUNC(GetStaticIntField)(JNIEnv *env, jclass clazz,
					 jfieldID fieldID);
  extern jlong JNIFUNC(GetStaticLongField)(JNIEnv *env, jclass clazz,
					   jfieldID fieldID);
  extern jfloat JNIFUNC(GetStaticFloatField)(JNIEnv *env, jclass clazz,
					     jfieldID fieldID);
  extern jdouble JNIFUNC(GetStaticDoubleField)(JNIEnv *env, jclass clazz,
					       jfieldID fieldID);
  extern jobject JNIFUNC(GetStaticObjectField)(JNIEnv *env, jclass clazz,
					       jfieldID fieldID);

  extern void JNIFUNC(SetStaticBooleanField)(JNIEnv *env, jclass clazz,
					     jfieldID fieldID, jboolean value);
  extern void JNIFUNC(SetStaticByteField)(JNIEnv *env, jclass clazz,
					  jfieldID fieldID, jbyte value);
  extern void JNIFUNC(SetStaticCharField)(JNIEnv *env, jclass clazz,
					  jfieldID fieldID, jchar value);
  extern void JNIFUNC(SetStaticShortField)(JNIEnv *env, jclass clazz,
					   jfieldID fieldID, jshort value);
  extern void JNIFUNC(SetStaticIntField)(JNIEnv *env, jclass clazz,
					 jfieldID fieldID, jint value);
  extern void JNIFUNC(SetStaticLongField)(JNIEnv *env, jclass clazz,
					  jfieldID fieldID, jlong value);
  extern void JNIFUNC(SetStaticFloatField)(JNIEnv *env, jclass clazz,
					   jfieldID fieldID, jfloat value);
  extern void JNIFUNC(SetStaticDoubleField)(JNIEnv *env, jclass clazz,
					    jfieldID fieldID, jdouble value);
  extern void JNIFUNC(SetStaticObjectField)(JNIEnv *env, jclass clazz,
					    jfieldID fieldID, jobject value);

  extern jstring JNIFUNC(NewString)(JNIEnv *env, const jchar *unicodeChars, jsize len);
  extern jsize JNIFUNC(GetStringLength)(JNIEnv *env, jstring string);
  extern jchar* JNIFUNC(GetStringChars)(JNIEnv *env, jstring string, jboolean *isCopy);
  extern void JNIFUNC(ReleaseStringChars)(JNIEnv *env, jstring string, const jchar *chars);
  extern jstring JNIFUNC(NewStringUTF)(JNIEnv *env, const char *bytes);
  extern jsize JNIFUNC(GetStringUTFLength)(JNIEnv *env, jstring string);
  extern const jbyte* JNIFUNC(GetStringUTFChars)(JNIEnv *env, jstring string, jboolean *isCopy);
  extern void JNIFUNC(ReleaseStringUTFChars)(JNIEnv *env, jstring string, const jbyte *utf);
  extern void JNIFUNC(GetStringRegion)(JNIEnv* env, jstring string, jsize start,
				       jsize len, jchar * buf );
  extern void JNIFUNC(GetStringUTFRegion)(JNIEnv *env, jstring string,
					  jsize start, jsize len, char * buf );
  extern const jchar *JNIFUNC(GetStringCritical)(JNIEnv *env, jstring string, 
						 jboolean *isCopy);
  extern void JNIFUNC(ReleaseStringCritical)(JNIEnv *env, jstring string,
					     const jchar *carray);

  extern jsize JNIFUNC(GetArrayLength)(JNIEnv *env, jarray array);
  extern jarray JNIFUNC(NewObjectArray)(JNIEnv *env, jsize length,
					jclass elementClass, jobject initialElement);
  extern jobject JNIFUNC(GetObjectArrayElement)(JNIEnv *env, jobjectArray array, jsize index);
  extern void JNIFUNC(SetObjectArrayElement)(JNIEnv *env, jobjectArray array,
					     jsize index, jobject value);

  extern jbooleanArray JNIFUNC(NewBooleanArray)(JNIEnv *env, jsize length);
  extern jboolean* JNIFUNC(GetBooleanArrayElements)(JNIEnv *env, jbooleanArray array,
						    jboolean* isCopy);
  extern void JNIFUNC(ReleaseBooleanArrayElements)(JNIEnv *env, jbooleanArray array,
						   jboolean* elems, jint mode);
  extern void JNIFUNC(GetBooleanArrayRegion)(JNIEnv *env, jbooleanArray array,
					     jsize start, jsize len, jboolean* buf);
  extern void JNIFUNC(SetBooleanArrayRegion)(JNIEnv *env, jbooleanArray array,
					     jsize start, jsize len, jboolean* buf);
  extern jbyteArray JNIFUNC(NewByteArray)(JNIEnv *env, jsize length);
  extern jbyte* JNIFUNC(GetByteArrayElements)(JNIEnv *env, jbyteArray array,
					      jboolean* isCopy);
  extern void JNIFUNC(ReleaseByteArrayElements)(JNIEnv *env, jbyteArray array,
						jbyte* elems, jint mode);
  extern void JNIFUNC(GetByteArrayRegion)(JNIEnv *env, jbyteArray array,
					  jsize start, jsize len, jbyte* buf);
  extern void JNIFUNC(SetByteArrayRegion)(JNIEnv *env, jbyteArray array,
					  jsize start, jsize len, jbyte* buf);
  extern jcharArray JNIFUNC(NewCharArray)(JNIEnv *env, jsize length);
  extern jchar* JNIFUNC(GetCharArrayElements)(JNIEnv *env, jcharArray array,
					      jboolean* isCopy);
  extern void JNIFUNC(ReleaseCharArrayElements)(JNIEnv *env, jcharArray array,
						jchar* elems, jint mode);
  extern void JNIFUNC(GetCharArrayRegion)(JNIEnv *env, jcharArray array,
					  jsize start, jsize len, jchar* buf);
  extern void JNIFUNC(SetCharArrayRegion)(JNIEnv *env, jcharArray array,
					  jsize start, jsize len, jchar* buf);
  extern jshortArray JNIFUNC(NewShortArray)(JNIEnv *env, jsize length);
  extern jshort* JNIFUNC(GetShortArrayElements)(JNIEnv *env, jshortArray array,
						jboolean* isCopy);
  extern void JNIFUNC(ReleaseShortArrayElements)(JNIEnv *env, jshortArray array,
						 jshort* elems, jint mode);
  extern void JNIFUNC(GetShortArrayRegion)(JNIEnv *env, jshortArray array,
					   jsize start, jsize len, jshort* buf);
  extern void JNIFUNC(SetShortArrayRegion)(JNIEnv *env, jshortArray array,
					   jsize start, jsize len, jshort* buf);
  extern jintArray JNIFUNC(NewIntArray)(JNIEnv *env, jsize length);
  extern jint* JNIFUNC(GetIntArrayElements)(JNIEnv *env, jintArray array,
					    jboolean* isCopy);
  extern void JNIFUNC(ReleaseIntArrayElements)(JNIEnv *env, jintArray array,
					       jint* elems, jint mode);
  extern void JNIFUNC(GetIntArrayRegion)(JNIEnv *env, jintArray array,
					 jsize start, jsize len, jint* buf);
  extern void JNIFUNC(SetIntArrayRegion)(JNIEnv *env, jintArray array,
					 jsize start, jsize len, jint* buf);
  extern jlongArray JNIFUNC(NewLongArray)(JNIEnv *env, jsize length);
  extern jlong* JNIFUNC(GetLongArrayElements)(JNIEnv *env, jlongArray array,
					      jboolean* isCopy);
  extern void JNIFUNC(ReleaseLongArrayElements)(JNIEnv *env, jlongArray array,
						jlong* elems, jint mode);
  extern void JNIFUNC(GetLongArrayRegion)(JNIEnv *env, jlongArray array,
					  jsize start, jsize len, jlong* buf);
  extern void JNIFUNC(SetLongArrayRegion)(JNIEnv *env, jlongArray array,
					  jsize start, jsize len, jlong* buf);
  extern jfloatArray JNIFUNC(NewFloatArray)(JNIEnv *env, jsize length);
  extern jfloat* JNIFUNC(GetFloatArrayElements)(JNIEnv *env, jfloatArray array,
						jboolean* isCopy);
  extern void JNIFUNC(ReleaseFloatArrayElements)(JNIEnv *env, jfloatArray array,
						 jfloat* elems, jint mode);
  extern void JNIFUNC(GetFloatArrayRegion)(JNIEnv *env, jfloatArray array,
					   jsize start, jsize len, jfloat* buf);
  extern void JNIFUNC(SetFloatArrayRegion)(JNIEnv *env, jfloatArray array,
					   jsize start, jsize len, jfloat* buf);
  extern jdoubleArray JNIFUNC(NewDoubleArray)(JNIEnv *env, jsize length);
  extern jdouble* JNIFUNC(GetDoubleArrayElements)(JNIEnv *env, jdoubleArray array,
						  jboolean* isCopy);
  extern void JNIFUNC(ReleaseDoubleArrayElements)(JNIEnv *env, jdoubleArray array,
						  jdouble* elems, jint mode);
  extern void JNIFUNC(GetDoubleArrayRegion)(JNIEnv *env, jdoubleArray array,
					    jsize start, jsize len, jdouble* buf);
  extern void JNIFUNC(SetDoubleArrayRegion)(JNIEnv *env, jdoubleArray array,
					    jsize start, jsize len, jdouble* buf);

  extern jint JNIFUNC(RegisterNatives)(JNIEnv *env, jclass clazz,
				       const JNINativeMethod *methods, jint nMethods);
  extern jint JNIFUNC(UnregisterNatives)(JNIEnv *env, jclass clazz);

  extern jint JNIFUNC(MonitorEnter)(JNIEnv *env, jobject obj);
  extern jint JNIFUNC(MonitorExit)(JNIEnv *env, jobject obj);

  extern jint JNIFUNC(GetJavaVM)(JNIEnv *env, JavaVM **vm);

  /* new for JDK 1.2 */

  /* local reference management */
  jint JNIFUNC(EnsureLocalCapacity)(JNIEnv *env, jint capacity);
  jint JNIFUNC(PushLocalFrame)(JNIEnv *env, jint capacity);
  jobject JNIFUNC(PopLocalFrame)(JNIEnv *env, jobject result);
  jobject JNIFUNC(NewLocalRef)(JNIEnv *env, jobject ref);

  /* weak global references */
  jweak JNIFUNC(NewWeakGlobalRef)(JNIEnv *env, jobject obj);
  void JNIFUNC(DeleteWeakGlobalRef)(JNIEnv *env, jweak obj);

  /* array operations */
  void* JNIFUNC(GetPrimitiveArrayCritical)(JNIEnv *env, jarray array,
					   jboolean *isCopy);
  void JNIFUNC(ReleasePrimitiveArrayCritical)(JNIEnv *env,
					      jarray array,
					      void *carray, jint mode);

  /* string operations */
  void JNIFUNC(GetStringRegion)(JNIEnv *env, jstring str, jsize start,
				jsize len, jchar *buf);
  void JNIFUNC(GetStringUTFRegion)(JNIEnv *env, jstring str,
				   jsize start, jsize len, char *buf);

  const jchar* JNIFUNC(GetStringCritical)(JNIEnv *env, jstring string,
					  jboolean *isCopy);
  void JNIFUNC(ReleaseStringCritical)(JNIEnv *env, jstring string,
				      const jchar *carray);

  /* reflection support */
  jmethodID JNIFUNC(FromReflectedMethod)(JNIEnv *env,
					 jobject method);
  jfieldID JNIFUNC(FromReflectedField)(JNIEnv *env,
				       jobject field);
  jobject JNIFUNC(ToReflectedMethod)(JNIEnv *env, jclass cls,
				     jmethodID methodID);
  jobject JNIFUNC(ToReflectedField)(JNIEnv *env, jclass cls,
				    jfieldID fieldID);

#ifdef __cplusplus
};
#endif

#endif /* _hungry_jniint_h */
