// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.icons.AllIcons;
import com.intellij.ide.presentation.Presentation;
import com.intellij.lang.Language;
import com.intellij.lang.properties.PropertiesLanguage;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiManager;
import com.intellij.psi.PsiType;
import com.intellij.psi.impl.FakePsiElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

/**
 * Navigation/Documentation wrapper element for config keys.
 */
@Presentation(typeName = "Spring Boot Configuration Key")
final class ConfigKeyDeclarationPsiElement extends FakePsiElement {

  private final PsiElement myNavigationParent;

  private final String myLibraryName;
  private final PsiElement myNavigationTarget;

  private final String myConfigKeyName;
  private final String mySourceTypeText;

  @Nullable
  private final PsiType myType;

  ConfigKeyDeclarationPsiElement(String libraryName,
                                 @NotNull PsiElement navigationParent,
                                 @NotNull PsiElement navigationTarget,
                                 String configKeyName,
                                 String sourceTypeText,
                                 @Nullable PsiType type) {
    myNavigationParent = navigationParent;
    myLibraryName = libraryName;

    myNavigationTarget = navigationTarget;
    myConfigKeyName = configKeyName;
    mySourceTypeText = sourceTypeText;
    myType = type;
  }

  @Override
  public PsiElement getParent() {
    return myNavigationParent;
  }

  @Override
  public Icon getIcon(boolean open) {
    // used with getPresentableText() when key resolves to multiple source-"classes"
    return AllIcons.Nodes.Class;
  }

  @NotNull
  @Override
  public PsiElement getNavigationElement() {
    return myNavigationTarget;
  }

  @Override
  public String getPresentableText() {
    // display "source" class FQN for multiple resolve targets
    return mySourceTypeText;
  }

  @Override
  public String getName() {
    return myConfigKeyName;
  }

  @Nullable
  @Override
  public String getLocationString() {
    return myLibraryName;
  }

  /**
   * Properties language, so that {@link SpringBootApplicationPropertiesDocumentationProvider} (always available) will
   * work when
   * <ul>
   * <li>resolved as PsiReference (e.g. in additional.json)</li>
   * <li>linking to another CKDPE via embedded link provided from {@link ConfigKeyDocumentationProviderBase#getDocumentationElementForLink(PsiManager, String, PsiElement)}</li>
   * </ul>
   *
   * @return PropertiesLanguage.
   */
  @NotNull
  @Override
  public Language getLanguage() {
    return PropertiesLanguage.INSTANCE;
  }

  @Nullable
  public PsiType getType() {
    return myType;
  }
}
