// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata;

import com.intellij.codeInsight.documentation.DocumentationManager;
import com.intellij.codeInsight.documentation.DocumentationManagerUtil;
import com.intellij.codeInsight.javadoc.JavaDocInfoGenerator;
import com.intellij.codeInsight.javadoc.JavaDocUtil;
import com.intellij.lang.documentation.AbstractDocumentationProvider;
import com.intellij.lang.documentation.DocumentationMarkup;
import com.intellij.lang.documentation.DocumentationProvider;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.beanProperties.BeanPropertyElement;
import com.intellij.psi.javadoc.PsiDocComment;
import com.intellij.psi.util.PropertyUtilBase;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.ui.ColorUtil;
import com.intellij.ui.JBColor;
import com.intellij.util.Function;
import com.intellij.util.ObjectUtils;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.xml.util.XmlUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Locale;

public abstract class ConfigKeyDocumentationProviderBase extends AbstractDocumentationProvider {

  private static final String CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX = "SPRING_BOOT_CONFIG_KEY_DECLARATION_PSI_ELEMENT_";
  private static final String CONFIG_KEY_LINK_SEPARATOR = "___";

  /**
   * Stores current Module when constructing {@link ConfigKeyDeclarationPsiElement}.
   */
  public static final Key<Module> CONFIG_KEY_DECLARATION_MODULE = Key.create("ConfigKeyDeclarationModule");

  /**
   * Set the flag to {@code true} for Psi file in order to mark its content as acceptable by application properties documentation provider.
   */
  public static final Key<Boolean> ELEMENT_IN_EXTERNAL_CONTEXT = Key.create("ElementInExternalContext");

  @NonNls
  private static final String UNKNOWN_TYPE = "(unknown type)";

  @Nullable
  @Override
  public String getQuickNavigateInfo(PsiElement element, PsiElement originalElement) {
    // see ConfigKeyDeclarationPsiElement.getLanguage()
    if (element instanceof ConfigKeyDeclarationPsiElement) {
      final ConfigKeyDeclarationPsiElement configKey = (ConfigKeyDeclarationPsiElement)element;

      StringBuilder sb = new StringBuilder();
      sb.append("<b>").append(configKey.getName()).append("</b>");
      sb.append(" [").append(XmlUtil.escape(configKey.getLocationString())).append("]\n");

      appendConfigKeyType(sb, element, configKey.getType());
      return sb.toString();
    }

    final String valueHintDocumentation = getValueHintDocumentation(element);
    if (valueHintDocumentation != null) {
      return valueHintDocumentation;
    }

    return super.getQuickNavigateInfo(element, originalElement);
  }

  @Nullable
  private static String getValueHintDocumentation(PsiElement element) {
    if (element instanceof ValueHintPsiElement) {
      final SpringBootApplicationMetaConfigKey.ValueHint hint =
        ((ValueHintPsiElement)element).getValueHint();
      return "<b>" + hint.getValue() + "</b>: " + hint.getDescriptionText().getFullText();
    }
    return null;
  }

  @Override
  public String generateDoc(PsiElement element, @Nullable PsiElement originalElement) {
    if (element instanceof ConfigKeyDeclarationPsiElement) {
      final ConfigKeyDeclarationPsiElement configKeyDeclarationPsiElement = (ConfigKeyDeclarationPsiElement)element;
      final String keyName = configKeyDeclarationPsiElement.getName();
      return getDocumentationTextForKey(ObjectUtils.chooseNotNull(element, originalElement), keyName);
    }

    final String valueHintDocumentation = getValueHintDocumentation(element);
    if (valueHintDocumentation != null) {
      return valueHintDocumentation;
    }

    return generateDocForProperty(element);
  }

  @Override
  public PsiElement getDocumentationElementForLink(PsiManager psiManager, String link, PsiElement context) {
    if (!StringUtil.startsWith(link, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX)) {
      return JavaDocUtil.findReferenceTarget(psiManager, link, context);
    }

    // return "fake" ConfigKeyDeclarationPsiElement
    String fqn = StringUtil.substringAfter(link, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX);
    assert fqn != null : link;
    String moduleName = StringUtil.substringBefore(fqn, CONFIG_KEY_LINK_SEPARATOR);
    assert moduleName != null : link;
    String key = StringUtil.substringAfter(fqn, CONFIG_KEY_LINK_SEPARATOR);

    final Module module = ModuleManager.getInstance(psiManager.getProject()).findModuleByName(moduleName);
    final SpringBootApplicationMetaConfigKey replacementKey =
      SpringBootApplicationMetaConfigKeyManager.getInstance().findCanonicalApplicationMetaConfigKey(module, key);
    if (replacementKey != null) {
      return replacementKey.getDeclaration();
    }
    return super.getDocumentationElementForLink(psiManager, link, context);
  }

  @Nullable
  protected abstract String getConfigKey(PsiElement configKeyElement);

  /**
   * Try to provide doc for "normal" property due to multi-resolve.
   */
  private String generateDocForProperty(PsiElement element) {
    String configKey = getConfigKey(element);
    if (configKey == null) {
      return generateDocForBeanProperty(element);
    }

    return getDocumentationTextForKey(element, configKey);
  }

  @Nullable
  private static String generateDocForBeanProperty(PsiElement element) {
    if (!(element instanceof BeanPropertyElement)) return null;

    BeanPropertyElement property = (BeanPropertyElement)element;
    final PsiElement originalElement = getOriginalDocumentationElement(property.getMethod());
    return getOriginalDocForPsiMember(originalElement);
  }

  private static void appendConfigKeyType(StringBuilder sb, PsiElement context, @Nullable PsiType psiType) {
    if (psiType == null) {
      sb.append(UNKNOWN_TYPE);
    }
    else {
      JavaDocInfoGenerator.generateType(sb, psiType, context);
    }
  }

  @Nullable
  private static String getDocumentationTextForKey(@Nullable PsiElement originalElement,
                                                   @Nullable String keyName) {
    if (originalElement == null) {
      return null;
    }

    Module module = ObjectUtils.chooseNotNull(ModuleUtilCore.findModuleForPsiElement(originalElement),
                                              originalElement.getUserData(CONFIG_KEY_DECLARATION_MODULE));
    final SpringBootApplicationMetaConfigKey key =
      SpringBootApplicationMetaConfigKeyManager.getInstance().findCanonicalApplicationMetaConfigKey(module, keyName);
    if (key == null) {
      return null;
    }

    StringBuilder sb = new StringBuilder(DocumentationMarkup.DEFINITION_START);
    sb.append("<b>").append(keyName).append("</b><br>");
    appendConfigKeyType(sb, originalElement, key.getType());
    sb.append(DocumentationMarkup.DEFINITION_END);


    sb.append(DocumentationMarkup.CONTENT_START);
    final String fullDescription = key.getDescriptionText().getFullText();
    final boolean hasDescription = StringUtil.isNotEmpty(fullDescription);
    if (hasDescription) {
      sb.append(fullDescription);
      sb.append("<br><br>");
    }

    final SpringBootApplicationMetaConfigKey.Deprecation deprecation = key.getDeprecation();
    if (deprecation != SpringBootApplicationMetaConfigKey.Deprecation.NOT_DEPRECATED) {
      if (deprecation.getLevel() == SpringBootApplicationMetaConfigKey.Deprecation.DeprecationLevel.ERROR) {
        sb.append("<font color='#").append(ColorUtil.toHex(JBColor.RED)).append("'><b>Deprecated</b></font>");
      }
      else {
        sb.append("<b>Deprecated</b>");
      }

      final String reasonText = deprecation.getReason().getFullText();
      if (StringUtil.isNotEmpty(reasonText)) {
        sb.append("  ").append(reasonText);
      }
      final String replacement = deprecation.getReplacement();
      if (replacement != null) {
        sb.append("<br><em>See:</em> ");
        DocumentationManagerUtil.createHyperlink(sb, CONFIG_KEY_DECLARATION_PSI_ELEMENT_PREFIX + module.getName() +
                                                     CONFIG_KEY_LINK_SEPARATOR + replacement,
                                                 replacement, true);
      }
      sb.append("<br>");
    }
    sb.append(DocumentationMarkup.CONTENT_END);

    sb.append(DocumentationMarkup.SECTIONS_START);

    final String defaultValue = key.getDefaultValue();
    if (defaultValue != null) {
      appendSection(sb, "Default", "<pre>" + defaultValue + "</pre>");
    }

    final SpringBootApplicationMetaConfigKey.ItemHint itemHint = key.getItemHint();
    final List<SpringBootApplicationMetaConfigKey.ValueHint> valueHints = itemHint.getValueHints();
    appendValueDescriptionTable(sb, valueHints,
                                SpringBootApplicationMetaConfigKey.ValueHint::getValue,
                                hint -> hint.getDescriptionText().getFullText());

    final PsiClass typeClass = PsiTypesUtil.getPsiClass(key.getType());
    if (typeClass != null &&
        typeClass.isEnum()) {
      final List<PsiField> enumConstants = ContainerUtil.findAll(typeClass.getFields(), Conditions.instanceOf(PsiEnumConstant.class));
      appendValueDescriptionTable(sb, enumConstants,
                                  field -> StringUtil.defaultIfEmpty(field.getName(), "<invalid>").toLowerCase(Locale.US),
                                  field -> {
                                    final PsiElement navigationElement = field.getNavigationElement();
                                    if (!(navigationElement instanceof PsiDocCommentOwner)) return "";

                                    final PsiDocComment comment = ((PsiDocCommentOwner)navigationElement).getDocComment();
                                    if (comment == null) {
                                      return "";
                                    }

                                    StringBuilder doc = new StringBuilder();
                                    for (PsiElement element : comment.getDescriptionElements()) {
                                      doc.append(StringUtil.replaceUnicodeEscapeSequences(element.getText()));
                                    }
                                    return doc.toString();
                                  });
    }

    sb.append(DocumentationMarkup.SECTIONS_END);

    // show original doc (method/field) last to prevent HTML problems
    if (StringUtil.isEmpty(fullDescription)) {
      PsiElement docElement = getOriginalDocumentationElement(key.getDeclaration().getNavigationElement());
      final String originalDoc = getOriginalDocForPsiMember(docElement);
      if (StringUtil.isNotEmpty(originalDoc)) {
        sb.append(DocumentationMarkup.CONTENT_START);
        sb.append("<em>Original documentation:</em><br><br>");
        sb.append(originalDoc);
        sb.append(DocumentationMarkup.CONTENT_END);
      }
    }
    return sb.toString();
  }

  private static void appendSection(StringBuilder sb, String sectionName, String sectionContent) {
    sb.append(DocumentationMarkup.SECTION_HEADER_START).append(sectionName).append(":")
      .append(DocumentationMarkup.SECTION_SEPARATOR);
    sb.append(sectionContent);
    sb.append(DocumentationMarkup.SECTION_END);
  }

  private static <V> void appendValueDescriptionTable(StringBuilder sb,
                                                      List<V> elements,
                                                      Function<V, String> valueFunction,
                                                      Function<V, String> descriptionFunction) {
    if (elements.isEmpty()) return;

    StringBuilder tableSb = new StringBuilder();
    tableSb.append("<table cellpadding=\"5\">");
    for (V value : elements) {
      tableSb.append("<tr>");
      tableSb.append("<td valign='top'><pre>").append(valueFunction.fun(value)).append("</pre></td>");
      tableSb.append("<td valign='top'>").append(descriptionFunction.fun(value)).append("</td>");
      tableSb.append("</tr>");
    }
    tableSb.append("</table>");

    appendSection(sb, "Values", tableSb.toString());
  }

  @Nullable
  private static String getOriginalDocForPsiMember(PsiElement docElement) {
    if (!(docElement instanceof PsiMember)) {
      return null;
    }

    final DocumentationProvider provider = DocumentationManager.getProviderFromElement(docElement);
    return provider.generateDoc(docElement, docElement);
  }

  /**
   * Fallback to underlying field (if navigationElement is setter/getter and field has javadoc), otherwise return given navigationElement.
   *
   * @param navigationElement Original navigation element.
   * @return Element to get documentation from.
   */
  @NotNull
  private static PsiElement getOriginalDocumentationElement(PsiElement navigationElement) {
    if (navigationElement instanceof PsiMethod) {
      final PsiField field = PropertyUtilBase.findPropertyFieldByMember((PsiMember)navigationElement);
      if (field != null && field.getDocComment() != null) return field;
    }
    return navigationElement;
  }
}
