// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.VolatileNotNullLazyValue;
import com.intellij.psi.PsiElement;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.graph.LazyModelDependenciesGraph;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.custom.CustomLocalComponentsDiscoverer;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

public abstract class AbstractSimpleLocalModel<T extends PsiElement> extends AbstractSimpleSpringModel implements LocalModel<T> {


  private final NotNullLazyValue<CommonSpringModel> myCustomDiscoveredBeansModel =
    new VolatileNotNullLazyValue<CommonSpringModel>() {
      @Override
      @NotNull
      protected CommonSpringModel compute() {
        return new BeansSpringModel(getModule(), new NotNullLazyValue<Collection<? extends SpringBeanPointer>>() {
          @NotNull
          @Override
          protected Collection<? extends SpringBeanPointer> compute() {
            return computeCustomBeans();
          }
        });
      }
    };

  protected static void addNotNullModel(@NotNull Set<? super Pair<LocalModel, LocalModelDependency>> models,
                                        @Nullable LocalModel model,
                                        @NotNull LocalModelDependency dependency) {
    if (model != null) {
      models.add(Pair.create(model, dependency));
    }
  }

  private Collection<? extends SpringBeanPointer> computeCustomBeans() {
    final Set<CommonSpringBean> customSpringComponents = ContainerUtil.newLinkedHashSet();
    for (CustomLocalComponentsDiscoverer discoverer : CustomLocalComponentsDiscoverer.EP_NAME.getExtensionList()) {
      customSpringComponents.addAll(discoverer.getCustomComponents(this));
    }
    return BeanService.getInstance().mapSpringBeans(customSpringComponents);
  }

  protected CommonSpringModel getCustomDiscoveredBeansModel() {
    return myCustomDiscoveredBeansModel.getValue();
  }

  @Override
  @NotNull
  public Set<LocalModel> getRelatedLocalModels() {
    return getDependentLocalModels().stream().map(pair -> pair.first).filter(model -> !this.equals(model)).collect(Collectors.toSet());
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "[" + getConfig() + "]";
  }

  @NotNull
  protected Object[] getOutsideModelDependencies(@NotNull LocalModel model) {
    final Project project = model.getConfig().getProject();
    return ArrayUtil.append(SpringModificationTrackersManager.getInstance(project).getOuterModelsDependencies(), model.getConfig());
  }

  @NotNull
  @Deprecated // to be removed in 2018.3, use LazyModelDependenciesGraph.getOrCreateLocalModelDependenciesGraph()
  public static LazyModelDependenciesGraph getOrCreateLocalModelDependenciesGraph(@NotNull final Module module,
                                                                                  @NotNull final Set<String> activeProfiles) {
    return LazyModelDependenciesGraph.getOrCreateLocalModelDependenciesGraph(module, activeProfiles);
  }
}
