// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.facet.SpringFileSet;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.custom.CustomComponentsDiscovererHelper;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

public abstract class SpringModel extends AbstractProcessableModel {

  @Nullable
  private final Module myModule;

  private Set<SpringModel> myDependencies = Collections.emptySet();

  @Nullable
  private final SpringFileSet myFileSet;

  protected SpringModel(@Nullable Module module) {
    this(module, null);
  }

  protected SpringModel(@Nullable final Module module,
                        @Nullable final SpringFileSet fileSet) {
    myFileSet = fileSet;
    myModule = module;
  }

  @Nullable
  public SpringFileSet getFileSet() {
    return myFileSet;
  }

  @NotNull
  public Set<SpringModel> getDependencies() {
    return myDependencies;
  }

  public void setDependencies(@NotNull final Set<SpringModel> dependencies) {
    myDependencies = dependencies;
  }

  @Override
  @NotNull
  public final Set<CommonSpringModel> getRelatedModels() {
    Set<CommonSpringModel> models = new LinkedHashSet<>();
    ContainerUtil.addAllNotNull(models, getRelatedModels(true));

    if (myModule != null && !myModule.isDisposed()) {
      ContainerUtil.addAllNotNull(models, CustomComponentsDiscovererHelper.getInstance(myModule).getCustomModels());
    }
    return models;
  }

  @NotNull
  public abstract Set<CommonSpringModel> getRelatedModels(boolean checkActiveProfiles);

  @NotNull
  @Deprecated // to remove in 2018.3, renamed to getRelatedModels();
  public final Set<CommonSpringModel> getModelsToProcess(){
    return getRelatedModels();
  }

  @NotNull
  @Deprecated // to remove in 2018.3, renamed to getRelatedModels(boolean checkActiveProfiles);
  public Set<CommonSpringModel> getModelsToProcess(boolean checkActiveProfiles) {
    return getRelatedModels(checkActiveProfiles);
  }

  @Override
  @Nullable
  public Module getModule() {
    return myModule;
  }

  public String toString() {
    final SpringFileSet fileSet = getFileSet();
    return getClass().getName() + (fileSet != null ? " fileset=" + fileSet.getId() : "");
  }

  @Nullable
  @Override
  public Set<String> getActiveProfiles() {
    if (myFileSet == null) {
      return null;
    }
    return myFileSet.getActiveProfiles();
  }

  public static final CommonSpringModel UNKNOWN = new CommonSpringModel() {

    @Override
    public boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                  @NotNull Processor<SpringBeanPointer> processor) {
      return true;
    }

    @Override
    public boolean processByName(@NotNull SpringModelSearchParameters.BeanName params,
                                 @NotNull Processor<SpringBeanPointer> processor) {
      return true;
    }

    @NotNull
    @Override
    public Collection<SpringBeanPointer> getAllCommonBeans() {
      return Collections.emptySet();
    }

    @Override
    public Module getModule() {
      return null;
    }


    @Override
    public Set<String> getActiveProfiles() {
      return Collections.emptySet();
    }
  };
}
