// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.converters;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.util.Condition;
import com.intellij.util.containers.MultiMap;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public class CustomConverterRegistry {

  public interface Provider extends Condition<GenericDomValue> {

    ExtensionPointName<Provider> EXTENSION_POINT_NAME = ExtensionPointName.create("com.intellij.spring.customConverterProvider");

    Converter getConverter();

    Class getConverterClass();
  }

  public static CustomConverterRegistry getRegistry() {
    return ourInstance;
  }

  private final static CustomConverterRegistry ourInstance = new CustomConverterRegistry();

  private final MultiMap<Class, Provider> myCustomConverters;

  private CustomConverterRegistry() {
    myCustomConverters = new MultiMap<>();
    for (Provider provider : Provider.EXTENSION_POINT_NAME.getExtensionList()) {
      myCustomConverters.putValue(provider.getConverterClass(), provider);
    }
  }

  @Nullable
  public Converter getCustomConverter(Class aClass, GenericDomValue context) {
    Collection<Provider> providers = myCustomConverters.get(aClass);
    for (Provider provider : providers) {
      if (provider.value(context)) {
        return provider.getConverter();
      }
    }

    return null;
  }

}
