// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.testContexts.profiles;

import com.intellij.ide.presentation.Presentation;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * @since 2017.2
 */
@Presentation(typeName = "@ActiveProfiles")
public class SpringJamActiveProfiles extends CommonModelElement.PsiBase implements SpringActiveProfile {

  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;
  private final PsiMember myPsiMember;

  public static final SemKey<JamAnnotationMeta> JAM_ANNO_META_KEY = ACTIVE_PROFILE_JAM_ANNOTATION_KEY.subKey("SpringJamActiveProfiles");
  private static final SemKey<SpringJamActiveProfiles> JAM_KEY = ACTIVE_PROFILE_JAM_KEY.subKey("SpringJamActiveProfiles");

  public static final JamClassMeta<SpringJamActiveProfiles> META = new JamClassMeta<>(null, SpringJamActiveProfiles.class, JAM_KEY);
  private static final JamStringAttributeMeta.Collection<String> VALUE_ATTR_META = JamAttributeMeta.collectionString(VALUE_ATTR_NAME);
  private static final JamStringAttributeMeta.Collection<String> PROFILES_ATTR_META = JamAttributeMeta.collectionString(PROFILES_ATTR_NAME);

  private static final JamAnnotationArchetype ARCHETYPE =
    new JamAnnotationArchetype().addAttribute(VALUE_ATTR_META).addAttribute(PROFILES_ATTR_META);
  public static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.ACTIVE_PROFILES, ARCHETYPE, JAM_ANNO_META_KEY);

  static {
    META.addAnnotation(ANNO_META);
  }

  public SpringJamActiveProfiles(@NotNull PsiMember psiMember) {
    myPsiMember = psiMember;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiMember);
  }

  @SuppressWarnings("unused")
  public SpringJamActiveProfiles(@NotNull PsiAnnotation annotation) {
    myPsiMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @Override
  @NotNull
  public Set<String> getActiveProfiles() {
    Set<String> profiles = ContainerUtil.newLinkedHashSet();

    addProfiles(profiles, VALUE_ATTR_META);
    addProfiles(profiles, PROFILES_ATTR_META);

    return profiles;
  }

  private void addProfiles(@NotNull Set<String> profiles,
                           @NotNull JamStringAttributeMeta.Collection<String> attrMeta) {
    for (JamStringAttributeElement<String> element : ANNO_META.getAttribute(getPsiElement(), attrMeta)) {
      if (element != null) {
        final String value = element.getValue();
        if (StringUtil.isNotEmpty(value)) profiles.add(value);
      }
    }
  }

  @Override
  @NotNull
  public PsiMember getPsiElement() {
    return myPsiMember;
  }

  @Override
  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }
}
