// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.xml;

import com.intellij.pom.references.PomService;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.DefaultSpringBeanQualifier;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.xml.DomReflectionUtil;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

public abstract class DomSpringBeanImpl extends AbstractDomSpringBean implements DomSpringBean {

  private static final Map<Class, BeanName> ourBeanNameCache =
    ConcurrentFactoryMap.createMap(key -> DomReflectionUtil.findAnnotationDFS(key, BeanName.class));

  private static final Map<Class, BeanType> ourBeanTypeCache =
    ConcurrentFactoryMap.createMap(key -> DomReflectionUtil.findAnnotationDFS(key, BeanType.class));

  /**
   * Use {@link BeanName} to provide custom bean name.
   * Overriding recommended when {@link #setName(String)} is also changed.
   * NB: do not use any resolving when overriding.
   *
   * @return Bean name.
   */
  @Override
  @SuppressWarnings("unchecked")
  @Nullable
  public String getBeanName() {
    final BeanName beanName = ourBeanNameCache.get(getClass());
    if (beanName == null) {
      return DomUtil.hasXml(getId()) ? getId().getRawText() : null;
    }

    final String value = beanName.value();
    if (!value.isEmpty()) {
      return value;
    }

    // we already check in indexing, but if model version is not incremented such errors might be introduced
    final Class<? extends BeanNameProvider> providerClass = beanName.provider();
    if (providerClass == BeanNameProvider.class) {
      throw new IllegalStateException("@BeanName: no value() given, provider() must be set" + getDomElementType());
    }

    final BeanNameProvider beanNameProvider = BeanProviderRegistry.getInstance().getBeanNameProvider(providerClass);
    return beanNameProvider.getBeanName(this);
  }

  @Override
  public void setName(@NotNull String newName) {
    if (DomUtil.hasXml(getId())) {
      getId().setStringValue(newName);
    }
  }

  /**
   * @return Bean's class name.
   * @see BeanType
   */
  @Override
  @SuppressWarnings("unchecked")
  @Nullable
  public final String getClassName() {
    final BeanType beanType = ourBeanTypeCache.get(getClass());
    if (beanType == null) {
      return null;
    }

    final String value = beanType.value();
    if (!value.isEmpty()) {
      return value;
    }

    // we already check in indexing, but if model version is not incremented such errors might be introduced
    final Class<? extends BeanTypeProvider> providerClass = beanType.provider();
    if (providerClass == BeanTypeProvider.class) {
      throw new IllegalStateException("@BeanType: no value() given, provider() must be set" + getDomElementType());
    }

    final BeanTypeProvider beanTypeProvider = BeanProviderRegistry.getInstance().getBeanTypeProvider(providerClass);
    return beanTypeProvider.getBeanType(this);
  }

  @Override
  @NotNull
  public PsiFile getContainingFile() {
    return DomUtil.getFile(this);
  }

  @Override
  @Nullable
  public PsiElement getIdentifyingPsiElement() {
    if (!isValid()) return null;
    return PomService.convertToPsi(getManager().getProject(),
                                   BeanService.getInstance().createSpringBeanPsiTarget(this));
  }

  @Override
  @NotNull
  public String[] getAliases() {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  @Override
  public PsiManager getPsiManager() {
    return PsiManager.getInstance(getManager().getProject());
  }

  @Override
  public Beans getBeansParent() {
    return DomUtil.getParentOfType(this, Beans.class, false);
  }

  @NotNull
  @Override
  public Collection<SpringQualifier> getSpringQualifiers() {
    return Collections.singleton(DefaultSpringBeanQualifier.create(this));
  }
}
