// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.references;

import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.PsiReferenceBase;
import com.intellij.psi.PsiReferenceProvider;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;

/**
 * @author Yann C&eacute;bron
 */
public class StaticStringValuesReferenceProvider extends PsiReferenceProvider {

  private final boolean allowOtherValues;
  private final String[] values;

  /**
   * Creates a reference provider with the given values for autocompletion.
   * Other values will *not* be highlighted as errors.
   *
   * @param values Autocompletion values.
   */
  public StaticStringValuesReferenceProvider(@NotNull @NonNls final String... values) {
    this(true, values);
  }

  /**
   * Creates a reference provider with the given values for autocompletion and optional error highlighting.
   *
   * @param allowOtherValues Set to false to enable error highlighting.
   * @param values           Autocompletion values.
   */
  public StaticStringValuesReferenceProvider(final boolean allowOtherValues, @NotNull @NonNls final String... values) {
    this.allowOtherValues = allowOtherValues;
    Arrays.sort(values); // make sure Arrays.binarySearch() works later on..
    this.values = values;
  }

  @Override
  @NotNull
  public PsiReference[] getReferencesByElement(@NotNull final PsiElement element,
                                               @NotNull final ProcessingContext context) {
    return new PsiReference[]{new PsiReferenceBase<PsiElement>(element) {
      @Override
      public PsiElement resolve() {
        final String myValue = getValue();
        if (allowOtherValues) {
          return myElement;
        }

        return Arrays.binarySearch(values, myValue) > -1 ? myElement : null;
      }

      @Override
      @NotNull
      public Object[] getVariants() {
        return values;
      }
    }};
  }
}