/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.toolWindow;

import com.intellij.ProjectTopics;
import com.intellij.facet.ProjectWideFacetAdapter;
import com.intellij.facet.ProjectWideFacetListener;
import com.intellij.facet.ProjectWideFacetListenersRegistry;
import com.intellij.ide.actions.ContextHelpAction;
import com.intellij.openapi.actionSystem.AnAction;
import com.intellij.openapi.actionSystem.PlatformDataKeys;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.application.ModalityState;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.ModuleListener;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ModuleRootEvent;
import com.intellij.openapi.roots.ModuleRootListener;
import com.intellij.openapi.ui.SimpleToolWindowPanel;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.wm.ToolWindow;
import com.intellij.openapi.wm.ToolWindowManager;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.ui.FinderRecursivePanel;
import com.intellij.ui.content.Content;
import com.intellij.ui.content.ContentManager;
import com.intellij.util.Function;
import com.intellij.util.messages.MessageBusConnection;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringBaseView extends SimpleToolWindowPanel {

  protected final Project myProject;

  protected FinderRecursivePanel myRootPanel;

  protected SpringBaseView(Project project) {
    super(false, true);
    myProject = project;

    refreshContentPanel();
  }

  protected abstract FinderRecursivePanel createRootPanel();

  private void refreshContentPanel() {
    myRootPanel = createRootPanel();
    myRootPanel.initPanel();
    setContent(myRootPanel);

    Disposer.register(myProject, myRootPanel);
  }

  /**
   * Invoked from Project/Facet settings.
   */
  protected void performFullUpdate() {
    ApplicationManager.getApplication().invokeLater(() -> {
      FinderRecursivePanel oldPanel = myRootPanel;
      remove(oldPanel);
      Disposer.dispose(oldPanel);

      refreshContentPanel();
    }, ModalityState.NON_MODAL, myProject.getDisposed());
  }

  protected void performDetailsUpdate() {
    FinderRecursivePanel panel = myRootPanel;
    while (true) {
      if (!(panel.getSecondComponent() instanceof FinderRecursivePanel)) {
        panel.updateRightComponent(true);
        break;
      }
      panel = (FinderRecursivePanel)panel.getSecondComponent();
    }
  }

  private void updateSelectedPath(Object... pathToSelect) {
    myRootPanel.updateSelectedPath(pathToSelect);
  }

  protected MessageBusConnection installProjectModuleListener() {
    ProjectWideFacetListener<? extends SpringFacet> myProjectWideFacetAdapter = new ProjectWideFacetAdapter<SpringFacet>() {
      @Override
      public void facetAdded(@NotNull SpringFacet facet) {
        performFullUpdate();
      }

      @Override
      public void facetRemoved(@NotNull SpringFacet facet) {
        performFullUpdate();
      }

      @Override
      public void facetConfigurationChanged(@NotNull SpringFacet facet) {
        performFullUpdate();
      }
    };
    ProjectWideFacetListenersRegistry.getInstance(myProject)
      .registerListener(SpringFacet.FACET_TYPE_ID, myProjectWideFacetAdapter);


    final MessageBusConnection messageBusConnection = myProject.getMessageBus().connect(myProject);
    messageBusConnection.subscribe(ProjectTopics.PROJECT_ROOTS, new ModuleRootListener() {
      @Override
      public void rootsChanged(@NotNull ModuleRootEvent event) {
        performFullUpdate();
      }
    });
    messageBusConnection.subscribe(ProjectTopics.MODULES, new ModuleListener() {
      @Override
      public void moduleAdded(@NotNull Project project, @NotNull Module module) {
        performFullUpdate();
      }

      @Override
      public void moduleRemoved(@NotNull Project project, @NotNull Module module) {
        performFullUpdate();
      }

      @Override
      public void modulesRenamed(@NotNull Project project,
                                 @NotNull List<Module> modules,
                                 @NotNull Function<Module, String> oldNameProvider) {
        performFullUpdate();
      }
    });
    return messageBusConnection;
  }

  /**
   * Performs recursive update for given selection path.
   *
   * @param project      Project.
   * @param pathToSelect Path to select.
   * @param requestFocus Focus requested.
   * @param tabName      Tab to select ({@link SpringToolWindowContent#displayName}).
   * @since 14
   */
  protected static void select(Project project, final Object[] pathToSelect, final boolean requestFocus, final String tabName) {
    final ToolWindow toolWindow = ToolWindowManager.getInstance(project).getToolWindow(SpringToolWindowContent.TOOLWINDOW_ID);
    assert toolWindow != null;

    Runnable runnable = () -> {
      final ContentManager contentManager = toolWindow.getContentManager();
      final Content content = contentManager.findContent(tabName);

      contentManager.setSelectedContentCB(content, requestFocus).doWhenDone(() -> {
        final SpringBaseView springBaseView = (SpringBaseView)content.getComponent();
        springBaseView.updateSelectedPath(pathToSelect);
      });
    };
    if (requestFocus) {
      toolWindow.activate(runnable);
    }
    else {
      runnable.run();
    }
  }

  @Nullable
  @Override
  public Object getData(@NotNull String dataId) {
    if (PlatformDataKeys.HELP_ID.is(dataId)) {
      return getHelpId();
    }
    return super.getData(dataId);
  }

  /**
   * @return Help ID for this view.
   * @since 2017.2
   */
  protected String getHelpId() {
    return "Reference.Spring.ToolWindow";
  }

  /**
   * @return Help action for use in toolbar.
   * @since 2017.2
   */
  protected AnAction getHelpAction() {
    return new ContextHelpAction(getHelpId());
  }
}
