// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.toolWindow;

import com.intellij.codeInsight.navigation.NavigationUtil;
import com.intellij.openapi.actionSystem.CommonDataKeys;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiType;
import com.intellij.spring.SpringPresentationProvider;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.ui.FinderRecursivePanel;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

/**
 * Provides unified rendering, editing and DataContext-integration.
 *
 * @since 2018.1
 */
public abstract class SpringBeanPointerPanelBase extends FinderRecursivePanel<SpringBeanPointer> {

  protected SpringBeanPointerPanelBase(@NotNull FinderRecursivePanel parent) {
    super(parent);
  }

  protected SpringBeanPointerPanelBase(@NotNull Project project,
                                       @Nullable String groupId) {
    super(project, groupId);
  }

  protected SpringBeanPointerPanelBase(@NotNull Project project, @Nullable FinderRecursivePanel parent, @Nullable String groupId) {
    super(project, parent, groupId);
  }

  @NotNull
  @Override
  protected String getItemText(SpringBeanPointer pointer) {
    return isNotDumb() && pointer.isValid() ? SpringPresentationProvider.getSpringBeanName(pointer) : "<invalid>";
  }

  @Nullable
  @Override
  protected Icon getItemIcon(SpringBeanPointer pointer) {
    return isNotDumb() && pointer.isValid() ? SpringPresentationProvider.getSpringIcon(pointer) : null;
  }

  @Nullable
  @Override
  protected VirtualFile getContainingFile(SpringBeanPointer pointer) {
    return pointer.isValid() ? pointer.getContainingFile().getVirtualFile() : null;
  }

  @Nullable
  @Override
  protected String getItemTooltipText(SpringBeanPointer pointer) {
    if (!isNotDumb()) return null;

    final CommonSpringBean bean = pointer.isValid() ? pointer.getSpringBean() : null;
    if (bean != null) {
      final PsiType beanType = bean.getBeanType(true);
      if (beanType != null) {
        return beanType.getCanonicalText();
      }
    }

    return null;
  }

  @Override
  protected boolean performEditAction() {
    final SpringBeanPointer springBeanPointer = getSelectedValue();
    if (springBeanPointer == null) return true;

    final PsiElement navigationElement = springBeanPointer.getPsiElement();
    if (navigationElement != null) {
      NavigationUtil.activateFileWithPsiElement(navigationElement);
    }
    return true;
  }

  @Nullable
  @Override
  public Object getData(@NotNull String dataId) {
    SpringBeanPointer selectedValue = getSelectedValue();
    if (selectedValue != null && selectedValue.isValid()) {
      if (CommonDataKeys.NAVIGATABLE.is(dataId) || CommonDataKeys.PSI_ELEMENT.is(dataId)) {
        return selectedValue.getSpringBean().getIdentifyingPsiElement();
      }

      if (CommonDataKeys.PSI_FILE.is(dataId)) {
        return selectedValue.getContainingFile();
      }
    }

    return super.getData(dataId);
  }

  protected boolean isNotDumb() {
    return !getProject().isDisposed() &&
           !DumbService.isDumb(getProject());
  }
}
