/*
 * Copyright (c) 2000-2006 JetBrains s.r.o. All Rights Reserved.
 */

package com.intellij.testFramework.fixtures.impl;

import com.intellij.facet.impl.FacetUtil;
import com.intellij.javaee.DeploymentDescriptorsConstants;
import com.intellij.javaee.artifact.JavaeeArtifactUtil;
import com.intellij.javaee.web.facet.WebFacet;
import com.intellij.javaee.web.facet.WebFacetType;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.StdModuleTypes;
import com.intellij.openapi.projectRoots.Sdk;
import com.intellij.openapi.projectRoots.impl.MockJdkWrapper;
import com.intellij.openapi.roots.ModifiableRootModel;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.roots.libraries.Library;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.vfs.VfsUtilCore;
import com.intellij.testFramework.IdeaTestUtil;
import com.intellij.testFramework.builders.WebModuleFixtureBuilder;
import com.intellij.testFramework.fixtures.IdeaProjectTestFixture;
import com.intellij.testFramework.fixtures.TestFixtureBuilder;
import com.intellij.testFramework.fixtures.WebModuleTestFixture;
import com.intellij.util.descriptors.ConfigFileInfoSet;

import java.util.ArrayList;
import java.util.List;

/**
 * Loaded by {@link WebModuleFixtureBuilder}.
 *
 * @author mike
 */
@SuppressWarnings("UnusedDeclaration")
public class WebModuleFixtureBuilderImpl extends JavaModuleFixtureBuilderImpl<WebModuleTestFixture> implements WebModuleFixtureBuilder {
  private final List<Pair<String, String>> myWebRoots = new ArrayList<>();
  private String myWebXml;

  public WebModuleFixtureBuilderImpl(final TestFixtureBuilder<? extends IdeaProjectTestFixture> fixtureBuilder) {
    super(StdModuleTypes.JAVA, fixtureBuilder);
  }

  @Override
  protected Module createModule() {
    final Module module = super.createModule();
    FacetUtil.addFacet(module, WebFacetType.getInstance());
    return module;
  }

  @Override
  protected WebModuleTestFixture instantiateFixture() {
    return new WebModuleTestFixtureImpl(this);
  }

  @Override
  public WebModuleFixtureBuilder addWebRoot(String rootPath, String relativePath) {
    myWebRoots.add(Pair.create(rootPath, relativePath));
    return this;
  }

  @Override
  public WebModuleFixtureBuilder setWebXml(final String path) {
    if (myWebXml != null) throw new IllegalStateException("Web xml is already set up to: " + myWebXml);
    myWebXml = path;
    return this;
  }

  @Override
  protected void libraryCreated(final Library library, final Module module) {
    super.libraryCreated(library, module);
    WebFacet webFacet = WebFacet.getInstances(module).iterator().next();
    JavaeeArtifactUtil.getInstance().addLibraryToAllArtifactsContainingFacet(library, webFacet);
  }

  @Override
  protected void initModule(final Module module) {
    super.initModule(module);

    WebFacet webFacet = WebFacet.getInstances(module).iterator().next();
    if (myWebXml != null) {
      final ConfigFileInfoSet descriptors = webFacet.getDescriptorsContainer().getConfiguration();

      descriptors.addConfigFile(DeploymentDescriptorsConstants.WEB_XML_META_DATA, "file://" + myWebXml);
    }

    for (String url : ModuleRootManager.getInstance(module).getSourceRootUrls()) {
      webFacet.addWebSourceRoot(url);
    }

    for (Pair<String, String> pair : myWebRoots) {
      String path = VfsUtilCore.pathToUrl(pair.first);
      String relativePath = pair.second;

      webFacet.addWebRoot(path, relativePath);
    }

    Sdk sdk = ModuleRootManager.getInstance(module).getSdk();
    assert sdk != null : module;
    sdk=IdeaTestUtil.addWebJarsTo(((MockJdkWrapper)sdk).getDelegate());
    ModifiableRootModel model = ModuleRootManager.getInstance(module).getModifiableModel();
    model.setSdk(sdk);
    model.commit();
  }
}
