/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.util;

import com.intellij.database.model.ObjectKind;
import com.intellij.openapi.util.Comparing;
import com.intellij.util.ObjectUtils;
import com.intellij.util.PairFunction;
import com.intellij.util.ThrowableConsumer;
import com.intellij.util.containers.WeakInterner;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;

/**
 * @author gregsh
 */
public class ObjectPath {
  private static final WeakInterner<ObjectPath> ourInterner = new WeakInterner<>();

  public final ObjectPath parent;

  public final ObjectKind kind;
  public final String name;

  private ObjectPath(@NotNull String name, @NotNull ObjectKind kind, @Nullable ObjectPath parent) {
    this.name = name;
    this.kind = kind;
    this.parent = parent;
  }

  public boolean isQuoted() {
    return true;
  }

  @Nullable
  public String getIdentity() {
    return null;
  }

  @NotNull
  public ObjectPath append(@NotNull String name, @NotNull ObjectKind kind) {
    return create(name, kind, true, null, this);
  }

  public ObjectPath append(@NotNull String name, @NotNull ObjectKind kind, boolean quoted, String identity) {
    return create(name, kind, quoted, identity, this);
  }

  public int getSize() {
    int result = 0;
    for (ObjectPath p = this; p != null; p = p.parent) {
      result ++;
    }
    return result;
  }

  @Nullable
  public ObjectPath getParent(int steps) {
    ObjectPath result = this;
    for (int i = steps; i > 0 && result != null; i--) result = result.parent;
    return result;
  }

  @Nullable
  public ObjectPath findParent(ObjectKind kind, boolean strict) {
    for (ObjectPath p = strict ? parent : this; p != null; p = p.parent) {
      if (p.kind == kind) return p;
    }
    return null;
  }

  public boolean isAncestorOf(@Nullable ObjectPath child, boolean strict) {
    if (strict && this == child) return false;
    while (child != null && child != this) child = child.parent;
    return this == child;
  }

  @NotNull
  public String getDisplayName() {
    return reduce(
      new StringBuilder(), (sb, p) ->
        (sb.length() > 0 && p.name.length() > 0 ? sb.append(".") : sb).append(p.name))
      .toString();
  }

  @NotNull
  public String getName() {
    return name;
  }

  public <T> T reduce(T t, @NotNull PairFunction<T, ObjectPath, T> reducer) {
    T result = t;
    if (parent != null) result = parent.reduce(t, reducer);
    return reducer.fun(result, this);
  }

  @NotNull
  public static ObjectPath create(@NotNull String name, @NotNull ObjectKind kind) {
    return create(name, kind, true, null, null);
  }

  public static ObjectPath create(@NotNull String name,
                                  @NotNull ObjectKind kind,
                                  boolean quoted,
                                  @Nullable String identity,
                                  @Nullable ObjectPath parent) {
    return ourInterner.intern(createInner(name, kind, quoted, identity, parent));
  }

  @NotNull
  private static ObjectPath createInner(@NotNull String name,
                                        @NotNull ObjectKind kind,
                                        boolean quoted,
                                        @Nullable String identity,
                                        @Nullable ObjectPath parent) {
    if (quoted && identity == null) return new ObjectPath(name, kind, parent);
    if (identity == null) return new ObjectPath(name, kind, parent) {
      @Override public boolean isQuoted() { return false; }
    };
    if (quoted) new ObjectPath(name, kind, parent) {
      @Override public String getIdentity() { return identity; }
    };
    return new ObjectPath(name, kind, parent) {
      @Override public boolean isQuoted() { return quoted; }
      @Override public String getIdentity() { return identity; }
    };
  }

  @NotNull
  public static ObjectPath create(@NotNull ObjectKind kind, @NotNull Iterable<String> names) {
    ObjectPath result = null;
    for (Iterator<String> it = names.iterator(); it.hasNext(); ) {
      result = create(it.next(), !it.hasNext() ? kind : ObjectKind.NONE, true, null, result);
    }
    return ObjectUtils.notNull(result);
  }

  public <T extends Throwable> void forEach(@NotNull ThrowableConsumer<ObjectPath, T> r) throws T {
    forEachInner(this, r);
  }

  private static <T extends Throwable> void forEachInner(@NotNull ObjectPath p, @NotNull ThrowableConsumer<ObjectPath, T> r) throws T {
    if (p.parent != null) forEachInner(p.parent, r);
    r.consume(p);
  }

  @Override
  public String toString() {
    String id = getIdentity();
    return (parent != null ? parent + "/" : "") +
           (kind == ObjectKind.NONE ? "" : kind.name() + ":") + name + (id == null ? "" : "@" + id);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    ObjectPath path = (ObjectPath)o;

    if (parent != path.parent) return false;
    if (!kind.equals(path.kind)) return false;
    if (!name.equals(path.name)) return false;
    if (!Comparing.equal(isQuoted(), path.isQuoted())) return false;
    if (!Comparing.equal(getIdentity(), path.getIdentity())) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = System.identityHashCode(parent);
    result = 31 * result + kind.hashCode();
    result = 31 * result + name.hashCode();
    result = 31 * result + Boolean.hashCode(isQuoted());
    result = 31 * result + Comparing.hashcode(getIdentity());
    return result;
  }
}
