/* openquicktime.h
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef OPENQUICKTIME_H
#define OPENQUICKTIME_H

#include "private.h"

#ifdef __cplusplus
extern "C" {
#endif


/* OpenQuicktime Codec Parameter Types */
#define QUICKTIME_UNKNOWN_PARAMETER         -1
#define QUICKTIME_STRING_PARAMETER           0
#define QUICKTIME_BOOLEAN_PARAMETER          1
#define QUICKTIME_INTEGER_PARAMETER          4
#define QUICKTIME_UNSIGNED_INTEGER_PARAMETER 5
#define QUICKTIME_DOUBLE_PARAMETER           8


/* Some codecs inherited from QT4L ... we keep them here for the moment but they are useless */

/* ===== compression formats for which codecs exist ====== */
/* DV works only for decoding. */
#define QUICKTIME_DV "dvc "
/* RGB uncompressed.  Allows alpha */
#define QUICKTIME_RAW  "raw "
/* Jpeg Photo */
#define QUICKTIME_JPEG "jpeg"
/* Concatenated png images.  Allows alpha */
#define QUICKTIME_PNG "png "
/* Motion JPEG-A. */
#define QUICKTIME_MJPA "mjpa"
/* YUV 4:2:2 */
#define QUICKTIME_YUV2 "yuv2"
/* YUV 4:2:0  NOT COMPATIBLE WITH STANDARD QUICKTIME */
#define QUICKTIME_YUV4 "yuv4"
/* ======== compression for which no codec exists ========== */
/* These are traditionally converted in hardware or not at all */

/* 8 bit Planar YUV 4:2:0 */
#define QUICKTIME_YUV420  "yv12"
/* 8 bit Planar YUV 4:1:1 */
#define QUICKTIME_YUV411  "y411"
/* 8 bit Packed YUV 4:2:2 */
#define QUICKTIME_YUV422 "yuv2"
/* 8 bit Planar YUV 4:4:4 */
#define QUICKTIME_YUV444  "v308"
/* 8 bit Planar YUVA 4:4:4:4 */
#define QUICKTIME_YUVA4444 "v408"
/* 10 bit Planar YUV 4:4:4 */
#define QUICKTIME_YUV444_10bit  "v410"

/* ======== Studies in different algorithms =============== */
/* YUV9.  What on earth for? */
#define QUICKTIME_YUV9 "YVU9"
/* RTjpeg, proprietary but fast? */
#define QUICKTIME_RTJ0 "RTJ0"

/* =================== Audio formats ======================= */
/* Unsigned 8 bit */
#ifndef QUICKTIME_RAW
#define QUICKTIME_RAW "raw "
#endif
/* IMA4 */
#define QUICKTIME_IMA4 "ima4"
/* Twos compliment 8, 16, 24 */
#define QUICKTIME_TWOS "twos"
/* ulaw */
#define QUICKTIME_ULAW "ulaw"

/* =========================== public interface ========================= */

/* return 1 if the file is a quicktime file */
OPENQUICKTIMELIB_API int quicktime_check_sig(char *path);

/* call this first to open the file and create all the objects */
OPENQUICKTIMELIB_API quicktime_t* quicktime_open(char *filename, int rd, int wr);
  int quicktime_init(quicktime_t *file);

/* make the quicktime file streamable */
OPENQUICKTIMELIB_API int quicktime_make_streamable(char *in_path, char *out_path);

/* Set various options in the file. */
OPENQUICKTIMELIB_API void quicktime_set_copyright(quicktime_t *file, char *string);
OPENQUICKTIMELIB_API void quicktime_set_name(quicktime_t *file, char *string);
OPENQUICKTIMELIB_API void quicktime_set_info(quicktime_t *file, char *string);
OPENQUICKTIMELIB_API char* quicktime_get_copyright(quicktime_t *file);
OPENQUICKTIMELIB_API char* quicktime_get_name(quicktime_t *file);
OPENQUICKTIMELIB_API char* quicktime_get_info(quicktime_t *file);

/* Read all the information about the file. */
/* Requires a MOOV atom be present in the file. */
/* If no MOOV atom exists return 1 else return 0. */
OPENQUICKTIMELIB_API int quicktime_read_info(quicktime_t *file);

/* set up tracks in a new file after opening and before writing */
/* returns the number of quicktime tracks allocated */
/* audio is stored two channels per quicktime track */
OPENQUICKTIMELIB_API int quicktime_set_audio(quicktime_t *file, 
	int channels, 
	long sample_rate, 
	int bits, 
	char *compressor);
/* Samplerate can be set after file is created */
OPENQUICKTIMELIB_API void quicktime_set_framerate(quicktime_t *file, float framerate);

/* video is stored one layer per quicktime track */
OPENQUICKTIMELIB_API int quicktime_set_video(quicktime_t *file, 
	int tracks, 
	int frame_w, 
	int frame_h, 
	float frame_rate, 
	char *compressor);

/* Set the depth of the track. */
OPENQUICKTIMELIB_API void quicktime_set_depth(quicktime_t *file, 
	int depth, 
	int track);

/* Set the colormodel for encoder input */
OPENQUICKTIMELIB_API void quicktime_set_cmodel(quicktime_t *file, int colormodel);

/* Set codec-dependant parameter 'param' to 'data' */
OPENQUICKTIMELIB_API int quicktime_set_audio_param (quicktime_t *file, int track,
			       const char* param, const void* data);
OPENQUICKTIMELIB_API int quicktime_set_video_param (quicktime_t *file, int track,
			       const char* param, const void* data);

/* Get codec-dependant parameter 'param' into 'data' */
OPENQUICKTIMELIB_API int quicktime_get_audio_param (quicktime_t *file, int track,
			       const char* param, void* data);
OPENQUICKTIMELIB_API int quicktime_get_video_param (quicktime_t *file, int track,
			       const char* param, void* data);



/* close the file and delete all the objects */
OPENQUICKTIMELIB_API int quicktime_close(quicktime_t *file);

/* get length information */
/* channel numbers start on 1 for audio and video */
OPENQUICKTIMELIB_API long quicktime_audio_length(quicktime_t *file, int track);
OPENQUICKTIMELIB_API long quicktime_video_length(quicktime_t *file, int track);

/* get position information */
OPENQUICKTIMELIB_API long quicktime_audio_position(quicktime_t *file, int track);
OPENQUICKTIMELIB_API long quicktime_video_position(quicktime_t *file, int track);

/* get file information */
OPENQUICKTIMELIB_API int quicktime_video_tracks(quicktime_t *file);
OPENQUICKTIMELIB_API int quicktime_audio_tracks(quicktime_t *file);

OPENQUICKTIMELIB_API int quicktime_has_audio(quicktime_t *file);
OPENQUICKTIMELIB_API long quicktime_sample_rate(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_audio_bits(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_track_channels(quicktime_t *file, int track);
OPENQUICKTIMELIB_API char* quicktime_audio_compressor(quicktime_t *file, int track);

OPENQUICKTIMELIB_API int quicktime_has_video(quicktime_t *file);
OPENQUICKTIMELIB_API int quicktime_video_width(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_video_height(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_video_depth(quicktime_t *file, int track);
OPENQUICKTIMELIB_API float quicktime_frame_rate(quicktime_t *file, int track);
OPENQUICKTIMELIB_API char* quicktime_video_compressor(quicktime_t *file, int track);

/* number of bytes of raw data in this frame */
OPENQUICKTIMELIB_API long quicktime_frame_size(quicktime_t *file, long frame, int track);

/* get the quicktime track and channel that the audio channel belongs to */
/* channels and tracks start on 0 */
OPENQUICKTIMELIB_API int quicktime_channel_location(quicktime_t *file, int *quicktime_track, int *quicktime_channel, int channel);

/* file positioning */
OPENQUICKTIMELIB_API int quicktime_seek_end(quicktime_t *file);
OPENQUICKTIMELIB_API int quicktime_seek_start(quicktime_t *file);

/* set position of file descriptor relative to a track */
OPENQUICKTIMELIB_API int quicktime_set_audio_position(quicktime_t *file, longest sample, int track);
OPENQUICKTIMELIB_API int quicktime_set_video_position(quicktime_t *file, longest frame, int track);

/* ========================== Access to raw data follows. */
/* write data for one quicktime track */
/* the user must handle conversion to the channels in this track */
OPENQUICKTIMELIB_API int quicktime_write_audio(quicktime_t *file, char *audio_buffer, long samples, int track);
OPENQUICKTIMELIB_API int quicktime_write_frame(quicktime_t *file, unsigned char *video_buffer, longest bytes, int track);

/* Read an entire chunk. */
/* read the number of bytes starting at the byte_start in the specified chunk */
/* You must provide enough space to store the chunk. */
OPENQUICKTIMELIB_API int quicktime_read_chunk(quicktime_t *file, char *output, int track, longest chunk, longest byte_start, longest byte_len);

/* read raw data */
OPENQUICKTIMELIB_API long quicktime_read_audio(quicktime_t *file, char *audio_buffer, long samples, int track);
OPENQUICKTIMELIB_API long quicktime_read_frame(quicktime_t *file, unsigned char *video_buffer, int track);

/* for reading frame using a library that needs a file descriptor */
/* Frame caching doesn't work here. */
OPENQUICKTIMELIB_API int quicktime_read_frame_init(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_read_frame_end(quicktime_t *file, int track);

/* ===================== Access to built in codecs follows. */

/* If the codec for this track is supported in the library return 1. */
OPENQUICKTIMELIB_API int quicktime_supported_video(quicktime_t *file, int track);
OPENQUICKTIMELIB_API int quicktime_supported_audio(quicktime_t *file, int track);

/* The codec can generate the color model with no downsampling */
OPENQUICKTIMELIB_API int quicktime_reads_cmodel(quicktime_t *file, 
		int colormodel, 
		int track);

/* The codec can write the color model with no upsampling */
OPENQUICKTIMELIB_API int quicktime_writes_cmodel(quicktime_t *file, 
		int colormodel, 
		int track);

/* Decode or encode the frame into a frame buffer. */

OPENQUICKTIMELIB_API int quicktime_encode_video(quicktime_t    *file, 
			   int             color_model,
			   unsigned char **row_pointers, 
			   int             track);

OPENQUICKTIMELIB_API int quicktime_decode_video(quicktime_t    *file, 
			   int             color_model,
			   unsigned char **row_pointers, 
			   int             track);

/* Decode or encode audio for a single channel into the buffer. */
/* Pass a buffer for the _i or the _f argument if you want int16 or float data. */
/* Notice that encoding requires an array of pointers to each channel. */
OPENQUICKTIMELIB_API int quicktime_decode_audio(quicktime_t *file, int16_t *output_i, float *output_f, long samples, int channel);
OPENQUICKTIMELIB_API int quicktime_encode_audio(quicktime_t *file, int16_t **input_i, float **input_f, long samples);

/* Dump the file structures for the currently opened file. */
OPENQUICKTIMELIB_API int quicktime_dump(quicktime_t *file);

/* Specify the number of cpus to utilize. */
OPENQUICKTIMELIB_API int quicktime_set_cpus(quicktime_t *file, int cpus);

/* Specify whether to read contiguously or not. */
/* preload is the number of bytes to read ahead. */
OPENQUICKTIMELIB_API void quicktime_set_preload(quicktime_t *file, longest preload);

OPENQUICKTIMELIB_API longest quicktime_byte_position(quicktime_t *file);

/* One keyframe table for each track */
OPENQUICKTIMELIB_API long quicktime_get_keyframe_before(quicktime_t *file, long frame, int track);
OPENQUICKTIMELIB_API long quicktime_get_keyframe_after(quicktime_t *file, long frame, int track);
OPENQUICKTIMELIB_API void quicktime_insert_keyframe(quicktime_t *file, long frame, int track);


/* Compressed Packets reading */
OPENQUICKTIMELIB_API longest quicktime_read_next_packet(quicktime_t *file, unsigned char *outbuf, int *isVideo, int *trak);


#ifdef __cplusplus
}
#endif

#endif
