/****************************************************************************
*                   glow.c
*
*  This module contains functions for the glow feature.
*
*****************************************************************************/

#include "frame.h"

#ifdef GlowPatch

#include "povray.h"
#include "vector.h"
#include "matrices.h"
#include "povproto.h"
#include "colour.h"
#include "express.h"
#include "pigment.h"
#include "warps.h"
#include "glow.h"

#include "parse.h"
#include "parstxtr.h"
#include "tokenize.h" 
#include "texture.h"

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

/*****************************************************************************
* Local typedefs
******************************************************************************/

/*****************************************************************************
* Static functions
******************************************************************************/

/*****************************************************************************
* Local variables
******************************************************************************/

/*****************************************************************************
*
* FUNCTION   Create_Glow()
*
*   
*
* INPUT
*
* OUTPUT
*
* RETURNS   pointer to a GLOW struct
*
* AUTHOR
*
*   Chris Huff
*
* DESCRIPTION
*
*   Creates and initializes a glow.
*
* CHANGES
*
*   2000.9.02: Creation.
*
******************************************************************************/
GLOW * Create_Glow()
{
	GLOW * New = (GLOW *)POV_MALLOC(sizeof(GLOW), "glow");
	Make_Vector(New->Center, 0, 0, 0);
	Make_Colour(New->Colour, 1, 1, 1);
	New->Glow_Type = 1;
	New->Radius = 1;
	New->Cutoff_Radius = 0;
	New->fade_pow = 1;
	New->Warps = NULL;

	return New;
}

/*****************************************************************************
*
* FUNCTION   Add_Glow()
*
*   
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Chris Huff
*
* DESCRIPTION
*
*   Adds a glow to the list in Frame.
*
* CHANGES
*
*   2000.9.02: Creation.
*
******************************************************************************/
void Add_Glow(GLOW * Glow)
{
	Glow->next = Frame.Glows;
	Frame.Glows = Glow;
}

/*****************************************************************************
*
* FUNCTION   Destroy_Glow()
*
*   
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Chris Huff
*
* DESCRIPTION
*
*   Destroys a glow.
*
* CHANGES
*
*   2000.9.02: Creation.
*
******************************************************************************/
void Destroy_Glow(GLOW * Glow)
{
	if(Glow->Warps != NULL)
	{
		Destroy_TPat_Fields(Glow->Warps);
		POV_FREE(Glow->Warps);
	}
	POV_FREE(Glow);
}

/*****************************************************************************
*
* FUNCTION   Destroy_Glow_List()
*
*   
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Chris Huff
*
* DESCRIPTION
*
*   Destroys a list of glows.
*
* CHANGES
*
*   2000.9.02: Creation.
*
******************************************************************************/
void Destroy_Glow_List(GLOW * Glow_List)
{
	GLOW * Glow = Glow_List;
	GLOW * Next_Glow = NULL;
	while(Glow != NULL)
	{
		Next_Glow = Glow->next;
		Destroy_Glow(Glow);
		Glow = Next_Glow;
	}
}

/*****************************************************************************
*
* FUNCTION   Do_Glow()
*
*   
*
* INPUT
*
* OUTPUT
*
* RETURNS 
*
* AUTHOR
*
*   Chris Huff
*
* DESCRIPTION
*
*   Calculates glow effect.
*
* CHANGES
*
*   2000.9.02: Creation.
*
******************************************************************************/
void Do_Glow(INTERSECTION * Isect, RAY * Ray, COLOUR Colour)
{
	VECTOR temp;
	VECTOR lightDir;/*direction of glow source*/
	DBL Depth = Isect->Depth;/*Distance to intersection*/
	DBL scattered_intensity;
	DBL cosA;/*cosine of the angle between the ray and the direction of the glow source*/
	DBL Dist;
	COLOUR scattered_light;
	GLOW * Glow = Frame.Glows;
	DBL d0;
	DBL denom;
	while(Glow != NULL)
	{
		if(Glow->Radius != 0)
		{
			/* cosA Computing */
			VSub(lightDir, Ray->Initial, Glow->Center);
			VDot(cosA, lightDir, Ray->Direction);

			/* d0 Computing */
			VScale(temp, Ray->Direction, -cosA);
			VAdd(temp, temp, Ray->Initial);
			if(Glow->Warps != NULL)
			{
				Warp_EPoint(temp, temp, Glow->Warps, FALSE);
			}
			VSub(temp, temp, Glow->Center);
			VLength(Dist, temp);
			if(Dist < Glow->Cutoff_Radius || Glow->Cutoff_Radius == 0)
			{
				/* scattered energy integration along ray */
				switch(Glow->Glow_Type)
				{
					case 0:/* A model, I(d) = 1/d^2 */
						d0 = Dist/Glow->Radius;
						scattered_intensity = (atan((Depth+cosA)/d0) - atan(cosA/d0))/d0;
						break;
		
					case 1:/* B model, I(d) = 1/(d^2+1) */
						denom = 1;
						VDot(d0, temp, temp);
						d0 /= Glow->Radius;/*or should this modify denom?*/
						denom = sqrt(d0 + 1);
						if(Depth >= Max_Distance) /* Optimization */
							scattered_intensity = (M_PI_2 - atan(cosA/denom))/denom;
						else
							scattered_intensity = (atan((Depth+cosA)/denom) - atan(cosA/denom))/denom;
						break;
		
					case 3:
						d0 = Dist/Glow->Radius;
						if((1-d0) > 0 && cosA < 0)
							scattered_intensity = sin((0, 1-d0)*M_PI_2);
						else
							scattered_intensity = 0;
						break;
		
					case 2:/*Solid density sphere model */
						if(cosA < 0)
							scattered_intensity = exp(-Dist/Glow->Radius);
						/*	scattered_intensity = exp(-Dist/(Glow->Radius+Depth/Glow->Radius));*/
						break;
				}
				if(Glow->fade_pow != 1)
				{
					if(Glow->fade_pow == 2)
						scattered_intensity *= scattered_intensity;
					else
						scattered_intensity = pow(scattered_intensity, Glow->fade_pow);
				}
			}
			else
			{
				scattered_intensity = 0;
			}
			Scale_Colour(scattered_light, Glow->Colour, scattered_intensity);
			Add_Colour(Colour, Colour, scattered_light);
		}
		Glow = Glow->next;
	}
}

void Parse_Glow(GLOW * Glow)
{
	MATRIX Matrix;
	int Warped = FALSE;
	TRANSFORM * Trans = Create_Transform();
	TRANSFORM * Temp_Trans = Create_Transform();
	TRANSFORM * Trans2;
	TURB * Local_Turb = NULL;
	VECTOR Local_Vector;
	TPATTERN * Warps = (TPATTERN *)POV_MALLOC(sizeof(TPATTERN), "TPATTERN");
	Init_TPat_Fields(Warps);
	Warps->Type = BOZO_PATTERN;/*Set it to a pattern type to allow turbulence*/

	Parse_Begin();

	EXPECT
		CASE(SIZE_TOKEN)
			/*Change to store reciprocal instead of straight value?*/
			Glow->Radius = Parse_Float();
		END_CASE

		CASE(RADIUS_TOKEN)
			Glow->Cutoff_Radius = Parse_Float();
		END_CASE

		CASE(FADE_POWER_TOKEN)
			Glow->fade_pow = Parse_Float();
		END_CASE

		CASE(TYPE_TOKEN)
			Glow->Glow_Type = (int)Parse_Float();
			/*YS sept 22 2000 added type checking*/
			if ( Glow->Glow_Type<0 || Glow->Glow_Type >3)
				Error("\nUnkonw glow type.\n");
			/*YS*/
		END_CASE

		CASE(LOCATION_TOKEN)
			Parse_Vector(Glow->Center);
		END_CASE

		CASE(COLOUR_TOKEN)
			Parse_Colour(Glow->Colour);
		END_CASE

		CASE (WARP_TOKEN)
			Warped = TRUE;
			Parse_Begin();
			Parse_Warp(&(Warps->Warps));
			Parse_End();
		END_CASE

		CASE (TRANSLATE_TOKEN)
			Parse_Vector(Local_Vector);
			Compute_Translation_Transform(Temp_Trans, Local_Vector);
			Compose_Transforms(Trans, Temp_Trans);
		END_CASE

		CASE (ROTATE_TOKEN)
			Parse_Vector(Local_Vector);
			Compute_Rotation_Transform(Temp_Trans, Local_Vector);
			Compose_Transforms(Trans, Temp_Trans);
		END_CASE

		CASE (SCALE_TOKEN)
			Parse_Scale_Vector(Local_Vector);
			Compute_Scaling_Transform(Trans, Local_Vector);
			Compose_Transforms(Trans, Temp_Trans);
		END_CASE

		CASE (MATRIX_TOKEN)
			Parse_Matrix(Matrix);
			Compute_Matrix_Transform(Temp_Trans, Matrix);
			Compose_Transforms(Trans, Temp_Trans);
		END_CASE

		CASE (TRANSFORM_TOKEN)
			Trans2 = Parse_Transform();
			Compose_Transforms(Trans, Trans2);
			Destroy_Transform(Trans2);
		END_CASE

		OTHERWISE
			UNGET
			EXIT
		END_CASE
	END_EXPECT

	Parse_End();
	if(Warped == TRUE)
	{	
		Glow->Warps = Warps;
	}
	else
	{
		Destroy_TPat_Fields(Warps);
		POV_FREE(Warps);
	}
	MTransPoint(Glow->Center, Glow->Center, Trans);
	Destroy_Transform(Trans);
	Destroy_Transform(Temp_Trans);
}

#endif

