/****************************************************************************
*                isosurf.c
*
*  This module implements the isosurface shapetype.
*  This module was written by R.Suzuki(rsuzuki@etl.go.jp) and 
*  D.Skarda&T.Bily (0rfelyus@atrey.karlin.mff.cuni.cz). 
*  This code is based on the code of 'boxes.c' in official POV-Ray 3.0.
*
*****************************************************************************/


#include "frame.h"
#include "povray.h"
#include "vector.h"
#include "povproto.h"
#include "boxes.h"
#include "bbox.h"
#include "spheres.h"
#include "matrices.h"
#include "objects.h"
#include "isosrf.h"
#include "f_expr.h"
/*dfs--12/23/99*/
#include "parse.h"
#include "tokenize.h"
/*--dfs*/
#ifdef POVISO

DBL (*IsoFunc)(FUNCTION *, VECTOR );
extern int Shadow_Test_Flag;

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

#define close(x, y) (fabs(x-y) < EPSILON ? 1 : 0)

/* Minimal intersection depth. */

#define DEPTH_TOLERANCE 1.0e-6

/* Two values are equal if their difference is small than CLOSE_TOLERANCE. */

#define CLOSE_TOLERANCE 1.0e-6

/* Side hit. */

#define SIDE_X_0 1
#define SIDE_X_1 2
#define SIDE_Y_0 3
#define SIDE_Y_1 4
#define SIDE_Z_0 5
#define SIDE_Z_1 6



/*****************************************************************************
* Static functions
******************************************************************************/
static int  All_IsoSurface_Intersections (OBJECT *Object, RAY *Ray, ISTACK *Depth_Stack);
static int  Inside_IsoSurface (VECTOR point, OBJECT *Object);
static void IsoSurface_Normal (VECTOR Result, OBJECT *Object, INTERSECTION *Inter);
static void Translate_IsoSurface (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Rotate_IsoSurface (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Scale_IsoSurface (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Transform_IsoSurface (OBJECT *Object, TRANSFORM *Trans);
static void Invert_IsoSurface (OBJECT *Object);



/*****************************************************************************
* Local variables
******************************************************************************/

METHODS IsoSurface_Methods =
{
  All_IsoSurface_Intersections,
  Inside_IsoSurface, IsoSurface_Normal, Default_UVCoord,
  Copy_IsoSurface, Translate_IsoSurface, Rotate_IsoSurface, Scale_IsoSurface, Transform_IsoSurface,
  Invert_IsoSurface, Destroy_IsoSurface
};

static VECTOR Int_Begin[30], Int_End[30];    
static DBL    Int_Lenght[30], Int_t_begin[30], Int_t_end[30];

FUNCTION *TmpFunc;
int   tempfunc=FALSE;

DBL EP_Best_Intersection;
static VECTOR VTmp;


/*****************************************************************************
*
* FUNCTION
*
*   All_IsoSurface_Intersections
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static int All_IsoSurface_Intersections(Object, Ray, Depth_Stack)
OBJECT *Object;
RAY *Ray;
ISTACK *Depth_Stack;
{
  ISOSURFACE *Isosrf = (ISOSURFACE *)Object;
  int   Side1, Side2, itrace=0, i_flg;
  DBL   Depth1=0, Depth2=0, len;
  RAY   New_Ray;
  VECTOR IPoint;
  VECTOR P, D;
  DBL   tmax, tmin, tmp=0;

  int   i=0,                /* count of intervals in stack - 1      */
        inf,
        IFound=FALSE;
  int   begin=0, end=0;
  FUNCTION *Func= Isosrf->Func;

 if (tempfunc) TmpFunc = Isosrf->Func;

 Increase_Counter(stats[Ray_IsoSurface_Bound_Tests]);


 if (Isosrf->bound_shape) 
 {
     if (Isosrf->Trans != NULL)
     {
        MInvTransPoint(New_Ray.Initial, Ray->Initial, Isosrf->Trans);
        MInvTransDirection(New_Ray.Direction, Ray->Direction, Isosrf->Trans);
        VLength(len, New_Ray.Direction);
        VInverseScaleEq(New_Ray.Direction, len);
        i_flg = Intersect_Sphere(&New_Ray, Isosrf->bounds[0], (Isosrf->bounds[1][X])*(Isosrf->bounds[1][X]), &Depth1, &Depth2);
        Depth1=Depth1/len;
        Depth2=Depth2/len;
     }
     else 
        i_flg = Intersect_Sphere(Ray, Isosrf->bounds[0], (Isosrf->bounds[1][X])*(Isosrf->bounds[1][X]), &Depth1, &Depth2);
     Decrease_Counter(stats[Ray_Sphere_Tests]);
     if (i_flg)   Decrease_Counter(stats[Ray_Sphere_Tests_Succeeded]);
 }
 else  
   i_flg = (Intersect_Box(Ray, (BOX *)Object, &Depth1, &Depth2, &Side1, &Side2));

 if (Depth1<0.) Depth1=0.;

 if (i_flg)  /* IsoSurface_Bound_Tests */
 {
     Increase_Counter(stats[Ray_IsoSurface_Bound_Tests_Succeeded]);
     if (Isosrf->Trans != NULL)
     {
        MInvTransPoint(P, Ray->Initial, Isosrf->Trans);
        MInvTransDirection(D, Ray->Direction, Isosrf->Trans);
     }
     else
     {
        Assign_Vector(P, Ray->Initial);
        Assign_Vector(D, Ray->Direction);
     }
     Isosrf->Inv3=1;

     if (Isosrf->closed)
     {
        VEvaluateRay(VTmp, P, Depth1, D); 
        tmp=iso_func(Func,VTmp);        
        if (Depth1>Isosrf->accuracy)
        {  
          if (tmp<0.)  /* The ray hits the bounding shape */
          {
            VEvaluateRay(IPoint, Ray->Initial, Depth1, Ray->Direction);
            if (Point_In_Clip(IPoint, Object->Clip))
            {
                push_entry_i1(Depth1,IPoint,Object,Side1,Depth_Stack);
                IFound=TRUE;
                itrace++;
                Isosrf->Inv3*=-1;
            }
          }
        }
        else
        {  
          if  (tmp < (Func->Max_gradient * Isosrf->accuracy *4.) )
          {    
             Depth1=Isosrf->accuracy*5;
             VEvaluateRay(VTmp, P, Depth1, D);
             if (iso_func(Func, VTmp) <0 )     Isosrf->Inv3=-1;
               /* Change the sign of the function (IPoint is in the bounding shpae.)*/
          }
          VEvaluateRay(VTmp, P, Depth2, D);
          if (iso_func(Func, VTmp) < 0.) 
          {
            VEvaluateRay(IPoint, Ray->Initial, Depth2, Ray->Direction);
            if (Point_In_Clip(IPoint, Object->Clip))
            {
              push_entry_i1(Depth2,IPoint,Object,Side2,Depth_Stack);
              IFound=TRUE;
            }
          }
        }
    }


    Side1=0;Side2=0;
    if (Isosrf->method<=0)
    {
      /*  METHOD 2   by R. Suzuki */
      tmax= Depth2=min(Depth2, EP_Best_Intersection);
      tmin= Depth1=min(Depth2, Depth1); 
      if ((tmax-tmin)< Isosrf->accuracy)    return (FALSE);
      Increase_Counter(stats[Ray_IsoSurface_Tests]);
      if ((Depth1 < Isosrf->accuracy) && (Isosrf->Inv3==1))
      {
         /* IPoint is on the isosurface */
         VEvaluateRay(VTmp, P, tmin, D);
         if  ( fabs( iso_func(Func, VTmp) ) <
                 (Func->Max_gradient * Isosrf->accuracy *4.) )
         {
             tmin=Isosrf->accuracy*5;
             VEvaluateRay(VTmp, P, tmin, D);
             if (iso_func(Func, VTmp) <0 )     Isosrf->Inv3=-1;
                 /* change the sign and go into the isosurface */
         }
      }

      for (; itrace<Isosrf->max_trace; itrace++)
      {
          if (iso_find_root(Isosrf, P,D, &tmin, &tmax, &Side1, &Side2)==FALSE)
            break;
          else
          {
             VEvaluateRay(IPoint, Ray->Initial, tmin, Ray->Direction);
             if (Point_In_Clip(IPoint, Object->Clip))
             {
                push_entry_i1(tmin,IPoint,Object,Side1,Depth_Stack);
                IFound = TRUE;
             }
          }
          tmin+=Isosrf->accuracy*5;
          if ((tmax-tmin)< Isosrf->accuracy) break;
          Isosrf->Inv3*=-1;
      }

      if (IFound) Increase_Counter(stats[Ray_IsoSurface_Tests_Succeeded]);
    }

    else
    {
      /*  METHOD 1  by D.Skarda & T.Bily */

      if (Depth1==Depth2) Depth1=0;
      if (Depth1<Isosrf->accuracy) Depth1+= Isosrf->accuracy*5; 
      if (Depth1>Depth2) return(FALSE);
      if((Isosrf->bounds[0][X]<=P[X])&&(P[X]<=Isosrf->bounds[1][X])&&  
         (Isosrf->bounds[0][Y]<=P[Y])&&(P[Y]<=Isosrf->bounds[1][Y])&&  
         (Isosrf->bounds[0][Z]<=P[Z])&&(P[Z]<=Isosrf->bounds[1][Z])) 
           if ((Depth1+= 4*Isosrf->accuracy) > Depth2) return (FALSE);

      Increase_Counter(stats[Ray_IsoSurface_Tests]);
   
      VScale (Int_Begin[begin], D, Depth1);
      VAddEq (Int_Begin[begin], P);

      VScale (Int_End[end], D, Depth2);
      VAddEq (Int_End[end], P);

      Int_t_begin[0]= Depth1;
      Int_t_end[0]= Depth2;

      inf= Isosrf->max_trace;

      VSub(VTmp, Int_Begin[begin], Int_End[end]);
      VLength(Int_Lenght[0], VTmp);

      inf= Isosrf->max_trace;
      i=0;

      VSub(VTmp, Int_Begin[begin], Int_End[end]);
      VLength(Int_Lenght[0], VTmp);

      while ((i>=0)&&(inf != 0))
      {
        if ((Int_End[end][X])<(Int_Begin[begin][X]))
          { func_XYZ_low[X]= Int_End[end][X];   func_XYZ_hi[X]=Int_Begin[begin][X]; }
        else
          { func_XYZ_low[X]= Int_Begin[begin][X]; func_XYZ_hi[X]=Int_End[end][X];   }
        if ((Int_End[end][Y])<(Int_Begin[begin][Y]))
          { func_XYZ_low[Y]= Int_End[end][Y];   func_XYZ_hi[Y]=Int_Begin[begin][Y]; }
        else
          { func_XYZ_low[Y]= Int_Begin[begin][Y]; func_XYZ_hi[Y]=Int_End[end][Y];   }
        if ((Int_End[end][Z])<(Int_Begin[begin][Z]))
          { func_XYZ_low[Z]= Int_End[end][Z];   func_XYZ_hi[Z]=Int_Begin[begin][Z]; }
        else
          { func_XYZ_low[Z]= Int_Begin[begin][Z]; func_XYZ_hi[Z]=Int_End[end][Z];   }

        evaluate_interval( Func, Int_Lenght[i] );

        if ( ((*cs_hi_top) * (*cs_low_top)) > 0.0)  /* does not contain 0 */
           { i--;  begin--; end--; }
        else
        {
          if (Int_Lenght[i] > (Isosrf->accuracy))
          {
            Int_Lenght[i+1]= (Int_Lenght[i]/= 2.0);

            Int_t_begin[i+1]= Int_t_begin[i];
            tmp= (Int_t_end[i]+Int_t_begin[i]) / 2.0;
            Int_t_end[i+1]= Int_t_begin[i]= tmp; 
            i++;
       
            Assign_Vector(Int_Begin[begin+1],Int_Begin[begin]);
            Assign_Vector(Int_End[end+1],Int_End[end]);
            end++;                  

            tmp= (Int_End[end][X] + Int_Begin[begin][X]) / 2.0;
            Int_Begin[begin][X]= tmp;
            Int_End[end][X]= tmp;

            tmp= ((Int_End[end][Y]) + (Int_Begin[begin][Y])) / 2.0;
            Int_Begin[begin][Y]= tmp;
            Int_End[end][Y]= tmp;

            tmp= ((Int_End[end][Z]) + (Int_Begin[begin][Z])) / 2.0;
            Int_Begin[begin][Z]= tmp;
            Int_End[end][Z]= tmp;
        
            begin++;
          }
          else
          {
            VScale (IPoint, Ray->Direction, Int_t_begin[i]);
            VAddEq (IPoint, Ray->Initial);
    
            if (Point_In_Clip (IPoint, Object->Clip))
            {
                push_entry_i1(Int_t_begin[i],IPoint,Object,Side1,Depth_Stack);
                IFound=TRUE;
            }
            inf--;
            i--;
            begin--;
            end--;
          }
        }
      }
      if (IFound) Increase_Counter(stats[Ray_IsoSurface_Tests_Succeeded]);
    }
  }
  return (IFound);
}






/*****************************************************************************
*
* FUNCTION
*
*   Inside_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static int Inside_IsoSurface(IPoint, Object)
VECTOR IPoint;
OBJECT *Object;
{
  VECTOR New_Point;
  ISOSURFACE *Isosrf = (ISOSURFACE *) Object;
  DBL r;


  /* Transform the point into box space. */

  if (Isosrf->Trans != NULL)
  {
    MInvTransPoint(New_Point, IPoint, Isosrf->Trans);
  }
  else
  {
    Assign_Vector(New_Point,IPoint);
  }

  if (Isosrf->bound_shape)
  {
    VSub(VTmp, Isosrf->bounds[0], IPoint);
    VDot(r, VTmp, VTmp);
    if (r > Sqr(Isosrf->bounds[1][X])) 
        return (Test_Flag(Isosrf, INVERTED_FLAG));
  }
  else
  {
    /* Test to see if we are outside the box. */
    if ((New_Point[X] < Isosrf->bounds[0][X]) || (New_Point[X] > Isosrf->bounds[1][X]))
      return (Test_Flag(Isosrf, INVERTED_FLAG));
    if ((New_Point[Y] < Isosrf->bounds[0][Y]) || (New_Point[Y] > Isosrf->bounds[1][Y]))
      return (Test_Flag(Isosrf, INVERTED_FLAG));
    if ((New_Point[Z] < Isosrf->bounds[0][Z]) || (New_Point[Z] > Isosrf->bounds[1][Z]))
      return (Test_Flag(Isosrf, INVERTED_FLAG));
  }

  if (Isosrf->Func->iso_func)
  {
    if (iso_func(Isosrf->Func,New_Point)>0) 
/*    if ((Isosrf->Func->iso_func)(Isosrf->Func, New_Point) > Isosrf->Func->threshold )
*/
    return (Test_Flag(Isosrf, INVERTED_FLAG));
   }

  /* Inside the box. */

  return (!Test_Flag(Isosrf, INVERTED_FLAG));
}



/*****************************************************************************
*
* FUNCTION
*
*   IsoSurface_Normal
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void IsoSurface_Normal(Result, Object, Inter)
OBJECT *Object;
VECTOR Result;
INTERSECTION *Inter;
{
  VECTOR New_Point, TPoint;
  ISOSURFACE *Isosrf = (ISOSURFACE *) Object;
  FUNCTION *Func=((ISOSURFACE *)Object)->Func;
  DBL func;

    switch (Inter->i1)
  {
    case SIDE_X_0: Make_Vector(Result, -1.0,  0.0,  0.0); break;
    case SIDE_X_1: Make_Vector(Result,  1.0,  0.0,  0.0); break;
    case SIDE_Y_0: Make_Vector(Result,  0.0, -1.0,  0.0); break;
    case SIDE_Y_1: Make_Vector(Result,  0.0,  1.0,  0.0); break;
    case SIDE_Z_0: Make_Vector(Result,  0.0,  0.0, -1.0); break;
    case SIDE_Z_1: Make_Vector(Result,  0.0,  0.0,  1.0); break;

    default: 

     /* Transform the point into the isosurface space */
     if (((ISOSURFACE *)Object)->Trans != NULL)
        MInvTransPoint(New_Point, Inter->IPoint, Isosrf->Trans);
     else Assign_Vector(New_Point, Inter->IPoint);
#ifdef IsoBlobPatch
     /* Modifications by Lummox JR, July 1999 */
     if(Isosrf->normal_type>0)
       {
       Assign_Vector(func_XYZ, New_Point);
       evaluate_normal(Func,Isosrf->accuracy);
       if(Func->sign>0) {Make_Vector(Result,norm_stack_top->x,norm_stack_top->y,norm_stack_top->z);}
       else {Make_Vector(Result,-norm_stack_top->x,-norm_stack_top->y,-norm_stack_top->z);}
       }
     /* This block was originally there except for the "else" */
     else if (Func->iso_func)
       {
       Assign_Vector(TPoint, New_Point);
       func=(Func->sign)*(Func->iso_func)(Func, TPoint);
       Assign_Vector(TPoint, New_Point);   TPoint[X]+= Isosrf->accuracy;
       Result[X]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
       Assign_Vector(TPoint, New_Point);   TPoint[Y]+= Isosrf->accuracy;
       Result[Y]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
       Assign_Vector(TPoint, New_Point);   TPoint[Z]+= Isosrf->accuracy;
       Result[Z]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
       }
     /* End Lummox JR's modifications */

  #else
     if (Isosrf->bound_shape)
     {
         VSub(Result, New_Point, Isosrf->bounds[0]);
         VLength(func, Result);
         if (fabs(func - Isosrf->bounds[1][X])<EPSILON)
         {
           VInverseScaleEq(Result, Isosrf->bounds[1][X]);
           break;
         }
     }

     if (Func->iso_func)
     {
       Assign_Vector(TPoint, New_Point);
       func=(Func->sign)*(Func->iso_func)(Func, TPoint);
       Assign_Vector(TPoint, New_Point);   TPoint[X]+= Isosrf->accuracy;
       Result[X]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
       Assign_Vector(TPoint, New_Point);   TPoint[Y]+= Isosrf->accuracy;
       Result[Y]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
       Assign_Vector(TPoint, New_Point);   TPoint[Z]+= Isosrf->accuracy;
       Result[Z]=Func->sign * (Func->iso_func)(Func, TPoint)-func;
      }
      #endif
     if ((Result[X]==0)&&(Result[Y]==0)&&(Result[Z]==0)) Result[X]=1.0;
     VNormalize(Result, Result);
  }


  /* Transform the point into the boxes space. */

  if (((ISOSURFACE *)Object)->Trans != NULL)
  {
    MTransNormal(Result, Result, ((ISOSURFACE *)Object)->Trans);

    VNormalize(Result, Result);
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Translate_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Translate_IsoSurface(Object, Vector, Trans)
OBJECT *Object;
VECTOR Vector;
TRANSFORM *Trans;
{
  Transform_IsoSurface(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Rotate_IsoSurface(Object, Vector, Trans)
OBJECT *Object;
VECTOR Vector;
TRANSFORM *Trans;
{
  Transform_IsoSurface(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Scale_IsoSurface(Object, Vector, Trans)
OBJECT *Object;
VECTOR Vector;
TRANSFORM *Trans;
{  
/* R.S.  Aug. 15 '96 */
  Transform_IsoSurface(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Invert_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Invert_IsoSurface(Object)
OBJECT *Object;
{
  Invert_Flag(Object, INVERTED_FLAG);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Transform_IsoSurface(Object, Trans)
OBJECT *Object;
TRANSFORM *Trans;
{
  ISOSURFACE *Isosrf = (ISOSURFACE *)Object;

  if (Isosrf->Trans == NULL)
  {
    Isosrf->Trans = Create_Transform();
  }

  Compose_Transforms(Isosrf->Trans, Trans);

  Compute_IsoSurface_BBox(Isosrf);
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

ISOSURFACE *Create_IsoSurface()
{
  ISOSURFACE *New;

  New = (ISOSURFACE *)POV_MALLOC(sizeof(ISOSURFACE), "Isosrf");

  INIT_OBJECT_FIELDS(New, ISOSURFACE_OBJECT, &IsoSurface_Methods)

  Make_Vector(New->bounds[0], -1.0, -1.0, -1.0);
  Make_Vector(New->bounds[1],  1.0,  1.0,  1.0);

  Make_BBox(New->BBox, -1.0, -1.0, -1.0, 2.0, 2.0, 2.0);

  New->Trans = NULL;
  New->Lib = NULL;
  New->max_trace=1;
  New->accuracy=0.001;
  New->Inv3=1;
  New->gradient_flag=1;
  New->grad_parm[0]=1.1;  
  New->grad_parm[1]=1.4;
  New->grad_parm[2]=0.99;
  New->Eval=0;
  New->closed=1;
  New->cache=FALSE;
  New->Inverted = FALSE;
  New->Func = NULL;
  New->method = 0;
  New->bound_shape = 0;
#ifdef IsoBlobPatch
  New->normal_type = 0; /* Lummox JR, July 1999 */
#endif
  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void *Copy_IsoSurface(Object)
OBJECT *Object;
{
  ISOSURFACE *New;

  New  = Create_IsoSurface();

  /* Copy Isosrf. */

  *New = *((ISOSURFACE *)Object);
  
  New->Trans = Copy_Transform(((ISOSURFACE *)Object)->Trans);

  if (((ISOSURFACE *)Object)->Lib != NULL) 
  {
      New->Lib= POV_MALLOC (sizeof(LIBRARY), "library struct");
      *(New->Lib)=*(((ISOSURFACE *)Object)->Lib);
      Load_Lib(New->Lib, ((ISOSURFACE *)Object)->Lib->lib_name);
  }
  New->Func=Copy_Function( ((ISOSURFACE *)Object)->Func );

  Load_Function(New->Func, New->Func->func_name);

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/
void Destroy_IsoSurface(Object)
OBJECT *Object;
{
	ISOSURFACE *Iso=(ISOSURFACE*)Object;
   Destroy_Transform(((ISOSURFACE *)Object)->Trans);
	/*YS sept 17 2000 Memory leak */
	/*YS oct 1 2000 fixed more 
		don't destroy functions before stats are printed out.
		When stage is STAGE_CLEANUP_PARSE we can destroy
		the declared functions. But if this function is called from Destroy_Frame()
		we don't destroy the functions. This will be done later on.
	*/
   if ( Iso->Func && Stage==STAGE_CLEANUP_PARSE)
	{
   		Destroy_Function(Iso->Func);
		POV_FREE(Iso->Func); 
		Iso->Func=NULL;
	}
   POV_FREE (Object);
}

/*****************************************************************************
*
* FUNCTION
*
*   Compute_IsoSurface_BBox
*
* INPUT
*
*   ISOSURFACE - IsoSurface
*
* OUTPUT
*
*   ISOSURFACE
*
* RETURNS
*
* AUTHOR
*
* DESCRIPTION
*
*   Calculate the bounding box of an Isosurface.
*
* CHANGES
*
******************************************************************************/

void Compute_IsoSurface_BBox(IsoSurface)
ISOSURFACE *IsoSurface;
{

  if (IsoSurface->bound_shape)
  {
      Make_BBox(IsoSurface->BBox, IsoSurface->bounds[0][X]-IsoSurface->bounds[1][X], 
                               IsoSurface->bounds[0][Y] - IsoSurface->bounds[1][X], 
                               IsoSurface->bounds[0][Z] - IsoSurface->bounds[1][X],
               IsoSurface->bounds[1][X]*2, 
               IsoSurface->bounds[1][X]*2,
               IsoSurface->bounds[1][X]*2);
  }
  else
  {
       Assign_BBox_Vect(IsoSurface->BBox.Lower_Left, IsoSurface->bounds[0]);
       VSub(IsoSurface->BBox.Lengths, IsoSurface->bounds[1], IsoSurface->bounds[0]);
  }

  if (IsoSurface->Trans != NULL)
  {
    Recompute_BBox(&IsoSurface->BBox, IsoSurface->Trans);
  }
}


/*****************************************************************************
*
* FUNCTION
*
*   iso_find_root
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

int iso_find_root(ISOSRF, P, D, Depth1, Depth2, Side1, Side2)
ISOSURFACE *ISOSRF;
VECTOR P, D;
DBL *Depth1, *Depth2;
int *Side1, *Side2;
{
  DBL dt,t21, l_b, l_e;
  ISO_Pair EP1, EP2;

      VLength(ISOSRF->Vlength, D);

    if (ISOSRF->cache) 
    {
        Increase_Counter(stats[Ray_IsoSurface_Cache]);
        VEvaluateRay(VTmp, P, *Depth1, D); 
        VSubEq(VTmp, ISOSRF->P);
        VLength(l_b, VTmp);
        VEvaluateRay(VTmp, P, *Depth2, D); 
        VSubEq(VTmp, ISOSRF->D);
        VLength(l_e, VTmp);
        if ((ISOSRF->fmax -    ISOSRF->Func->Max_gradient*max(l_b, l_e)) > 0.) 
        {
            Increase_Counter(stats[Ray_IsoSurface_Cache_Succeeded]);
            return(FALSE);
        }
    }

    Assign_Vector(ISOSRF->P,P);
    Assign_Vector(ISOSRF->D,D);

    ISOSRF->cache=FALSE;
    EP1.t=*Depth1; EP1.f=iso_fn(ISOSRF, Depth1);
    ISOSRF->fmax=EP1.f;
    if ((ISOSRF->closed==0)&&(EP1.f < 0.)) ISOSRF->Inv3*=-1;

    EP2.t=*Depth2; EP2.f=iso_fn(ISOSRF, Depth2);
    ISOSRF->fmax = min(EP2.f, ISOSRF->fmax) ;
 
    t21=(*Depth2-*Depth1) ;
    if ((ISOSRF->gradient_flag)&&(ISOSRF->Func->Max_gradient > ISOSRF->grad_parm[0]))
         ISOSRF->Func->Max_gradient*=ISOSRF->grad_parm[2];
    dt=ISOSRF->Func->Max_gradient* ISOSRF->Vlength * t21;
    if (iso_find_root_R(ISOSRF, &EP1, &EP2,dt,t21)) 
    {
        *Depth1=ISOSRF->tl;
        return(TRUE);
    }
    else if (!Shadow_Test_Flag) 
    {
        ISOSRF->cache=TRUE;
        VEvaluateRay(ISOSRF->P, P, EP1.t, D); 
        VEvaluateRay(ISOSRF->D, P, EP2.t, D); 
        return(FALSE);
    }
    return(FALSE);
}

/*****************************************************************************
*
* FUNCTION
*
*   iso_find_root_R
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/
int iso_find_root_R (ISOSRF, EP1, EP2, dt, t21)
ISOSURFACE *ISOSRF;
ISO_Pair *EP1, *EP2;
DBL dt,t21;
{
    ISO_Pair EPa;
    DBL  temp;

    if (ISOSRF->Eval)    {
       temp = fabs((EP2->f - EP1->f)/(t21*ISOSRF->Vlength));
       if (ISOSRF->Func->gradient < temp) ISOSRF->Func->gradient = temp;
       if (ISOSRF->gradient_flag) 
       {
          if (ISOSRF->Func->Max_gradient < temp*ISOSRF->grad_parm[1]) 
          { 
              ISOSRF->Func->Max_gradient=temp*ISOSRF->grad_parm[1]*ISOSRF->grad_parm[1]; 
              dt=ISOSRF->Func->Max_gradient* ISOSRF->Vlength * t21;
          }
       }
    }

    if (t21< ISOSRF->accuracy){
        if(EP2->f < 0) {ISOSRF->tl= EP2->t;return TRUE;}
        else return FALSE; 
    }
    if ((EP1->f + EP2->f -dt)< 0) {
        t21*=.5; dt*=.5;
        EPa.t=EP1->t + t21;
        EPa.f=iso_fn(ISOSRF, &EPa.t);
        if (Shadow_Test_Flag && (EPa.f<0)) {ISOSRF->tl=EPa.t;return TRUE;}
        ISOSRF->fmax= min(EPa.f, ISOSRF->fmax);
        if(!iso_find_root_R(ISOSRF,EP1,&EPa,dt,t21)) 
                     return(iso_find_root_R(ISOSRF,&EPa,EP2,dt,t21));
        else return TRUE;        
    }
    else return FALSE;
}


/*****************************************************************************
*
* FUNCTION
*
*   iso_fn
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

#ifdef _WIN32
__inline DBL iso_fn(ISOSRF, t)
#else
DBL iso_fn(ISOSRF, t)
#endif
ISOSURFACE *ISOSRF;
DBL *t;
{
    FUNCTION *Func=ISOSRF->Func;

    VEvaluateRay(VTmp, ISOSRF->P, *t, ISOSRF->D);
    return(
        (DBL)(ISOSRF->Inv3)*(Func->sign)
          *( (Func->iso_func) (Func, VTmp)- Func->threshold) );
}

/*****************************************************************************
*
* FUNCTION
*
*   iso_func
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/
#ifdef _WIN32
__inline DBL iso_func(Func, VPos)
#else
DBL iso_func(Func, VPos)
#endif
FUNCTION *Func;
VECTOR VPos;
{
	DBL temp=(Func->iso_func)(Func,VPos);
  
  return((Func->sign)*( temp- Func->threshold) );
/*  return((Func->sign)*( (Func->iso_func) (Func, VPos)- Func->threshold) ); */
}



/*****************************************************************************
* FUNCTION
*   Load_Lib
* INPUT
* OUTPUT
* RETURNS
* AUTHOR
*   R. Suzuki
* DESCRIPTION
*   -
* CHANGES
*   -
******************************************************************************/
/*YS initialisation routine i_dat3d */
/*this one modified by David Sharp to also read in .df3 files*/
DBL i_dat3dinitialize(LIBRARY *Lib) 
{
  DBL ret=0;
  unsigned long i,im3;/* dfs-- =(int)Lib->parm[0]*(int)Lib->parm[1]*(int)Lib->parm[2];*/
  short j;
  float f,*f3d;
  FILE *INFILE;
  void  *flg;
  long lj;
  /*dfs--*/
  int h;
  int df3datasize=0;
  unsigned char df3header[6]; /* bugfix David Sharp */
  /*--dfs*/
  /*dfs--12/23/99*/
  int LibParm3=(int)Lib->parm[3];
  POV_ARRAY *a;
  /*--dfs*/
  /*dfs--12/23/99*/
  im3=(int)Lib->parm[0]*(int)Lib->parm[1]*(int)Lib->parm[2];
  if(im3==0){
       Warning(0, "i_dat3d sample size (x_samples*y_samples*z_samples) is zero.");
       if(Lib->parm[0]==0.0)
          Lib->parm[0]=1.0;
       if(Lib->parm[1]==0.0)
          Lib->parm[1]=1.0;
       if(Lib->parm[2]==0.0)
          Lib->parm[2]=1.0;
       im3=(int)Lib->parm[0]*(int)Lib->parm[1]*(int)Lib->parm[2];
  }
  if(Lib->pClientDataSource==PCLIENTDATA_FILE){
  /*--dfs*/
    if (Lib->pClient) 
    {
       if (LibParm3==0)
          flg=(INFILE=fopen((char *)(Lib->pClient),"rt"));
       else
          flg=(INFILE=fopen((char *)(Lib->pClient),"rb"));
        if (flg!=NULL)
        {
            /*dfs--*/
        if (LibParm3==5 ||LibParm3==6)
        {
           /*read df3 file header*/
            df3datasize=1;
            for(h=0;h<3;h++)
            {
              df3header[2*h]=fgetc(INFILE);
              df3header[2*h+1]=fgetc(INFILE);
              df3datasize*=(  (df3header[2*h] << 8)| (df3header[2*h+1])  );
            }
          }
          /*--dfs*/
          im3=df3datasize>im3?df3datasize:im3;
          f3d = (float*)POV_MALLOC(sizeof(float)*(im3+3),"Lib");
          Lib->pClientData = f3d;
          for (i=0;i<im3;i++) 
          {
            switch (LibParm3)
             {
              case 0:  fscanf(INFILE,"%f",&f);  break;
              case 1:  f=(float)fgetc(INFILE);  break;
              case 2:  fread(&j,2,1,INFILE); f=j;break;
              case 3:  fread(&lj,4,1,INFILE); f=(float)lj; break;
              case 4:  fread(&f,4,1,INFILE);break;
              /*dfs--*/
              case 5:
              case 6:  f=((float)fgetc(INFILE))/255.0; break;
              /*--dfs*/
             }
             f3d[i+3]=f;
          }
          fclose(INFILE);
          /*dfs--*/
          if(LibParm3==6)
          {
             f3d[0]=(float) (  (df3header[0] << 8) |(df3header[1])  );
             f3d[1]=(float) (  (df3header[2] << 8) |(df3header[3])  );
             f3d[2]=(float) (  (df3header[4] << 8) |(df3header[5])  );
          }  /*--dfs*/
          else
          {
             f3d[0]=(float)Lib->parm[0];
             f3d[1]=(float)Lib->parm[1];
             f3d[2]=(float)Lib->parm[2];
          }
          ret=0.;
         }
         else 
         {
          Error("Error opening file '%s' Can not initialise i_dat3d library!\n",(char *)(Lib->pClient));
           return ret;
       }
    }
  /*dfs--12/23/99*/
  }
  
  else if(Lib->pClientDataSource==PCLIENTDATA_ARRAY){
       a=(POV_ARRAY *) Lib->pClient;
       if(a->Type==FLOAT_ID_TOKEN){

          f3d = (float*)POV_MALLOC(sizeof(float)*(im3+3),"Lib");
          Lib->pClientData = f3d;
          if(im3>a->Total){
              Error("Array is smaller than i_dat3d sample dimensions");
          }
          for (i=0;i<im3;i++) {
                if(a->DataPtrs[i]!=NULL){
                    f=(float)(*(double *)(a->DataPtrs[i]));
                    f3d[i+3]=f;
                }
                else
                   Error("Attempt to access uninitialized array element");
          }
          f3d[0]=(float)Lib->parm[0];
          f3d[1]=(float)Lib->parm[1];
          f3d[2]=(float)Lib->parm[2];
          ret=0.0;
       }
       else{
         Error("Array data for i_dat3d is not numerical");
       }
  }
  /*--dfs*/
  return ret;
}
/*YS finish routine i_dat3d */
DBL i_dat3dfinish(LIBRARY *Lib) 
{
	if (Lib->pClientData) 
	POV_FREE (Lib->pClientData);   
	return 0.;
}

/*YS quick and dirty way of doing things*/
#define NUMLIBS 3		 

typedef struct {
	char libname[30];
	INIT_FUNC initfunc;
	INIT_FUNC finishfunc;
}initlibstruct;

initlibstruct istruct[NUMLIBS]={
                                              {"i_dat3d",i_dat3dinitialize,i_dat3dfinish},
                                              {"i_algbr",0,0},
                                              {"i_nfunc",0,0}
                                              };

void Init_Lib(LIBRARY *Lib)
{
	int x;
	short found=FALSE;
	short ret=FALSE;
	for (x=0; x<NUMLIBS; x++)
	{
		if ( strcmp(Lib->lib_name,istruct[x].libname)==0)
		{
			if (istruct[x].initfunc!=0)  /*lib has an init routine*/
				ret=(int)istruct[x].initfunc(Lib);
			found=TRUE;
			break;
		}
	}
 	if ( !found )
 	   Error ("library '%s' not build in.\n",Lib->lib_name);
	if ( ret)
		Error("Error while initialising library %s\n",Lib->lib_name);
}
/*****************************************************************************
* FUNCTION
*   Finish_Lib
* INPUT
* OUTPUT
* RETURNS
* AUTHOR
*   R. Suzuki
* DESCRIPTION
*   -
* CHANGES
*   -
******************************************************************************/
void Finish_Lib(LIBRARY *Lib)
{
	int x;
	for (x=0; x<NUMLIBS; x++)
	{
		if ( strcmp(Lib->lib_name,istruct[x].libname)==0)
		{
			if (istruct[x].finishfunc!=0)  /*lib has an init routine*/
			{
				(int)istruct[x].finishfunc(Lib);
				break;
			}
		}
	}
    /*dfs--12/23/99*/
    if(Lib->pClientDataSource==PCLIENTDATA_FILE)
     /*--dfs*/
    if (Lib->pClient) 
      POV_FREE(Lib->pClient); 
}

void Load_Lib(LIBRARY *Lib,char *libname)
{

  Lib->lib_name = POV_MALLOC (strlen(libname) + 1, "lib_name");
  Lib->pnum=0;
  Lib->parm=NULL;
  Lib->pClient=NULL;
  Lib->pClientData=NULL;
  /*dfs--12/23/99*/
  Lib->pClientDataSource=0;
  /*--dfs*/
  strcpy (Lib->lib_name, libname);
}


/*****************************************************************************
*
* FUNCTION
*
*   Load_Function
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*   RS May '97
*   -
*
******************************************************************************/
void Load_Function(FUNCTION *Func, char *funcname)
{
	ISO_FUNC find_iso_func (char * fname);

  Func->iso_func= find_iso_func(funcname);
  if (!Func->iso_func) 
    Error ("Function '%s' not build in! Cannot load function %s.",funcname,funcname);
  if (strcmp(funcname, "imp_func")==0)   
    Func->isosf = NULL; /*Func->isosf = ISOSRF;*/
}


#else
	static char dmm[2];
#endif
