#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <linux/types.h>
#include <linux/kd.h>
#include <linux/keyboard.h>
#include "ksyms.h"

int use_names = 1;
static int fd;

int get_bind(unsigned char index, unsigned char table) {
	struct kbentry ke;
	ke.kb_index = index;
	ke.kb_table = table;
	if (ioctl(fd, KDGKBENT, (unsigned long)&ke)) {
		fprintf(stderr, "KDGKBENT at index %d in table %d", index, table);
		perror("");
		exit(1);
	}
	return ke.kb_value;
}

void print_keysym(int code) {
	int t;
	int v;
	const char *fmt = " %-16s";
	char *p;
	static char tmp[4] = "' '";

	t = KTYP(code);
	v = KVAL(code);
	if (t == KT_META) {
		t = 0;
		fmt = " Meta-%-11s";
	}
	if (t < syms_size && v < syms[t].size && (p = syms[t].table[v])[0]) {
		if (!use_names && !t && v >= 32 && v < 127) {
			tmp[1] = v;
			p = tmp;
		}
		printf(fmt, p);
	} else
		printf(" 0x%04x          ", code);
}

void dump_symbols(void) {
	int t;
	int v;
	char *p;

	printf("# symbols recognized by dumpkeys\n");
	for (t = 0; t < syms_size; t++)
		for (v = 0; v < syms[t].size; v++)
			if ((p = syms[t].table[v])[0])
				printf("%s\n", p);
}

void usage(void) {
	fprintf(stderr, "usage: dumpkeys [-s]\n");
	exit(1);
}

void print_mod(int x) {
	if (x & (1 << KG_SHIFT))
		printf("shift   ");
	if (x & (1 << KG_ALTGR))
		printf("altgr   ");
	if (x & (1 << KG_CTRL))
		printf("control ");
	if (x & (1 << KG_ALT))
		printf("alt     ");
}

void print_bind(int buf[], int i, int j) {
	printf("\t");
	print_mod(j);
	printf("keycode %3d =", i);
	print_keysym(buf[j]);
	printf("\n");
}

main (int argc, char *argv[]) {
	int i;
	int j;
	int k;
	int c;
	int count;
	int vcount;
	int vmax;
	int vlast;
	struct kbsentry fbuf;
	int buf[NR_KEYMAPS];
	char *p;
	char dump = 0;

	extern int optind;

	while ((c = getopt(argc, argv, "sn")) != -1) {
		switch (c) {
			case 's':
				dump = 1;
				break;
			case 'n':
				use_names = 0;
				break;
			case '?':
				usage();
		}
	}

	if (optind < argc)
		usage();

	if (dump) {
		dump_symbols();
		exit(0);
	}

	if ((fd = open("/dev/console", O_RDONLY)) < 0) {
		perror("when opening /dev/console");
		exit(1);
	}

	for (i = 0; i < NR_KEYS; i++) {
		for (j = 0; j < NR_KEYMAPS; j++)
			buf[j] = get_bind(i, j);

		for (j = NR_KEYMAPS - 1; j >= 0; j--)
			if (buf[j] != K_HOLE)
				break;
		vmax = j;
		vlast = j;
		vcount = 0;
		for (j; j > 0; j--) {
			if (buf[j] != K_HOLE)
				vcount++;
			else if (vmax - j + 1 > 2 * vcount)
				vlast = j - 1;
		}

		count = 0;
		for (j = 1; j < NR_KEYMAPS; j++)
			if (buf[0] != buf[j])
				count++;

		printf("keycode %3d =", i);
		if (
			buf[0] >= 'a' && buf[0] <= 'z' &&
			buf[1] == buf[0] - 32 &&
			buf[4] == buf[0] - 96 &&
			buf[5] == buf[0] - 96 &&
			KTYP(buf[8]) == KT_META && KVAL(buf[8]) == buf[0] &&
			KTYP(buf[9]) == KT_META && KVAL(buf[9]) == buf[0]-32 &&
			KTYP(buf[12]) == KT_META && KVAL(buf[12]) == buf[0]-96 &&
			KTYP(buf[13]) == KT_META && KVAL(buf[13]) == buf[0]-96
		) {
			print_keysym(buf[0]);
			printf("\n");
			for (j = 2; j < NR_KEYMAPS; j++) {
				switch (j) {
					default:
						if (buf[j] != K_HOLE)
							print_bind(buf, i, j);
					case 4: case 5:
					case 8: case 9:
					case 12: case 13:
				}
			}
		} else if (count < NR_KEYMAPS / 2) {
			if (buf[0] != K_HOLE)
				print_keysym(buf[0]);
			printf("\n");
			for (j = 1; j < NR_KEYMAPS; j++)
				if (buf[0] != buf[j])
					print_bind(buf, i, j);
		} else {
			for (k = 0; k <= vlast; k++)
				print_keysym(buf[k]);
			printf("\n");
			for (; k <= vmax; k++)
				if (buf[k] != K_HOLE)
					print_bind(buf, i, k);
		}
	}
	for (i = 0; i < NR_FUNC; i++) {
		fbuf.kb_func = i;
		if (ioctl(fd, KDGKBSENT, (unsigned long)&fbuf)) {
			fprintf(stderr, "KDGKBSENT at index %d", i);
			perror("");
			exit(1);
		}
		printf("string %s = \"", syms[KT_FN].table[i]);
		for (p = fbuf.kb_string; *p; p++)
			if (isgraph(*p))
				putchar(*p);
			else
				printf("\\%03o", *p);
		printf("\"\n");
	}
	exit(0);
}
