/*
 *   Copyright (C) International Business Machines Corp., 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/* defines and includes common among the fsck.jfs modules */
#include "xfsckint.h"
#include <errno.h>

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * superblock buffer pointer
 *
 *      defined in xchkdsk.c
 */
extern struct superblock *sb_ptr;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * fsck aggregate info structure pointer
 *
 *      defined in xchkdsk.c
 */
extern struct fsck_agg_record *agg_recptr;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * For message processing
 *
 *      defined in xchkdsk.c
 */

extern char *verbose_msg_ptr;

extern char *Vol_Label;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * Device information.
 *
 *      defined in xchkdsk.c
 */
extern HFILE Dev_IOPort;
extern uint32_t Dev_blksize;

/* VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
 *
 * The following are internal to this file
 *
 */

/*
 * Since access to the directory index may not be sequential, keep a number
 * of pages resident
 */
#define NUM_INDEX_BUFS 16

struct dir_index_page {
	long long address;
	struct dir_index_page *next;
	struct dir_index_page *prev;
	struct dir_table_slot table[512];
};

struct dir_index_page *dir_index_buffers;

struct dir_index_page *free_di_page;	/* free list */
struct dir_index_page *lru_di_page;	/* least-recently-used */
struct dir_index_page *mru_di_page;	/* most-recently-used */

int allocate_dir_index_buffers(void)
{
	int i;
	struct dir_index_page *page, *last;

	dir_index_buffers = malloc(sizeof(struct dir_index_page) *
				   NUM_INDEX_BUFS);
	if (! dir_index_buffers) {
		fsck_send_msg(MSG_OSO_INSUFF_MEMORY);
		return ENOMEM;
	}

	page = dir_index_buffers;
	last = NULL;

	for (i = 0; i < NUM_INDEX_BUFS; i++) {
		page->address = 0;
		page->next = last;
		page->prev = NULL;	/* Not used on free list */
		last = page;
		page++;
	}
	free_di_page = last;
	lru_di_page = mru_di_page = NULL;

	return FSCK_OK;
}


static struct dir_table_slot *read_index_page(long long address)
{
	struct dir_index_page *page;
	int rc;

	/* Search cache.  Is it worthwhile to build a hash? */
	for (page = mru_di_page; page; page = page->next) {
		if (page->address == address) {
			/* hit! */
			if (page != mru_di_page) {
				/* remove from linked list */
				if (page == lru_di_page)
					lru_di_page = page->prev;
				else
					page->next->prev = page->prev;
				page->prev->next = page->next;
				/* add back at front (mru) */
				mru_di_page->prev = page;
				page->next = mru_di_page;
				page->prev = NULL;
				mru_di_page = page;
			}
			return page->table;
		}
	}
	if (free_di_page) {
		page = free_di_page;
		free_di_page = page->next;
	} else {
		page = lru_di_page;
		lru_di_page = page->prev;
		lru_di_page->next = NULL;
	}
	page->address = address;
	rc = ujfs_rw_diskblocks(Dev_IOPort, address << sb_ptr->s_l2bsize,
				PSIZE, page->table, GET);
	if (rc) {
		page->next = free_di_page;
		free_di_page = page;
		return NULL;
	}

	/* Insert into lru list */
	page->next = mru_di_page;
	page->prev = NULL;
	mru_di_page = page;
	if (page->next)
		page->next->prev = page;
	else
		lru_di_page = page;

	return page->table;
}

#ifdef NOT_YET
static int write_index_page(struct dir_table_slot *table)
{
	int rc;

	/* Should only be called on most recent page */
	if (table != mru_di_page->table) {
		/* TODO: replace with more appropriate message & rc */
		printf("You messed up, Shaggy.  write_index_page mismatch.\n");
		return EIO;
	}

	rc = ujfs_rw_diskblocks(Dev_IOPort,
				mru_di_page->address << sb_ptr->s_l2bsize,
				PSIZE, table, PUT);

	return rc;
}
#endif

void verify_dir_index(struct fsck_inode_record *inorecptr,
		      struct dinode *inoptr,
		      struct dtreeQelem *this_Qel,
		      int dtindex,
		      uint cookie)
{
	int64_t blkno;
	int8_t match_found;
	int64_t offset;
	int64_t page_addr;
	int rc;
	struct dir_table_slot *slot;
	struct dir_table_slot *table;
	xad_t *xad_ptr;

	if (!inorecptr->check_dir_index)
		return;

	/*
	 * FIXME: Don't treat cookie == 0 as an error right now since the
	 * file system will fix this at runtime.  When fsck rebuilds the
	 * index entirely, we will want to go ahead and rebuild if a zero
	 * is found.
	 */
	if (cookie == 0)
		return;

	if ((cookie < 2) || (cookie >= inoptr->di_next_index)) {
		fsck_send_msg(fsck_BAD_COOKIE);
		goto error_found;
		return;
	}

	if (inoptr->di_next_index <= (MAX_INLINE_DIRTABLE_ENTRY + 1)) {
		table = inoptr->di_dirtable;
		slot = &table[cookie - 2];
	} else {
		offset = (cookie - 2) * sizeof (struct dir_table_slot);
		blkno = (offset >> L2PSIZE) << (L2PSIZE - sb_ptr->s_l2bsize);
		rc = xTree_search(inoptr, blkno, &xad_ptr, &match_found);
		if (rc || !match_found) {
			fsck_send_msg(fsck_PAGE_NOT_FOUND);
			goto error_found;
		}
		table = read_index_page(addressXAD(xad_ptr));
		if (!table) {
			fsck_send_msg(fsck_READ_FAILED);
			goto error_found;
		}
		slot = &table[(cookie - 2) % 512];
	}

	if (this_Qel->node_level)
		page_addr = this_Qel->node_addr;
	else
		page_addr = 0;

	if ((slot->flag == 0) || (slot->slot != dtindex) ||
	    (addressDTS(slot) != page_addr)) {
		fsck_send_msg(fsck_BAD_ENTRY);
		goto error_found;
	}
	return;

error_found:
	inorecptr->check_dir_index = 0;
	inorecptr->rebuild_dirtable = 1;
	agg_recptr->corrections_needed = 1;
	return;
}
