/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gul-string.h"
#include "js-console.h"
#include "nsCOMPtr.h"
#include "JSConsoleListener.h"
#include "xpconnect/nsIScriptError.h"
#include "nsString.h"
#include "mozilla-embed-shell.h"

#include <glib/gi18n.h>

/* Implementation file */
NS_IMPL_ISUPPORTS1(JSConsoleListener, nsIConsoleListener)

JSConsoleListener::JSConsoleListener()
{
  NS_INIT_ISUPPORTS();
  /* member initializers and constructor code */
}

JSConsoleListener::~JSConsoleListener()
{
  /* destructor code */
}

/* void observe (in nsIConsoleMessage aMessage); */
NS_IMETHODIMP JSConsoleListener::Observe(nsIConsoleMessage *aMessage)
{
	PRUnichar *utmp;
	PRUint32 itmp;
	guint sline_offset = 0;
	GString *code_ref;
	gchar *msg;
	gchar *error_message;

	aMessage->GetMessage (&utmp);

	msg = gul_string_strip_newline (NS_ConvertUCS2toUTF8(utmp).get());
	nsMemory::Free (utmp);

	nsCOMPtr<nsIScriptError> error = do_QueryInterface (aMessage);
	if (error) /* best interface, more infos */
	{
		/* build a message */
		PRUint32 flags;
		JSConsoleMessageType type;
		error->GetFlags (&flags);

		if (flags == nsIScriptError::errorFlag ||
		    flags == nsIScriptError::exceptionFlag ||
		    flags == nsIScriptError::strictFlag)
		{
			type = JS_CONSOLE_ERROR;
		}
		else if (flags == nsIScriptError::warningFlag)
		{
			type = JS_CONSOLE_WARNING;
		}
		else
		{
			type = JS_CONSOLE_MESSAGE;
		}

		code_ref = g_string_new(NULL);

		error->GetSourceName (&utmp);
		if (utmp) 
		{
			char *tmp;
			tmp = gul_string_strip_newline (NS_ConvertUCS2toUTF8(utmp).get());

			if (strcmp(tmp,"") != 0)
			{
				code_ref = g_string_append (code_ref, "In ");
				code_ref = g_string_append (code_ref, tmp);
				code_ref = g_string_append (code_ref, ", ");
			}

			g_free (tmp);
			nsMemory::Free (utmp);
		}

		error->GetLineNumber (&itmp);
		if (itmp)
		{
			char *tmp;
			tmp = g_strdup_printf ("%d", itmp);
			code_ref = g_string_append (code_ref, "Line ");
			code_ref = g_string_append (code_ref, tmp);
			code_ref = g_string_append (code_ref, ", ");
			g_free (tmp);
		}

		error->GetSourceLine (&utmp);
		if (utmp) 
		{
			sline_offset = strlen (code_ref->str);
			
			char *tmp;
			tmp = gul_string_strip_newline (NS_ConvertUCS2toUTF8(utmp).get());
			
			code_ref = g_string_append (code_ref, tmp);
			if (strlen (tmp) > 0)
				code_ref = g_string_append (code_ref, "\n");
			g_free (tmp);
			nsMemory::Free (utmp);
		}
		
		sline_offset = strlen (msg) + sline_offset + 1;
		error->GetColumnNumber (&itmp);
		error_message = g_strconcat (msg, 
					     "\n",
					     code_ref->str,
					     NULL);
		javascript_console_append (JAVASCRIPT_CONSOLE(mozilla_js_console),
				   	   error_message, type, 
				   	   sline_offset + itmp);
		g_free (error_message);

		g_string_free (code_ref, TRUE);
	}
	else
	{
		javascript_console_append (JAVASCRIPT_CONSOLE(mozilla_js_console), msg, 
				           JS_CONSOLE_MESSAGE, 0);
	}

	g_free (msg);

	return NS_OK;
}
