/*****************************************************************

Copyright (c) 2001 John Firebaugh <jfirebaugh@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qlayout.h>
#include <qtimer.h>
#include <kdebug.h>
#include <karrowbutton.h>
#include <kwindowlistmenu.h>
#include <kconfig.h>
#include <kapplication.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "taskbar.h"
#include "taskbarcontainer.h"
#include "taskbarcontainer.moc"

#define WINDOWLISTBUTTON_SIZE 12

#define BUTTON_MIN_HEIGHT 18
#define BUTTON_HEIGHT 20
#define BUTTON_MAX_WIDTH 800
#define BUTTON_MIN_WIDTH 20

TaskBarContainer::TaskBarContainer( bool enableFrame, QWidget *parent, const char *name )
    : FittsLawFrame( parent, name )
    , direction( KPanelApplet::Up )
    , showWindowListButton( true )
    , windowListButton(0)
    , windowListMenu(0)
{
    setBackgroundOrigin( AncestorOrigin );

    uint margin;
    if ( enableFrame )
    {
	setFrameStyle( Sunken | StyledPanel );
	margin = frameWidth();
    }
    else
    {
	setFrameStyle( NoFrame );
	margin = 0;
    }

    layout = new QBoxLayout( this, QApplication::reverseLayout() ?
                                   QBoxLayout::RightToLeft :
                                   QBoxLayout::LeftToRight );
    layout->setMargin( margin );

    // scrollable taskbar
    taskBar = new TaskBar( this );
    layout->addWidget( taskBar );

    connect( taskBar, SIGNAL( containerCountChanged() ), SIGNAL( containerCountChanged() ) );

    // read settings and setup layout
    configure();
}

TaskBarContainer::~TaskBarContainer()
{
    delete windowListMenu;
}

void TaskBarContainer::configure()
{
    KConfig c( "ktaskbarrc", false, false );
    c.setGroup( "General" );

    setFont( c.readFontEntry( "taskbarFont" ) );
    showWindowListButton = c.readBoolEntry( "ShowWindowListBtn", false );

    if ( showWindowListButton )
    {
        // window list button
        windowListButton = new KArrowButton( this );
        windowListMenu= new KWindowListMenu;
        connect( windowListButton, SIGNAL( pressed() ), SLOT( showWindowListMenu() ) );
        connect( windowListMenu, SIGNAL( aboutToHide() ), SLOT( windowListMenuAboutToHide() ) );
        //windowListButton->setPopup(windowListMenu);
        windowListButton->setMinimumSize( WINDOWLISTBUTTON_SIZE, WINDOWLISTBUTTON_SIZE );
        layout->insertWidget( 0, windowListButton );
    }
    else
    {
        delete windowListButton;
        windowListButton = 0;
        delete windowListMenu;
        windowListMenu = 0;
    }
}

void TaskBarContainer::preferences()
{
    kapp->startServiceByDesktopName( "kcmtaskbar", QStringList(),
                                     0, 0, 0, "", true);
}

void TaskBarContainer::orientationChange( Orientation o )
{
    if( o == Horizontal ) {
        if (windowListButton)
        {
            windowListButton->setFixedWidth( WINDOWLISTBUTTON_SIZE );
            windowListButton->setMaximumHeight( BUTTON_MAX_WIDTH );
        }
        layout->setDirection( QApplication::reverseLayout() ?
                              QBoxLayout::RightToLeft :
                              QBoxLayout::LeftToRight );
    } else {
        if (windowListButton)
        {
            windowListButton->setMaximumWidth( BUTTON_MAX_WIDTH );
            windowListButton->setFixedHeight( WINDOWLISTBUTTON_SIZE );
        }
        layout->setDirection( QBoxLayout::TopToBottom );
    }
    taskBar->setOrientation( o );
    layout->activate();
}

void TaskBarContainer::popupDirectionChange( KPanelApplet::Direction d )
{
    direction = d;

    ArrowType at = UpArrow;

    switch( d ) {
        case KPanelApplet::Up:
            at = UpArrow;
            break;
        case KPanelApplet::Down:
            at = DownArrow;
            break;
        case KPanelApplet::Left:
            at = LeftArrow;
            break;
        case KPanelApplet::Right:
            at = RightArrow;
            break;
    }
    taskBar->setArrowType( at );

    if (windowListButton)
        windowListButton->setArrowType( at );
}

void TaskBarContainer::showWindowListMenu()
{
    if (!windowListMenu)
        return;

    windowListMenu->init();

    // calc popup menu position
    QPoint pos( mapToGlobal( QPoint(0,0) ) );

    switch( direction ) {
        case KPanelApplet::Right:
            pos.setX( pos.x() + width() );
            break;
        case KPanelApplet::Left:
            pos.setX( pos.x() - windowListMenu->sizeHint().width() );
            break;
        case KPanelApplet::Down:
            pos.setY( pos.y() + height() );
            break;
        case KPanelApplet::Up:
            pos.setY( pos.y() - windowListMenu->sizeHint().height() );
        default:
            break;
    }

    disconnect( windowListButton, SIGNAL( pressed() ), this, SLOT( showWindowListMenu() ) );
    windowListMenu->exec( pos );
    QTimer::singleShot(100, this, SLOT(reconnectWindowListButton()));
}

void TaskBarContainer::windowListMenuAboutToHide()
{
    // this ensures that when clicked AGAIN, the window list button doesn't cause the
    // window list menu to show again. usability, you see. hoorah.
    windowListButton->setDown( false );
}

void TaskBarContainer::reconnectWindowListButton()
{
    connect( windowListButton, SIGNAL( pressed() ), SLOT( showWindowListMenu() ) );
}

QSize TaskBarContainer::sizeHint( KPanelExtension::Position p, QSize maxSize) const
{
    QSize size = taskBar->sizeHint( p, maxSize );
    if ( (p == KPanelExtension::Left || p == KPanelExtension::Right) && showWindowListButton ) {
        return QSize( size.width(), size.height() + WINDOWLISTBUTTON_SIZE );
    }
    return size;
}
