/***********************************************************************
 *
 * PROJECT:	  PMake
 * MODULE:	  Customs -- SunOS 4.x dependencies
 * FILE:	  os-sunos4.c
 *
 * AUTHOR:  	  Adam de Boor: Sep  9, 1989
 *
 * ROUTINES:
 *	Name	  	    Description
 *	----	  	    -----------
 *	OS_Init	    	    Initialize module and return mask of criteria
 *	    	    	    to be considered in determination.
 *	OS_Exit	    	    Undo any initialization.
 *	OS_Idle	    	    Return machine idle time, in seconds
 *	OS_Load	    	    Return machine load factor
 *	OS_Swap	    	    Return free swap space
 *	OS_Proc	    	    Return number of free processes
 *
 * REVISION HISTORY:
 *	Date	  Name	    Description
 *	----	  ----	    -----------
 *	9/ 9/89	  ardeb	    Initial version
 *	9/91      stolcke   Free processes support added
 *	2/92      stolcke   Verbose error and log messages added
 *	2/92      stolcke   OS_Exit added
 *
 * DESCRIPTION:
 *	OS-dependent functions for SunOS 4.x.
 *
 *	These functions are responsible for the determination of the
 *	current state of the system, returning it as a set of numbers
 *	in a customs-standard form, whatever form they may be in in the
 *	system itself.
 *
 *	The format used is the same as that transmitted for the AVAIL_SET
 *	RPC.
 *
 * 	Copyright (c) Berkeley Softworks 1989
 * 	Copyright (c) Adam de Boor 1989
 *
 * 	Permission to use, copy, modify, and distribute this
 * 	software and its documentation for any non-commercial purpose
 *	and without fee is hereby granted, provided that the above copyright
 * 	notice appears in all copies.  Neither Berkeley Softworks nor
 * 	Adam de Boor makes any representations about the suitability of this
 * 	software for any purpose.  It is provided "as is" without
 * 	express or implied warranty.
 *
 *
 ***********************************************************************/
#ifndef lint
static char *rcsid =
"$Id: os-sunos4.c,v 1.14 1994/10/22 02:37:09 stolcke Exp $ ICSI (Berkeley)";
#endif /* not lint */

#include    <sys/types.h>
#include    <sys/param.h>
#include    <nlist.h>
#include    <fcntl.h>
#include    <sys/stat.h>
#include    <vm/anon.h>
#include    <kvm.h>
#include    <sys/time.h>
#include    <sys/proc.h>
#include    <time.h>
#include    <sys/file.h>
#include    <utmp.h>

#include    "customsInt.h"

static struct nlist kAddrs[] = {
{	"_avenrun"  },	    /* Load averages	    	    	*/
#define AVENRUN	    0
{	"_nproc"    },      /* Max. number of processes		*/
#define NPROC       1
#ifndef NO_SWAP
{	"_anoninfo" },	    /* Swap space stats	    	    	*/
#define ANONINFO    2
#endif /* NO_SWAP */
{	""  	    }
};

#define DEV_CONSOLE	"/dev/console"
#define DEV_KEYBOARD    "/dev/kbd"
#define DEV_MOUSE	"/dev/mouse"
#define UTMP_FILE	"/etc/utmp"

static kvm_t	*kmem = NULL;	 /* handle for reading /dev/kmem */
static int	utmpf;		 /* Descriptor to /etc/utmp */



/***********************************************************************
 *				OS_Init
 ***********************************************************************
 * SYNOPSIS:	    Initialize this module
 * CALLED BY:	    Avail_Init
 * RETURN:	    Mask of bits indicating what things we can check
 * SIDE EFFECTS:    kmem and kbd are set. kAddrs filled in.
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *
 ***********************************************************************/
int
OS_Init()
{
    int	retMask = AVAIL_EVERYTHING;

    /*
     * Open the utmp file.  If it's not there we cannot determine idle times.
     */
    if ((utmpf = open (UTMP_FILE, O_RDONLY, 0)) < 0) {
	/*
	 * If couldn't open/stat keyboard, we can't tell how long the machine's
	 * been idle.
	 */
	xlog (XLOG_WARNING,
		"OS_Init: cannot open or stat %s: disabling idle time check", 
		UTMP_FILE
	     );
	retMask &= ~AVAIL_IDLE;
    }

    /*
     * Open the kernel. If that fails, kvm_open will have given the
     * error for us, so we can just exit.
     */
    kmem = kvm_open(NULL, NULL, NULL, O_RDONLY, NULL);
    if (kmem == NULL) {
	xlog (XLOG_WARNING,
		"OS_Init: cannot read kernel: disabling load/swap/proc checks");
	return (retMask & ~(AVAIL_LOAD | AVAIL_SWAP | AVAIL_PROC));
    }

    /*
     * Locate the other structures in the kernel that we need.
     */
    switch (kvm_nlist(kmem, kAddrs)) {
	case -1:
	    xlog (XLOG_WARNING,
		    "OS_Init: cannot locate kernel symbols: disabling load/swap/proc checks");
	    retMask &= ~(AVAIL_LOAD | AVAIL_SWAP | AVAIL_PROC);
	    break;
	case 0:
	    /*
	     * No symbols unfound -- this is good.
	     */
	    break;
	default:
	    if (kAddrs[AVENRUN].n_type == 0) {
		/*
		 * Couldn't find _avenrun symbol, so can't determine load
		 * average.
		 */
	        xlog (XLOG_WARNING,
			"OS_Init: cannot locate %s: disabling load check",
			kAddrs[AVENRUN].n_name);
		retMask &= ~AVAIL_LOAD;
	    }
	    if (kAddrs[NPROC].n_type == 0) {
		/*
		 * Couldn't find process count symbols, so can't determine free
		 * process info.
		 */
	        xlog (XLOG_WARNING,
			"OS_Init: cannot locate %s: disabling proc check",
			kAddrs[NPROC].n_name);
		retMask &= ~AVAIL_PROC;
	    }
#ifndef NO_SWAP
	    if (kAddrs[ANONINFO].n_type == 0) {
		/*
		 * If couldn't find _anoninfo, we can't figure out the
		 * swap space situation.
		 */
	        xlog (XLOG_WARNING,
			"OS_Init: cannot locate %s: disabling swap check",
			kAddrs[ANONINFO].n_name);
		retMask &= ~AVAIL_SWAP;
	    }
#endif
	    break;
    }
#ifdef NO_SWAP
    retMask &= ~AVAIL_SWAP;
#endif
    
    return (retMask);
}

/***********************************************************************
 *				OS_Exit
 ***********************************************************************
 * SYNOPSIS:	    Deinitialize this module
 * CALLED BY:	    CustomsRestart()
 * RETURN:	    Nothing
 * SIDE EFFECTS:    kmem, kbd and mouse are closed.
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	stolcke	2/20/92		Initial Revision
 *
 ***********************************************************************/
void
OS_Exit()
{
    if (kmem) {
	kvm_close(kmem);
    }
    if (utmpf >= 0) {
	close(utmpf);
    }
    return;
}

/***********************************************************************
 *				OS_Swap
 ***********************************************************************
 * SYNOPSIS:	    Find the percentage of the system's swap space that
 *	    	    isn't being used.
 * CALLED BY:	    Avail_Local
 * RETURN:	    The percentage of free swap space
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *	The number of blocks is kept in the "anoninfo" structure in the
 *	kernel. The numbers are actually given in pages, but we only
 *	deal with percentages, so we don't care.
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *
 ***********************************************************************/
int
OS_Swap()
{
#ifdef NO_SWAP
    return (0);
#else /* !NO_SWAP */
    struct anoninfo ani;
    int free;

    if (kvm_read(kmem,
		 kAddrs[ANONINFO].n_value,
		 (char *)&ani,
		 sizeof(ani)) != sizeof(ani))
    {
	/*
	 * Couldn't find out -- assume worst case.
	 */
	xlog (XLOG_ERROR,
		"OS_Swap: can no longer read %s", kAddrs[ANONINFO].n_name);
	return (0);
    }

    /*
     * Convert free to percentage and return it.
     * 10/27/91: apparently, ani_resv+ani_free != ani_max, and it's
     * when ani_resv == ani_max that the system stops allowing
     * memory allocations, so base the percentage on the amount
     * left over after the reserved pages are taken out, rather than
     * on the pages actually free. -- ardeb
     */
    free = ((ani.ani_max-ani.ani_resv) * 100) / ani.ani_max;

    if (verbose)
	xlog (XLOG_DEBUG, "OS_Swap: max swap = %d, reserved = %d (%d%% free)",
		ani.ani_max, ani.ani_resv, free);

    return (free);
#endif /* NO_SWAP */
}


/***********************************************************************
 *				OS_Proc
 ***********************************************************************
 * SYNOPSIS:	    Find the number of additional processes that can be
 *	    	    created.
 * CALLED BY:	    Avail_Local
 * RETURN:	    Number of free process slots
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *	Count current no. of processes by scanning process list,
 *	then subtract from max. no. of processes.
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	stolcke	9/21/91		Initial Revision
 *
 ***********************************************************************/
int
OS_Proc()
{
    int cproc = 0;
    int nproc;
    struct proc *p;

    if (kvm_read(kmem,
		 kAddrs[NPROC].n_value,
		 (char *)&nproc,
		 sizeof(nproc)) != sizeof(nproc) ||
        kvm_setproc(kmem) < 0)
    {
	xlog (XLOG_ERROR,
		"OS_Proc: can no longer read %s", kAddrs[NPROC].n_name);
	return (0);
    }

    while (p = kvm_nextproc(kmem)) {
	if ( p->p_stat )
		cproc++;
    }

    if (verbose)
	xlog (XLOG_DEBUG, "OS_Proc: nproc = %d, #proc = %d, free = %d",
	       nproc, cproc, nproc - cproc);

    return (nproc - cproc);
}


/***********************************************************************
 *				OS_Load
 ***********************************************************************
 * SYNOPSIS:	    Return the current load average in standard form
 * CALLED BY:	    Avail_Local
 * RETURN:	    The current load as a 32-bit fixed-point number
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *
 ***********************************************************************/
unsigned int
OS_Load()
{
    double          result;
    unsigned long   avenrun[3];

    if (kvm_read(kmem,
		 kAddrs[AVENRUN].n_value,
		 avenrun,
		 sizeof(avenrun)) != sizeof(avenrun))
    {
	xlog (XLOG_ERROR,
		"OS_Load: can no longer read %s", kAddrs[AVENRUN].n_name);
	return (0);
    }

#define FLOAT(v) ((double)(v)/FSCALE)

#ifdef ALL_LOAD
    /*
     * Find largest of the three averages and return that
     */
    if (avenrun[0] > avenrun[1]) {
	if (avenrun[0] > avenrun[2]) {
	    result = FLOAT(avenrun[0]);
	} else {
	    result = FLOAT(avenrun[2]);
	}
    } else if (avenrun[1] > avenrun[2]) {
	result = FLOAT(avenrun[1]);
    } else {
	result = FLOAT(avenrun[2]);
    }
#else
    /*
     * Just return the 1-minute average.
     */
    result = FLOAT(avenrun[0]);
#endif
    if (verbose)
	xlog (XLOG_DEBUG, "OS_Load: avenrun = %.2lf, %.2lf, %.2lf",
		FLOAT(avenrun[0]), FLOAT(avenrun[1]), FLOAT(avenrun[2]));

    return (result * LOADSCALE);
}

/*
 * Use generic code for OS_Idle()
 */
#include "os-idle.c"
