/*
 * manpath.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.  
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Changed PATH->manpath algorithm
 * Added: an empty string in MANPATH denotes the system path
 * Added: use LANG to search in /usr/man/<locale>
 * Lots of other minor things, including spoiling the indentation.
 * aeb - 940315
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "config.h"
#include "gripes.h"
#include "man.h"

#ifdef STDC_HEADERS
#include <stdlib.h>
#else
extern int fprintf ();
extern int strcmp ();
extern int strncmp ();
extern char *memcpy ();
extern char *getenv();
extern char *malloc();
extern void free ();
extern int exit ();
#endif

extern int is_directory ();

extern int debug;

extern struct dirs cfdirlist;
extern struct dir srchlist;	/* linked list, 1st entry unused */

/*
 * Add a directory to the manpath list if it isn't already there.
 */
void
add_to_srchlist (dir, lang, perrs)
     char *dir;
     char *lang;
     int perrs;
{
    add_to_list(dir, lang, perrs); /* first try with $LANG */
    add_to_list(dir, (char *) 0, perrs); /* then without */
}

add_to_list (dir, lang, perrs)
     char *dir;
     char *lang;
     int perrs;
{
  int status;
  struct dir *dp;

  if (!lang)
    lang = "";

  /* only add absolute paths */
  if (*dir != '/') {
      char cwd[BUFSIZ];

      if (!getcwd(cwd, sizeof(cwd)))
	return; /* cwd not readable, or pathname very long */
      if (strlen(dir) + strlen(lang) + strlen(cwd) + 3 > sizeof(cwd))
	return;
      if (!strncmp (dir, "./", 2))
	dir += 2;
      if (!strncmp (dir, "../", 3)) {
	  dir += 3;
	  *rindex (cwd, '/') = 0;
      }
      strcat (cwd, "/");
      strcat (cwd, dir);
      if (*lang) {
	  strcat (cwd, "/");
	  strcat (cwd, lang);
      }
      dir = cwd;
  } else if (*lang) {
      char cwd[BUFSIZ];

      strcpy (cwd, dir);
      strcat (cwd, "/");
      strcat (cwd, lang);
      dir = cwd;
  }

  dp = &srchlist;
  while (dp->nxt) {
      dp = dp->nxt;
      if (!strcmp (dp->dir, dir)) {
	  if (debug)
	    fprintf (stderr, "but %s is already in the manpath\n", dir);
	  return;
      }
  }

  /*
   * Not found -- add it.
   */
  status = is_directory(dir);

  if (status < 0 && perrs) {
      fprintf (stderr, "Warning: couldn't stat file %s!\n", dir);
  }
  else if (status == 0 && perrs) {
      fprintf (stderr, "Warning: %s isn't a directory!\n", dir);
  }
  else if (status == 1) {
      if (debug)
	fprintf (stderr, "adding %s to manpath\n", dir);

      dp->nxt = (struct dir *) malloc(sizeof(struct dir));
      if (!dp->nxt)
	gripe_alloc (sizeof(struct dir), "dp");
      dp = dp->nxt;
      dp->nxt = 0;
      dp->dir = strdup (dir);
  }
}

void
to_srchlist(s, perrs)
     char *s;
     int perrs;
{
  void get_manpath ();
  char *lang, *path;

  if (*s) {
      add_to_srchlist (s, (char *) 0, perrs);
      return;
  }

  /* empty string: insert default path */
  lang = getenv("LANG");
  path = getenv ("PATH");
  get_manpath (path, lang, perrs);
}

/*
 * Input: a non-NULL string, with : as separator
 * For each entry in the string, call dirfn.
 */
void
split (pathstring, dirfn, perrs)
     char *pathstring;
     void (*dirfn)();
     int perrs;
{
    register char *p, *q, *r;

    p = strdup(pathstring);
    for (q = p; ; ) {
	r = index(q, ':');
	if (r) {
	    *r = 0;
	    dirfn (q, perrs);
	    q = r+1;
	} else {
	    dirfn (q, perrs);
	    break;
	}
    }
    free (p);
}
	
/*
 * If the environment variable MANPATH is set, return it.
 * If the environment variable PATH is set and has a nonzero length,
 * try to determine the corresponding manpath, otherwise, return the
 * default manpath.
 *
 * The man.config file is used to map system wide bin directories
 * to top level man page directories.
 *
 * For directories which are in the user's path but not in the
 * man.config file, see if there is a subdirectory `man' or `MAN'.
 * If so, add that directory to the path.  Example:  user has
 * $HOME/bin in his path and the directory $HOME/bin/man exists -- the
 * directory $HOME/bin/man will be added to the manpath.
 */

/*
 * Use user-specified path, or MANPATH, or "" as manpath.
 * An empty string denotes the path derived from PATH and man.config
 * Put directories in srchlist, and also return a string.
 */
void
manpath (manp, perrs)
     char *manp;
     int perrs;
{
  if (manp == NULL && (manp = getenv ("MANPATH")) == NULL)
    manp = "";

  split (manp, to_srchlist, perrs);
}

prmanpath ()
{
  register struct dir *dp, *dp0;
    
  dp0 = dp = srchlist.nxt;
  while (dp) {
      if (dp != dp0)
	printf(":");
      printf("%s", dp->dir);
      dp = dp->nxt;
  }
  printf("\n");
}

/*
 * For each directory in the user's path, see if it is one of the
 * directories listed in the man.config file.  If so, and it is
 * not already in the manpath, add it.  If the directory is not listed
 * in the manpath.config file, see if there is a subdirectory `man' or
 * `MAN'.  If so, and it is not already in the manpath, add it.
 *
 * Example:  user has <dir>/bin in his path and the directory
 * <dir>/bin/man exists -- the directory <dir>/bin/man will be added
 * to the manpath.
 * Try also <dir>/man ?and <dir>?, and, if LANG is set, <dir>/$LANG/man.
 * aeb - 940320
 */
void
get_manpath (path, lang, perrs)
     char *path, *lang;
     int perrs;
{
  register int len;
  register char *t;
  register char *p;
  register char *end;
  register struct dir *dp;
  register struct dirs *dlp;
  void add_to_srchlist ();
  char *find_man_subdir ();

  if (path) {
    for (p = strdup(path); ; p = end+1) {
      if ((end = strchr(p, ':')) != NULL)
	*end = '\0';
	  
      if (debug)
	fprintf (stderr, "\npath directory %s ", p);
	  
      /*
       * The directory we're working on is in the config file.
       * If we haven't added it to the list yet, do.
       */
      if (*p)
	for (dlp = cfdirlist.nxt; dlp; dlp = dlp->nxt)
	  if (!strcmp (p, dlp->bindir)) {
	      if (debug)
		fprintf (stderr, "is in the config file\n");
		  
	      add_to_srchlist (dlp->mandir, lang, perrs);
	      goto found;
	  }
	  
      /*
       * The directory we're working on isn't in the config file.  See if it
       * has man or MAN subdirectories.  If so, and this subdirectory hasn't
       * been added to the list, do. (Try also a few other places nearby.)
       */
      if (debug)
	fprintf (stderr, "is not in the config file\n");
	  
      t = find_man_subdir (p);
      if (t != NULL) {
	  if (debug)
	    fprintf (stderr, "but there is a man directory nearby\n");
	      
	  add_to_srchlist (t, lang, perrs);
	  free (t);
      } else {
	  if (debug)
	    fprintf (stderr, "and we found no man directory nearby\n");
      }
	  
    found:
	  
      if (!end)
	break;
    }
  }

  if (debug)
    fprintf (stderr, "\nadding mandatory man directories\n\n");

  for (dlp = cfdirlist.nxt; dlp; dlp = dlp->nxt)
    if (dlp->mandatory)
	add_to_srchlist (dlp->mandir, lang, perrs);

}

/*
 * Check to see if the current directory has man or MAN
 * or ../man or ../man1 or ../man8 subdirectories. 
 */
char *
find_man_subdir (p)
     register char *p;
{
  int len;
  register char *t, *sp;

  len = strlen (p);

  t = malloc ((unsigned) len + 20);
  if (t == NULL)
    gripe_alloc (len+20, "p\n");

  memcpy (t, p, len);
  strcpy (t + len, "/man");
  
  if (is_directory (t) == 1)
    return t;

  strcpy (t + len, "/MAN");
  
  if (is_directory (t) == 1)
    return t;

  /* find parent directory */
  t[len] = 0;
  if (sp = rindex (t, '/')) {
      *sp = 0;
      len = sp - t;
  } else {
      strcpy (t + len, "/..");
      len += 3;
  }

  /* look for the situation with packagedir/bin and packagedir/man */
  strcpy (t + len, "/man");

  if (is_directory (t) == 1)
    return t;

  /* look for the situation with pkg/bin and pkg/man1 or pkg/man8 */
  /* (looking for all man[1-9] would probably be a waste of stats) */
  strcpy (t + len, "/man1");

  if (is_directory (t) == 1) {
      t[len] = 0;
      return t;
  }

  strcpy (t + len, "/man8");

  if (is_directory (t) == 1) {
      t[len] = 0;
      return t;
  }

  return NULL;
}
