/*
 * ldd - print shared library dependencies
 *
 * usage: ldd [-vV] prog ...
 *        -v: print ldd version
 *        -V: print ld.so version
 *        prog ...: programs to check
 *
 * Copyright 1993, David Engel
 *
 * This program may be used for any purpose as long as this
 * copyright notice is kept.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <a.out.h>
#include <errno.h>
#include <sys/wait.h>
#include "config.h"

/* see if prog is a binary file */
int is_bin(char *argv0, char *prog)
{
    int res = 0;
    FILE *file;
    struct exec exec;

    /* must be able to open it for reading */
    if ((file = fopen(prog, "rb")) == NULL)
	fprintf(stderr, "%s: can't open %s (%s)\n", argv0, prog,
		strerror(errno));
    else
    {
	/* then see if it's Z, Q or OMAGIC */
	if (fread(&exec, sizeof exec, 1, file) < 1)
	    fprintf(stderr, "%s: can't read header from %s\n", argv0, prog);
	else if (N_MAGIC(exec) != ZMAGIC && N_MAGIC(exec) != QMAGIC &&
                 N_MAGIC(exec) != OMAGIC)
	    fprintf(stderr, "%s: %s is not {Z,Q,O}MAGIC\n", argv0, prog);
	else
	    res = 1; /* looks good */

	fclose(file);
    }

    return res;
}

int main(int argc, char **argv, char **envp)
{
    int i;
    int vprinted = 0;
    /* this must be volatile to work with -O, GCC bug? */
    volatile loadptr loader = (loadptr)LDSO_ADDR;
  
    while ((i = getopt(argc, argv, "vV")) != EOF)
	switch (i)
	{
	case 'v':
	    /* print our version number */
	    printf("%s: version %s\n", argv[0], VERSION);
	    vprinted = 1;
	    break;
	case 'V':
	    /* print the version number of ld.so */
	    if (uselib(LDSO_IMAGE))
	    {
		fprintf(stderr, "%s: can't load dynamic linker %s (%s)\n",
			argv[0], LDSO_IMAGE, strerror(errno));
		exit(1);
	    }
	    loader(FUNC_VERS, NULL);
	    vprinted = 1;
	    break;
	}

    /* must specify programs is -v or -V not used */
    if (optind >= argc && !vprinted)
    {
	printf("usage: %s [-vV] prog ...\n", argv[0]);
	exit(0);
    }

    /* print the dependencies for each program */
    for (i = optind; i < argc; i++)
    {
	pid_t pid;
	char buff[1024];

	/* make sure it's a binary file */
	if (!is_bin(argv[0], argv[i]))
	    continue;

	/* print the program name if doing more than one */
	if (optind < argc-1)
	{
	    printf("%s:\n", argv[i]);
	    fflush(stdout);
	}

	/* save the name in the enviroment, ld.so may need it */
	sprintf(buff, "%s=%s", LDD_ARGV0, argv[i]);
	putenv(buff);

	/* now fork and exec prog with argc = 0 */
	if ((pid = fork()) < 0)
	{
	    sprintf(buff, "%s: can't fork (%s)", argv[0], strerror(errno));
	    exit(1);
	}
	else if (pid == 0)
	{
	    execl(argv[i], NULL);
	    sprintf(buff, "%s: can't execute %s (%s)", argv[0], argv[i],
		    strerror(errno));
	    exit(1);
	}
	else
	{
	    /* then wait for it to complete */
	    waitpid(pid, NULL, 0);
	}
    }

    exit(0);
}
