------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 4                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.83 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Exp_Ch3;  use Exp_Ch3;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Util; use Exp_Util;
with Itypes;   use Itypes;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Ch5;  use Sem_Ch5;
with Sinfo;    use Sinfo;
with Sinfo.CN; use Sinfo.CN;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Exp_Ch4 is

   ------------------------
   --  Local Subprograms --
   ------------------------

   function Expand_Array_Equality
     (Loc : Source_Ptr; Typ : Entity_Id; Lhs, Rhs : Node_Id) return Node_Id;
   --  Expand an array equality into an expression-action containing a local
   --  function implementing this equality, and a call to it. Loc is the
   --  location for the generated nodes. Typ is the type of the array, and
   --  Lhs, Rhs are the array expressions to be compared.

   procedure Expand_Boolean_Operator (N : Node_Id);
   --  Common expansion processing for Boolean operators (And, Or, Xor)

   procedure Expand_Comparison_Operator (N : Node_Id);
   --  This routine handles expansion of the comparison operators (N_Op_Lt,
   --  N_Op_Le, N_Op_Gt, N_Op_Ge). Since the code is basicallly similar with
   --  the addition of some outer

   function Expand_Composite_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id;
   --  Local recursive function used to expand equality for nested
   --  composite types. Used by Expand_Record_Equality, Expand_Array_Equality.

   procedure Expand_Concatenation (Node : Node_Id; Ops : List_Id);
   --  This routine handles expansion of concatenation operations, where
   --  N is the N_Op_Concat or N_Concat_Multiple node being expanded, and
   --  Ops is the list of operands (at least two are present).

   function Make_Array_Comparison_Op
     (Typ : Entity_Id; Loc : Source_Ptr; Equal : Boolean) return Node_Id;
   --  Comparisons between arrays are expanded in line. This function
   --  produces the body of the implementation of (a > b), or (a >= b), when
   --  a and b are one-dimensional arrays of some discrete type. The original
   --  node is then expanded into the appropriate call to this function.

   function Make_Boolean_Array_Op (N : Node_Id) return Node_Id;
   --  Boolean operations on boolean arrays are expanded in line. This
   --  function produce the body for (a and b), (a or b), or (a xor b).

   function Tagged_Membership (N : Node_Id) return Node_Id;
   --  Construct the expression corresponding to the tagged membership test.
   --  Deals with a second operand being (or not) a class-wide type.

   ---------------------------
   -- Expand_Array_Equality --
   ---------------------------

   --  Expand an equality function for multi-dimentionnal arrays. Here is
   --  an example of such a function for Nb_Dimension = 2

   --  function Enn (A : arr; B : arr) return boolean is
   --     J1 : integer := B'first (1);
   --     J2 : integer := B'first (2);

   --  begin
   --     if A'length (1) /= B'length (1) then
   --        return false;
   --     else
   --        for I1 in A'first (1) .. A'last (1) loop
   --           if A'length (2) /= B'length (2) then
   --              return false;
   --           else
   --              for I2 in A'first (2) .. A'last (2) loop
   --                 if A (I1, I2) /=  B (J1, J2) then
   --                    return false;
   --                 end if;
   --                 J2 := Integer'succ (J2);
   --              end loop;
   --           end if;
   --           J1 := Integer'succ (J1);
   --        end loop;
   --     end if;
   --     return true;
   --  end Enn;

   function Expand_Array_Equality
     (Loc      : Source_Ptr;
      Typ      : Entity_Id;
      Lhs, Rhs : Node_Id)
      return     Node_Id
   is
      Decls       : List_Id := New_List;
      Index_List1 : List_Id := New_List;
      Index_List2 : List_Id := New_List;
      Index       : Entity_Id := First_Index (Typ);
      Index_Type  : Entity_Id;
      Formals     : List_Id;
      Result      : Node_Id;
      Stats       : Node_Id;
      Func_Name   : Entity_Id;
      Func_Body   : Node_Id;

      A : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uA);
      B : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uB);

      function Component_Equality (Typ : Entity_Id) return Node_Id;
      --  Create one statement to compare corresponding components, designated
      --  by a full set of indices.

      function Loop_One_Dimension (N : Int) return Node_Id;
      --  Loop over the n'th dimension of the arrays. The single statement
      --  in the body of the loop is a loop over the next dimension, or
      --  the comparison of corresponding components.

      ------------------------
      -- Component_Equality --
      ------------------------

      function Component_Equality (Typ : Entity_Id) return Node_Id is
         Test : Node_Id;
         L, R : Node_Id;

      begin
         --  if a(i1...) /= b(j1...) then return false; end if;

         L := Make_Indexed_Component (Loc,
                Prefix => Make_Identifier (Loc, Chars (A)),
                Expressions => Index_List1);

         R := Make_Indexed_Component (Loc,
                Prefix => Make_Identifier (Loc, Chars (B)),
                Expressions => Index_List2);

         Test := Expand_Composite_Equality (Loc, Component_Type (Typ), L, R);

         return
           Make_If_Statement (Loc,
             Condition => Make_Op_Not (Loc, Right_Opnd => Test),
             Then_Statements => New_List (
               Make_Return_Statement (Loc,
                 Expression => New_Occurrence_Of (Standard_False, Loc))));

      end Component_Equality;

      ------------------------
      -- Loop_One_Dimension --
      ------------------------

      function Loop_One_Dimension (N : Int) return Node_Id is
         I : constant Entity_Id := Make_Defining_Identifier (Loc,
                                                  New_Internal_Name ('I'));
         J : constant Entity_Id := Make_Defining_Identifier (Loc,
                                                  New_Internal_Name ('J'));
         Stats : Node_Id;

      begin
         if N > Number_Dimensions (Typ) then
            return Component_Equality (Typ);

         else
            --  Generate the following:

            --  j: index_type := b'first (n);
            --  ...
            --  if a'length (n) /= b'length (n) then
            --    return false;
            --  else
            --     for i in a'range (n) loop
            --        --  loop over remaining dimensions.
            --        j := index_type'succ (j);
            --     end loop;
            --  end if;

            --  retrieve index type for current dimension.

            Index_Type := Base_Type (Etype (Index));
            Append (New_Reference_To (I, Loc), Index_List1);
            Append (New_Reference_To (J, Loc), Index_List2);

            --  Declare index for j as a local variable to the function.
            --  Index i is a loop variable.

            Append_To (Decls,
              Make_Object_Declaration (Loc,
                Defining_Identifier => J,
                Object_Definition   => New_Reference_To (Index_Type, Loc),
                Expression =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (B, Loc),
                    Attribute_Name => Name_First,
                    Expressions => New_List (
                        Make_Integer_Literal (Loc, UI_From_Int (N))))));

            Stats :=
              Make_If_Statement (Loc,
                Condition =>
                  Make_Op_Ne (Loc,
                    Left_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (A, Loc),
                        Attribute_Name => Name_Length,
                        Expressions => New_List (
                          Make_Integer_Literal (Loc, UI_From_Int (N)))),
                    Right_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (B, Loc),
                        Attribute_Name => Name_Length,
                        Expressions => New_List (
                          Make_Integer_Literal (Loc, UI_From_Int (N))))),

                Then_Statements => New_List (
                  Make_Return_Statement (Loc,
                    Expression => New_Occurrence_Of (Standard_False, Loc))),

                Else_Statements => New_List (
                  Make_Loop_Statement (Loc,
                    Identifier => Empty,
                    Iteration_Scheme =>
                      Make_Iteration_Scheme (Loc,
                        Loop_Parameter_Specification =>
                          Make_Loop_Parameter_Specification (Loc,
                            Defining_Identifier => I,
                            Discrete_Subtype_Definition =>
                              Make_Attribute_Reference (Loc,
                                Prefix => New_Reference_To (A, Loc),
                                Attribute_Name => Name_Range,
                                Expressions => New_List (
                                  Make_Integer_Literal (Loc,
                                    Intval => UI_From_Int (N)))))),

                    Statements => New_List (
                      Loop_One_Dimension (N + 1),
                      Make_Assignment_Statement (Loc,
                        Name => New_Reference_To (J, Loc),
                        Expression =>
                          Make_Attribute_Reference (Loc,
                            Prefix => New_Reference_To (Index_Type, Loc),
                            Attribute_Name => Name_Succ,
                            Expressions => New_List (
                              New_Reference_To (J, Loc))))))));

            Index := Next_Index (Index);
            return Stats;
         end if;
      end Loop_One_Dimension;

   ------------------------------------------
   -- Processing for Expand_Array_Equality --
   ------------------------------------------

   begin
      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => A,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => B,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      Func_Name := Make_Defining_Identifier (Loc,  New_Internal_Name ('E'));

      Stats := Loop_One_Dimension (1);

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark => New_Reference_To (Standard_Boolean, Loc)),
          Declarations               =>  Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Stats,
                Make_Return_Statement (Loc,
                  Expression => New_Occurrence_Of (Standard_True, Loc)))));

         Set_Has_Completion (Func_Name, True);

         Result :=
           Make_Expression_Actions (Loc,
             Actions    => New_List (Func_Body),
             Expression => Make_Function_Call (Loc,
               Name => New_Reference_To (Func_Name, Loc),
               Parameter_Associations => New_List (Lhs, Rhs)));

         return Result;
   end Expand_Array_Equality;

   -----------------------------
   -- Expand_Boolean_Operator --
   -----------------------------

   --  Expansion happens only for the array type cases. The expansion is
   --  to an expression actions node that declares a function to perform
   --  the desired operation, followed by a call to it. The construction
   --  of the function is performed by Make_Boolean_Array_Op.

   procedure Expand_Boolean_Operator (N : Node_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Typ       : constant Entity_Id  := Etype (N);
      Result    : Node_Id;
      Func_Body : Node_Id;
      Func_Name : Entity_Id;

   begin
      if Is_Array_Type (Typ) then
         Func_Body := Make_Boolean_Array_Op (N);
         Func_Name := Defining_Unit_Name (Specification (Func_Body));

         Result :=
           Make_Expression_Actions (Loc,
             Actions => New_List (Func_Body),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (Func_Name, Loc),
                 Parameter_Associations =>
                   New_List (Left_Opnd (N), Right_Opnd (N))));

         Replace_Substitute_Tree (N, Result);
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_Boolean_Operator;

   ---------------------------------
   -- Expand_Class_Wide_Allocator --
   ---------------------------------

   --  The Node N is assumed to be a N_Allocator with a N_Qualified_Expression
   --  in its expression field

   procedure Expand_Class_Wide_Allocator (
     N                  : Node_Id;
     Acc_Type           : Entity_Id;
     Class_Wide_Subtype : Entity_Id)
   is
      Exp        : constant Node_Id := Expression (Expression (N));
      Loc        : constant Source_Ptr := Sloc (N);
      List_Decl  : List_Id;
      Equiv_Type : Entity_Id;
      New_N      : Node_Id;

   begin
      --  Expand_Class_Wide_Allocator is called directly from the semantics,
      --  so check to see whether expansion is active before proceeding

      if not Expander_Active then
         return;
      end if;

      List_Decl := Expand_Class_Wide_Subtype (Loc, Class_Wide_Subtype, Exp);
      Equiv_Type := Equivalent_Type (Class_Wide_Subtype);
      New_N :=
        Make_Expression_Actions (Loc,
          Actions => List_Decl,
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Acc_Type, Loc),
              Expression =>
                Make_Allocator (Loc,
                  Expression =>
                    Make_Qualified_Expression (Loc,
                      Subtype_Mark => New_Reference_To (Equiv_Type, Loc),
                      Expression =>
                        Make_Unchecked_Type_Conversion (Loc,
                          Subtype_Mark => New_Reference_To (Equiv_Type, Loc),
                          Expression => New_Copy (Exp))))));

      Transfer_Itypes (N, New_N);
      Rewrite_Substitute_Tree (N, New_N);
      Analyze (N);

   end Expand_Class_Wide_Allocator;

   --------------------------------
   -- Expand_Comparison_Operator --
   --------------------------------

   --  Expansion is only required in the case of array types. The form of
   --  the expansion is:

   --     [body for greater_nn; boolean_expression]

   --  The body is built by Make_Array_Comparison_Op, and the form of the
   --  Boolean expression depends on the operator involved.

   procedure Expand_Comparison_Operator (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Op1 : Node_Id             := Left_Opnd (N);
      Op2 : Node_Id             := Right_Opnd (N);
      Typ : constant Node_Id    := Base_Type (Etype (Op1));

      Result    : Node_Id;
      Expr      : Node_Id;
      Func_Body : Node_Id;
      Func_Name : Entity_Id;

   --   ??? can't Op1, Op2 be constants, aren't assignments to Op1, Op2
   --   below redundant, if not why not? RBKD

   begin
      if Is_Array_Type (Typ) then

         --  For <= the Boolean expression is
         --    greater__nn (op2, op1, true)

         if Chars (N) = Name_Op_Le then
            Func_Body := Make_Array_Comparison_Op (Typ, Loc, True);
            Op1  := Right_Opnd (N);
            Op2  := Left_Opnd  (N);

         --  For < the Boolean expression is
         --    greater__nn (op2, op1)

         elsif Chars (N) = Name_Op_Lt then
            Func_Body := Make_Array_Comparison_Op (Typ, Loc, False);
            Op1  := Right_Opnd (N);
            Op2  := Left_Opnd  (N);

         --  For >= the Boolean expression is
         --    op1 = op2 or else greater__nn (op1, op2)

         elsif Chars (N) = Name_Op_Ge then
            Func_Body := Make_Array_Comparison_Op (Typ, Loc, True);

         --  For > the Boolean expression is
         --    greater__nn (op1, op2)

         elsif Chars (N) = Name_Op_Gt then
            Func_Body := Make_Array_Comparison_Op (Typ, Loc, False);
         else
            return;
         end if;

         Func_Name := Defining_Unit_Name (Specification (Func_Body));
         Expr :=
           Make_Function_Call (Loc,
             Name => New_Reference_To (Func_Name, Loc),
             Parameter_Associations => New_List (Op1, Op2));

         Result :=
           Make_Expression_Actions (Loc,
             Actions => New_List (Func_Body),
             Expression => Expr);

         Rewrite_Substitute_Tree (N, Result);
         Analyze (N);
         Resolve (N, Standard_Boolean);
      end if;
   end Expand_Comparison_Operator;

   -------------------------------
   -- Expand_Composite_Equality --
   -------------------------------

   --  This function is only called for comparing internal fields of composite
   --  types when these fields are themselves composites. This is a special
   --  case because it is not possible to respect normal Ada visibility rules.

   function Expand_Composite_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id
   is
      Full_Type : Entity_Id;
      Prim      : Elmt_Id;

   begin
      if Ekind (Typ) in Private_Kind then
         Full_Type := Full_Declaration (Typ);
      else
         Full_Type := Typ;
      end if;

      Full_Type := Base_Type (Full_Type);

      if Is_Array_Type (Full_Type) then

         if Is_Scalar_Type (Component_Type (Full_Type)) then
            return Make_Op_Eq (Loc, Left_Opnd  => Lhs, Right_Opnd => Rhs);
         else
            return Expand_Array_Equality (Loc, Full_Type, Lhs, Rhs);
         end if;

      elsif Is_Tagged_Type (Full_Type) then

         --  Call the primitive operation "=" of this type

         if Is_Class_Wide_Type (Full_Type) then
            Full_Type := Etype (Full_Type);
         end if;

         Prim := First_Elmt (Primitive_Operations (Full_Type));

         while Chars (Node (Prim)) /= Name_Op_Eq loop
            Prim := Next_Elmt (Prim);
            pragma Assert (Present (Prim));
         end loop;

         return
           Make_Function_Call (Loc,
             Name => New_Reference_To (Node (Prim), Loc),
             Parameter_Associations => New_List (Lhs, Rhs));

      elsif Is_Record_Type (Full_Type) then
         return Expand_Record_Equality (Loc, Full_Type, Lhs, Rhs);
      else
         --  if it is neither a record nor an array, it can be the
         --  full declaration of a scalar private type for instance.

         return Make_Op_Eq (Loc, Left_Opnd => Lhs, Right_Opnd => Rhs);
      end if;
   end Expand_Composite_Equality;

   --------------------------
   -- Expand_Concatenation --
   --------------------------

   --  We construct the following expression actions node, where Atyp is
   --  the base type of the array involved and Ityp is the index type
   --  of this array:

   --    [function Cnn (S1 : Atyp; S2 : Atyp; .. Sn : Atyp) return Atyp is
   --        L : constant Ityp := S1'Length + S2'Length + ... Sn'Length;
   --        R : Atyp (S1'First .. S1'First + L - 1)
   --        P : Ityp := S1'First;
   --
   --     begin
   --        R (P .. P + S1'Length - 1) := S1;
   --        P := P + S1'Length;
   --        R (P .. P + S2'Length - 1) := S2;
   --        P := P + S2'Length;
   --        ...
   --        R (P .. P + Sn'Length - 1) := Sn;
   --        P := P + Sn'Length;
   --        return R;
   --     end Cnn;
   --
   --     Cnn (operand1, operand2, ... operandn)]

   --  Note: the low bound is not quite right, to be fixed later ???

   procedure Expand_Concatenation (Node : Node_Id; Ops : List_Id) is
      Loc   : constant Source_Ptr := Sloc (Node);
      Atyp  : constant Entity_Id  := Base_Type (Etype (Node));
      Ityp  : constant Entity_Id  := Etype (First_Index (Atyp));
      N     : constant Nat        := List_Length (Ops);

      Op    : Node_Id;
      Pspec : List_Id;
      Lexpr : Node_Id;
      Slist : List_Id;
      Alist : List_Id;
      Decls : List_Id;
      Func  : Node_Id;

      function L return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uL);
      end L;

      function Nam (J : Nat) return Name_Id is
      begin
         return New_External_Name ('S', J);
      end Nam;

      function One return Node_Id is
      begin
         return Make_Integer_Literal (Loc, Uint_1);
      end One;

      function P return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uP);
      end P;

      function R return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uR);
      end R;

      function S1first return Node_Id is
      begin
         return
           Make_Attribute_Reference (Loc,
             Prefix => Make_Identifier (Loc, Nam (1)),
             Attribute_Name => Name_First);
      end S1first;

      function Slength (J : Nat) return Node_Id is
      begin
         return
           Make_Attribute_Reference (Loc,
             Prefix => Make_Identifier (Loc, Nam (J)),
             Attribute_Name => Name_Length);
      end Slength;

   --  Start of processing for Expand_Concatenation

   begin
      --  Construct parameter specification list

      Pspec := New_List;

      for J in 1 .. N loop
         Append_To (Pspec,
           Make_Parameter_Specification (Loc,
             Defining_Identifier => Make_Defining_Identifier (Loc, Nam (J)),
             Parameter_Type => New_Reference_To (Atyp, Loc)));
      end loop;

      --  Construct expression for total length of result

      Lexpr := Slength (1);

      for J in 2 .. N loop
         Lexpr := Make_Op_Add (Loc, Lexpr, Slength (J));
      end loop;

      --  Construct list of statements

      Slist := New_List;

      for J in 1 .. N loop
         Append_To (Slist,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Slice (Loc,
                 Prefix => R,
                 Discrete_Range =>
                   Make_Range (Loc,
                     Low_Bound => P,
                     High_Bound =>
                       Make_Op_Subtract (Loc,
                         Left_Opnd  => Make_Op_Add (Loc, P, Slength (J)),
                         Right_Opnd => One))),
             Expression => Make_Identifier (Loc, Nam (J))));

         Append_To (Slist,
           Make_Assignment_Statement (Loc,
             Name       => P,
             Expression => Make_Op_Add (Loc, P, Slength (J))));
      end loop;

      Append_To (Slist, Make_Return_Statement (Loc, Expression => R));

      --  Construct list of arguments for the call

      Alist := New_List;
      Op := First (Ops);

      for J in 1 .. N loop
         Append_To (Alist, New_Copy (Op));
         Op := Next (Op);
      end loop;

      --  Construct the declarations for the function

      Decls := New_List (
        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uL),
          Object_Definition   => New_Reference_To (Ityp, Loc),
          Constant_Present    => True,
          Expression          => Lexpr),

        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uR),

          Object_Definition =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (Atyp, Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints => New_List (
                    Make_Range (Loc,
                      Low_Bound  => S1first,
                      High_Bound =>
                        Make_Op_Subtract (Loc,
                          Left_Opnd => Make_Op_Add (Loc, S1first, L),
                          Right_Opnd => One)))))),

        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uP),
          Object_Definition   => New_Reference_To (Ityp, Loc),
          Expression          => S1first));

      --  Now construct the expression actions node and do the replace

      Func := Make_Defining_Identifier (Loc, New_Internal_Name ('C'));

      Rewrite_Substitute_Tree (Node,
        Make_Expression_Actions (Loc,
          Actions => New_List (
            Make_Subprogram_Body (Loc,
              Specification =>
                Make_Function_Specification (Loc,
                  Defining_Unit_Name       => Func,
                  Parameter_Specifications => Pspec,
                  Subtype_Mark => New_Reference_To (Atyp, Loc)),
              Declarations => Decls,
              Handled_Statement_Sequence =>
                Make_Handled_Sequence_Of_Statements (Loc, Slist))),
          Expression =>
            Make_Function_Call (Loc, New_Reference_To (Func, Loc), Alist)));

      Analyze (Node);
      Resolve (Node, Atyp);
      Set_Is_Inlined (Func);
   end Expand_Concatenation;

   ------------------------
   -- Expand_N_Allocator --
   ------------------------

   --  If the allocator is for a type which requires initialization, and
   --  there is no initial value (i.e. the operand is a subtype indication
   --  rather than a qualifed expression), then we must generate a call to
   --  the initialization routine. This is done using an expression actions
   --  node:
   --
   --     [Pnnn : constant ptr_T := new (T); Init (Pnnn.all,...); Pnnn]
   --
   --  Here ptr_T is the pointer type for the allocator, and T is the
   --  subtype of the allocator. A special case arises if the designated
   --  type of the access type is a task or contains tasks. In this case
   --  the call to Init (Temp.all ...) is replaced by code that ensures
   --  that the tasks get activated (see Exp_Ch9.Build_Task_Allocate_Block
   --  for details). In addition, if the type T is a task T, then the first
   --  argument to Init must be converted to the task record type.

   procedure Expand_N_Allocator (N : Node_Id) is
      PtrT  : constant Entity_Id  := Etype (N);
      Loc   : constant Source_Ptr := Sloc (N);
      Temp  : Entity_Id;
      Node  : Node_Id;

   begin

      if Nkind (Expression (N)) = N_Qualified_Expression then
         declare
            T   : constant Entity_Id := Entity (Subtype_Mark (Expression (N)));
            Exp : constant Node_Id := Expression (Expression (N));
            Act : constant List_Id := New_List;
         begin
            if Is_Controlled (T) then

               --    output:  [
               --              Temp : constant ptr_T := new T'(Expression);
               --              Temp._tag := T'tag;
               --              Adjust (Finalizable (Temp.all));
               --              Attach_To_Final_List (Finalizable (Temp.all));
               --              Temp]

               --  we analyze by hand the new internal allocator to avoid
               --  any recursion and inappropriate call to Initialize

               Node := Relocate_Node (N);
               Set_Analyzed (Node, True);

               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

               Append_To (Act,
                 Make_Object_Declaration (Loc,
                   Defining_Identifier => Temp,
                   Constant_Present    => True,
                   Object_Definition   => New_Reference_To (PtrT, Loc),
                   Expression          => Node));

               Append_To (Act,
                 Make_Assignment_Statement (Loc,
                   Name =>
                     Make_Selected_Component (Loc,
                       Prefix => New_Reference_To (Temp, Loc),
                       Selector_Name =>
                         New_Reference_To (Tag_Component (T), Loc)),
                   Expression =>
                     Make_Unchecked_Type_Conversion (Loc,
                       Subtype_Mark =>
                         New_Reference_To (RTE (RE_Tag), Loc),
                       Expression =>
                         New_Reference_To (Access_Disp_Table (T), Loc))));

               --  The previous assignment has to be done in any case

               Set_Assignment_OK (Name (Next (First (Act))));

               Append_To (Act,
                   Make_Adjust_Call (
                     Ref => Make_Explicit_Dereference (Loc,
                              Prefix => New_Reference_To (Temp, Loc)),
                     Typ => T));

               Append_To (Act,
                   Make_Attach_Ctrl_Object (
                     Ref => Make_Explicit_Dereference (Loc,
                              Prefix => New_Reference_To (Temp, Loc)),
                     Scop => Scope (PtrT)));

               Rewrite_Substitute_Tree (N,
                 Make_Expression_Actions (Loc,
                   Actions    => Act,
                   Expression => New_Reference_To (Temp, Loc)));
               Analyze (N);
            else
               null;
            end if;
         end;

      --  in this case, an initialization routine is needed

      else
         declare
            T     : constant Entity_Id  := Entity (Expression (N));
            Init  : constant Entity_Id  := Base_Init_Proc (T);
            Arg1  : Node_Id;
            Args  : List_Id;
            Discr : Elmt_Id;
            Eact  : Node_Id;

         begin
            --  Nothing to do if no initialization routine required

            if No (Init) then
               null;

            --  Else we have the case that definitely needs a call

            else
               Node := N;
               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

               --  Construct argument list for the initialization routine call

               Arg1 :=
                 Make_Explicit_Dereference (Loc,
                   Prefix => New_Reference_To (Temp, Loc));

               --  The initialization procedure expects a specific type.
               --  if the context is access to class wide, indicate that
               --  the object being allocated has the right specific type.

               if Is_Class_Wide_Type (Designated_Type (PtrT)) then
                  Arg1 :=
                    Make_Type_Conversion (Loc,
                      Subtype_Mark => New_Reference_To (T,  Loc),
                      Expression => Arg1);
               end if;

               --  If designated type is a task type, then the first argument
               --  in the Init routine has to be unchecked converted to the
               --  corresponding record type, since that's what Init expects.

               if Is_Task_Type (T) then
                  Arg1 :=
                    Make_Unchecked_Type_Conversion (Loc,
                      Subtype_Mark =>
                        New_Reference_To (Corresponding_Record_Type (T), Loc),
                      Expression => Arg1);
               end if;

               Args := New_List (Arg1);

               --  For the task case, pass the Master_Id of the access type
               --  as the value of the _Master parameter, and _Chain as the
               --  value of the _Chain parameter (_Chain will be defined as
               --  part of the generated code for the allocator).

               if Has_Tasks (T) then

                  if No (Master_Id (PtrT)) then

                     --  The designated type was an incomplete type, and
                     --  the access type did not get expanded. Salvage
                     --  it now. This may be a more general problem.

                     Expand_N_Full_Type_Declaration (Parent (PtrT));
                  end if;

                  Append_To (Args, New_Reference_To (Master_Id (PtrT), Loc));
                  Append_To (Args, Make_Identifier (Loc, Name_uChain));
               end if;

               --  Add discriminants if discriminated type

               if Has_Discriminants (T) then
                  Discr := First_Elmt (Discriminant_Constraint (T));

                  while Present (Discr) loop
                     Append (New_Copy (Elists.Node (Discr)), Args);
                     Discr := Next_Elmt (Discr);
                  end loop;
               end if;

               --  We set the allocator as analyzed so that when we analyze the
               --  expression actions node, we do not get an unwanted recursive
               --  expansion of the allocator expression.

               Set_Analyzed (N, True);

               --  Now we can rewrite the allocator. First see if it is
               --  already in an expression actions node, which will often
               --  be the case, because this is how we handle the case of
               --  discriminants being present. If so, we can just modify
               --  that expression actions node that is there, otherwise
               --  we must create an expression actions node.

               Eact := Parent (N);

               if Nkind (Eact) = N_Expression_Actions
                 and then Expression (Eact) = N
               then
                  Node := N;

               else
                  Rewrite_Substitute_Tree (N,
                    Make_Expression_Actions (Loc,
                      Actions    => New_List,
                      Expression => New_Copy (N)));

                  Eact := N;
                  Node := Expression (N);
               end if;

               --  Now we modify the expression actions node as follows

               --    input:   [... ; new T]

               --    output:  [... ;
               --              Temp : constant ptr_T := new (T);
               --              Init (Temp.all, ...);
               --      <CTRL>  Attach_To_Final_List (Finalizable (Temp.all));
               --      <CTRL>  Initialize (Finalizable (Temp.all));
               --              Temp]

               --  Here ptr_T is the pointer type for the allocator, and T
               --  is the subtype of the allocator.

               Append_To (Actions (Eact),
                 Make_Object_Declaration (Loc,
                   Defining_Identifier => Temp,
                   Constant_Present    => True,
                   Object_Definition   => New_Reference_To (PtrT, Loc),
                   Expression          => Node));

               --  Case of designated type is task or contains task

               if Has_Tasks (T) then
                  Build_Task_Allocate_Block (Actions (Eact), Node, Args);

               else
                  Append_To (Actions (Eact),
                    Make_Procedure_Call_Statement (Loc,
                      Name => New_Reference_To (Init, Loc),
                      Parameter_Associations => Args));
               end if;

               if Is_Controlled (T) then
                  Append_List_To (Actions (Eact),
                    Make_Init_Attach_Calls (
                      Ref => Make_Explicit_Dereference (Loc,
                               Prefix => New_Reference_To (Temp, Loc)),
                      Scop => Scope (PtrT),
                      Typ => T));
               end if;

               Set_Expression (Eact, New_Reference_To (Temp, Loc));
               Analyze (Eact);

            end if;
         end;
      end if;
   end Expand_N_Allocator;

   ------------------------------
   -- Expand_N_Concat_Multiple --
   ------------------------------

   procedure Expand_N_Concat_Multiple (N : Node_Id) is
   begin
      Expand_Concatenation (N, Expressions (N));
   end Expand_N_Concat_Multiple;

   ---------------------
   -- Expand_N_Op_And --
   ---------------------

   --  This is really just a renaming of Expand_Boolean_Operator ???

   procedure Expand_N_Op_And (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_And;

   ------------------------
   -- Expand_N_Op_Concat --
   ------------------------

   procedure Expand_N_Op_Concat (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Lhs      : Node_Id   := Left_Opnd (N);
      Rhs      : Node_Id   := Right_Opnd (N);
      Ltyp     : Entity_Id := Base_Type (Etype (Lhs));
      Rtyp     : Entity_Id := Base_Type (Etype (Rhs));
      Comp_Typ : Entity_Id := Base_Type (Component_Type (Etype (N)));

   begin
      --  If left operand is a single component, replace by an aggregate
      --  of the form (1 => operand), as required by concatenation semantics.

      if Ltyp = Comp_Typ then
         Lhs :=
           Make_Aggregate (Loc,
             Component_Associations => New_List (
               Make_Component_Association (Loc,
                 Choices    => New_List (Make_Integer_Literal (Loc, Uint_1)),
                 Expression => New_Copy (Lhs))));
         Ltyp := Base_Type (Etype (N));
      end if;

      --  Similar handling for right operand

      if Rtyp = Comp_Typ then
         Rhs :=
           Make_Aggregate (Loc,
             Component_Associations => New_List (
               Make_Component_Association (Loc,
                 Choices    => New_List (Make_Integer_Literal (Loc, Uint_1)),
                 Expression => New_Copy (Rhs))));
         Rtyp := Base_Type (Etype (N));
      end if;

      --  Handle case of concatenating Standard.String with runtime call

      if Ltyp = Standard_String and then Rtyp = Standard_String then
         Rewrite_Substitute_Tree (N,
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (RE_Str_Concat), Loc),
             Parameter_Associations =>
                     New_List (New_Copy (Lhs), New_Copy (Rhs))));

         Analyze (N);
         Resolve (N, Standard_String);

      --  For other than Standard.String, use general routine

      else
         Expand_Concatenation (N, New_List (Lhs, Rhs));
      end if;

   end Expand_N_Op_Concat;

   --------------------
   -- Expand_N_Op_Eq --
   --------------------

   procedure Expand_N_Op_Eq (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Lhs     : constant Node_Id    := Left_Opnd (N);
      Rhs     : constant Node_Id    := Right_Opnd (N);
      Typ     : Entity_Id  := Etype (Lhs);
      Eq_Prim : Entity_Id;

   begin
      if Ekind (Typ) = E_Private_Type then
         Typ := Full_Declaration (Typ);
      end if;

      Typ := Base_Type (Typ);

      if  Is_Array_Type (Typ) then

         if Is_Scalar_Type (Component_Type (Typ)) then

            --  The case of two constrained arrays can be left to Gigi

            if Nkind (Lhs) /= N_Expression_Actions
              and then Nkind (Rhs) /= N_Expression_Actions
            then
               null;

               --  Kludge to avoid a bug in Gigi (work only for Strings) ???

            elsif Typ = Standard_String then
               Rewrite_Substitute_Tree (N,
                 Make_Function_Call (Loc,
                   Name => New_Reference_To (RTE (RE_Str_Equal), Loc),
                   Parameter_Associations =>
                     New_List (New_Copy (Lhs), New_Copy (Rhs))));

               Analyze (N);
               Resolve (N, Standard_Boolean);

            --  Other cases, we hope Gigi will not blow up ???

            else
               null;
            end if;
         else
            Replace_Substitute_Tree (N,
              Expand_Array_Equality
                (Loc, Typ, New_Copy (Lhs), New_Copy (Rhs)));

            Analyze (N);
            Resolve (N, Standard_Boolean);
         end if;

      elsif Is_Record_Type (Typ) then

         if Has_Discriminants (Typ)
           and then Present (Variant_Part (Component_List (
                               Type_Definition (Parent (Typ)))))
         then

            --  ???
            --  in this case a function has to be expanded and called using
            --  the same model as for initialization procedures  (use of
            --  the case statement in the record definition).
            --  It has to be dealt with as a special case because in the
            --  simple case (record without variant part), we prefer to
            --  generate a big expression which will be optimized by the
            --  back-end.

            Unimplemented (N, "?complex equality of discriminated records");

         else
            declare
               L : Node_Id := New_Copy (Lhs);
               R : Node_Id := New_Copy (Rhs);

            begin
               --  ??? L and R should be marked evaluate_once because the
               --  equality is expanded into a bunch of component equalities
               --  but in this case this flag seems to cause big trouble in
               --  the generated code, so it is commented out till further
               --  investigation.
               --  Set_Evaluate_Once (L, True);
               --  Set_Evaluate_Once (R, True);

               Replace_Substitute_Tree (N,
                 Expand_Record_Equality (Loc, Typ, L, R));
               Analyze (N);
               Resolve (N, Standard_Boolean);
            end;
         end if;
      end if;
   end Expand_N_Op_Eq;

   -----------------------
   -- Expand_N_Op_Expon --
   -----------------------

   procedure Expand_N_Op_Expon (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Typ   : constant Entity_Id  := Etype (N);
      Btyp  : constant Entity_Id  := Root_Type (Typ);
      Max   : constant Uint       := Uint_4;
      Min   : constant Uint       := Uint_Minus_4;
      Base  : constant Node_Id    := New_Copy (Left_Opnd (N));
      Exp   : constant Node_Id    := New_Copy (Right_Opnd (N));
      Expv  : Uint;
      Xnode : Node_Id;
      Temp  : Node_Id;
      Rent  : RE_Id;
      Ent   : Entity_Id;

   begin
      --  At this point the exponentiation must be dynamic since the static
      --  case has already been folded after Resolve by Eval_Op_Expon.

      --  Test for case of literal right argument

      if Nkind (Exp) = N_Integer_Literal then
         Expv := Intval (Exp);

         if (Ekind (Typ) in Float_Kind
               and then UI_Ge (Expv, Min)
               and then UI_Le (Expv, Max))
           or else
            (Ekind (Typ) in Integer_Kind
               and then UI_Ge (Expv, Uint_0)
               and then UI_Le (Expv, Max))
         then
            Expv := UI_Abs (Expv);

            --  X ** 0 = 1 (or 1.0)

            if Expv = Uint_0 then
               if Ekind (Typ) in Integer_Kind then
                  Xnode := Make_Integer_Literal (Loc, Intval => Uint_1);
               else
                  Xnode :=
                    Make_Real_Literal (Loc,
                      Numerator   => Uint_1,
                      Denominator => Uint_1);
               end if;

            --  X ** 1 = X

            elsif Expv = Uint_1 then
               Xnode := Base;

            --  X ** 2 = X * X

            elsif Expv = Uint_2 then
               Set_Evaluate_Once (Base, True);
               Xnode :=
                 Make_Op_Multiply (Loc,
                   Left_Opnd  => Base,
                   Right_Opnd => Base);

            --  X ** 3 = X * X * X

            elsif Expv = Uint_3 then
               Set_Evaluate_Once (Base, True);
               Xnode :=
                 Make_Op_Multiply (Loc,
                   Left_Opnd =>
                     Make_Op_Multiply (Loc,
                       Left_Opnd  => Base,
                       Right_Opnd => Base),

                   Right_Opnd  => Base);

            --  X ** 4 = {Xnnn : constant base'type := base * base} Xnnn * Xnnn

            elsif Expv = Uint_4 then
               Set_Evaluate_Once (Base, True);
               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('X'));

               Xnode :=
                 Make_Expression_Actions (Loc,
                   Actions => New_List (
                     Make_Object_Declaration (Loc,
                       Defining_Identifier => Temp,
                       Constant_Present    => True,
                       Object_Definition   => New_Reference_To (Typ, Loc),
                       Expression =>
                         Make_Op_Multiply (Loc,
                           Left_Opnd  => Base,
                           Right_Opnd => Base))),
                   Expression =>
                     Make_Op_Multiply (Loc,
                       Left_Opnd  => New_Reference_To (Temp, Loc),
                       Right_Opnd => New_Reference_To (Temp, Loc)));
            end if;

            --  For non-negative case, we are all set

            if not UI_Is_Negative (Intval (Exp)) then
               Rewrite_Substitute_Tree (N, Xnode);

            --  For negative cases, take reciprocal (base must be real)

            else
               Set_Paren_Count (Xnode, 1);
               Replace_Substitute_Tree (N,
                 Make_Op_Divide (Loc,
                   Left_Opnd   =>
                     Make_Real_Literal (Loc,
                       Numerator   => Uint_1,
                       Denominator => Uint_1),
                   Right_Opnd  => Xnode));
            end if;

            Analyze (N);
            Resolve (N, Typ);
            return;

         --  Don't fold cases of large literal exponents, and also don't fold
         --  cases of integer bases with negative literal exponents.

         end if;

      --  Don't fold cases where exponent is not integer literal

      end if;

      --  Fall through if exponentiation must be done using a runtime routine
      --  First deal with modular case.

      if Is_Modular_Integer_Type (Btyp) then

         --  Non-binary case, we call the special exponentiation routine for
         --  the non-binary case, converting the argument to Long_Long_Integer
         --  and passing the modulus value. Then the result is converted back
         --  to the base type.

         if Non_Binary_Modulus (Btyp) then

            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression   =>
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (RTE (RE_Xp_NBM), Loc),
                    Parameter_Associations => New_List (
                      Make_Type_Conversion (Loc,
                        Subtype_Mark =>
                          New_Reference_To (Standard_Integer, Loc),
                        Expression => Base),
                      Make_Integer_Literal (Loc, Modulus (Btyp)),
                      Exp))));

         --  Binary case, in this case, we call one of two routines, either
         --  the unsigned integer case, or the unsigned long long integer
         --  case, with the final conversion doing the required truncation.

         else
            if UI_To_Int (Esize (Btyp)) <= Standard_Integer_Size then
               Ent := RTE (RE_Xp_BMI);
            else
               Ent := RTE (RE_Xp_BML);
            end if;

            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression   =>
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (Ent, Loc),
                    Parameter_Associations => New_List (
                      Make_Type_Conversion (Loc,
                        Subtype_Mark =>
                          New_Reference_To (Etype (First_Formal (Ent)), Loc),
                        Expression   => Base),
                      Exp))));
         end if;

         --  Common exit point for modular type case

         Analyze (N);
         Resolve (N, Typ);
         return;

      --  Signed integer cases

      elsif Btyp = Standard_Integer then
         Rent := RE_Xp_I;
      elsif Btyp = Standard_Short_Integer then
         Rent := RE_Xp_SI;
      elsif Btyp = Standard_Short_Short_Integer then
         Rent := RE_Xp_SSI;
      elsif Btyp = Standard_Long_Integer then
         Rent := RE_Xp_LI;
      elsif (Btyp = Standard_Long_Long_Integer
        or else Btyp = Universal_Integer)
      then
         Rent := RE_Xp_LLI;

      --  Floating-point cases

      elsif Btyp = Standard_Float then
         Rent := RE_Xp_F;
      elsif Btyp = Standard_Short_Float then
         Rent := RE_Xp_SF;
      elsif Btyp = Standard_Long_Float then
         Rent := RE_Xp_LF;
      else
         pragma Assert (Btyp = Standard_Long_Long_Float
                         or else Btyp = Universal_Real);
         Rent := RE_Xp_LLF;
      end if;

      --  Common processing for integer cases and floating-point cases.
      --  If we are in the base type, we can call runtime routine directly

      if Typ = Btyp then
         Replace_Substitute_Tree (N,
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (Rent), Loc),
             Parameter_Associations => New_List (Base, Exp)));

      --  Otherwise we have to introduce conversions

      else
         Replace_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression   =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (RTE (Rent), Loc),
                 Parameter_Associations => New_List (
                   Make_Type_Conversion (Loc,
                     Subtype_Mark => New_Reference_To (Btyp, Loc),
                     Expression   => Base),
                   Exp))));
      end if;

      Analyze (N);
      Resolve (N, Typ);
      return;

   end Expand_N_Op_Expon;

   --------------------
   -- Expand_N_Op_Ge --
   --------------------

   procedure Expand_N_Op_Ge (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Ge;

   --------------------
   -- Expand_N_Op_Gt --
   --------------------

   procedure Expand_N_Op_Gt (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Gt;

   --------------------
   -- Expand_N_Op_In --
   --------------------

   --  Expansion is only required for the tagged case. See specification of
   --  Tagged_Membership function for details of required transformation.

   procedure Expand_N_Op_In (N : Node_Id) is
      Typ : constant Entity_Id := Etype (N);

   begin
      if Is_Tagged_Type (Etype (Etype (Right_Opnd (N)))) then
         Replace_Substitute_Tree (N, Tagged_Membership (N));
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_N_Op_In;

   --------------------
   -- Expand_N_Op_Le --
   --------------------

   procedure Expand_N_Op_Le (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Le;

   --------------------
   -- Expand_N_Op_Lt --
   --------------------

   procedure Expand_N_Op_Lt (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Lt;

   ---------------------
   -- Expand_N_Op_Not --
   ---------------------

   --  If the argument of negation is a Boolean array type, generate the
   --  following in line function definition:

   --     function Nnnn (A : arr) is
   --       B : arr; (or arr (A'range) if arr is unconstrained)
   --     begin
   --       for I in a'range loop
   --          B (I) := not A (I);
   --       end loop;
   --       return B;
   --     end Nnnn;

   procedure Expand_N_Op_Not (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : constant Entity_Id  := Etype (N);
      A   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uA);
      B   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uB);
      I   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uI);
      A_I : Node_Id;
      B_I : Node_Id;

      Func_Name      : Entity_Id;
      Func_Body      : Node_Id;
      Loop_Statement : Node_Id;
      Result         : Node_Id;
      Type_Of_B      : Node_Id;

   begin
      if not Is_Array_Type (Typ) then
         return;
      end if;

      A_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (A, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      B_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (B, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => Make_Identifier (Loc, Chars (A)),
                      Attribute_Name => Name_Range))),

          Statements => New_List (
            Make_Assignment_Statement (Loc,
              Name       => B_I,
              Expression => Make_Op_Not (Loc, A_I))));


      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('N'));

      if Is_Constrained (Typ) then
         Type_Of_B := New_Reference_To (Typ, Loc);
      else
         Type_Of_B :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Constraint   =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (Make_Attribute_Reference (Loc,
                      Attribute_Name => Name_Range,
                      Prefix => New_Reference_To (A, Loc)))));
      end if;

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name => Func_Name,
              Parameter_Specifications => New_List (
                Make_Parameter_Specification (Loc,
                  Defining_Identifier => A,
                  Parameter_Type      => New_Reference_To (Typ, Loc))),
              Subtype_Mark => New_Reference_To (Typ,  Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => B,
              Object_Definition   => Type_Of_B)),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Loop_Statement,
                Make_Return_Statement (Loc,
                  Expression =>
                    Make_Identifier (Loc, Chars (B))))));

      Result :=
        Make_Expression_Actions (Loc,
          Actions => New_List (Func_Body),
          Expression =>
            Make_Function_Call (Loc,
              Name => New_Reference_To (Func_Name, Loc),
              Parameter_Associations =>
                New_List (Right_Opnd (N))));

      Replace_Substitute_Tree (N, Result);
      Analyze (N);
      Resolve (N, Typ);
   end Expand_N_Op_Not;

   ------------------------
   -- Expand_N_Op_Not_In --
   ------------------------

   --  Expansion is only required for the tagged case. See specification of
   --  Tagged_Membership function for details of required transformation.

   procedure Expand_N_Op_Not_In (N : Node_Id) is
      Typ : constant Entity_Id := Etype (N);

   begin
      if Is_Tagged_Type (Etype (Etype (Right_Opnd (N)))) then
         Replace_Substitute_Tree (N,
           Make_Op_Not (Sloc (N), Tagged_Membership (N)));
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_N_Op_Not_In;

   --------------------
   -- Expand_N_Op_Or --
   --------------------

   procedure Expand_N_Op_Or (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_Or;

   ---------------------
   -- Expand_N_Op_Xor --
   ---------------------

   procedure Expand_N_Op_Xor (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_Xor;

   --------------------
   -- Expand_N_Slice --
   --------------------

   --  Build an implicit subtype declaration to represent the type delivered
   --  by the slice. This subtype has an Ekind of E_Slice_Subtype, which is
   --  a special kind of type used just for this purpose. Logically, what is
   --  needed is a full array subtype declaration, but that would take a lot
   --  of nodes. On the other hand if we don't provide any kind of subtype
   --  for the slice, Gigi gets really confused. The compromise of building
   --  a special kind of very economical subtype declaration node, and then
   --  putting a bit of specialized code in Gigi to deal with this special
   --  declaration meets the need with minimum overhead.

   --  The procesing consists of building this subtype and then resetting the
   --  Etype of the slice node to have this type.

   procedure Expand_N_Slice (N : Node_Id) is
      Impl_Subtype : Entity_Id;

   begin
      --  First we build a defining occurrence for the "slice subtype"

      Impl_Subtype := New_Itype (N);
      Set_Ekind (Impl_Subtype, E_Slice_Subtype);
      Set_Component_Type (Impl_Subtype, Component_Type (Etype (N)));
      Set_Slice_Range (Impl_Subtype, Discrete_Range (N));
      Set_Etype (Impl_Subtype, Etype (N));

      --  The Etype of the existing Slice node is reset to this anymous
      --  subtype. This node will be marked as Analyzed when we return and
      --  nothing else needs to be done to it.

      Set_Etype (N, Impl_Subtype);
   end Expand_N_Slice;

   ------------------------------
   -- Expand_N_Type_Conversion --
   ------------------------------

   --  For tagged types, the conversion is a view conversion, and the node
   --  must be converted to an unchecked type conversion, since we don't
   --  want Gigi doing anything to such a node.

   procedure Expand_N_Type_Conversion (N : Node_Id) is
      Expr : constant Node_Id := Expression (N);
      T    : constant Entity_Id := Etype (N);

   begin
      --  If it is a tagged type, that's a view conversion and the node
      --  gets converted to an unchecked type conversion since we don't
      --  want Gigi doing anything to such a node

      if Is_Variable (Expr)
        and then Is_Tagged_Type (T)
        and then Is_Tagged_Type (Etype (Expr))
      then
         Change_Conversion_To_Unchecked (N);
      end if;
   end Expand_N_Type_Conversion;

   ----------------------------
   -- Expand_Record_Equality --
   ----------------------------

   --  For non-variant records Equality is expanded into:

   --    Lhs.Cmp1 = Rhs.Cmp1
   --      and then Lhs.Cmp2 = Rhs.Cmp2
   --      and then ...
   --      and then Lhs.Cmpn = Rhs.Cmpn

   --  The expression is folded by the back-end for adjacent fields. This
   --  function can be called for tagged records but only in the case of
   --  implemetation of predefined equality (see Predefined_Primitives_Bodies)

   function Expand_Record_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id
   is
      function Build (Comp : Entity_Id) return Node_Id;
      --  Build recursively the sequence of components equalities using the
      --  equality (predefined or primitive) for each component type.

      function Build (Comp : Entity_Id) return Node_Id is
         Eq_Node   : Node_Id;
         L, R      : Node_Id;
         Next_Comp : Entity_Id := Next_Component (Comp);

      begin
         --  The inherited Components are skipped (they are part of the parent)

         while Present (Next_Comp)
                 and then Next_Comp /= Original_Record_Component (Next_Comp)
         loop
            Next_Comp := Next_Component (Next_Comp);
         end loop;

         L :=
           Make_Selected_Component (Loc,
             Prefix => Lhs,
             Selector_Name => New_Reference_To (Comp, Loc));

         R :=
           Make_Selected_Component (Loc,
             Prefix => Rhs,
             Selector_Name => New_Reference_To (Comp, Loc));

         --  Recursive call to deal with recursive composite types

         Eq_Node := Expand_Composite_Equality (Loc, Etype (Comp), L, R);

         if No (Next_Comp) then
            return Eq_Node;
         else
            return
              Make_Op_And_Then (Loc,
                Left_Opnd  => Eq_Node,
                Right_Opnd => Build (Next_Comp));
         end if;
      end Build;

   --  Start of processing for Expand_Record_Equality

   begin
      --  Generates the following code: (assuming that component C2 is
      --  also a record)

      --  <if no component>
      --  True
      --  <else>
      --   Lhs.C1 = Rhs.C1
      --     and then Lhs.C2.C1=Rhs.C2.C1 and then ... Lhs.C2.Cn=Rhs.C2.Cn
      --     and then ...
      --     and then Lhs.Cmpn = Rhs.Cmpn

      if No (First_Component (Typ)) then
         return New_Reference_To (Standard_True, Loc);
      else
         return Build (First_Component (Typ));
      end if;
   end Expand_Record_Equality;

   ------------------------------
   -- Make_Array_Comparison_Op --
   ------------------------------

   --  This is a hand-coded expansion of the following generic function:

   --  generic
   --    type elem is  (<>);
   --    type index is (<>);
   --    type a is array (index range <>) of elem;
   --
   --  function Gnnn (X : a; Y: a) return boolean is
   --    J : index := Y'first;
   --
   --  begin
   --    if X'length = 0 then
   --       return false;
   --
   --    elsif Y'length = 0 then
   --       return true;
   --
   --    else
   --      for I in X'range loop
   --        if X (I) = Y (J) then
   --          if J = Y'last then
   --            exit;
   --          else
   --            J := index'succ (J);
   --          end if;
   --
   --        else
   --           return X (I) > Y (J);
   --        end if;
   --      end loop;
   --
   --      return X'length > Y'length;
   --    end if;
   --  end Gnnn;

   --  If the flag Equal is true, the procedure generates the body for
   --  >= instead. This only affects the last return statement.

   --  Note that since we are essentially doing this expansion by hand, we
   --  do not need to generate an actual or formal generic part, just the
   --  instantiated function itself.

   function Make_Array_Comparison_Op
     (Typ : Entity_Id; Loc : Source_Ptr; Equal : Boolean) return Node_Id
   is
      X : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uX);
      Y : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uY);
      I : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uI);
      J : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uJ);

      Index : constant Entity_Id := Base_Type (Etype (First_Index (Typ)));

      Loop_Statement : Node_Id;
      Loop_Body      : Node_Id;
      If_Stat        : Node_Id;
      Inner_If       : Node_Id;
      Final_Expr     : Node_Id;
      Func_Body      : Node_Id;
      Func_Name      : Entity_Id;
      Formals        : List_Id;
      Length1        : Node_Id;
      Length2        : Node_Id;

   begin
      --  if J = Y'last then
      --     exit;
      --  else
      --     J := index'succ (J);
      --  end if;

      Inner_If :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd => New_Reference_To (J, Loc),
              Right_Opnd =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (Y, Loc),
                  Attribute_Name => Name_Last)),

          Then_Statements => New_List (
                Make_Exit_Statement (Loc)),

          Else_Statements =>
            New_List (
              Make_Assignment_Statement (Loc,
                Name => New_Reference_To (J, Loc),
                Expression =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (Index, Loc),
                    Attribute_Name => Name_Succ,
                    Expressions => New_List (New_Reference_To (J, Loc))))));

      --  if X (I) = Y (J) then
      --     if ... end if;
      --  else
      --     return X (I) > Y (J);
      --  end if;

      Loop_Body :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd =>
                Make_Indexed_Component (Loc,
                  Prefix      => New_Reference_To (X, Loc),
                  Expressions => New_List (New_Reference_To (I, Loc))),

              Right_Opnd =>
                Make_Indexed_Component (Loc,
                  Prefix      => New_Reference_To (Y, Loc),
                  Expressions => New_List (New_Reference_To (J, Loc)))),

          Then_Statements => New_List (Inner_If),

          Else_Statements => New_List (
            Make_Return_Statement (Loc,
              Expression =>
                Make_Op_Gt (Loc,
                  Left_Opnd =>
                    Make_Indexed_Component (Loc,
                      Prefix      => New_Reference_To (X, Loc),
                      Expressions => New_List (New_Reference_To (I, Loc))),

                  Right_Opnd =>
                    Make_Indexed_Component (Loc,
                      Prefix      => New_Reference_To (Y, Loc),
                      Expressions => New_List (
                        New_Reference_To (J, Loc)))))));

      --  for I in X'range loop
      --     if ... end if;
      --  end loop;

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (X, Loc),
                      Attribute_Name => Name_Range))),

          Statements => New_List (Loop_Body));

      --    if X'length = 0 then
      --       return false;
      --    elsif Y'length = 0 then
      --       return true;
      --    else
      --      for ... loop ... end loop;
      --      return X'length > Y'length;
      --    --  return X'length >= Y'length to implement >=.
      --    end if;

      Length1 :=
        Make_Attribute_Reference (Loc,
          Prefix => New_Reference_To (X, Loc),
          Attribute_Name => Name_Length);

      Length2 :=
        Make_Attribute_Reference (Loc,
          Prefix => New_Reference_To (Y, Loc),
          Attribute_Name => Name_Length);

      if Equal then
         Final_Expr := Make_Op_Ge (Loc,
            Left_Opnd  => Length1,
            Right_Opnd => Length2);
      else
         Final_Expr := Make_Op_Gt (Loc,
            Left_Opnd  => Length1,
            Right_Opnd => Length2);
      end if;

      If_Stat :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (X, Loc),
                  Attribute_Name => Name_Length),
              Right_Opnd =>
                Make_Integer_Literal (Loc, Uint_0)),

          Then_Statements =>
            New_List (
              Make_Return_Statement (Loc,
                Expression => New_Reference_To (Standard_False, Loc))),

          Elsif_Parts => New_List (
            Make_Elsif_Part (Loc,
              Condition =>
                Make_Op_Eq (Loc,
                  Left_Opnd =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (Y, Loc),
                      Attribute_Name => Name_Length),
                  Right_Opnd =>
                    Make_Integer_Literal (Loc, Uint_0)),

              Then_Statements =>
                New_List (
                  Make_Return_Statement (Loc,
                     Expression => New_Reference_To (Standard_True, Loc))))),

          Else_Statements => New_List (
            Loop_Statement,
            Make_Return_Statement (Loc,
              Expression => Final_Expr)));


      --  (X : a; Y: a)

      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => X,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => Y,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      --  function Gnnn (...) return boolean is
      --    J : index := Y'first;
      --  begin
      --    if ... end if;
      --  end Gnnn;

      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('G'));

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark => New_Reference_To (Standard_Boolean, Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => J,
              Object_Definition   => New_Reference_To (Index, Loc),
              Expression =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (Y, Loc),
                  Attribute_Name => Name_First))),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (If_Stat)));

      return Func_Body;

   end Make_Array_Comparison_Op;

   ---------------------------
   -- Make_Boolean_Array_Op --
   ---------------------------

   --  For logical operations on boolean arrays, expand in line the
   --  following, replacing 'and' with 'or' or 'xor' where needed:

   --    function Annn (A : arr; B: arr) is
   --       C : arr;   (or arr (A'range) if arr is unconstrained)
   --    begin
   --       for I in A'range loop
   --          C (i) := A (i) and B (i);
   --       end loop;
   --       return C;
   --    end Annn;

   function Make_Boolean_Array_Op (N : Node_Id) return Node_Id is

      Loc : Source_Ptr := Sloc (N);
      Typ : Entity_Id := Etype (Left_Opnd (N));

      A   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uA);
      B   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uB);
      C   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uC);
      I   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uI);

      A_I : Node_Id;
      B_I : Node_Id;
      C_I : Node_Id;
      Op  : Node_Id;

      Formals        : List_Id;
      Func_Name      : Entity_Id;
      Func_Body      : Node_Id;
      Loop_Statement : Node_Id;
      Type_Of_C      : Node_Id;

   begin
      A_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (A, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      B_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (B, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      C_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (C, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      if Nkind (N) = N_Op_And then
         Op :=
           Make_Op_And (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);

      elsif Nkind (N) = N_Op_Or then
         Op :=
           Make_Op_Or (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);

      else
         Op :=
           Make_Op_Xor (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);
      end if;

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (A, Loc),
                      Attribute_Name => Name_Range))),

          Statements => New_List (
            Make_Assignment_Statement (Loc,
              Name       => C_I,
              Expression => Op)));

      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => A,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => B,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

      if Is_Constrained (Typ) then
         Type_Of_C := New_Reference_To (Typ, Loc);
      else
         Type_Of_C :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Constraint   =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (Make_Attribute_Reference (Loc,
                      Attribute_Name => Name_Range,
                      Prefix => New_Reference_To (A, Loc)))));
      end if;

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark             => New_Reference_To (Typ, Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => C,
              Object_Definition   => Type_Of_C)),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Loop_Statement,
                Make_Return_Statement (Loc,
                  Expression => New_Reference_To (C, Loc)))));

      return Func_Body;

   end Make_Boolean_Array_Op;

   ------------------------
   --  Tagged_Membership --
   ------------------------

   --  There are two different cases to consider depending on whether
   --  the right operand is a class-wide type or not. If not we just
   --  compare the actual tag of the left expr to the target type tag:
   --
   --     Left_Expr.Tag = Tag_Of (Right_Type);
   --
   --  If it is a class-wide type, it is more complex. We use the table of
   --  ancestors accessed by the "Tags" field of the Dispatch table. We have
   --  to ensure that the inheritance depth of the operand if greater or equal
   --  than the target types's and that they are on the inheritance path :
   --
   --  <action>
   --    N : Integer := Left_Expr.Tag.all.Inheritance_Depth -
   --                     Tag_Of (Right_Type).all.Inheritance_Depth;
   --  <expression>
   --    (N >= 0)
   --      and then Left_Expr.Tag.all.Tags.all (N) = Tag_Of (Right_Type)
   --
   --  the real expressions are a bit more complicated due to type conversions

   function Tagged_Membership (N : Node_Id) return Node_Id is
      Left       : constant Node_Id    := Left_Opnd  (N);
      Right      : constant Node_Id    := Right_Opnd (N);
      Sloc_N     : constant Source_Ptr := Sloc (N);

      Left_Type  : Entity_Id;
      Right_Type : Entity_Id;
      Var_N      : Node_Id;

   begin
      Left_Type  := Etype (Left);
      Right_Type := Etype (Right);

      if Is_Class_Wide_Type (Left_Type) then
         Left_Type := Etype (Left_Type);
      end if;

      if not Is_Class_Wide_Type (Right_Type) then

         --  Left_Type (Left)._Tag =
         --    System.Tag (Access_Disp_Table (Right_Type));

         return
           Make_Op_Eq (Sloc_N,
             Left_Opnd =>
               Make_Selected_Component (Sloc_N,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Sloc_N,
                     Subtype_Mark => New_Reference_To (Left_Type, Sloc_N),
                     Expression => New_Copy (Left)),
                 Selector_Name =>
                   New_Reference_To (Tag_Component (Left_Type), Sloc_N)),
             Right_Opnd =>
               Make_Unchecked_Type_Conversion (Sloc_N,
                 Subtype_Mark => New_Reference_To (RTE (RE_Tag), Sloc_N),
                 Expression =>
                   New_Reference_To (
                     Access_Disp_Table (Right_Type), Sloc_N)));

      else
         --  Replace N by expression-actions
         --
         --    <actions>
         --    N : Integer :=
         --      Acc_Dt (Left_Type(Left)._Tag).all.Inheritance_Depth
         --        - Access_Disp_Table (Right_Type).all.Inheritance_Depth;

         Var_N := Make_Defining_Identifier (Sloc_N,  New_Internal_Name ('N'));

         --  Use the root type of the class

         Right_Type := Etype (Right_Type);

         return
           Make_Expression_Actions (Sloc_N,
             Actions => New_List (
               Make_Object_Declaration (Sloc_N,
                 Defining_Identifier => Var_N,

                 Object_Definition   =>
                   New_Reference_To (Standard_Integer, Sloc_N),

                 Expression =>
                   Make_Op_Subtract (Sloc_N,
                     Left_Opnd =>
                       Make_Selected_Component (Sloc_N,
                         Prefix =>  Make_DT_Access (Sloc_N, Left, Left_Type),
                         Selector_Name =>
                           Make_DT_Component (Sloc_N, Left_Type, 1)),

                     Right_Opnd =>
                       Make_Selected_Component (Sloc_N,
                         Prefix =>
                           Make_Explicit_Dereference (Sloc_N,
                             Prefix =>
                               Make_Identifier (Sloc_N,
                                 Chars (Access_Disp_Table (Right_Type)))),

                       Selector_Name =>
                         Make_DT_Component (Sloc_N, Right_Type, 1))))),

         --  (N >= 0)
         --  and then
         --    (Acc_Dt (Left_Type (Left).__Tag).all.Tags.all (N)
         --     = System.Tag (Access_Disp_Table (Right_Type)))

             Expression =>
               Make_Op_And_Then (Sloc_N,
                 Left_Opnd =>
                   Make_Op_Ge (Sloc_N,
                     Left_Opnd => New_Reference_To (Var_N, Sloc_N),
                     Right_Opnd => Make_Integer_Literal (Sloc_N, Uint_0)),

                 Right_Opnd =>
                   Make_Op_Eq (Sloc_N,
                     Left_Opnd =>
                       Make_Indexed_Component (Sloc_N,
                         Prefix =>
                           Make_Explicit_Dereference (Sloc_N,
                             Prefix =>
                               Make_Selected_Component (Sloc_N,
                                 Prefix =>
                                   Make_DT_Access (Sloc_N, Left, Left_Type),
                                 Selector_Name =>
                                   Make_DT_Component (Sloc_N, Left_Type, 2))),
                         Expressions =>
                           New_List (New_Reference_To (Var_N, Sloc_N))),

                     Right_Opnd =>
                       Make_Unchecked_Type_Conversion (Sloc_N,
                         Subtype_Mark =>
                           New_Reference_To (RTE (RE_Tag), Sloc_N),
                         Expression =>
                           New_Reference_To (
                             Access_Disp_Table (Right_Type), Sloc_N)))));
      end if;
   end Tagged_Membership;

end Exp_Ch4;
