------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.57 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;     use Atree;
with Debug;     use Debug;
with Einfo;     use Einfo;
with Elists;    use Elists;
with Errout;    use Errout;
with Itypes;    use Itypes;
with Lib;       use Lib;
with Nlists;    use Nlists;
with Nmake;     use Nmake;
with Rtsfind;   use Rtsfind;
with Sem;       use Sem;
with Sem_Ch13;  use Sem_Ch13;
with Sem_Res;   use Sem_Res;
with Sem_Util;  use Sem_Util;
with Sinfo;     use Sinfo;
with Sinput;    use Sinput;
with Snames;    use Snames;
with Stand;     use Stand;
with Table;
with Tbuild;    use Tbuild;
with Ttypes;    use Ttypes;
with Uintp;     use Uintp;

package body Exp_Util is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id;
   --  Produce a Range node whose bounds are:
   --    Index_Typ'first .. Index_Typ'First + Length (Literal_Typ)
   --  this is used for expanding declarations like X : String := "sdfgdfg";

   function Make_Constraints_From_Expr
     (Expr     : Node_Id;
      Expr_Typ : Entity_Id;
      Unc_Type : Entity_Id)
      return     List_Id;
   --  Produce a list of constraints from the expression Expr. Unc_Type is
   --  either an unconstrained array (of any dimension) or an unconstrained
   --  discriminated type.
   --  this is used for expanding declarations like X : Unc_Type := Expr;

   ----------------
   -- Local Data --
   ----------------

   --  The following table is used to save values of the Expander_Active
   --  flag when they are saved by Expander_Mode_Save_And_Set. We use an
   --  extendible table (which is a bit of overkill) because it is easier
   --  than figuring out a maximum value or bothering with range checks!

   package Expander_Flags is new Table (
     Table_Component_Type => Boolean,
     Table_Index_Type     => Int,
     Table_Low_Bound      => 0,
     Table_Initial        => 32,
     Table_Increment      => 200,
     Table_Name           => "Expander_Flags");

   ------------------------
   -- Build_Runtime_Call --
   ------------------------

   function Build_Runtime_Call
     (Loc  : Source_Ptr;
      RE   : Rtsfind.RE_Id)
      return Node_Id
   is
   begin
      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE), Loc));
   end Build_Runtime_Call;

   ---------------------------
   -- Expander_Mode_Restore --
   ---------------------------

   procedure Expander_Mode_Restore is
   begin
      Expander_Active := Expander_Flags.Table (Expander_Flags.Last);
      Expander_Flags.Decrement_Last;

      if Errors_Detected /= 0 then
         Expander_Active := False;
      end if;
   end Expander_Mode_Restore;

   --------------------------------
   -- Expander_Mode_Save_And_Set --
   --------------------------------

   procedure Expander_Mode_Save_And_Set (Status : Boolean) is
   begin
      Expander_Flags.Increment_Last;
      Expander_Flags.Table (Expander_Flags.Last) := Expander_Active;
      Expander_Active := Status;
   end Expander_Mode_Save_And_Set;

   -------------------------------
   -- Expand_Class_Wide_Subtype --
   -------------------------------

   --  Create a record type used as an equivalent of any member of the class
   --  which takes its size from exp.

   --  Generate the following code:

   --   type Equiv_T is record
   --     _parent :  T (List of discriminant constaints taken from Exp);
   --     Ext__50 : Storage_Array (1 .. (Exp'size - Typ'size) / Storage_Unit);
   --   end Equiv_T;

   function Expand_Class_Wide_Subtype
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Exp  : Node_Id)
      return List_Id
   is
      Root_Type   : constant Entity_Id := Etype (Typ);
      Equiv_Type  : Entity_Id;
      Range_Type  : Entity_Id;
      Str_Type    : Entity_Id;
      List_Def    : List_Id := Empty_List;
      Constr_Root : Entity_Id;
      Sizexpr     : Node_Id;

   begin
      if not Has_Discriminants (Root_Type) then
         Constr_Root := Root_Type;
      else
         Constr_Root :=
           Make_Defining_Identifier (Loc, New_Internal_Name ('R'));

         --  subtype cstr__n is T (List of discr constaints taken from Exp)

         Append_To (List_Def,
           Make_Subtype_Declaration (Loc,
             Defining_Identifier => Constr_Root,
               Subtype_Indication =>
                 Make_Subtype_Indication (Loc,
                   Subtype_Mark => New_Reference_To (Root_Type, Loc),
                   Constraint =>
                     Make_Index_Or_Discriminant_Constraint (Loc,
                       Constraints =>
                         Make_Constraints_From_Expr (
                           Exp, Etype (Exp), Root_Type)))));
      end if;

      --  subtype rg__xx is Storage_Offset range
      --                           (Exp'size - typ'size) / Storage_Unit

      Range_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('G'));

      Sizexpr :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => Exp,
              Attribute_Name => Name_Size),
          Right_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => New_Reference_To (Constr_Root, Loc),
              Attribute_Name => Name_Size));

      Set_Paren_Count (Sizexpr, 1);

      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Range_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Offset), Loc),
              Constraint => Make_Range_Constraint (Loc,
                Range_Expression =>
                  Make_Range (Loc,
                    Low_Bound => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_Op_Divide (Loc,
                        Left_Opnd => Sizexpr,
                        Right_Opnd => Make_Integer_Literal (Loc,
                          Intval =>
                            UI_From_Int (System_Storage_Unit))))))));

      --  subtype str__nn is Storage_Array (rg__x);

      Str_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('S'));
      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Str_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (New_Reference_To (Range_Type, Loc))))));

      --  type Equiv_T is record
      --    _parent : Tnn;
      --    E : Str_Type;
      --  end Equiv_T;

      Equiv_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

      Append_To (List_Def,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Equiv_Type,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List => Make_Component_List (Loc,
                Component_Declarations => New_List (
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc, Name_uParent),
                    Subtype_Indication => New_Reference_To (Constr_Root, Loc)),
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc,
                        Chars => New_Internal_Name ('X')),
                    Subtype_Indication => New_Reference_To (Str_Type, Loc))),
                Variant_Part => Empty))));

      Set_Equivalent_Type (Typ, Equiv_Type);
      return List_Def;
   end Expand_Class_Wide_Subtype;

   ------------------------------
   -- Expand_Subtype_From_Expr --
   ------------------------------

   --  For an unconstained type T, change  "Val : T := Expr;" into :
   --
   --  <if Expr is a Slice>
   --    Val : T ([Slice_Range (Expr)]) := Expr;
   --
   --  <elsif Expr is a String Literal>
   --    Val : T (T'First .. T'First + Length (string literal) - 1) := Expr;
   --
   --  <elsif Expr is Constrained>
   --    Val : Type_Of_Expr := Expr;
   --
   --  <else>
   --    type Axxx is access all T;
   --    Rval : Axxx := Expr'ref;
   --    <if Expr is a function call>
   --      Val : T (contraints taken from Rval) renames Rval.all;
   --    <else>
   --      Val : T (contraints taken from Rval) := Rval.all;

   procedure Expand_Subtype_From_Expr (N : Node_Id; T : Entity_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Exp     : constant Node_Id    := Expression (N);
      Obj_Def : constant Node_Id    := Object_Definition (N);
      Exp_Typ : constant Entity_Id  := Etype (Exp);

      New_Def       : Node_Id;
      Ref_Node      : Node_Id;
      Ref_Type_Node : Node_Id;
      New_Exp       : Node_Id;
      Ref_Type      : Entity_Id;
      Ref_Id        : Entity_Id;

   begin
      if Ekind (Exp_Typ) = E_Slice_Subtype then

         New_Def :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Copy (Obj_Def),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List (Slice_Range (Exp_Typ))));
         Rewrite_Substitute_Tree (Obj_Def, New_Def);

      elsif Ekind (Exp_Typ) = E_String_Literal_Subtype then

         New_Def :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Copy (Obj_Def),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List (
                   Make_Literal_Range (Loc,
                     Literal_Typ => Exp_Typ,
                     Index_Typ   => Etype (First_Index (T))))));
         Rewrite_Substitute_Tree (Obj_Def, New_Def);

      elsif Is_Constrained (Exp_Typ) then

         New_Def := New_Reference_To (Exp_Typ, Loc);
         Rewrite_Substitute_Tree (Obj_Def, New_Def);

      else
         --  Expand: type Axxx is access all T;

         Ref_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
         Ref_Type_Node :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Ref_Type,
             Type_Definition =>
              Make_Access_To_Object_Definition (Loc,
                 All_Present => True,
                 Subtype_Indication => New_Reference_To (T, Loc)));

         Insert_Before (N, Ref_Type_Node);
         Analyze (Ref_Type_Node);

         --  Expand: Rval : Axxx := Expr'ref;

         Ref_Id :=
           Make_Defining_Identifier (Loc,
             Chars =>
               New_External_Name (Chars (Defining_Identifier (N)), 'R'));

         Ref_Node := Make_Object_Declaration (Loc,
           Defining_Identifier => Ref_Id,
           Object_Definition   => New_Reference_To (Ref_Type, Loc),
           Expression          => Make_Reference (Loc, Exp));

         Insert_Before (N, Ref_Node);
         Analyze (Ref_Node);

         --  New expression : "Rval.all"

         New_Exp := Make_Explicit_Dereference (Loc,
                      Prefix => New_Reference_To (Ref_Id, Loc));

         --  New subtype Indication : "T (constraints)"

         New_Def := Make_Subtype_Indication (Loc,
           Subtype_Mark => Obj_Def,
           Constraint =>
             Make_Index_Or_Discriminant_Constraint (Loc,
               Constraints =>
                 Make_Constraints_From_Expr (New_Exp, Exp_Typ, T)));

         if False

            --  ??? test disabled temporarily require some more work
            --  Nkind (Exp) = N_Function_Call
            --    or else (Nkind (Exp) = N_Expression_Actions
            --          and then Nkind (Expression (Exp)) = N_Function_Call)

         then
            Rewrite_Substitute_Tree (N,
              Make_Object_Renaming_Declaration (Loc,
                Defining_Identifier => Defining_Identifier (N),
                Subtype_Mark        => New_Def,
                Name                => New_Exp));
         else
            Rewrite_Substitute_Tree (N,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Defining_Identifier (N),
                Aliased_Present     => Aliased_Present (N),
                Constant_Present    => Constant_Present (N),
                Object_Definition   => New_Def,
                Expression          => New_Exp));
         end if;
      end if;
   end Expand_Subtype_From_Expr;

   ------------------------------------
   -- Insert_List_Before_And_Analyze --
   ------------------------------------

   procedure Insert_List_Before_And_Analyze (N : Node_Id; L : List_Id) is
      Node : Node_Id;
      Lend : Node_Id;

   begin
      --  Capture first and last nodes in list

      Node := First (L);
      Lend := Last (L);

      --  Now do the insertion

      Insert_List_Before (N, L);

      --  The insertion does not change the Id's of any of the nodes in the
      --  list, and the are still linked, so we can simply loop from the
      --  first to the last to get them analyzed.

      loop
         Analyze (Node);
         exit when Node = Lend;
         Node := Next (Node);
      end loop;

   end Insert_List_Before_And_Analyze;

   --------------------------------
   -- Make_Constraints_From_Expr --
   --------------------------------

   --  1. Expr is an uncontrained array expression
   --    -> (Expr'first(1)..Expr'Last(1), ... , Expr'first(n)..Expr'last(n))
   --  2. Expr is a unconstrained discriminated type expression
   --    -> (Expr.Discr1, ... , Expr.Discr_n)

   function Make_Constraints_From_Expr
     (Expr     : Node_Id;
      Expr_Typ : Entity_Id;
      Unc_Type : Entity_Id)
      return     List_Id
   is
      Loc         : constant Source_Ptr := Sloc (Expr);
      List_Constr : List_Id := New_List;
      D           : Entity_Id;
      Elmt        : Elmt_Id;

   begin

      if Is_Array_Type (Expr_Typ) then
         for I in 1 .. Number_Dimensions (Expr_Typ) loop
            Append_To (List_Constr,
              Make_Range (Loc,
                Low_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => Expr,
                    Attribute_Name => Name_First,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc, Intval => UI_From_Int (I)))),
                High_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => Expr,
                    Attribute_Name => Name_Last,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc,
                        Intval => UI_From_Int (I))))));
         end loop;
      else

         D := First_Discriminant (Unc_Type);
         while (Present (D)) loop

            Append_To (List_Constr,
              Make_Selected_Component (Loc,
                Prefix =>  Expr,
                Selector_Name => New_Reference_To (D, Loc)));

            D := Next_Discriminant (D);
         end loop;
      end if;

      return List_Constr;
   end Make_Constraints_From_Expr;

   ------------------------
   -- Make_Literal_Range --
   ------------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id
   is
   begin
         return
           Make_Range (Loc,
             Low_Bound =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Occurrence_Of (Index_Typ, Loc),
                 Attribute_Name => Name_First),
             High_Bound =>
               Make_Op_Subtract (Loc,
                  Left_Opnd =>
                    Make_Op_Add (Loc,
                      Left_Opnd =>
                        Make_Attribute_Reference (Loc,
                          Prefix => New_Occurrence_Of (Index_Typ, Loc),
                          Attribute_Name => Name_First),
                      Right_Opnd => Make_Integer_Literal (Loc,
                        String_Literal_Length (Literal_Typ))),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_1)));
   end Make_Literal_Range;

   ----------------------
   -- Make_Tagged_Copy --
   ----------------------
   --  Generate :
   --    declare
   --       subtype _A is Storage_Offset range (1 .. Address_Size)/Store_Unit;
   --                                            3 * Address_Size  <CTRL>
   --       subtype _B is Storage_Array (_A);
   --       type _C is record
   --          D : Tags.Tag;
   --          E : Finalizable_Ptr;   <CTRL>
   --          F : Finalizable_Ptr;   <CTRL>
   --          G : _B;
   --       end record;
   --   begin
   --      _C!(Lhr).G := _C!(Rhs).G;
   --   end;

   function Make_Tagged_Copy (
     Loc : Source_Ptr;
     Lhs : Node_Id;
     Rhs : Node_Id;
     Typ : Entity_Id)
     return Node_Id
   is
      Decls      : constant List_Id    := New_List;
      Stmts      : constant List_Id    := New_List;
      Comp_List  : constant List_Id    := New_List;
      Ptrs_Size  : Node_Id;
      Sizexpr    : Node_Id;

      A : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
      B : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('B'));
      C : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('C'));

   begin
      --  Sizexpr : rhs'size - Standard_Address_Size
      --     or     rhs'size - 3 * Standard_Address_Size   <CTRL>
      --  (we assume all 3 pointers, the tag and the finalization pointers,
      --   are thin pointers)

      Ptrs_Size :=
        Make_Integer_Literal (Loc,
          Intval => UI_From_Int (System_Address_Size));

      if Is_Controlled (Typ) then
         Ptrs_Size :=
           Make_Op_Multiply (Loc,
             Left_Opnd =>
               Make_Integer_Literal (Loc, Intval => UI_From_Int (3)),
             Right_Opnd => Ptrs_Size);
      end if;

      Sizexpr :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => Rhs,
              Attribute_Name => Name_Size),
          Right_Opnd => Ptrs_Size);

      Set_Paren_Count (Sizexpr, 1);

      --  Subtype _A is Storage_Offset range 1 .. size_exp/Storage_Unit;

      Append_To (Decls,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => A,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark =>
                New_Reference_To (RTE (RE_Storage_Offset), Loc),
              Constraint => Make_Range_Constraint (Loc,
                Range_Expression =>
                  Make_Range (Loc,
                    Low_Bound => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_Op_Divide (Loc,
                        Left_Opnd => Sizexpr,
                        Right_Opnd => Make_Integer_Literal (Loc,
                          Intval =>
                            UI_From_Int (System_Storage_Unit))))))));

      --  subtype _B is Storage_Array (_A);

      Append_To (Decls,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => B,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark =>
                New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (New_Reference_To (A, Loc))))));

      --  type _C is record
      --     D : Tags.Tag;
      --     E : Finalizable_Ptr;                            <CTRL>
      --     F : Finalizable_Ptr;                            <CTRL>
      --     G : _B;
      --  end record;

      Append_To (Comp_List,
        Make_Component_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_D),
          Subtype_Indication  => New_Reference_To (RTE (RE_Tag), Loc)));

      if Is_Controlled (Typ) then

         Append_To (Comp_List,
            Make_Component_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_E),
              Subtype_Indication  =>
                New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));

         Append_To (Comp_List,
            Make_Component_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_F),
              Subtype_Indication  =>
                New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));
      end if;

      Append_To (Comp_List,
        Make_Component_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_G),
          Subtype_Indication  => New_Reference_To (B, Loc)));

      Append_To (Decls,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => C,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List =>
                Make_Component_List (Loc,
                  Component_Declarations => Comp_List,
                  Variant_Part => Empty))));

      --  _C!(Lhr).G := _C!(Rhs).G

      Append_To (Stmts,
        Make_Assignment_Statement (Loc,
          Name =>
            Make_Selected_Component (Loc,
              Prefix =>
                Make_Unchecked_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (C, Loc),
                  Expression => New_Copy (Lhs)),
              Selector_Name => Make_Identifier (Loc, Name_G)),

         Expression =>
            Make_Selected_Component (Loc,
              Prefix =>
                Make_Unchecked_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (C, Loc),
                  Expression => New_Copy (Rhs)),
              Selector_Name => Make_Identifier (Loc, Name_G))));

      --  This assignment must work even for constant target (it is
      --  used for initializing tagged object)

      Set_Assignment_OK (Name (First (Stmts)));

      return
        Make_Block_Statement (Loc,
          Identifier => Empty,
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc, Stmts));
   end Make_Tagged_Copy;

   ----------------------
   -- Is_Unconstrained --
   ----------------------

   function Is_Unconstrained (N : Node_Id) return Boolean is
      Typ : Entity_Id := Etype (N);

   begin
      if Ekind (Typ) in Private_Kind then
         Typ := Full_Declaration (Typ);
      end if;

      return (Has_Discriminants (Typ) or else Is_Array_Type (Typ))
               and then not Is_Constrained (Typ);
   end Is_Unconstrained;

   ------------------------
   -- Protect_Statements --
   ------------------------

   procedure Protect_Statements (N : Node_Id; E : Entity_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Stm : constant Node_Id    := Handled_Statement_Sequence (N);

   begin
      --  If the existing statement sequence has no exception handlers, then
      --  all we need to do is to add the specified cleanup call. If there
      --  are exception handlers present, then we have to wrap an extra
      --  block around to hold the cleanup call because of the current rule
      --  that a block cannot have both a cleanup and exception handlers.

      if No (Exception_Handlers (Stm)) and then No (Identifier (Stm)) then
         Set_Identifier (Stm, New_Occurrence_Of (E, Loc));

      else
         Set_Handled_Statement_Sequence (N,
           Make_Handled_Sequence_Of_Statements (Loc,
             Statements => New_List (
               Make_Block_Statement (Loc,
                 Identifier => Empty,
                 Handled_Statement_Sequence => Stm)),
             Identifier => New_Occurrence_Of (E, Loc)));
      end if;

   end Protect_Statements;

   ---------------------
   -- Traceback_Store --
   ---------------------

   procedure Traceback_Store (N : Node_Id; Anal : Boolean := True) is
      Loc        : constant Source_Ptr := Sloc (N);
      Call       : Node_Id;
      Vname      : Name_Id;
      Kind       : Node_Kind;
      Prv        : Node_Id;
      Unum       : Unit_Number_Type;
      Unit_Node  : Node_Id;
      Sparm      : Node_Id;

   begin
      --  Immediate return if traceback flag is off, nothing to do

      if not Debug_Flag_B then
         return;
      end if;

      --  Immediate return if the node we are inserting before is another
      --  traceback store node (stops an infinite loop!)

      if Nkind (N) = N_Procedure_Call_Statement
        and then Nkind (Name (N)) = N_Identifier
        and then Entity (Name (N)) = RTE (RE_Store_TB)
      then
         return;
      end if;

      --  Immediate return if we are inserting immediately after another
      --  traceback store node (this is just to avoid unnecssary calls)

      Prv := Prev (N);

      if Present (Prv)
        and then Nkind (Prv) = N_Procedure_Call_Statement
        and then Nkind (Name (Prv)) = N_Identifier
        and then Entity (Name (Prv)) = RTE (RE_Store_TB)
      then
         return;
      end if;

      --  Immediate return if not main unit, or a subunit. All other units
      --  are being compiled for subsidiary use and do not need traceback
      --  calls inserted (furthermore, if they did have calls inserted, we
      --  get into trouble with System.Traceback itself!

      Unum := Get_Sloc_Unit_Number (Loc);

      if Unum /= Main_Unit
        and then Nkind (Unit (Cunit (Unum))) /= N_Subunit
      then
         return;
      end if;

      --  Conditions for the generation of the traceback call are met. The
      --  call to be generated has one of the following two forms (the second
      --  being used if there is no currently visible subprogram):

      --    Store_TB (linenum, Version_x'Address, _TB_Snam'Address);
      --    Store_TB (linenum, Version_x'Address, Null_Address);

      --  Now build the call

--      Call :=
--        Make_Procedure_Call_Statement (Loc,
--          Name => New_Reference_To (RTE (RE_Store_TB), Loc),
--
--          Parameter_Associations => New_List (
--            Make_Integer_Literal (Loc,
--              Intval => UI_From_Int (Int (Get_Line_Number (Loc)))),
--
--            Make_Attribute_Reference (Loc,
--              Prefix => Make_Identifier (Loc, Vname),
--              Attribute_Name => Name_Address),
--
--            Sparm));
--
--      Insert_Before (N, Call);
--
--      if Anal then
--        Analyze (Call);
--      end if;
--  ??? to be filled in later!
   end Traceback_Store;

   ----------------------------
   -- Wrap_Cleanup_Procedure --
   ----------------------------

   procedure Wrap_Cleanup_Procedure (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Stseq : constant Node_Id    := Handled_Statement_Sequence (N);
      Stmts : constant List_Id    := Statements (Stseq);

   begin
      Prepend_To (Stmts, Build_Runtime_Call (Loc, RE_Abort_Defer));
      Append_To  (Stmts, Build_Runtime_Call (Loc, RE_Abort_Undefer));
   end Wrap_Cleanup_Procedure;

end Exp_Util;
