------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             G N A T F D R V                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.18 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Csets;       use Csets;
with Comperr;
with Errout;      use Errout;
with Xref_Tab;    use Xref_Tab;
with Excep;       use Excep;
with Frontend;
with Gnatvsn;     use Gnatvsn;
with Xref;        use Xref;
with Lib;         use Lib;
with Namet;       use Namet;
with Opt;         use Opt;
with Osint;       use Osint;
with Output;      use Output;
with Par;
with Snames;
with Sprint;      use Sprint;
with Stringt;
with System.Assertions;
with Treepr;      use Treepr;
with Types;       use Types;
with Uintp;
with Usage;

procedure Gnatfdrv is

   Total_Warnings : Nat := 0;
   --  Counts total warnings in all files

   Total_Errors : Nat := 0;
   --  Counts total errors in all files

begin
   --  The following package initializations are done once for the complete
   --  set of main source files. It is in particular important that the
   --  names table not be reinitialized between compilations, since we use
   --  name table indexes in the source file table.

   Osint.Initialize (Compiler);
   Xref.Initialize;
   Csets.Initialize;
   Uintp.Initialize;
   Errout.Initialize;
   Namet.Initialize;
   Snames.Initialize;
   Stringt.Initialize;

   if Verbose_Mode or Full_List then
      Write_Eol;
      Write_Eol;
      Write_Str ("GNAT Front End/XREF Tool Version ");
      Write_Str (Gnat_Version_String);
      Write_Str (" (C) Copyright NYU, 1992,1993,1994");
      Write_Eol;
   end if;

   --  Output usage information if no files

   if not More_Source_Files then
      Usage;
      Exit_Program (E_Fatal);
   end if;

   --  Loop through files

   while More_Source_Files loop

      --  The outer block is here to handle an unrecoverable error if one
      --  is signalled (by raising the Unrecoverable_Error exception).

      begin
         --  The inner block is here to handle an assert error or constraint
         --  error. We need the nested blocks because the handling of these
         --  exceptions can end up raising an Unrecoverable_Error exception.

         begin
            Frontend;

            --  Update total error counts

            Total_Warnings := Total_Warnings + Warnings_Detected;
            Total_Errors := Total_Errors + Errors_Detected;

            --  Let the Xref gather what it needs

            if Total_Errors = 0
              and then (With_Warnings or Spec_REQs_Flag or Body_REQs_Flag)
            then
               Gather_Info (Cunit (Main_Unit));
            end if;

         --  Exception handler catches fatal internal errors

         exception

            when System.Assertions.Assert_Failure =>
               Comperr.Compiler_Abort ("Assert_Failure raised");

            when Constraint_Error =>
               Comperr.Compiler_Abort ("Constraint_Error raised");

            when Program_Error =>
               Comperr.Compiler_Abort ("Program_Error raised");

            when Storage_Error =>
               Comperr.Compiler_Abort ("Storage_Error raised");

         end;

      --  This is the handler for the outer block

      exception
         when Unrecoverable_Error =>
            Total_Warnings := Total_Warnings + Warnings_Detected;
            Total_Errors := Total_Errors + Errors_Detected;
            Errout.Finalize;
            Set_Standard_Error;
            Write_Str ("compilation of ");
            Write_Name (Unit_File_Name (Main_Unit));
            Write_Str (" abandoned");
            Write_Eol;
            Set_Standard_Output;
            Tree_Dump;
            Source_Dump;
            Exit_Program (E_Errors);
      end;

      Errout.Finalize;
      Namet.Finalize;
   end loop;

   if Total_Errors = 0 and With_Warnings then
      Write_Xref;
   end if;

   --  All done. Set proper exit status

   if Total_Errors > 0 then
      Exit_Program (E_Errors);
   else
      Exit_Program (E_Success);
   end if;

end Gnatfdrv;
