------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ C H 1 2                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.148 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_Util; use Exp_Util;
with Lib;      use Lib;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch6;  use Sem_Ch6;
with Sem_Ch7;  use Sem_Ch7;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch10; use Sem_Ch10;
with Sem_Ch13; use Sem_Ch13;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinfo.CN; use Sinfo.CN;
with Snames;   use Snames;
with Uname;    use Uname;
with Table;
with Tbuild;   use Tbuild;

package body Sem_Ch12 is

   use Atree.Unchecked_Access;
   --  This package performs untyped traversals of the tree, therefore it
   --  needs direct access to the fields of a node.

   -----------------------------------------------------------
   --  Implementation of generic analysis and instantiation --
   -----------------------------------------------------------

   --  GNAT implements generics by macro expansion. No attempt is made to
   --  share generic instantions (for now). Analysis of a generic definition
   --  does not perform any expansion action, but the expander must be called
   --  on the tree for each instantiation, because the expansion may of course
   --  depend  on the generic actuals. All of this is best achieved as follows:
   --
   --  a) Semantic analysis of a generic unit is performed on  a copy of the
   --  tree for the  generic unit. All tree modifications that follow analysis
   --  do not affect the original tree. Links are kept between the original
   --  tree and the copy,  in order to recognize non-local references within
   --  the generic, and propagate them to each instance (recall that name
   --  resolution is done on the generic declaration: generics are not really
   --  macros!). This is summarized in the following diagram:
   --
   --              .-----------.               .----------.
   --              |  semantic |<--------------|  generic |
   --              |    copy   |               |    unit  |
   --              |           |==============>|          |
   --              |___________|    global     |__________|
   --                             references     |   |  |
   --                                            |   |  |
   --                                          .-----|--|.
   --                                          |  .-----|---.
   --                                          |  |  .----------.
   --                                          |  |  |  generic |
   --                                          |__|  |          |
   --                                             |__| instance |
   --                                                |__________|
   --
   --  b) Each instantiation copies the original tree, and inserts into it a
   --  series of declarations that describe the mapping between generic formals
   --  and actuals. For example, a generic In OUT parameter is  an object
   --  renaming of the corresponing actual, etc. Generic IN parameters are
   --  constant declarations.
   --
   --  c) In order to give the right visibility for these renamings, we use
   --  a different scheme for package and subprogram instantiations. For
   --  packages, the list of renamings is inserted into the package
   --  specification, before the visible declarations of the package. The
   --  renamings are analyzed before any of the text of the instance, and are
   --  thus visible at the right place. Furthermore, outside of the instance,
   --  the generic parameters are visible and denote their corresponding
   --  actuals.

   --  For subprograms, we create a container package to hold the renamings
   --  and the subprogram instance itself. Analysis of the package makes the
   --  renaming declarations visible to the subprogram. after analyzing the
   --  package, the defining entity for the subprogram is touched-up so that
   --  it appears declared in the current scope, and not inside the container
   --  package.

   --  If the instantiation is a compilation unit, the container package is
   --  given the same name as the subprogram instance. This ensures that
   --  the elaboration procedure called by the binder, using the compilation
   --  unit name, calls in fact the elaboration procedure for the package.

   --  Not surprisingly, private types complicate this approach. By saving in
   --  the original generic object the non-local references, we guarantee that
   --  the proper entities are referenced at the point of instantiation.
   --  However, for private types, this by itself does not insure that the
   --  proper VIEW of the entity is used (the full type may be visible at the
   --  point of generic definition, but not at instantiation, or viceversa).
   --  In  order to reference the proper view, we special-case any reference
   --  to private types in the generic object, by saving boths views, one in
   --  the generic and one in the semantic copy. At time of instantiation, we
   --  check whether the two views are consistent, and exchange declarations if
   --  necessary, in  order to restore the correct visibility. Similarly, if
   --  the instance view is private when the generic view was not, we perform
   --  the exchange.  After completing the instantiation,  we restore the
   --  current visibility. The flag Has_Private_View marks identifiers in the
   --  the generic unit that require checking.

   -----------------------
   -- Local subprograms --
   -----------------------

   --  All these routines need documentation ???

   procedure Analyze_Formal_Array_Type   (T : in out Entity_Id; Def : Node_Id);
   --  Really need documentation as to why T is in out ???

   procedure Analyze_Formal_Decimal_Fixed_Point (T : Entity_Id; Def : Node_Id);

   procedure Analyze_Formal_Derived_Type
     (N   : Node_Id;
      T   : Entity_Id;
      Def : Node_Id);

   procedure Analyze_Formal_Discrete_Type       (T : Entity_Id; Def : Node_Id);
   procedure Analyze_Formal_Floating_Type       (T : Entity_Id; Def : Node_Id);
   procedure Analyze_Formal_Signed_Integer_Type (T : Entity_Id; Def : Node_Id);
   procedure Analyze_Formal_Modular_Type        (T : Entity_Id; Def : Node_Id);
   procedure Analyze_Formal_Ordinary_Fixed_Type (T : Entity_Id; Def : Node_Id);

   procedure Analyze_Formal_Private_Type
     (N   : Node_Id;
      T   : Entity_Id;
      Def : Node_Id);

   procedure Analyze_Generic_Formal_Part        (N : Node_Id);
   procedure Analyze_Generic_Access_Type        (T : Entity_Id; Def : Node_Id);

   function Analyze_Associations
     (Formals : List_Id;
      Actuals : List_Id)
      return List_Id;

   procedure Analyze_Subprogram_Instantiation
     (N : Node_Id;
      K : Entity_Kind);

   procedure Build_Instance_Compilation_Unit_Nodes
     (N        : Node_Id;
      Act_Body : Node_Id;
      Act_Decl : Node_Id);
   --  This procedure is used in the case where the generic instance of a
   --  subprogram body or package body is a library unit. In this case, the
   --  original library unit node for the generic instantiation must be
   --  replaced by the resulting generic body, and a link made to a new
   --  compilation unit node for the generic declaration. The argument N is
   --  the original generic instantiation. Act_Body and Act_Decl are the body
   --  and declaration of the instance (either package body and declaration
   --  nodes or subprogram body and declaration nodes depending on the case).
   --  On return, the node N has been rewritten with the actual body.

   function Get_Instance_Of  (A : Entity_Id) return Entity_Id;
   --  Retrieve actual associated with given generic parameter.
   --  If parameter is uninstantiated still, return generic.

   procedure Set_Instance_Of (A : Entity_Id; B : Entity_Id);
   --  Associate generic type or package parameter with corresponding
   --  instance. Used for semantic checks at instantiation time.

   --  The functions Instantiate_XXX perform various legality checks and build
   --  the declarations for instantiated generic parameters.
   --  Need to describe what the parameters are ???

   function Instantiate_Object
      (Formal : Node_Id;
       Actual : Node_Id)
      return Node_Id;

   function Instantiate_Type
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id;

   function Instantiate_Formal_Subprogram
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id;

   function Instantiate_Formal_Package
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id;

   procedure Load_Parent_Of_Generic (N : Entity_Id; Spec : Node_Id);
   --  If the generic appears in a separate non-generic library unit,
   --  load the corresponding body to retrieve the body of the generic.

   procedure Inherit_Context (Gen_Decl : Node_Id);
   --  If a generic is a compilation unit, its instantiation has semantic
   --  dependences on the context units of the generic. Eventually these
   --  dependences will be reflected in actual ali files for generic units.
   --  In the meantime, the simplest is to attach the with clauses of the
   --  generic compilation to the compilation that contains the instance.

   function Associated_Node (N : Node_Id) return Node_Id;
   --  Nodes in a generic unit that have an entity field are linked to the
   --  corresponding nodes in the semantic copy, so that non-local references
   --  in the copy can be marked in the original generic nodes. The link
   --  overlaps the Entity field of the node, and must be reset correctly
   --  after collecting global references.

   procedure Set_Associated_Node
     (Gen_Node  : Node_Id;
      Copy_Node : Node_Id);
   --  Establish the link between an identifier in the generic unit, and the
   --  corresponding node in the semantic copy.

   -------------------------------------------
   -- Data structures for generic renamings --
   -------------------------------------------

   --  Need more documentation of what Assoc and the table are for ???

   type Assoc is record
      Gen_Id : Entity_Id;
      Act_Id : Entity_Id;
   end record;

   package Generic_Renamings is new Table
     (Table_Component_Type => Assoc,
      Table_Index_Type     => Int,
      Table_Low_Bound      => 0,
      Table_Initial        => 10,
      Table_Increment      => 10,
      Table_Name           => "Generic_Renamings");

   Exchanged_Views : Elist_Id;
   --  This list holds the private views that have been exchanged during
   --  instantiation to restore the visibility of the generic declaration.
   --  (see comments above). After instantiation, the current visibility is
   --  reestablished by means of a traversal of this list.

   procedure Restore_Private_Views;

   ------------------------------------
   -- Structures for Error Reporting --
   ------------------------------------

   Instantiation_Node  : Node_Id;
   --  Used by subprograms that validate instantiation of formal parameters
   --  where there might be no actual on which to place the error message.

   Instantiation_Error : exception;
   --   ??? this needs documentation

   -------------------------------------------
   --  Analyze_Generic_Package_Declaration  --
   -------------------------------------------

   procedure Analyze_Generic_Package_Declaration (N : Node_Id) is
      Id : Entity_Id;
      New_N : Node_Id;
      Save_Parent : Node_Id;

   begin
      --  Create copy of generic unit, and save for instantiation.
      --  If the unit is a child unit, do not copy the specifications
      --  for the parent,  which are not part of the generic tree.

      Save_Parent := Parent_Spec (N);
      Set_Parent_Spec (N, Empty);

      New_N := Copy_Generic_Node (N, Empty, Instantiating => False);
      Set_Parent_Spec (New_N, Save_Parent);
      Rewrite_Substitute_Tree (N, New_N);
      Id  := Defining_Unit_Simple_Name (Specification (N));

      --  Expansion is not applied to generic units.

      Expander_Mode_Save_And_Set (False);

      Enter_Name (Id);
      Set_Ekind (Id, E_Generic_Package);
      Set_Etype (Id, Standard_Void_Type);
      New_Scope (Id);
      Analyze_Generic_Formal_Part (N);

      --  After processing the generic formals, analysis proceeds
      --  as for a non-generic package.

      Analyze (Specification (N));

      Save_Global_References (Original_Node (N));
      Expander_Mode_Restore;
      End_Package_Scope (Id);

   end Analyze_Generic_Package_Declaration;

   ---------------------------------------------
   --  Analyze_Generic_Subprogram_Declaration --
   ---------------------------------------------

   procedure Analyze_Generic_Subprogram_Declaration (N : Node_Id) is
      Spec    : Node_Id;
      Id      : Entity_Id;
      Formals : List_Id;
      New_N   : Node_Id;
      Save_Parent : Node_Id;

   begin
      --  Create copy of generic unit,and save for instantiation.
      --  If the unit is a child unit, do not copy the specifications
      --  for the parent, which are not part of the generic tree.

      Save_Parent := Parent_Spec (N);
      Set_Parent_Spec (N, Empty);

      New_N := Copy_Generic_Node (N, Empty,  Instantiating => False);
      Set_Parent_Spec (New_N, Save_Parent);
      Rewrite_Substitute_Tree (N, New_N);

      Spec := Specification (N);
      Id := Defining_Unit_Simple_Name (Spec);

      --  Expansion is not applied to generic units.

      Expander_Mode_Save_And_Set (False);

      Enter_Name (Id);

      if Nkind (Spec) = N_Function_Specification then
         Set_Ekind (Id, E_Generic_Function);
      else
         Set_Ekind (Id, E_Generic_Procedure);
         Set_Etype (Id, Standard_Void_Type);
      end if;

      New_Scope (Id);
      Analyze_Generic_Formal_Part (N);

      Formals := Parameter_Specifications (Spec);

      if Present (Formals) then
         Process_Formals (Id, Formals, Spec);
      end if;

      if Nkind (Spec) = N_Function_Specification then
         Find_Type (Subtype_Mark (Spec));
         Set_Etype (Id, Entity (Subtype_Mark (Spec)));
      end if;

      Save_Global_References (Original_Node (N));
      Expander_Mode_Restore;
      End_Scope;

   end Analyze_Generic_Subprogram_Declaration;

   ----------------------------------
   --  Analyze_Generic_Formal_Part --
   ----------------------------------

   procedure Analyze_Generic_Formal_Part (N : Node_Id) is
      Gen_Parm_Decl : Node_Id;

   begin
      --  The generic formals are processed in the scope of the generic
      --  unit, where they are directly visible. The scope is installed
      --  by the caller.

      Gen_Parm_Decl := First (Generic_Formal_Declarations (N));

      while Present (Gen_Parm_Decl) loop
         Analyze (Gen_Parm_Decl);
         Gen_Parm_Decl := Next (Gen_Parm_Decl);
      end loop;
   end Analyze_Generic_Formal_Part;

   ----------------------
   --  Is_In_Main_Unit --
   ----------------------

   function Is_In_Main_Unit (N : Node_Id) return Boolean is
      Unum : constant Unit_Number_Type := Get_Sloc_Unit_Number (Sloc (N));
      Current_Unit : Node_Id;

   begin
      if Unum = Main_Unit then
         return True;
      elsif Nkind (N) = N_Compilation_Unit then
         return False;
      end if;

      Current_Unit := Parent (N);
      while Present (Current_Unit)
        and then Nkind (Current_Unit) /= N_Compilation_Unit
      loop
         Current_Unit := Parent (Current_Unit);
      end loop;

      --  The instantiation node is in the main unit, or else the current
      --  node (perhaps as the result of nested instantiations) is in the
      --  main unit, or in the declaration of the main unit, which in this
      --  last case must be a body.

      return Unum = Main_Unit
        or else Current_Unit = Cunit (Main_Unit)
        or else Current_Unit = Library_Unit (Cunit (Main_Unit))
        or else (Present (Library_Unit (Current_Unit))
          and then Is_In_Main_Unit (Library_Unit (Current_Unit)));
   end Is_In_Main_Unit;

   -----------------------------------
   -- Analyze_Package_Instantiation --
   -----------------------------------

   procedure Analyze_Package_Instantiation (N : Node_Id) is
      Loc           : constant Source_Ptr := Sloc (N);
      Actuals       : constant List_Id    := Generic_Associations (N);
      Gen_Id        : constant Node_Id    := Name (N);

      Act_Decl      : Node_Id;
      Act_Decl_Id   : Entity_Id;
      Act_Spec      : Node_Id;
      Act_Tree      : Node_Id;

      Formals       : List_Id;
      Gen_Decl      : Node_Id;
      Gen_Unit      : Entity_Id;

      Renaming_List : List_Id;
      Unit_Renaming : Node_Id;

   begin
      --  Make node global for error reporting.

      Instantiation_Node := N;

      if Nkind (N) = N_Package_Instantiation then
         Act_Decl_Id := New_Copy (Defining_Unit_Simple_Name (N));
      else
         --  Instantiation of a formal package.

         Act_Decl_Id := Defining_Identifier (N);
      end if;

      Find_Name (Gen_Id);
      Gen_Unit := Entity (Gen_Id);

      --  If renaming, indicate this is an instantiation of renamed unit

      if Present (Renamed_Object (Gen_Unit)) then
         Gen_Unit := Renamed_Object (Gen_Unit);
         Set_Entity (Gen_Id, Gen_Unit);
      end if;

      --  Verify that it is the name of a generic package

      if Etype (Gen_Unit) = Any_Type then
         return;
      end if;

      if Ekind (Gen_Unit) /= E_Generic_Package then
         Error_Msg_N
            ("expect name of generic package in instantiation", Gen_Id);

      else
         Gen_Decl := Get_Declaration_Node (Gen_Unit);

         --  Initialize renamings map, for error checking, and the list
         --  that holds private entities whose views have changed between
         --  generic definition and instantiation.

         Exchanged_Views := New_Elmt_List;
         Generic_Renamings.Set_Last (0);

         --  Copy original generic tree, to produce text for instantiation.

         Act_Tree := Copy_Generic_Node
                   (Original_Node (Gen_Decl), Empty, Instantiating => True);

         Act_Spec := Specification (Act_Tree);
         Formals := Generic_Formal_Declarations (Act_Tree);
         Renaming_List := Analyze_Associations (Formals, Actuals);
         Set_Defining_Unit_Name (Act_Spec, Act_Decl_Id);
         Set_Generic_Parent (Act_Spec, Gen_Unit);

         --  References to the generic in its own declaration or its body
         --  are references to the instance. Add a renaming declaration for
         --  the generic unit itself.

         Unit_Renaming :=
           Make_Package_Renaming_Declaration (Loc,
             Defining_Unit_Name =>
               Make_Defining_Identifier (Loc, Chars (Gen_Id)),
             Name => Make_Identifier (Loc, Chars (Act_Decl_Id)));

         --  Prepend (Unit_Renaming, Renaming_List); ???

         --  The renaming declarations are the first local declarations of
         --  the new unit.

         if Is_Non_Empty_List (Visible_Declarations (Act_Spec)) then
            Insert_List_Before
              (First (Visible_Declarations (Act_Spec)), Renaming_List);
         else
            Set_Visible_Declarations (Act_Spec, Renaming_List);
         end if;

         Act_Decl := Make_Package_Declaration (Loc,
           Specification => Act_Spec);

         if Nkind (Parent (N)) /= N_Compilation_Unit then
            Insert_Before (N, Act_Decl);
            Analyze (Act_Decl);

         else
            --  Place declaration on current node so context is complete
            --  for analysis (including nested instantiations).

            Set_Unit (Parent (N), Act_Decl);
            Analyze (Act_Decl);
            Set_Unit (Parent (N), N);
         end if;

         --  Save the instantiation node, for subsequent instantiation
         --  of the body, if there is one and we are generating code for
         --  the current unit. Mark the unit as having a body, to avoid
         --  a premature error message.

         if Unit_Requires_Body (Act_Decl_Id)
           and then Is_In_Main_Unit (N)
           and then Expander_Active
         then
            Pending_Instantiations.Increment_Last;
            Pending_Instantiations.Table (Pending_Instantiations.Last) :=
                      (N, Act_Decl);

         elsif Nkind (Parent (N)) = N_Compilation_Unit then
            Set_Parent_Spec (Act_Decl, Parent_Spec (N));
            Rewrite_Substitute_Tree (N, Act_Decl);
         end if;

         Set_Has_Completion (Act_Decl_Id);

         Restore_Private_Views;
         Inherit_Context (Gen_Decl);
      end if;

   exception
      when Instantiation_Error =>
         Error_Msg_N ("instantiation abandoned",  N);
   end Analyze_Package_Instantiation;

   ------------------------------
   -- Instantiate_Package_Body --
   ------------------------------

   procedure Instantiate_Package_Body
     (N        : Node_Id;
      Act_Decl : Node_Id)
   is
      Gen_Unit      : constant Entity_Id := Entity (Name (N));
      Gen_Decl      : constant Node_Id   := Get_Declaration_Node (Gen_Unit);
      Act_Decl_Id   : constant Entity_Id :=
                        Defining_Unit_Name (Specification (Act_Decl));
      Gen_Body      : Node_Id;
      Gen_Body_Id   : Node_Id;
      Act_Body      : Node_Id;
      Act_Body_Id   : Entity_Id;

   begin
      Gen_Body_Id := Corresponding_Body (Gen_Decl);

      if No (Gen_Body_Id) then
         Load_Parent_Of_Generic (N, Specification (Gen_Decl));
         Gen_Body_Id := Corresponding_Body (Gen_Decl);
      end if;

      if Present (Gen_Body_Id) then

         Gen_Body := Get_Declaration_Node (Gen_Body_Id);
         Act_Body := Copy_Generic_Node
                (Original_Node (Gen_Body), Empty, Instantiating => True);
         Act_Body_Id := Defining_Unit_Simple_Name (Act_Body);
         Set_Chars (Act_Body_Id, Chars (Act_Decl_Id));
         Set_Corresponding_Spec (Act_Body, Act_Decl_Id);
         Exchanged_Views := New_Elmt_List;

         --  If the instantiation is a library unit, then build
         --  the resulting compilation unit nodes for the instance

         if Nkind (Parent (N)) = N_Compilation_Unit then
            Build_Instance_Compilation_Unit_Nodes (N, Act_Body, Act_Decl);

         --  If the instantiation is not a library unit, then place the
         --  body to the current declarative part.  The specification has
         --  already been entered at the point of its instantiation.

         else
            Insert_Before (N, Act_Body);
         end if;

         Analyze (Act_Body);
         Inherit_Context (Gen_Body);
         Restore_Private_Views;

      --  Case of package that does not need a body

      else
         --  If the instantiation of the declaration is a library unit,
         --  rewrite the original package instantiation as a package
         --  declaration in the compilation unit node.

         if Nkind (Parent (N)) = N_Compilation_Unit then
            Set_Parent_Spec (Act_Decl, Parent_Spec (N));
            Rewrite_Substitute_Tree (N, Act_Decl);

         --  If the instantiation is not a library unit, then append the
         --  declaration to the list of implicitly generated entities.

         else
            Insert_Before (N, Act_Decl);
         end if;
      end if;
   end Instantiate_Package_Body;

   ---------------------------------
   -- Instantiate_Subprogram_Body --
   ---------------------------------

   procedure Instantiate_Subprogram_Body
     (N        : Node_Id;
      Act_Decl : Node_Id)
   is
      Gen_Unit      : constant Entity_Id := Entity (Name (N));
      Gen_Decl      : constant Node_Id   := Get_Declaration_Node (Gen_Unit);
      Act_Decl_Id   : constant Entity_Id :=
                        Defining_Unit_Name (Specification (Act_Decl));
      Gen_Body      : Node_Id;
      Gen_Body_Id   : Node_Id;
      Act_Body      : Node_Id;
      Act_Body_Id   : Entity_Id;
      Loc           : constant Source_Ptr := Sloc (N);
      Pack_Id       : Entity_Id := Defining_Unit_Name (Parent (Act_Decl));
      Pack_Body     : Node_Id;
      Unit_Renaming : Node_Id;

   begin
      Gen_Body_Id := Corresponding_Body (Gen_Decl);

      if No (Gen_Body_Id) then
         Load_Parent_Of_Generic (N, Specification (Gen_Decl));
         Gen_Body_Id := Corresponding_Body (Gen_Decl);
      end if;

      if Present (Gen_Body_Id) then

         Gen_Body := Get_Declaration_Node (Gen_Body_Id);
         Act_Body := Copy_Generic_Node
                (Original_Node (Gen_Body), Empty, Instantiating => True);
         Act_Body_Id := Defining_Unit_Simple_Name (Specification (Act_Body));
         Set_Chars (Act_Body_Id, Chars (Act_Decl_Id));
         Set_Corresponding_Spec (Act_Body, Act_Decl_Id);
         Exchanged_Views := New_Elmt_List;

         --  Inside its body, a reference to the generic unit is a reference
         --  to the instance. The corresponding renaming is the first
         --  declaration in the body.

         Unit_Renaming :=
           Make_Subprogram_Renaming_Declaration (Loc,
             Specification =>
               Copy_Generic_Node (
                 Specification (Original_Node (Gen_Body)),
                 Empty,
                 Instantiating => True),
                Name => New_Occurrence_Of (Act_Decl_Id, Loc));

         --  The subprogram body is placed in the body of a dummy package
         --  body, whose spec contains the subprogram declaration as well
         --  as the renaming declarations for the generic parameters.

         Pack_Body := Make_Package_Body (Loc,
           Defining_Unit_Name => New_Copy (Pack_Id),
           Declarations       => New_List (Unit_Renaming, Act_Body));

         Set_Corresponding_Spec (Pack_Body, Pack_Id);

         --  If the instantiation is a library unit, then build
         --  the resulting compilation unit nodes for the instance
         --  The declaration of the enclosing package is the grandparent
         --  of the subprogram declaration. First replace the instantiation
         --  node as the unit of the corresponding compilation.

         if Nkind (Parent (N)) = N_Compilation_Unit then
            Set_Unit (Parent (N), N);
            Build_Instance_Compilation_Unit_Nodes (N, Pack_Body,
               Parent (Parent (Act_Decl)));

            Analyze (N);

         --  If the instantiation is not a library unit, then place the
         --  body in the current declarative part.

         else
            Insert_Before (N, Pack_Body);
            Analyze (Pack_Body);
         end if;

         Inherit_Context (Gen_Body);
         Restore_Private_Views;

      else
         --  Body not found. Error was emitted already.
         null;
      end if;

   end Instantiate_Subprogram_Body;

   -------------------------------------
   -- Analyze_Procedure_Instantiation --
   -------------------------------------

   procedure Analyze_Procedure_Instantiation (N : Node_Id) is
   begin
      Analyze_Subprogram_Instantiation (N, E_Procedure);
   end Analyze_Procedure_Instantiation;

   ------------------------------------
   -- Analyze_Function_Instantiation --
   ------------------------------------

   procedure Analyze_Function_Instantiation (N : Node_Id) is
   begin
      Analyze_Subprogram_Instantiation (N, E_Function);
   end Analyze_Function_Instantiation;

   ------------------------------------
   -- Analyze_Subprogram_Instantiation --
   ------------------------------------

   procedure Analyze_Subprogram_Instantiation
     (N : Node_Id;
      K : Entity_Kind)
   is
      Loc           : constant Source_Ptr := Sloc (N);
      Actuals       : constant List_Id    := Generic_Associations (N);
      Gen_Id        : constant Node_Id    := Name (N);

      Act_Decl_Id   : Entity_Id := New_Copy (Defining_Unit_Simple_Name (N));
      Act_Decl      : Node_Id;
      Act_Spec      : Node_Id;
      Act_Tree      : Node_Id;

      Formals       : List_Id;
      Gen_Unit      : Entity_Id;
      Gen_Decl      : Node_Id;
      Renaming_List : List_Id;
      Spec          : Node_Id;

      procedure Analyze_Instance_And_Renamings;
      --  The instance must be analyzed in a context that includes the
      --  mappings of generic parameters into actuals. We create a package
      --  declaration for this purpose. After analysis,  we reset the scope
      --  of the instance to be the current one, rather than the bogus package.

      procedure Analyze_Instance_And_Renamings is
         Pack_Decl : Node_Id;
         Pack_Id   : Entity_Id;

      begin
         if Nkind (Parent (N)) = N_Compilation_Unit then

            --  The container package has the same name as the instantiation,
            --  to insure that the binder calls the elaboration procedure
            --  with the right name.

            Pack_Id := Make_Defining_Identifier (Loc, Chars (Act_Decl_Id));

         else
            Pack_Id := Make_Defining_Identifier (Loc, New_Internal_Name ('P'));
         end if;

         Pack_Decl := Make_Package_Declaration (Loc,
           Specification => Make_Package_Specification (Loc,
             Defining_Unit_Name  => Pack_Id,
             Visible_Declarations => Renaming_List));

         if Nkind (Parent (N)) /= N_Compilation_Unit then
            Insert_Before (N, Pack_Decl);
            Set_Has_Completion (Pack_Id);

         else
            --  Place declaration on current node so context is complete
            --  for analysis (including nested instantiations), and for
            --  use in  a context_clause (see Analyze_With_Clause).

            Set_Unit (Parent (N), Pack_Decl);
         end if;

         Analyze (Pack_Decl);

         --  Body of the enclosing package is supplied when instantiating
         --  the subprogram body, after semantic  analysis is completed.

         --  Insert subprogram entity into current scope, so that
         --  visiblity is correct for callers. First remove subprogram
         --  from visibility, so that subsequent insertion works properly.

         if Nkind (Parent (N)) = N_Compilation_Unit then

            --  Skip package as well.

            Set_Name_Entity_Id
               (Chars (Act_Decl_Id), Homonym (Homonym (Act_Decl_Id)));

         else
            Set_Name_Entity_Id (Chars (Act_Decl_Id), Homonym (Act_Decl_Id));
         end if;

         New_Overloaded_Entity (Act_Decl_Id);

      end Analyze_Instance_And_Renamings;

   --  Start processing for Analyze_Subprogram_Instantiation

   begin
      --  Make node global for error reporting.

      Instantiation_Node := N;

      Find_Name (Gen_Id);
      Gen_Unit := Entity (Gen_Id);

      --  If renaming, indicate that this is instantiation of renamed unit

      if Present (Renamed_Object (Gen_Unit)) then
         Gen_Unit := Renamed_Object (Gen_Unit);
         Set_Entity (Gen_Id, Gen_Unit);
      end if;

      if Etype (Gen_Unit) = Any_Type then return; end if;

      --  Verify that it is a generic subprogram of the right kind.

      if K = E_Procedure and then Ekind (Gen_Unit) /= E_Generic_Procedure then
         Error_Msg_N
            ("expect name of generic procedure in instantiation", Gen_Id);

      elsif K = E_Function and then Ekind (Gen_Unit) /= E_Generic_Function then
         Error_Msg_N
            ("expect name of generic function in instantiation", Gen_Id);

      else
         Gen_Decl := Get_Declaration_Node (Gen_Unit);
         Spec := Specification (Gen_Decl);

         --  Initialize renamings map, for error checking.

         Exchanged_Views := New_Elmt_List;
         Generic_Renamings.Set_Last (0);

         --  Copy original generic tree, to produce text for instantiation.

         Act_Tree := Copy_Generic_Node
                   (Original_Node (Gen_Decl), Empty, Instantiating => True);

         Act_Spec := Specification (Act_Tree);
         Formals := Generic_Formal_Declarations (Act_Tree);
         Renaming_List := Analyze_Associations (Formals, Actuals);

         Set_Defining_Unit_Name (Act_Spec, Act_Decl_Id);
         Set_Generic_Parent (Act_Spec, Gen_Unit);
         Act_Decl :=
           Make_Subprogram_Declaration (Loc,
             Specification => Act_Spec);

         Append (Act_Decl, Renaming_List);
         Set_Has_Completion (Act_Decl_Id);

         if Chars (Gen_Unit) = Name_Unchecked_Conversion
           and then Has_Convention_Intrinsic (Gen_Unit)
            --  ??? probably should be testing Is_Intrinsic_Subprogram here
         then
            --  Empty body for subprogram, which is special-cased.

            Analyze_Instance_And_Renamings;
            Validate_Unchecked_Conversion (N, Act_Decl_Id);
            Set_Has_Convention_Intrinsic (Act_Decl_Id);
            --  Probably should be setting Is_Intrinsic_Subprogram ???
            return;
         end if;

         Analyze_Instance_And_Renamings;
         Inherit_Context (Gen_Decl);
         Restore_Private_Views;

         --  If the context requires a full instantiation, mark node for
         --  subsequent construction of the body.

         if Is_In_Main_Unit (N)
           and then Expander_Active
         then
            Pending_Instantiations.Increment_Last;
            Pending_Instantiations.Table (Pending_Instantiations.Last) :=
              (N, Act_Decl);
         end if;
      end if;

   exception
      when Instantiation_Error =>
         Error_Msg_N ("instantiation abandoned",  N);

   end Analyze_Subprogram_Instantiation;

   ------------------------
   -- Instantiate_Record --
   ------------------------

   function Instantiate_Record
     (Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
      return         Node_Id
   is
   begin
      return Copy_Generic_Node (Type_Definition (Parent (Parent_Type)),
                                Empty,
                                Instantiating => True);
   end Instantiate_Record;

   ----------------------------
   -- Load_Parent_Of_Generic --
   ----------------------------

   procedure Load_Parent_Of_Generic (N : Entity_Id; Spec : Node_Id) is
   begin
      if Get_Sloc_Unit_Number (Sloc (N)) /=
         Get_Sloc_Unit_Number (Sloc (Spec))
      then
         --  Find body of parent of spec, and analyze it

         Load_Needed_Body (Cunit (Get_Sloc_Unit_Number (Sloc (Spec))));
      end if;
   end Load_Parent_Of_Generic;

   ---------------------
   -- Inherit_Context --
   ---------------------

   procedure Inherit_Context (Gen_Decl : Node_Id) is
      Current_Context : List_Id;
      Item  : Node_Id;
      New_I : Node_Id;

   begin
      if Nkind (Parent (Gen_Decl)) = N_Compilation_Unit then
         Current_Context := Context_Items (Cunit (Main_Unit));

         Item := First (Context_Items (Parent (Gen_Decl)));
         while Present (Item) loop

            if Nkind (Item) = N_With_Clause then
               New_I := New_Copy (Item);
               Set_Implicit_With (New_I, True);
               Append (New_I, Current_Context);
            end if;

            Item := Next (Item);
         end loop;
      end if;
   end Inherit_Context;

   --------------------------
   -- Analyze_Associations --
   --------------------------

   function Analyze_Associations
     (Formals : List_Id;
      Actuals : List_Id)
      return    List_Id
   is
      Actual      : Node_Id;
      Assoc       : List_Id := New_List;
      Formal      : Node_Id;
      Match       : Node_Id;
      Named       : Node_Id;
      First_Named : Node_Id := Empty;
      Num_Matched : Int := 0;
      Num_Actuals : Int := 0;

      function Matching_Actual (F : Entity_Id) return Node_Id;
      --  Find actual that corresponds to a given a formal parameter. If the
      --  actuals are positional,  return the next one, if any. If the actuals
      --  are named, scan the parameter associations to find the right one.

      function Matching_Actual (F : Entity_Id) return Node_Id is
         Found : Node_Id;

      begin
         --  End of list of purely positional parameters

         if No (Actual) then
            Found := Empty;

         --  Case of positional parameter correspond to current formal

         elsif No (Selector_Name (Actual)) then
            Found := Explicit_Generic_Actual_Parameter (Actual);
            Num_Matched := Num_Matched + 1;
            Actual := Next (Actual);

         --  Otherwise scan list of named actuals to find the one with the
         --  desired name. All remaining actuals have explicit names.

         else
            Found := Empty;

            while Present (Actual) loop
               if Chars (Selector_Name (Actual)) = Chars (F) then
                  Found := Explicit_Generic_Actual_Parameter (Actual);
                  Num_Matched := Num_Matched + 1;
                  exit;
               end if;

               Actual := Next (Actual);
            end loop;

            --  Reset for subsequent searches.

            Actual := First_Named;
         end if;

         return Found;
      end Matching_Actual;

   --  Start processing for Analyze_Associations

   begin
      --  If named associations are present, save the first named association
      --  (it may of course be Empty) to facilitate subsequent name search.

      if Present (Actuals) then
         First_Named := First (Actuals);

         while Present (First_Named)
           and then No (Selector_Name (First_Named))
         loop
            Num_Actuals := Num_Actuals + 1;
            First_Named := Next (First_Named);
         end loop;
      end if;

      Named := First_Named;
      while Present (Named) loop
         if No (Selector_Name (Named)) then
            Error_Msg_N ("invalid positional actual after named one", Named);

            --  No point in  countinuing with associations.

            raise Instantiation_Error;
         end if;

         Num_Actuals := Num_Actuals + 1;
         Named := Next (Named);
      end loop;

      if Present (Formals) then
         Formal := First (Formals);

         if Present (Actuals) then
            Actual := First (Actuals);

         --  All formals should have default values

         else
            Actual := Empty;
         end if;

         while Present (Formal) loop

            case Nkind (Formal) is
               when N_Formal_Object_Declaration =>
                  Match := Matching_Actual (Defining_Identifier (Formal));
                  Append (Instantiate_Object (Formal, Match), Assoc);

               when N_Formal_Type_Declaration =>
                  Match := Matching_Actual (Defining_Identifier (Formal));
                  if No (Match) then
                     Error_Msg_NE ("missing actual for instantiation of &",
                        Instantiation_Node, Defining_Identifier (Formal));
                     raise Instantiation_Error;

                  else
                     Analyze (Match);
                     Append_To (Assoc, Instantiate_Type (Formal, Match));

                     --  Even though the internal type appears as a subtype
                     --  of the actual, it inherits all operations and they
                     --  are directly visible. This is equivalent to a use
                     --  type clause on  the actual.

                     Append_To (Assoc,
                       Make_Use_Type_Clause (Sloc (Match),
                         Subtype_Marks => New_List (
                           New_Occurrence_Of (Entity (Match), Sloc (Match)))));
                  end if;

               when N_Formal_Subprogram_Declaration =>
                  Append (Instantiate_Formal_Subprogram (Formal,
                              Matching_Actual (Defining_Unit_Name (
                                                 Specification (Formal)))),
                          Assoc);

               when N_Formal_Package_Declaration =>
                  Match := Matching_Actual (Defining_Identifier (Formal));

                  if No (Match) then
                     Error_Msg_NE ("missing actual for instantiation of &",
                        Instantiation_Node, Defining_Identifier (Formal));
                     raise Instantiation_Error;
                  else
                     Analyze (Match);
                     Append
                       (Instantiate_Formal_Package (Formal, Match), Assoc);
                  end if;
               when others => null;

            end case;

            Formal := Next (Formal);
         end loop;

         if Num_Actuals > Num_Matched then
            Error_Msg_N
              ("unmatched actuals in instantiation", Instantiation_Node);
         end if;

      elsif Present (Actuals) then
         Error_Msg_N
           ("too many actuals in generic instantiation", Instantiation_Node);
      end if;

      return Assoc;
   end Analyze_Associations;

   ---------------------------------------
   -- Analyze_Formal_Object_Declaration --
   ---------------------------------------

   procedure Analyze_Formal_Object_Declaration (N : Node_Id) is
      E  : constant Node_Id := Expression (N);
      Id : Node_Id;
      K  : Entity_Kind;
      T  : Node_Id;

   begin
      --  Determine the mode of the formal object

      if Out_Present (N) then
         K := E_Generic_In_Out_Parameter;

         if not In_Present (N) then
            Error_Msg_N ("formal generic objects cannot have mode OUT", N);
         end if;

      else
         K := E_Generic_In_Parameter;
      end if;

      Find_Type (Subtype_Mark (N));
      T := Entity (Subtype_Mark (N));

      if K = E_Generic_In_Parameter then
         if Is_Limited_Type (T) then
            Error_Msg_N
             ("generic formal of mode IN must not be of limited type", N);
         end if;

         if Present (E) then
            Analyze (E);
            Resolve (E, T);

            --  if is_deferred_constant(opt_init) then
            --  errmsg("Deferred constant cannot be default expression",
            --  " for a generic parameter","7.4.3");
            --  end if;   ???

         end if;

      --  Case of generic IN OUT parameter.

      else
         --  Constraints will be inherited from actual. This is described
         --  by regarding the subtype of the in out parameter as an extra
         --  generic parameter, obtained from the actual at instantiation.
         --  subtype_mark := make_generic_subtype (subtype_mark);

         if Present (E) then
            Error_Msg_N
             ("initialization not allowed for `IN OUT` formals", N);
         end if;
      end if;

      Id := Defining_Identifier (N);
      Enter_Name (Id);
      Set_Ekind (Id, K);
      Set_Etype (Id, T);
   end Analyze_Formal_Object_Declaration;

   -------------------------------------
   -- Analyze_Formal_Type_Declaration --
   -------------------------------------

   procedure Analyze_Formal_Type_Declaration (N : Node_Id) is
      Def : constant Node_Id := Formal_Type_Definition (N);
      T   : Entity_Id;

   begin
      T := Defining_Identifier (N);

      --  Enter the new name, and branch to specific routine.

      case Nkind (Def) is
         when N_Formal_Private_Type_Definition
                        => Analyze_Formal_Private_Type (N, T, Def);

         when N_Formal_Derived_Type_Definition
                        => Analyze_Formal_Derived_Type (N, T, Def);

         when N_Formal_Discrete_Type_Definition
                        => Analyze_Formal_Discrete_Type (T, Def);

         when N_Formal_Signed_Integer_Type_Definition
                        => Analyze_Formal_Signed_Integer_Type (T, Def);

         when N_Formal_Modular_Type_Definition
                        => Analyze_Formal_Modular_Type (T, Def);

         when N_Formal_Floating_Point_Definition
                        => Analyze_Formal_Floating_Type (T, Def);

         when N_Formal_Ordinary_Fixed_Point_Definition
                        => Analyze_Formal_Ordinary_Fixed_Type (T, Def);

         when N_Formal_Decimal_Fixed_Point_Definition
                        => Analyze_Formal_Decimal_Fixed_Point (T, Def);

         when N_Array_Type_Definition
                        => Analyze_Formal_Array_Type (T, Def);

         when N_Access_To_Object_Definition
                        => Analyze_Generic_Access_Type (T, Def);

         when others =>
            pragma Assert (False); null;

      end case;

      Set_Is_Generic_Type (T);

   end Analyze_Formal_Type_Declaration;

   ---------------------------------
   -- Analyze_Formal_Private_Type --
   ---------------------------------

   procedure Analyze_Formal_Private_Type
     (N   : Node_Id;
      T   : Entity_Id;
      Def : Node_Id)
   is
   begin
      Enter_Name (T);

      if Limited_Present (Def) then
         Set_Ekind (T, E_Limited_Private_Type);
      else
         Set_Ekind (T, E_Private_Type);
      end if;

      Set_Etype (T, T);

      if Present (Discriminant_Specifications (N)) then
         New_Scope (T);
         Process_Discriminants (N);
         End_Scope;
      end if;

      --  For tagged type, set as tagged and initialize dispatch table

      if Tagged_Present (Def) then
         Set_Is_Tagged_Type (T);
         Make_Class_Wide_Type (T);
      end if;
   end Analyze_Formal_Private_Type;

   ---------------------------------
   -- Analyze_Formal_Derived_Type --
   ---------------------------------

   procedure Analyze_Formal_Derived_Type
     (N   : Node_Id;
      T   : Entity_Id;
      Def : Node_Id)
   is
      Loc     : constant Source_Ptr := Sloc (Def);
      New_N   : Node_Id;
      New_Def : Node_Id;

   begin
      if Private_Present (Def) then
         New_N :=
           Make_Private_Extension_Declaration (Loc,
             Defining_Identifier         => T,
             Discriminant_Specifications => Discriminant_Specifications (N),
             Has_Unknown_Discriminants   => Has_Unknown_Discriminants (N),
             Subtype_Indication          => Subtype_Mark (Def));

      else
         New_N :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => T,
             Discriminant_Specifications =>
               Discriminant_Specifications (Parent (T)),
              Type_Definition =>
                Make_Derived_Type_Definition (Loc,
                  Subtype_Indication => Subtype_Mark (Def)));
      end if;

      Rewrite_Substitute_Tree (N,  New_N);
      Analyze (N);
   end Analyze_Formal_Derived_Type;

   ----------------------------------
   -- Analyze_Formal_Discrete_Type --
   ----------------------------------

   --  The operations defined for a discrete types are those of an
   --  enumeration type. The size is set to an arbitrary value, for use
   --  in analyzing the generic unit.

   procedure Analyze_Formal_Discrete_Type (T : Entity_Id; Def : Node_Id) is
      Loc    : constant Source_Ptr := Sloc (Def);
      Bounds : Node_Id;
      Lo     : Node_Id;
      Hi     : Node_Id;

   begin
      Enter_Name (T);
      Set_Ekind (T, E_Enumeration_Type);
      Set_Etype (T, T);
      Set_Esize (T, Esize (Standard_Integer));

      --  For semantic analysis,  the bounds of the type must be set to some
      --  non-static value. The simplest is to create attribute nodes for
      --  those bounds, that refer to the type itself. These bounds are never
      --  analyzed but serve as place-holders.

      Lo :=
        Make_Attribute_Reference (Loc,
          Attribute_Name => Name_First,
          Prefix => New_Reference_To (T, Loc));
      Set_Etype (Lo, T);

      Hi :=
        Make_Attribute_Reference (Loc,
          Attribute_Name => Name_Last,
          Prefix => New_Reference_To (T, Loc));
      Set_Etype (Hi, T);

      Set_Scalar_Range (T,
        Make_Range (Loc,
          Low_Bound => Lo,
          High_Bound => Hi));

   end Analyze_Formal_Discrete_Type;


   ----------------------------------------
   -- Analyze_Formal_Signed_Integer_Type --
   ----------------------------------------

   procedure Analyze_Formal_Signed_Integer_Type
     (T   : Entity_Id;
      Def : Node_Id)
   is
      Base : constant Entity_Id :=
        New_Internal_Entity
          (E_Signed_Integer_Type, Current_Scope, Sloc (Def), 'G');

   begin
      Enter_Name (T);

      Set_Ekind        (T, E_Signed_Integer_Subtype);
      Set_Etype        (T, Base);
      Set_Esize        (T, Esize (Standard_Integer));
      Set_Scalar_Range (T, Scalar_Range (Standard_Integer));

      Set_Is_Generic_Type (Base);
      Set_Esize           (Base, Esize (Standard_Integer));
      Set_Etype           (Base, Base);
      Set_Scalar_Range    (Base, Scalar_Range (Standard_Integer));
   end Analyze_Formal_Signed_Integer_Type;

   ---------------------------------
   -- Analyze_Formal_Modular_Type --
   ---------------------------------

   procedure Analyze_Formal_Modular_Type (T : Entity_Id; Def : Node_Id) is
   begin
      Enter_Name (T);
      Unimplemented (Def, "formal modular types");
   end Analyze_Formal_Modular_Type;

   ----------------------------------
   -- Analyze_Formal_Floating_Type --
   ---------------------------------

   procedure Analyze_Formal_Floating_Type (T : Entity_Id; Def : Node_Id) is
      --  the various semantic attributes are taken from the predefined type
      --  Float, just so that all of them are initialized. Their values are
      --  never used because no constant folding or expansion takes place in
      --  the generic itself.

      Base : constant Entity_Id :=
        New_Internal_Entity
          (E_Float_Type, Current_Scope, Sloc (Def), 'G');

   begin
      Enter_Name (T);
      Set_Ekind (T, E_Float_Subtype);
      Set_Etype (T, Base);
      Set_Etype (Base, Base);
      Set_Is_Generic_Type (Base);
      Set_Esize (T, Esize (Standard_Float));
      Set_Esize (Base, Esize (Standard_Float));
      Set_Digits_Value (T,  Digits_Value (Standard_Float));
      Set_Digits_Value (Base,  Digits_Value (Standard_Float));
      Set_Scalar_Range (T, Scalar_Range (Standard_Float));
      Set_Scalar_Range (Base, Scalar_Range (Standard_Float));
   end Analyze_Formal_Floating_Type;

   ----------------------------------------
   -- Analyze_Formal_Ordinary_Fixed_Type --
   ----------------------------------------

   procedure Analyze_Formal_Ordinary_Fixed_Type
     (T   : Entity_Id;
      Def : Node_Id)
   is
   begin
      Enter_Name (T);
      Unimplemented (Def, "formal ordinary fixed types");
   end Analyze_Formal_Ordinary_Fixed_Type;

   ----------------------------------------
   -- Analyze_Formal_Decimal_Fixed_Point --
   ----------------------------------------

   procedure Analyze_Formal_Decimal_Fixed_Point
     (T   : Entity_Id;
      Def : Node_Id)
   is
   begin
      Enter_Name (T);
      Unimplemented (Def, "formal decimal fixed types");
   end Analyze_Formal_Decimal_Fixed_Point;

   -------------------------------
   -- Analyze_Formal_Array_Type --
   -------------------------------

   procedure Analyze_Formal_Array_Type
     (T   : in out Entity_Id;
      Def : Node_Id)
   is
      J : Node_Id;

   begin
      --  Treated like a non-generic array declaration, with
      --  additional semantic checks.

      Enter_Name (T);
      Array_Type_Declaration (T, Def);

      if Is_Incomplete_Type (Component_Type (T))
        and then not Is_Generic_Type (Component_Type (T))
      then
         Error_Msg_N ("premature usage of incomplete type", Def);

      elsif Is_Internal (Component_Type (T)) then
         Error_Msg_N
           ("only a subtype mark is allowed in a formal", Def);
      end if;

      J := First_Index (T);

      while Present (J) loop
         if Is_Internal (Etype (J)) then
            Error_Msg_N
              ("only a subtype mark is allowed in a formal", Def);
         end if;

         J := Next_Index (J);
      end loop;
   end Analyze_Formal_Array_Type;

   ---------------------------------
   -- Analyze_Generic_Access_Type --
   ---------------------------------

   procedure Analyze_Generic_Access_Type (T : Entity_Id; Def : Node_Id) is
   begin
      Enter_Name (T);
      Access_Type_Declaration (T, Def);

      if Is_Incomplete_Type (Designated_Type (T))
        and then not Is_Generic_Type (Designated_Type (T))
      then
         Error_Msg_N ("premature usage of incomplete type", Def);

      elsif Is_Internal (Designated_Type (T)) then
         Error_Msg_N
           ("only a subtype mark is allowed in a formal", Def);
      end if;
   end Analyze_Generic_Access_Type;

   -------------------------------
   -- Analyze_Formal_Subprogram --
   -------------------------------

   procedure Analyze_Formal_Subprogram (N : Node_Id) is
      Spec : constant Node_Id := Specification (N);

   begin
      --  Default name is resolved at the point of instantiation

      if Box_Present (N) then
         null;

      --  Else default is bound at the point of generic declaration

      elsif Present (Default_Name (N)) then
         Find_Name (Default_Name (N));
      end if;

      Analyze_Subprogram_Declaration (N);
      Set_Has_Completion (Defining_Unit_Name (Spec));
   end Analyze_Formal_Subprogram;

   ----------------------------
   -- Analyze_Formal_Package --
   ----------------------------

   procedure Analyze_Formal_Package (N : Node_Id) is
      Formal    : Entity_Id := Defining_Identifier (N);
      Gen_Id    : constant Node_Id   := Name (N);
      Gen_Decl  : Node_Id;
      Gen_Unit  : Entity_Id;
      New_N     : Node_Id;

   begin

      Find_Name (Gen_Id);
      Gen_Unit := Entity (Gen_Id);

      if Ekind (Gen_Unit) /= E_Generic_Package then
         Error_Msg_N ("Expect generic package name", Gen_Id);
         return;
      end if;

      --  The formal package is treated like a regular instance, but only
      --  the specification needs to be instantiated, to make entities visible.
      --  If there are no generic associations, the generic parameters appear
      --  as local entities and are instantiated like them.

      if not Box_Present (N) then
         Analyze_Package_Instantiation (N);

      else
         Gen_Decl := Get_Declaration_Node (Gen_Unit);
         New_N := Copy_Generic_Node (
                 Original_Node (Gen_Decl), Empty, Instantiating => True);
         Set_Defining_Unit_Name (Specification (New_N), Formal);
         Rewrite_Substitute_Tree (N, New_N);
         Analyze (N);

         --  Inside the generic unit, the formal package is a regular
         --  package, but no body is needed for it. Note that after
         --  instantiation, the defining_unit_name we need is in the
         --  new tree and not in the original. (see Package_Instantiation).

         Formal := Defining_Unit_Name (Specification (N));
         Set_Ekind (Formal,  E_Package);
         Set_Has_Completion (Formal,  True);
      end if;

   end Analyze_Formal_Package;

   ---------------------------------------------
   --  Build_Instance_Compilation_Unit_Nodes  --
   ---------------------------------------------

   procedure Build_Instance_Compilation_Unit_Nodes
     (N        : Node_Id;
      Act_Body : Node_Id;
      Act_Decl : Node_Id)
   is
      Decl_Cunit : Node_Id;
      Body_Cunit : Node_Id;
      Citem      : Node_Id;

   begin
      --  A new compilation unit node is built for the instance declaration

      Decl_Cunit := New_Node (N_Compilation_Unit, Sloc (N));
      Set_Context_Items (Decl_Cunit, Empty_List);
      Set_Unit          (Decl_Cunit, Act_Decl);
      Set_Parent_Spec   (Act_Decl, Parent_Spec (N));
      Set_Body_Required (Decl_Cunit, True);

      --  We use the original instantiation compilation unit as the resulting
      --  compilation unit of the instance, since this is the main unit.

      Rewrite_Substitute_Tree (N, Act_Body);
      Body_Cunit := Parent (N);

      --  The two compilation unit nodes are linked by the Library_Unit field

      Set_Library_Unit  (Decl_Cunit, Body_Cunit);
      Set_Library_Unit  (Body_Cunit, Decl_Cunit);

      --  The context clause items on the instantiation, which are now
      --  attached to the body compilation unit (since the body overwrote
      --  the orginal instantiation node), semantically belong on the spec,
      --  so copy them there. It's harmless to leave them on the body as well.
      --  In fact one could argue that they belong in both places.

      Citem := First (Context_Items (Body_Cunit));
      while Present (Citem) loop
         Append (New_Copy (Citem), Context_Items (Decl_Cunit));
         Citem := Next (Citem);
      end loop;

   end Build_Instance_Compilation_Unit_Nodes;

   ------------------------
   -- Instantiate_Object --
   ------------------------

   function Instantiate_Object
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id
   is
      Formal_Id  : constant Entity_Id  := Defining_Identifier (Formal);
      Type_Id    : constant Node_Id    := Subtype_Mark (Formal);
      Loc        : constant Source_Ptr := Sloc (Actual);
      Decl_Node  : Node_Id;

   begin
      if Get_Instance_Of (Formal_Id) /= Formal_Id then
         Error_Msg_N ("Duplicate instantiation of generic parameter", Actual);
      end if;

      if Out_Present (Formal) then

         if No (Actual) then
            Error_Msg_NE ("missing actual for instantiation of &",
                Instantiation_Node, Formal_Id);
            raise Instantiation_Error;
         end if;

         --  An IN OUT generic actual must be a name. The instantiation
         --  is a renaming declaration.

         Decl_Node :=
           Make_Object_Renaming_Declaration (Loc,
             Defining_Identifier => New_Copy (Formal_Id),
             Subtype_Mark        => New_Copy (Type_Id),
             Name                => New_Copy (Actual));

      else
         --  The instantiation of a generic formal in-parameter is a
         --  constant declaration.

         if Present (Actual) then
            Decl_Node := Make_Object_Declaration (Loc,
              Defining_Identifier => New_Copy (Formal_Id),
              Constant_Present => True,
              Object_Definition => New_Copy (Type_Id),
              Expression => New_Copy (Actual));

         elsif Present (Expression (Formal)) then

            --  Use default to construct declaration.

            Decl_Node := Make_Object_Declaration (Loc,
              Defining_Identifier => New_Copy (Formal_Id),
              Constant_Present => True,
              Object_Definition => New_Copy (Type_Id),
              Expression => New_Copy (Expression (Formal)));
         else
            Error_Msg_NE ("missing actual for instantiation of &",
                Instantiation_Node, Formal_Id);
            raise Instantiation_Error;
         end if;

         --  Analyze expression now, before analyzing the declaration,
         --  to avoid possible name conflict with the object being declared.

         Analyze (Expression (Decl_Node));
      end if;

      return Decl_Node;
   end Instantiate_Object;

   ----------------------
   -- Instantiate_Type --
   ----------------------

   function Instantiate_Type
     (Formal : Node_Id;
      Actual : Node_Id)
      return   Node_Id
   is
      Loc       : constant Source_Ptr := Sloc (Actual);
      Gen_T     : constant Entity_Id  := Defining_Identifier (Formal);
      Def       : constant Node_Id    := Formal_Type_Definition (Formal);
      Act_T     : Entity_Id;
      Decl_Node : Node_Id;

      procedure Validate_Array_Type_Instance;
      procedure Validate_Access_Type_Instance;
      procedure Validate_Derived_Type_Instance;
      procedure Validate_Private_Type_Instance;

      ----------------------------------
      -- Validate_Array_Type_Instance --
      ----------------------------------

      procedure Validate_Array_Type_Instance is
         A_Index : Node_Id;
         G_Index : Node_Id;
         Num     : Int;

      begin
         if not Is_Array_Type (Act_T) then
            Error_Msg_NE ("expect array type in instantiation of &",
                                                      Actual,  Gen_T);
            raise Instantiation_Error;

         elsif Nkind (Def) = N_Constrained_Array_Definition then
            if not (Is_Constrained (Act_T)) then
               Error_Msg_NE
                 ("expect constrained array in instantiation of &",
                  Actual, Gen_T);
               raise Instantiation_Error;

            else
               G_Index := First (Discrete_Subtype_Definitions (Def));
               Num := 0;
               while Present (G_Index) loop
                  Num := Num + 1;
                  G_Index := Next_Index (G_Index);
               end loop;

               if Num /= Number_Dimensions (Act_T) then
                  Error_Msg_NE
                    ("dimensions of actual do not match formal &",
                     Actual, Gen_T);
                  raise Instantiation_Error;
               end if;
            end if;

         else
            if Is_Constrained (Act_T) then
               Error_Msg_NE
                 ("expect unconstrained array in instantiation of &",
                  Actual, Gen_T);
               raise Instantiation_Error;

            else
               G_Index := First (Subtype_Marks (Def));

               while Present (G_Index) loop
                  Num := Num + 1;
                  G_Index := Next_Index (G_Index);
               end loop;

               if Num /= Number_Dimensions (Act_T) then
                  Error_Msg_NE
                    ("dimensions of actual do not match formal &",
                     Actual, Gen_T);
                  raise Instantiation_Error;
               end if;
            end if;
         end if;
      end Validate_Array_Type_Instance;

      -----------------------------------
      -- Validate_Access_Type_Instance --
      -----------------------------------

      procedure Validate_Access_Type_Instance is
      begin
         if not Is_Access_Type (Act_T) then
            Error_Msg_NE
              ("expect access type in instantiation of &", Actual,  Gen_T);
            raise Instantiation_Error;
         end if;
      end Validate_Access_Type_Instance;

      ------------------------------------
      -- Validate_Private_Type_Instance --
      ------------------------------------

      procedure Validate_Private_Type_Instance is
      begin
         --  many semantic checks forthcoming. ???
         null;
      end Validate_Private_Type_Instance;

      ------------------------------------
      -- Validate_Derived_Type_Instance --
      ------------------------------------

      procedure Validate_Derived_Type_Instance is
         T : Node_Id := Subtype_Mark (Def);

      begin
         if Present (Entity (T))
           and then Root_Type (Entity (T)) /= Root_Type (Act_T)
         then
            Error_Msg_NE
               ("expect type derived from & in instantiation",
                Actual,  Entity (T));
            raise Instantiation_Error;
         end if;
      end Validate_Derived_Type_Instance;

   --  Start of processing for Instantiate_Type

   begin

      if Get_Instance_Of (Gen_T) /= Gen_T then
         Error_Msg_N ("Duplicate instantiation of generic type", Actual);
         return Error;

      elsif not Is_Entity_Name (Actual)
        or else not Is_Type (Entity (Actual))
      then
         Error_Msg_NE
           ("expect valid subtype mark to instantiate & ", Actual, Gen_T);
         raise Instantiation_Error;

      else
         Act_T := Entity (Actual);
         Set_Instance_Of (Gen_T, Act_T);

         if not Is_Abstract (Gen_T)
           and then Is_Abstract (Act_T)
         then
            Error_Msg_N
              ("actual of non-abstract formal cannot be abstract", Actual);
         end if;

         if Is_Scalar_Type (Gen_T) then
            Set_Instance_Of (Etype (Gen_T), Etype (Act_T));
         end if;
      end if;

      case Nkind (Def) is
         when N_Formal_Private_Type_Definition =>
            Validate_Private_Type_Instance;

         when N_Formal_Derived_Type_Definition =>
            Validate_Derived_Type_Instance;

         when N_Formal_Discrete_Type_Definition =>
            if not Is_Discrete_Type (Act_T) then
               Error_Msg_NE ("expect discrete type in instantiation of &",
                       Actual, Gen_T);
                  raise Instantiation_Error;
            end if;

         when N_Formal_Signed_Integer_Type_Definition =>
            if not Is_Signed_Integer_Type (Act_T) then
               Error_Msg_NE
                  ("expect signed integer type in instantiation of &",
                                                   Actual, Gen_T);
                  raise Instantiation_Error;
            end if;

         when N_Formal_Modular_Type_Definition =>
            null;

         when N_Formal_Floating_Point_Definition =>
            if not Is_Float_Type (Act_T) then
               Error_Msg_NE
                  ("expect float type in instantiation of &", Actual, Gen_T);
                  raise Instantiation_Error;
            end if;

         when N_Formal_Ordinary_Fixed_Point_Definition =>
            if not Is_Fixed_Type (Act_T) then
               Error_Msg_NE
                  ("expect fixed type in instantiation of &", Actual, Gen_T);
                  raise Instantiation_Error;
            end if;

         when N_Formal_Decimal_Fixed_Point_Definition =>
            null;

         when N_Array_Type_Definition =>
            Validate_Array_Type_Instance;

         when N_Access_To_Object_Definition =>
            Validate_Access_Type_Instance;

         when others =>
            pragma Assert (False); null;

      end case;

      Decl_Node :=
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => New_Copy (Gen_T),
          Subtype_Indication  => New_Reference_To (Act_T, Loc));

      return Decl_Node;
   end Instantiate_Type;

   -----------------------------------
   -- Instantiate_Formal_Subprogram --
   -----------------------------------

   function Instantiate_Formal_Subprogram
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id
   is
      Loc        : constant Source_Ptr := Sloc (Actual);
      Formal_Sub : constant Entity_Id :=
                     Defining_Unit_Name (Specification (Formal));
      Decl_Node  : Node_Id;
      New_Act    : Node_Id;

      procedure Valid_Actual_Subprogram (Act : Node_Id);
      --  Documentation for this spec needed ???

      procedure Valid_Actual_Subprogram (Act : Node_Id) is
      begin
         if not Is_Entity_Name (Act)
           and then Nkind (Act) /= N_Operator_Symbol
         then
            if Etype (Act) /= Any_Type then
               Error_Msg_NE ("Expect subprogram name to instantiate &",
                                        Instantiation_Node, Formal_Sub);
            end if;

            --  In any case, instantiation cannot continue.

            raise Instantiation_Error;
         end if;
      end Valid_Actual_Subprogram;

   --  Start processing for Instantiate_Formal_Subprogram

   begin
      if Present (Actual) then
         if Nkind (Actual) = N_Operator_Symbol then
            Find_Name (Actual);
         else
            Analyze (Actual);
         end if;

         Valid_Actual_Subprogram (Actual);

         Decl_Node :=
           Make_Subprogram_Renaming_Declaration (Loc,
             Specification => New_Copy (Specification (Formal)),
             Name => Actual);

      elsif Present (Default_Name (Formal)) then
         Decl_Node :=
           Make_Subprogram_Renaming_Declaration (Loc,
             Specification => New_Copy (Specification (Formal)),
             Name => New_Occurrence_Of (Entity (Default_Name (Formal)), Loc));

      elsif Box_Present (Formal) then

         --  Actual is resolved at the point of instantiation.

         New_Act := Make_Identifier (Loc, Chars (Formal_Sub));
         Decl_Node :=
           Make_Subprogram_Renaming_Declaration (Loc,
             Specification => New_Copy (Specification (Formal)),
             Name => New_Act);

         Analyze (New_Act);
         Valid_Actual_Subprogram (New_Act);

      else
         Error_Msg_NE
           ("missing actual for instantiation of &",
            Instantiation_Node, Defining_Identifier (Formal));
         raise Instantiation_Error;
      end if;

      return Decl_Node;

   end Instantiate_Formal_Subprogram;

   --------------------------------
   -- Instantiate_Formal_Package --
   --------------------------------

   function Instantiate_Formal_Package
     (Formal : Node_Id;
      Actual : Node_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (Actual);

   begin
      Analyze (Actual);

      if not Is_Entity_Name (Actual)
        or else  Ekind (Entity (Actual)) /= E_Package
        or else Generic_Parent (Parent (Entity (Actual)))
                                     /= Entity (Name (Formal))
      then
         Error_Msg_N ("expect package instance to instantiate formal",
           Actual);
         raise Instantiation_Error;

      else
         Set_Instance_Of (Defining_Identifier (Formal), Entity (Actual));
         return
           Make_Package_Renaming_Declaration (Loc,
             Defining_Unit_Name => New_Copy (Defining_Identifier (Formal)),
             Name               => New_Reference_To (Entity (Actual), Loc));
      end if;

   end Instantiate_Formal_Package;

   -----------------------
   -- Copy_Generic_Node --
   -----------------------

   function Copy_Generic_Node
     (N             : Node_Id;
      Parent_Id     : Node_Id;
      Instantiating : Boolean)
      return Node_Id
   is
      New_N : Node_Id;

      function Copy_Generic_Descendant (D : Union_Id) return Union_Id;
      --  Check the given value of one of the Fields referenced by the
      --  current node to determine whether to copy it recursively. The
      --  field may hold a Node_Id, a List_Id, or an Elist_Id, or a plain
      --  value (Sloc, Uint, Char) in which case it need not be copied.

      function Copy_Generic_List
        (L         : List_Id;
         Parent_Id : Node_Id)
         return List_Id;
      --  ??? documentation

      function Copy_Generic_Elist (E : Elist_Id) return Elist_Id;
      --  ??? documentation

      -----------------------------
      -- Copy_Generic_Descendant --
      -----------------------------

      function Copy_Generic_Descendant (D : Union_Id) return Union_Id is
      begin
         if D in Node_Range then
            if D = Union_Id (Empty) then
               return D;

            else
               return Union_Id (Copy_Generic_Node
                               (Node_Id (D), New_N, Instantiating));
            end if;

         elsif D in List_Range then
            if D = No_List or else Is_Empty_List (List_Id (D)) then
               return Union_Id (D);
            else
               return Union_Id (Copy_Generic_List (List_Id (D), New_N));
            end if;

         elsif D in Elist_Range then
            if D = No_Elist or else Is_Empty_Elmt_List (Elist_Id (D)) then
               return Union_Id (D);
            else
               return Union_Id (Copy_Generic_Elist (Elist_Id (D)));
            end if;

         else
            --  Field is not Id of copyable structure: return as is

            return D;
         end if;
      end Copy_Generic_Descendant;

      -----------------------
      -- Copy_Generic_List --
      -----------------------

      function Copy_Generic_List
        (L         : List_Id;
         Parent_Id : Node_Id)
         return List_Id
      is
         N      : Node_Id;
         New_L  : List_Id;

      begin
         N := First (L);

         if N = Empty then
            return L;

         else
            New_L := New_List;
            Set_Parent (New_L, Parent_Id);
            N := First (L);

            while Present (N) loop
               Append (Copy_Generic_Node (N, Empty, Instantiating), New_L);
               N := Next (N);
            end loop;

            return New_L;
         end if;

      end Copy_Generic_List;

      ------------------------
      -- Copy_Generic_Elist --
      ------------------------

      function Copy_Generic_Elist (E : Elist_Id) return Elist_Id is
         M : Elmt_Id;
         L : Elist_Id := New_Elmt_List;

      begin
         M := First_Elmt (E);

         while Present (M) loop
            Append_Elmt
              (Copy_Generic_Node (Node (M), Empty, Instantiating), L);
            M := Next_Elmt (M);
         end loop;

         return L;
      end Copy_Generic_Elist;

      ------------------------
      -- Check_Private_View --
      ------------------------

      procedure Check_Private_View is
         T : Entity_Id := Etype (N);

      begin

         if Present (T) then
            if Ekind (T) in Private_Kind
              and then not Has_Private_View (N)
              and then Present (Full_Declaration (T))
            then
               --  In the generic, the full type was visible. Save the
               --  private entity,  for subsequent exchange.

               Append_Elmt (Full_Declaration (T), Exchanged_Views);
               Exchange_Declarations (T);

            elsif Has_Private_View (N)
              and then Ekind (T) not in Private_Kind
            then
               --  Only the private declaration was visible in the generic.

               Append_Elmt (T, Exchanged_Views);
               Exchange_Declarations (Etype (Associated_Node (N)));
            end if;
         end if;
      end Check_Private_View;

   --  Start processing for Copy_Generic_Node

   begin
      if N = Empty then
         return N;
      end if;

      New_N := New_Copy (N);

      if not Is_List_Member (N) then
         Set_Parent (New_N, Parent_Id);
      end if;

      --  If defining identifier, then all fields have been copied already

      if Nkind (New_N) in N_Entity then
         null;

      elsif    (Nkind (New_N) = N_Identifier
        or else Nkind (New_N) = N_Character_Literal
        or else Nkind (New_N) = N_Expanded_Name
        or else Nkind (New_N) = N_Operator_Symbol
        or else Nkind (New_N) in N_Op)
      then
         if not Instantiating then

            --  Link both nodes in order to assign subsequently the
            --  entity of the copy to the original node, in case this
            --  is a global reference.

            Set_Associated_Node (N, New_N);
            Set_Associated_Node (New_N, Empty);

         else
            --  If the associated node is still defined, the entity in
            --  it is global, and must be copied to the instance.

            if Present (Associated_Node (N)) then
               if Nkind (Associated_Node (N)) = Nkind (N) then
                  Set_Entity (New_N, Entity (Associated_Node (N)));
                  Check_Private_View;

               elsif Nkind (Associated_Node (N)) = N_Function_Call then

                  --  Name resolves to a call to parameterless function.

                  Set_Entity (New_N, Entity (Name (Associated_Node (N))));

               else
                  Set_Entity (New_N, Empty);
               end if;
            end if;

         end if;

         if Nkind (N) = N_Expanded_Name
           or else Nkind (N) in N_Op
         then
            --  Complete the copy of remaining descendants.

            Set_Field2 (New_N, Copy_Generic_Descendant (Field2 (N)));
            Set_Field3 (New_N, Copy_Generic_Descendant (Field3 (N)));
         end if;

      else
         --  For all remaining nodes, copy recursively their descendants.

         Set_Field1 (New_N, Copy_Generic_Descendant (Field1 (N)));
         Set_Field2 (New_N, Copy_Generic_Descendant (Field2 (N)));
         Set_Field3 (New_N, Copy_Generic_Descendant (Field3 (N)));
         Set_Field4 (New_N, Copy_Generic_Descendant (Field4 (N)));
         Set_Field5 (New_N, Copy_Generic_Descendant (Field5 (N)));

         if (Nkind (N) = N_Package_Body_Stub
              or else Nkind (N) = N_Subprogram_Body_Stub)
           and then not Instantiating
         then
            --  Subunits of generic units must be loaded and analyzed at the
            --  point the stubs occur. A more permissive system might defer
            --  this analysis to the point of instantiation, but this seems too
            --  complicated for now.

            declare
               Context      : List_Id;
               Subunit_Name : constant Unit_Name_Type := Get_Unit_Name (N);
               Subunit      : Node_Id;
               New_Subunit  : Node_Id;
               Parent_Unit  : Node_Id;
               Unum         : Unit_Number_Type;
               Lib          : Node_Id;

            begin
               Unum := Load (Subunit_Name, True, N);
               Subunit :=  Cunit (Unum);

               --  After semantic analysis all non-local references are marked,
               --  and the context clause of the subunit becomes irrelevant. At
               --  instantiation time, only the body (package or subprogram)
               --  must be copied, so we only retain the body of the subunit.

               Rewrite_Substitute_Tree (N, Proper_Body (Unit (Subunit)));

               --  The subunit is copied in order to perform semantic analysis
               --  and then attached to the copy of the enclosing generic.

               --  Only the proper body needs to be copied. Library_Unit and
               --  context clause are simply inherited by the generic copy.

               Set_Proper_Body (Unit (Subunit), Copy_Generic_Node (
                   Proper_Body (Unit (Subunit)),
                                             Empty, Instantiating => False));
               Set_Library_Unit (New_N, Subunit);
               Inherit_Context (Unit (Subunit));

            end;

         end if;
      end if;

      return New_N;
   end Copy_Generic_Node;

   ----------------------------
   -- Save_Global_References --
   ----------------------------

   procedure Save_Global_References (N : Node_Id) is
      E  : Entity_Id;
      N2 : Node_Id;

      ---------------
      -- Is_Global --
      ---------------

      --  Examine the scope of an entity, and the scope of the scope,
      --  etc, until we find either Standard, in which case the entity
      --  is global, or the generic unit itself, which indicates that
      --  the entity is local. If the entity is the generic unit itself,
      --  as in the case of a recursive call, the entity is local as well.

      function Is_Global (E : Entity_Id) return Boolean is
         Se  : Entity_Id := Scope (E);
         S   : Entity_Id := Current_Scope;

      begin
         if E = S then
            return False;
         end if;

         while Se /= S loop
            if Se = Standard_Standard then
               return true;
            else
               Se := Scope (Se);
            end if;
         end loop;

         return False;
      end Is_Global;

      ----------------------------
      -- Save_Global_Descendant --
      ----------------------------

      procedure Save_Global_Descendant (D : Union_Id) is
         N1 : Node_Id;

      begin
         if D in Node_Range then
            if D = Union_Id (Empty) then
               null;

            elsif Nkind (Node_Id (D)) /= N_Compilation_Unit then
               Save_Global_References (Node_Id (D));
            end if;

         elsif D in List_Range then
            if D = No_List or else Is_Empty_List (List_Id (D)) then
               null;
            else
               N1 := First (List_Id (D));
               while Present (N1) loop
                  Save_Global_References (N1);
                  N1 := Next (N1);
               end loop;
            end if;

         --  Element list or other non-node field, nothing to do

         else
            null;
         end if;
      end Save_Global_Descendant;

      ------------------
      -- Reset_Entity --
      ------------------

      procedure Reset_Entity is
      begin
         N2 := Associated_Node (N);
         E := Entity (N2);

         if Present (E) then
            if Is_Global (E) then
               Set_Etype (N,  Etype (N2));

               if (Ekind (Etype (N))) in Private_Kind
                 and then Present (Full_Declaration (Etype (N2)))
               then
                  Set_Has_Private_View (N);
                  Set_Etype (N2, Full_Declaration (Etype (N2)));
               end if;
            else
               --  Entity is local.  Mark generic node as unresolved.
               --  Note that now it does not have an entity.
               Set_Associated_Node (N, Empty);
               Set_Etype  (N, Empty);
            end if;

         elsif Nkind (Parent (N)) = N_Selected_Component
           and then Nkind (Parent (N2)) = N_Expanded_Name
           and then Is_Global (Entity (Parent (N2)))
         then
            Change_Selected_Component_To_Expanded_Name (Parent (N));
            Set_Associated_Node (Parent (N), Parent (N2));
            Set_Etype (Parent (N), Etype (Parent (N2)));
            Save_Global_Descendant (Field2 (N));
            Save_Global_Descendant (Field3 (N));

         else
            --  Entity is local.  Reset in generic unit,  so that node
            --  is resolved anew at the point of instantiation.

            Set_Associated_Node (N, Empty);
            Set_Etype (N, Empty);
         end if;
      end Reset_Entity;

   --  Start processing for Save_Global_References

   begin
      if N = Empty then
         null;

      elsif (Nkind (N) = N_Character_Literal
        or else Nkind (N) = N_Operator_Symbol)
        and then Nkind (N) = Nkind (Associated_Node (N))
      then
         Reset_Entity;

      elsif Nkind (N) in N_Op then

         --  Inequality is rewritten as the negation of equality, and
         --  resolved as such. Make thesame replacement in the generic.

         if Nkind (N) = N_Op_Ne then
            declare
               Eq  : Node_Id := Make_Op_Eq  (Sloc (N), Node_Id (Field2 (N)),
                                                       Node_Id (Field3 (N)));
               Neg : Node_Id := Make_Op_Not (Sloc (N), Eq);
            begin
               Rewrite_Substitute_Tree (N, Neg);
               Save_Global_References (N);
            end;

         elsif Nkind (N) = Nkind (Associated_Node (N)) then
            Reset_Entity;

         else
            --  Node may be transformed into a call to a user-defined operator.

            N2 := Associated_Node (N);

            if Nkind (N2) = N_Function_Call then
               E := Entity (Name (N2));

               if Present (E)
                 and then Is_Global (E)
               then
                  Set_Etype (N, Etype (N2));
               else
                  Set_Associated_Node (N, Empty);
                  Set_Etype (N, Empty);
               end if;
            end if;
         end if;

         --  Complete the check on operands.

         if Nkind (N) /= N_Op_Ne then
            Save_Global_Descendant (Field2 (N));
            Save_Global_Descendant (Field3 (N));
         end if;

      elsif Nkind (N) = N_Identifier then
         if Nkind (N) = Nkind (Associated_Node (N)) then
            Reset_Entity;
         else
            N2 := Associated_Node (N);

            if Nkind (N2) = N_Function_Call then
               E := Entity (Name (N2));

               --  Name resolves to a call to parameterless function. If
               --  original entity is global,  mark node as resolved.

               if Present (E)
                 and then Is_Global (E)
               then
                  Set_Etype (N, Etype (N2));
               else
                  Set_Associated_Node (N, Empty);
                  Set_Etype (N, Empty);
               end if;

            elsif Nkind (N2) = N_Integer_Literal
              or else Nkind (N2) = N_Real_Literal
            then
               --  Name resolves to named number that is constant-folded.
               --  Perform the same replacement in generic.

               Rewrite_Substitute_Tree (N,  New_Copy (N2));
               Set_Analyzed (N, False);
            else
               null;
            end if;
         end if;

      elsif Nkind (N) in N_Entity then
         null;

      else
         Save_Global_Descendant (Field1 (N));
         Save_Global_Descendant (Field2 (N));
         Save_Global_Descendant (Field3 (N));
         Save_Global_Descendant (Field4 (N));
         Save_Global_Descendant (Field5 (N));

      end if;
   end Save_Global_References;

   ---------------------
   -- Associated_Node --
   ---------------------

   function Associated_Node (N : Node_Id) return Node_Id is
      Assoc : Node_Id := Node4 (N);
      --  ??? what is Node4 being used for here?

   begin
      if Nkind (Assoc) /= Nkind (N) then
         return Assoc;
      else
         --  If the node is part of an inner generic, it may itself have been
         --  remapped into a further generic copy. Node4 is otherwise used for
         --  the entity of the node, and will be of a different node kind, or
         --  else N has been rewritten as a literal or function call.

         while Present (Node4 (Assoc))
           and then Nkind (Node4 (Assoc)) = Nkind (Assoc)
         loop
            Assoc := Node4 (Assoc);
         end loop;

         --  Follow and additional link in case the final node was rewritten.
         --  This can only happen with nested generic units.

         if (Nkind (Assoc) = N_Identifier or else Nkind (Assoc) in N_Op)
           and then Present (Node4 (Assoc))
           and then (Nkind (Node4 (Assoc)) = N_Function_Call
                       or else Nkind (Node4 (Assoc)) = N_Integer_Literal
                       or else Nkind (Node4 (Assoc)) = N_Real_Literal)
         then
            Assoc := Node4 (Assoc);
         end if;

         return Assoc;
      end if;
   end Associated_Node;

   -------------------------
   -- Set_Associated_Node --
   -------------------------

   procedure Set_Associated_Node
     (Gen_Node  : Node_Id;
      Copy_Node : Node_Id)
   is
   begin
      Set_Node4 (Gen_Node,  Copy_Node);
   end Set_Associated_Node;

   ---------------------
   -- Get_Instance_Of --
   ---------------------

   function Get_Instance_Of (A : Entity_Id) return Entity_Id is
   begin
      for J in 0 .. Generic_Renamings.Last - 1 loop
         if Chars (A) = Chars (Generic_Renamings.Table (J).Gen_Id) then
            return Generic_Renamings.Table (J).Act_Id;
         end if;
      end loop;

      --  On exit, entity is not instantiated: not a generic parameter,
      --  or else parameter of an inner generic unit.

      return A;
   end Get_Instance_Of;

   ---------------------
   -- Set_Instance_Of --
   ---------------------

   procedure Set_Instance_Of (A : Entity_Id; B : Entity_Id) is
   begin
      Generic_Renamings.Table (Generic_Renamings.Last) := (A, B);
      Generic_Renamings.Increment_Last;
   end Set_Instance_Of;

   ---------------------------
   -- Restore_Private_Views --
   ---------------------------

   procedure Restore_Private_Views is
      E : Elmt_Id;

   begin
      E := First_Elmt (Exchanged_Views);
      while Present (E) loop
         Exchange_Declarations (Node (E));
         E := Next_Elmt (E);
      end loop;
   end Restore_Private_Views;

end Sem_Ch12;
