
/********************************************************************
 ********************************************************************
 ******                                                        ******
 ******  Reduction Library for Modula-2*                       ******
 ******                                                        ******
 ******  allows for efficient reduction operations on          ******
 ******  arbitrarily large and distributed arrays              ******
 ******                                                        ******
 ******  C implementation by Ernst A. Heinz                    ******
 ******                                                        ******
 ******  Last change:  March 25, 1994                          ******
 ******                                                        ******
 ********************************************************************
 ********************************************************************/

#include "msReduce.LINX.h"


/********************************************************************
 ********************************************************************
 *******              NORMAL REDUCTION ROUTINES               *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template of normal reduction routines
 */

#define ReduceFunc(FuncName,elemType,GlobalReduce,accu,initVal,RED_OP)	\
    elemType								\
    FuncName(x, n, l)							\
    register elemType *x;						\
    register LONGCARD n;						\
    LONGCARD l;								\
    {									\
      register elemType accu;						\
									\
      GlobalReduce(elemType,accu,initVal,RED_OP);			\
    }

/*
 *  generic template of normal reduction code with accumulation operator INC_OP
 */

#define GenericReduce1(elemType,accu,initVal,INC_OP)	\
    accu = initVal;					\
    for (; n>0; n--, x++)				\
      accu INC_OP *x;					\
    return (accu);

/*
 *  generic template of normal min/max reduction code with compare operator CMP_OP
 */

#define GenericReduce2(elemType,accu,initVal,CMP_OP)	\
    accu = initVal;					\
    {							\
      register elemType tmp;				\
							\
      for (; n>0; n--, x++) {				\
        tmp = *x;					\
        if (tmp CMP_OP accu)				\
	  accu = tmp;					\
      }							\
    };							\
    return (accu);


/****************************************************************
 *****  normal routines with reduction operator "addition"  *****
 ****************************************************************/

#define ReduceAddFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce1,sum,initVal,+=)

ReduceAddFunc( SReduceAddCh, CHAR,      '\0' )
ReduceAddFunc( SReduceAddSC, SHORTCARD, 0    )
ReduceAddFunc( SReduceAddC,  CARDINAL,  0    )
ReduceAddFunc( SReduceAddSI, SHORTINT,  0    )
ReduceAddFunc( SReduceAddI,  INTEGER,   0    )
ReduceAddFunc( SReduceAddR,  REAL,      0.0  )
ReduceAddFunc( SReduceAddLR, LONGREAL,  0.0  )


/*******************************************************************
 *****  normal routines with reduction operator "logical AND"  *****
 *******************************************************************/

#define ReduceAndFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce1,and,initVal,&=)

ReduceAndFunc( SReduceAndCh, CHAR,      '\377'     )
ReduceAndFunc( SReduceAndSC, SHORTCARD, 0xFFFF     )
ReduceAndFunc( SReduceAndC,  CARDINAL,  0xFFFFFFFF )
ReduceAndFunc( SReduceAndSI, SHORTINT,  0xFFFF     )
ReduceAndFunc( SReduceAndI,  INTEGER,   0xFFFFFFFF )


/***************************************************************
 *****  normal routines with reduction operator "maximum"  *****
 ***************************************************************/

#define ReduceMaxFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce2,max,initVal,>)

ReduceMaxFunc( SReduceMaxCh, CHAR,      MIN_CHAR      )
ReduceMaxFunc( SReduceMaxSC, SHORTCARD, MIN_SHORTCARD )
ReduceMaxFunc( SReduceMaxC,  CARDINAL,  MIN_LONGCARD  )
ReduceMaxFunc( SReduceMaxSI, SHORTINT,  MIN_SHORTINT  )
ReduceMaxFunc( SReduceMaxI,  INTEGER,   MIN_LONGINT   )
ReduceMaxFunc( SReduceMaxR,  REAL,      MIN_REAL      )
ReduceMaxFunc( SReduceMaxLR, LONGREAL,  MIN_LONGREAL  )


/***************************************************************
 *****  normal routines with reduction operator "minimum"  *****
 ***************************************************************/

#define ReduceMinFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce2,min,initVal,<)

ReduceMinFunc( SReduceMinCh, CHAR,      MAX_CHAR      )
ReduceMinFunc( SReduceMinSC, SHORTCARD, MAX_SHORTCARD )
ReduceMinFunc( SReduceMinC,  CARDINAL,  MAX_LONGCARD  )
ReduceMinFunc( SReduceMinSI, SHORTINT,  MAX_SHORTINT  )
ReduceMinFunc( SReduceMinI,  INTEGER,   MAX_LONGINT   )
ReduceMinFunc( SReduceMinR,  REAL,      MAX_REAL      )
ReduceMinFunc( SReduceMinLR, LONGREAL,  MAX_LONGREAL  )


/**********************************************************************
 *****  normal routines with reduction operator "multiplication"  *****
 **********************************************************************/

#define ReduceMulFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce1,prod,initVal,*=)

ReduceMulFunc( SReduceMulCh, CHAR,      '\001' )
ReduceMulFunc( SReduceMulSC, SHORTCARD, 1      )
ReduceMulFunc( SReduceMulC,  CARDINAL,  1      )
ReduceMulFunc( SReduceMulSI, SHORTINT,  1      )
ReduceMulFunc( SReduceMulI,  INTEGER,   1      )
ReduceMulFunc( SReduceMulR,  REAL,      1.0    )
ReduceMulFunc( SReduceMulLR, LONGREAL,  1.0    )


/******************************************************************
 *****  normal routines with reduction operator "logical OR"  *****
 ******************************************************************/

#define ReduceOrFunc(FuncName,elemType,initVal) \
    ReduceFunc(FuncName,elemType,GenericReduce1,or,initVal,|=)

ReduceOrFunc( SReduceOrCh, CHAR,      '\0' )
ReduceOrFunc( SReduceOrSC, SHORTCARD, 0    )
ReduceOrFunc( SReduceOrC,  CARDINAL,  0    )
ReduceOrFunc( SReduceOrSI, SHORTINT,  0    )
ReduceOrFunc( SReduceOrI,  INTEGER,   0    )


/********************************************************************
 ********************************************************************
 *******              MASKED REDUCTION ROUTINES               *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template for masked reduction routines
 */

#define MaskReduceFunc(FuncName,elemType,GlobalMaskReduce,accu,initVal,RED_OP)	\
    elemType									\
    FuncName(mask, nm, lm, x, n, l)						\
    register BOOLEAN *mask;							\
    LONGCARD nm, lm;								\
    register elemType *x;							\
    register LONGCARD n;							\
    LONGCARD l;									\
    {										\
      register elemType accu;							\
										\
      if (nm!=n) return;							\
										\
      GlobalMaskReduce(elemType,accu,initVal,RED_OP);				\
    }

/*
 *  generic template of masked reduction code with accumulation operator INC_OP
 */

#define GenericMaskReduce1(elemType,accu,initVal,INC_OP)\
    accu = initVal;					\
    for (; n>0; n--, mask++, x++)			\
      if (*mask)					\
        accu INC_OP *x;					\
    return (accu);

/*
 *  generic template of masked min/max reduction code with compare operator CMP_OP
 */

#define GenericMaskReduce2(elemType,accu,initVal,CMP_OP)\
    accu = initVal;					\
    {							\
      register elemType tmp;				\
							\
      for (; n>0; n--, mask++, x++)			\
        if (*mask) {					\
          tmp = *x;					\
          if (tmp CMP_OP accu)				\
	    accu = tmp;					\
	}						\
    };							\
    return (accu);


/****************************************************************
 *****  masked routines with reduction operator "addition"  *****
 ****************************************************************/

#define MaskReduceAddFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce1,sum,initVal,+=)

MaskReduceAddFunc( SMaskReduceAddCh, CHAR,      '\0' )
MaskReduceAddFunc( SMaskReduceAddSC, SHORTCARD, 0    )
MaskReduceAddFunc( SMaskReduceAddC,  CARDINAL,  0    )
MaskReduceAddFunc( SMaskReduceAddSI, SHORTINT,  0    )
MaskReduceAddFunc( SMaskReduceAddI,  INTEGER,   0    )
MaskReduceAddFunc( SMaskReduceAddR,  REAL,      0.0  )
MaskReduceAddFunc( SMaskReduceAddLR, LONGREAL,  0.0  )


/*******************************************************************
 *****  masked routines with reduction operator "logical AND"  *****
 *******************************************************************/

#define MaskReduceAndFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce1,and,initVal,&=)

MaskReduceAndFunc( SMaskReduceAndCh, CHAR,      '\377'     )
MaskReduceAndFunc( SMaskReduceAndSC, SHORTCARD, 0xFFFF     )
MaskReduceAndFunc( SMaskReduceAndC,  CARDINAL,  0xFFFFFFFF )
MaskReduceAndFunc( SMaskReduceAndSI, SHORTINT,  0xFFFF     )
MaskReduceAndFunc( SMaskReduceAndI,  INTEGER,   0xFFFFFFFF )


/***************************************************************
 *****  masked routines with reduction operator "maximum"  *****
 ***************************************************************/

#define MaskReduceMaxFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce2,max,initVal,>)

MaskReduceMaxFunc( SMaskReduceMaxCh, CHAR,      MIN_CHAR      )
MaskReduceMaxFunc( SMaskReduceMaxSC, SHORTCARD, MIN_SHORTCARD )
MaskReduceMaxFunc( SMaskReduceMaxC,  CARDINAL,  MIN_LONGCARD  )
MaskReduceMaxFunc( SMaskReduceMaxSI, SHORTINT,  MIN_SHORTINT  )
MaskReduceMaxFunc( SMaskReduceMaxI,  INTEGER,   MIN_LONGINT   )
MaskReduceMaxFunc( SMaskReduceMaxR,  REAL,      MIN_REAL      )
MaskReduceMaxFunc( SMaskReduceMaxLR, LONGREAL,  MIN_LONGREAL  )


/***************************************************************
 *****  masked routines with reduction operator "minimum"  *****
 ***************************************************************/

#define MaskReduceMinFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce2,min,initVal,<)

MaskReduceMinFunc( SMaskReduceMinCh, CHAR,      MAX_CHAR      )
MaskReduceMinFunc( SMaskReduceMinSC, SHORTCARD, MAX_SHORTCARD )
MaskReduceMinFunc( SMaskReduceMinC,  CARDINAL,  MAX_LONGCARD  )
MaskReduceMinFunc( SMaskReduceMinSI, SHORTINT,  MAX_SHORTINT  )
MaskReduceMinFunc( SMaskReduceMinI,  INTEGER,   MAX_LONGINT   )
MaskReduceMinFunc( SMaskReduceMinR,  REAL,      MAX_REAL      )
MaskReduceMinFunc( SMaskReduceMinLR, LONGREAL,  MAX_LONGREAL  )


/**********************************************************************
 *****  masked routines with reduction operator "multiplication"  *****
 **********************************************************************/

#define MaskReduceMulFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce1,prod,initVal,*=)

MaskReduceMulFunc( SMaskReduceMulCh, CHAR,      '\001' )
MaskReduceMulFunc( SMaskReduceMulSC, SHORTCARD, 1      )
MaskReduceMulFunc( SMaskReduceMulC,  CARDINAL,  1      )
MaskReduceMulFunc( SMaskReduceMulSI, SHORTINT,  1      )
MaskReduceMulFunc( SMaskReduceMulI,  INTEGER,   1      )
MaskReduceMulFunc( SMaskReduceMulR,  REAL,      1.0    )
MaskReduceMulFunc( SMaskReduceMulLR, LONGREAL,  1.0    )


/******************************************************************
 *****  masked routines with reduction operator "logical OR"  *****
 ******************************************************************/

#define MaskReduceOrFunc(FuncName,elemType,initVal) \
    MaskReduceFunc(FuncName,elemType,GenericMaskReduce1,or,initVal,|=)

MaskReduceOrFunc( SMaskReduceOrCh, CHAR,      '\0' )
MaskReduceOrFunc( SMaskReduceOrSC, SHORTCARD, 0    )
MaskReduceOrFunc( SMaskReduceOrC,  CARDINAL,  0    )
MaskReduceOrFunc( SMaskReduceOrSI, SHORTINT,  0    )
MaskReduceOrFunc( SMaskReduceOrI,  INTEGER,   0    )
