/*
 *  linux/ibcs/sysconf.c
 *
 *  Copyright (C) 1994  Mike Jagdis (jaggy@purplet.demon.co.uk)
 *
 * $Id: sysconf.c,v 1.3 1994/04/12 10:38:40 mike Exp $
 * $Source: /var/CVS/ibcs/sysconf.c,v $
 */


#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/stddef.h>
#include <linux/limits.h>
#include <linux/unistd.h>
#include <linux/segment.h>
#include <linux/ptrace.h>
#include <linux/config.h>
#include <linux/fcntl.h>

#include <asm/segment.h>
#include <asm/system.h>
#include <linux/fs.h>
#include <linux/sys.h>

#include <ibcs/ibcs.h>

#ifdef IBCS_TRACE
#include <ibcs/trace.h>
#endif


/* The sysconf() call is supposed to give applications access to various
 * kernel parameters. According to SCO's man page this a POSIX mandated
 * function. Perhaps it should be moved across as a native Linux call?
 *
 * N.B. SCO only has sysconf in the Xenix group. Therefore this is based
 * on the Xenix spec. Is SVR4 the same? Wyse Unix V.3.2.1A doesn't have
 * sysconf documented at all.
 *
 * N.B. 0-7 are required (by who?). Other values may be defined for
 * various systems but there appears no guarantee that they match across
 * platforms. Thus, unless we can identify what system the executable
 * was compiled for, we probably prefer to have extensions fail. Hell,
 * nothing important is going to use this obscure stuff anyway...
 */
#define _SC_ARG_MAX	0
#define _SC_CHILD_MAX	1
#define _SC_CLK_TCK	2
#define _SC_NGROUPS_MAX	3
#define _SC_OPEN_MAX	4
#define _SC_JOB_CONTROL	5
#define _SC_SAVED_IDS	6
#define _SC_VERSION	7


int ibcs_sysconf(int name) {
	switch (name) {
		case _SC_ARG_MAX: {
			/* From limits.h */
			return (ARG_MAX);
		}

		case _SC_CHILD_MAX: {
			/* From limits.h */
			return (CHILD_MAX);
		}

		case _SC_CLK_TCK: {
			extern unsigned long volatile jiffies;

			/* What is this? It's supposed to be evaluated
			 * at run time - jiffies?
			 */
			return (jiffies);
		}

		case _SC_NGROUPS_MAX: {
			/* From limits.h */
			return (NGROUPS_MAX);
		}

		case _SC_OPEN_MAX: {
			/* From limits.h */
			return (OPEN_MAX);
		}

		case _SC_JOB_CONTROL: {
			return (1);
		}

		case _SC_SAVED_IDS: {
			return (1);
		}

		case _SC_VERSION: {
			/* The version of the POSIX standard we conform
			 * to. SCO defines _POSIX_VERSION as 198808L
			 * sys/unistd.h. What are we?
			 */
			return (198808L);
		}
	}

#ifdef IBCS_TRACE
	if ((ibcs_trace & TRACE_API) || ibcs_func_p->trace) {
		printk(KERN_DEBUG "iBCS2 unsupported sysconf call %d\n", name);
	}
#endif

	return -EINVAL;
}
