/*
 * man-config.c
 *
 * Read the man.config file
 *
 * Input line types:
 *	MANBIN		/usr/bin/man
 *	MANPATH         /usr/X386/man	[/var/catman/X386]
 *	MANPATH_MAP     /usr/bin     /usr/man
 *	FSSTND
 *	NROFF           /usr/bin/groff -Tascii -mandoc
 *	.gz             /usr/bin/gunzip -c
 *	# Comment
 */

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#else
char *rindex ();
#endif

#include <stdio.h>
#include "config.h"
#include "man.h"
#include "paths.h"
#include "gripes.h"

extern int debug;

int fsstnd = 0;

#define whitespace(x) ((x) == ' ' || (x) == '\t')

void
addval (buf)
     char *buf;
{
  int i, len;
  char *bp;

  for (i = 0; i < sizeof(paths)/sizeof(paths[0]); i++) {
      len = strlen (paths[i].name);
      bp = buf + len;
      if(!strncmp (buf, paths[i].name, len) && (!*bp || whitespace(*bp))) {
	  while(whitespace(*bp))
	    bp++;
	  paths[i].path = strdup(bp);
	  return;
      }
  }
  gripe1 (UNRECOGNIZED_LINE, buf);
}

char *
getval (cmd)
     char *cmd;
{
  int i;

  for (i = 0; i < sizeof(paths)/sizeof(paths[0]); i++)
      if (!strcmp (cmd, paths[i].name))
	return paths[i].path;	/* never NULL */
  gripe1 (GETVAL_ERROR, cmd);
  return "";			/* impossible */
}

static struct xp {
    char *extension;		/* non-null, including initial . */
    char *expander;
    struct xp *nxt;
} uncompressors;		/* linked list, 1st entry unused */

void
addext (bp)
     char *bp;
{
  char *p, csv;
  struct xp *xpp;

  xpp = &uncompressors;
  while (xpp->nxt)
    xpp = xpp->nxt;
  xpp->nxt = (struct xp *) my_malloc (sizeof(struct xp));
  xpp = xpp->nxt;
  xpp->nxt = 0;

  p = bp;
  while(*p && !whitespace(*p))
    p++;
  csv = *p;
  *p = 0;
  xpp->extension = strdup(bp);

  *p = csv;
  while(whitespace(*p))
    p++;
  xpp->expander = strdup(p);
}

char *
get_expander (file)
     char *file;
{
  struct xp *xp;
  char *extp;

  extp = rindex (file, '.');
  if (extp != NULL) {
      for (xp = uncompressors.nxt; xp; xp = xp->nxt)
	if (!strcmp (extp, xp->extension))
	  return (xp->expander);
  }
  return NULL;
}

struct dirs cfdirlist;		/* linked list, 1st entry unused */

void
adddir (bp, mandatory)
     char *bp;
     int mandatory;
{
  int i;
  struct dirs *dlp;

  while (whitespace(*bp))
    bp++;
  if (*bp == 0)
    gripe (PARSE_ERROR_IN_CONFIG);

  dlp = &cfdirlist;
  while (dlp->nxt)
    dlp = dlp->nxt;
  dlp->nxt = (struct dirs *) my_malloc (sizeof(struct dirs));
  dlp = dlp->nxt;
  dlp->mandatory = mandatory;
  dlp->nxt = 0;

  if (!mandatory) {
      i = 0;
      while (*bp && !whitespace(*bp)) {
	  if (i < MAXPATHLEN - 1)
	    dlp->bindir[i++] = *bp;
	  bp++;
      }
      dlp->bindir[i] = 0;

      while (whitespace(*bp))
	bp++;
  } else {
      dlp->bindir[0] = 0;
  }

  i = 0;
  while (*bp && !whitespace(*bp)) {
      if (i < MAXPATHLEN - 1)
	dlp->mandir[i++] = *bp;
      bp++;
  }
  dlp->mandir[i] = 0;

  while (whitespace(*bp))
    bp++;
      
  i = 0;
  while (*bp && !whitespace(*bp)) {
      if (i < MAXPATHLEN - 1)
	dlp->catdir[i++] = *bp;
      bp++;
  }
  dlp->catdir[i] = 0;

  if (debug) {
      if (dlp->mandatory)
	gripe1 (FOUND_MANDIR, dlp->mandir);
      else
	gripe2 (FOUND_MAP, dlp->bindir, dlp->mandir);
      if (dlp->catdir[0])
	gripe1 (FOUND_CATDIR, dlp->catdir);
  }
}

void
read_config_file (cf)
     char *cf;
{
  char *bp;
  char *p;
  char buf[BUFSIZ];
  FILE *config;

  if (cf == NULL)
    cf = CONFIG_FILE;

  if ((config = fopen (cf, "r")) == NULL) {
      perror (cf);
      gripe1 (CONFIG_OPEN_ERROR, cf);
      return;
  }

  while ((bp = fgets (buf, BUFSIZ, config)) != NULL) {
      while (whitespace(*bp))
	bp++;

      for (p = bp; *p && *p != '#' && *p != '\n'; p++) ;
      if (!*p) {
	  gripe (LINE_TOO_LONG);
	  gripe1 (BAD_CONFIG_FILE, cf);
	  return;
      }
      while (p > bp && whitespace(p[-1]))
	p--;
      *p = 0;
      
      if (*bp == 0)
	continue;

      if (!strncmp ("MANPATH_MAP", bp, 11))
	adddir (bp+11, 0);
      else if (!strncmp ("MANPATH", bp, 7))
	adddir (bp+7, 1);
      else if(!strncmp ("MANDATORY_MANPATH", bp, 17))/* backwards compatible */
	adddir (bp+17, 1);
      else if (!strncmp ("FSSTND", bp, 6))
	fsstnd = 1;
      else if (*bp == '.')
	addext (bp);
      else
	addval (bp);
  }
}

