------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ U T I L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.133 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Errout;   use Errout;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Opt;      use Opt;
with Scans;    use Scans;
with Scn;      use Scn;
with Sem;      use Sem;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Type; use Sem_Type;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Snames;   use Snames;
with Stand;    use Stand;
with Style;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Sem_Util is

   ------------------------------
   -- Access_Checks_Suppressed --
   ------------------------------

   function Access_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Access_Checks
        or else Suppress_Access_Checks (E);
   end Access_Checks_Suppressed;

   -------------------------------------
   -- Accessibility_Checks_Suppressed --
   -------------------------------------

   function Accessibility_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Accessibility_Checks
        or else Suppress_Accessibility_Checks (E);
   end Accessibility_Checks_Suppressed;

   ------------------------
   -- Apply_Access_Check --
   ------------------------

   procedure Apply_Access_Check (N : Node_Id; Typ : Entity_Id) is
   begin
      if not Access_Checks_Suppressed (Typ) then
         Set_Do_Access_Check (N, True);
      end if;
   end Apply_Access_Check;

   ------------------------------
   -- Apply_Discriminant_Check --
   ------------------------------

   procedure Apply_Discriminant_Check (N : Node_Id; Typ : Entity_Id) is
   begin
      if not Discriminant_Checks_Suppressed (Typ) then
         Set_Do_Discriminant_Check (N, True);
      end if;
   end Apply_Discriminant_Check;

   -----------------------
   -- Apply_Range_Check --
   -----------------------

   --  A range constraint may be applied in any of the following contexts:
   --  object declaration, subtype declaration, derived declaration
   --  assignment, function/procedure/entry call, type conversion

   procedure Apply_Range_Check
     (N           : Node_Id;
      Source_Type : Entity_Id;
      Target_Type : Entity_Id)
   is
      Checks_On : constant Boolean :=
                    not Index_Checks_Suppressed (Target_Type)
                    and not Range_Checks_Suppressed (Target_Type);

   begin
      --  Don't worry about range checks if we have a previous error

      if Source_Type = Any_Type then
         return;

      --  Confine the range checks currently to only signed integer types and
      --  enumeration types since support for floating point and fixed point
      --  types is too limited to do useful checks at this time and checks
      --  for modular types need to be better understood by us???

      elsif not Is_Discrete_Type (Source_Type)
        or else Is_Modular_Integer_Type (Source_Type)
      then
         return;

      --  For literals, we can tell if the constraint error will be raised
      --  at compile time, so we never need a dynamic check, but if the
      --  exception will be raised, then post the usual warning, and replace
      --  the literal with a raise constraint error expression.

      elsif Is_Static_Expression (N) then

         declare
            LB        : constant Node_Id := Type_Low_Bound (Target_Type);
            UB        : constant Node_Id := Type_High_Bound (Target_Type);
            LB_Static : constant Boolean :=
                          Is_Static_Expression (Type_Low_Bound (Target_Type));
            UB_Static : constant Boolean :=
                          Is_Static_Expression (Type_High_Bound (Target_Type));
            Litval    : constant Uint := Expr_Value (N);

         begin
            --  If literal is outside a static bound, raise the warning

            if (LB_Static and then UI_Lt (Litval, Expr_Value (LB)))
              or else (UB_Static and then UI_Gt (Litval, Expr_Value (UB)))
            then
               --  Temporary code, replace literal value with the low or
               --  high bound as appropriate, this avoids trouble in Gigi
               --  and can be removed when we have a proper raise constraint
               --  error node that gets rid of the original expression. ???

               if LB_Static then
                  Rewrite_Substitute_Tree
                    (N, Make_Integer_Literal (Sloc (N), Expr_Value (LB)));
               else
                  Rewrite_Substitute_Tree
                    (N, Make_Integer_Literal (Sloc (N), Expr_Value (UB)));
               end if;

               Set_Etype (N, Target_Type);
               Constraint_Error_Warning (N, "static value out of range?!");

            --  If both bounds are static, and we passed the above checks
            --  then there is nothing to do, the literal is in range

            elsif LB_Static and UB_Static then
               null;

            --  Otherwise the check is needed

            else
               Set_Do_Range_Check (N, Checks_On);
            end if;
         end;

      --  Here for the case of a non-static expression, we need a runtime
      --  check unless the source type range is guaranteed to be in the
      --  range of the target type.

      else
         if not In_Subrange_Of (Source_Type, Target_Type) then
            Set_Do_Range_Check (N, Checks_On);
         end if;
      end if;
   end Apply_Range_Check;

   --------------------------
   -- Check_Fully_Declared --
   --------------------------

   procedure Check_Fully_Declared (T : Entity_Id; N : Node_Id) is
   begin
      if Ekind (T) = E_Incomplete_Type
        or else Has_Private_Component (T)
      then
         Error_Msg_NE
           ("premature usage of private or incomplete type &", N, T);
      end if;
   end Check_Fully_Declared;

   ------------------------------
   -- Constraint_Error_Warning --
   ------------------------------

   procedure Constraint_Error_Warning (N : Node_Id; Msg : String) is
      P    : Node_Id;
      Warn : Boolean;

   begin
      --  Should we generate a warning? The answer is not quite yes. The very
      --  annoying exception occurs in the case of a short circuit operator
      --  where the left operand is static and decisive. Climb parents to see
      --  if that is the case we have here.

      Warn := True;
      P := N;

      loop
         P := Parent (P);
         exit when Nkind (P) not in N_Subexpr;

         if (Nkind (P) = N_Op_And_Then
             and then Is_Static_Expression (Left_Opnd (P))
             and then Is_False (Expr_Value (Left_Opnd (P))))
           or else (Nkind (P) = N_Op_Or_Else
             and then Is_Static_Expression (Left_Opnd (P))
             and then Is_True (Expr_Value (Left_Opnd (P))))
         then
            Warn := False;
            exit;
         end if;
      end loop;

      if Warn then
         Raise_Warning (N, Standard_Constraint_Error, Msg);
      end if;

      Create_Raise_Expression (N, Standard_Constraint_Error);

      Set_Raises_Constraint_Error (N, True);
      Set_Potentially_Static      (N, True);
   end Constraint_Error_Warning;

   -----------------------------
   -- Create_Raise_Expression --
   -----------------------------

   procedure Create_Raise_Expression (N : Node_Id; Excep_Id : Entity_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Typ   : constant Entity_Id  := Etype (N);
      Pstat : constant Boolean    := Potentially_Static (N);

   begin
      Rewrite_Substitute_Tree (N, Make_Raise_Constraint_Error (Loc));
      Set_Analyzed (N, True);
      Set_Etype (N, Typ);
      Set_Raises_Constraint_Error (N, True);

      --  The resulting node is potentially static if the original node was
      --  potentially static. Raising constraint error affects staticness
      --  but not potential staticness.

      Set_Potentially_Static (N, Pstat);
      return;
   end Create_Raise_Expression;

   --------------------
   -- Current_Entity --
   --------------------

   --  The currently visible definition for a given identifier is the
   --  one most chained at the start of the visibility chain, i.e. the
   --  one that is referenced by the Node_Id value of the name of the
   --  given identifier.

   function Current_Entity (N : Node_Id) return Entity_Id is
   begin
      return Get_Name_Entity_Id (Chars (N));
   end Current_Entity;

   -----------------------------
   -- Current_Entity_In_Scope --
   -----------------------------

   function Current_Entity_In_Scope (N : Node_Id) return Entity_Id is
      E : Entity_Id;

   begin
      E := Get_Name_Entity_Id (Chars (N));

      while Present (E)
        and then Scope (E) /= Current_Scope
      loop
         E := Homonym (E);
      end loop;

      return E;
   end Current_Entity_In_Scope;

   -------------------
   -- Current_Scope --
   -------------------

   function Current_Scope return Entity_Id is
      C : constant Entity_Id := Scope_Stack.Table (Scope_Stack.last).Entity;

   begin
      if Present (C) then
         return C;
      else
         return Standard_Standard;
      end if;
   end Current_Scope;

   -------------------------------
   -- Defining_Unit_Simple_Name --
   -------------------------------

   function Defining_Unit_Simple_Name (N : Node_Id) return Entity_Id is
      Nam : Node_Id := Defining_Unit_Name (N);

   begin
      if Nkind (Nam) in N_Entity then
         return Nam;
      else
         return Defining_Identifier (Nam);
      end if;
   end Defining_Unit_Simple_Name;

   ------------------------------------
   -- Discriminant_Checks_Suppressed --
   ------------------------------------

   function Discriminant_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Discriminant_Checks
        or else Suppress_Discriminant_Checks (E);
   end Discriminant_Checks_Suppressed;

   --------------------------------
   -- Division_Checks_Suppressed --
   --------------------------------

   function Division_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Division_Checks
        or else Suppress_Division_Checks (E);
   end Division_Checks_Suppressed;

   -----------------------------
   -- Enclosing_Dynamic_Scope --
   -----------------------------

   function Enclosing_Dynamic_Scope (E : Entity_Id) return Entity_Id is
      S  : Entity_Id := E;

   begin
      while S /= Standard_Standard
        and then Ekind (S) /= E_Block
        and then Ekind (S) /= E_Function
        and then Ekind (S) /= E_Procedure
        and then Ekind (S) /= E_Task_Type
        and then Ekind (S) /= E_Entry
      loop
         S := Scope (S);
      end loop;

      return S;
   end Enclosing_Dynamic_Scope;

   -------------------------------
   -- Enclosing_Lib_Unit_Entity --
   -------------------------------

   function Enclosing_Lib_Unit_Entity (N : Node_Id) return Entity_Id is
      Unit_Node   : Node_Id := Enclosing_Lib_Unit_Node (N);
      Unit_Kind   : constant Node_Kind := Nkind (Unit (Unit_Node));
      Unit_Entity : Entity_Id;

   begin
      if Unit_Kind = N_Package_Body
        or else (Unit_Kind = N_Subprogram_Body
                  and then not Acts_As_Spec (Unit_Node))
      then
         Unit_Entity := Corresponding_Spec (Unit (Unit_Node));

      elsif Unit_Kind = N_Package_Instantiation
        or else Unit_Kind = N_Procedure_Instantiation
        or else Unit_Kind = N_Function_Instantiation
        or else Unit_Kind = N_Package_Renaming_Declaration
        or else Unit_Kind in N_Generic_Renaming_Declaration
      then
         Unit_Entity := Defining_Unit_Simple_Name (Unit (Unit_Node));

      elsif Unit_Kind = N_Subunit then

         while Nkind (Unit (Unit_Node)) = N_Subunit loop
            Unit_Node := Library_Unit (Unit_Node);
         end loop;

         Unit_Entity := Defining_Unit_Simple_Name (Specification (Unit
           (Library_Unit (Unit_Node))));

      else
         Unit_Entity := Defining_Unit_Simple_Name (Specification (Unit (
           Unit_Node)));
      end if;

      return Unit_Entity;
   end Enclosing_Lib_Unit_Entity;

   -----------------------------
   -- Enclosing_Lib_Unit_Node --
   -----------------------------

   function Enclosing_Lib_Unit_Node (N : Node_Id) return Node_Id is
      Current_Node : Node_Id := N;

   begin
      while Present (Current_Node)
        and then Nkind (Current_Node) /= N_Compilation_Unit
      loop
         Current_Node := Parent (Current_Node);
      end loop;

      if Nkind (Current_Node) /= N_Compilation_Unit then
         return Empty;
      end if;

      return Current_Node;
   end Enclosing_Lib_Unit_Node;

   -----------------------------------
   -- Elaboration_Checks_Suppressed --
   -----------------------------------

   function Elaboration_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Elaboration_Checks
        or else Suppress_Elaboration_Checks (E);
   end Elaboration_Checks_Suppressed;

   ----------------
   -- Enter_Name --
   ----------------

   procedure Enter_Name (Def_Id : Node_Id) is
      E : constant Entity_Id := Current_Entity (Def_Id);
      S : constant Entity_Id := Current_Scope;

   begin
      --  Add new name to current scope declarations. Check for duplicate
      --  declaration, which may or may not be a genuine error.

      if Present (E) and then Scope (E) = S then

         --  Case of previous entity entered because of a missing declaration
         --  or else a bad subtype indication. Best is to use the new entity,
         --  and make the previous one invisible.

         if Etype (E) = Any_Type then
            Set_Is_Immediately_Visible (E, False);

         --  Case of renaming declaration constructed for package instances.
         --  if there is an explicit declaration with the same identifier,
         --  the renaming is not immediately visible any longer, but remains
         --  visible through selected component notation.

         elsif Nkind (Parent (E)) = N_Package_Renaming_Declaration
           and then not Comes_From_Source (E)
         then
            Set_Is_Immediately_Visible (E, False);

         --  Case of genuine duplicate declaration

         else
            --  Case of duplicate definition of entity in Standard, this can
            --  only occur as the result of trying to declare a package with
            --  the same name as an entity in Standard. Abandon compilation
            --  in this case, since we have serious visibility problems that
            --  would be hard to recover nicely from.

            if S = Standard_Standard then
               Error_Msg_N
                 ("library package conflicts with name in Standard", Def_Id);
               raise Unrecoverable_Error;

            --  In all other cases, simply keep previous declaration visible,
            --  and give some usable attributes to the new entity to minimize
            --  cascaded error messages

            else
               Set_Ekind (Def_Id, E_Variable);
               Set_Etype (Def_Id, Any_Type);
               Set_Scope (Def_Id,  S);
               Error_Msg_Sloc := Sloc (E);
               Error_Msg_N
                 ("declaration of& conflicts with#", Def_Id);
               return;
            end if;
         end if;
      end if;

      --  The kind E_Void insures that premature uses of the entity will be
      --  detected. Any_Type insures that no cascaded errors will occur.

      Set_Ekind (Def_Id, E_Void);
      Set_Etype (Def_Id, Any_Type);

      Set_Is_Immediately_Visible (Def_Id);
      Set_Current_Entity         (Def_Id);
      Set_Homonym                (Def_Id, E);
      Append_Entity              (Def_Id, S);
      Set_Public_Status          (Def_Id);

   end Enter_Name;

   ------------------
   -- First_Actual --
   ------------------

   function First_Actual (Node : Node_Id) return Node_Id is
      N : Node_Id;

   begin
      if No (Parameter_Associations (Node)) then
         return Empty;
      end if;

      N := First (Parameter_Associations (Node));

      if Nkind (N) = N_Parameter_Association then
         return First_Named_Actual (Node);
      else
         return N;
      end if;
   end First_Actual;

   --------------------------
   -- Get_Declaration_Node --
   --------------------------

   function Get_Declaration_Node (Unit_Id : Entity_Id) return Node_Id is
      N : Node_Id := Parent (Unit_Id);

   begin
      --  Predefined operators do not have a full function declaration.

      if Ekind (Unit_Id) = E_Operator then
         return N;
      end if;

      while Nkind (N) /= N_Generic_Package_Declaration
        and then Nkind (N) /= N_Generic_Subprogram_Declaration
        and then Nkind (N) /= N_Package_Declaration
        and then Nkind (N) /= N_Package_Body
        and then Nkind (N) /= N_Package_Renaming_Declaration
        and then Nkind (N) /= N_Subprogram_Declaration
        and then Nkind (N) /= N_Subprogram_Body
        and then Nkind (N) /= N_Subprogram_Body_Stub
        and then Nkind (N) /= N_Subprogram_Renaming_Declaration
        and then Nkind (N) not in N_Generic_Renaming_Declaration
      loop
         N := Parent (N);
         pragma Assert (Present (N));
      end loop;

      return N;
   end Get_Declaration_Node;

   ----------------------
   -- Get_Index_Bounds --
   ----------------------

   procedure Get_Index_Bounds (I : Node_Id; L, H : out Node_Id) is
      Kind : constant Node_Kind := Nkind (I);

   begin
      if Kind = N_Range then
         L := Low_Bound (I);
         H := High_Bound (I);

      elsif Kind = N_Subtype_Indication then
         L := Low_Bound (Range_Expression (Constraint (I)));
         H := High_Bound (Range_Expression (Constraint (I)));

      else
         pragma Assert (Is_Entity_Name (I) and then Is_Type (Entity (I)));
         L := Low_Bound (Scalar_Range (Entity (I)));
         H := High_Bound (Scalar_Range (Entity (I)));

      end if;
   end Get_Index_Bounds;

   ------------------------
   -- Get_Name_Entity_Id --
   ------------------------

   function Get_Name_Entity_Id (Id : Name_Id) return Entity_Id is
   begin
      return Entity_Id (Get_Name_Table_Info (Id));
   end Get_Name_Entity_Id;

   ---------------------------
   -- Has_Private_Component --
   ---------------------------

   function Has_Private_Component (Type_Id : Entity_Id) return Boolean is
      Btype     : constant Entity_Id := Base_Type (Type_Id);
      Component : Entity_Id;

   begin
      if Is_Private_Type (Btype) then
         return No (Underlying_Type (Btype))
           and then not Is_Generic_Type (Btype);

      elsif Is_Array_Type (Btype) then
         return Has_Private_Component (Component_Type (Btype));

      elsif Is_Record_Type (Btype) then

         Component := First_Entity (Btype);
         while Present (Component) loop
            if Has_Private_Component (Etype (Component)) then
               return True;
            end if;

            Component := Next_Entity (Component);
         end loop;

         return False;

      else
         return False;
      end if;
   end Has_Private_Component;

   --------------------------
   -- Has_Tagged_Component --
   --------------------------

   function Has_Tagged_Component (Typ : Entity_Id) return Boolean is
      Comp : Entity_Id;

   begin
      if Is_Private_Type (Typ)
        and then Present (Underlying_Type (Typ))
      then
         return Has_Tagged_Component (Underlying_Type (Typ));

      elsif Is_Array_Type (Typ) then
         return Has_Tagged_Component (Component_Type (Typ));

      elsif Is_Tagged_Type (Typ) then
         return True;

      elsif Is_Record_Type (Typ) then
         Comp := First_Component (Typ);

         while Present (Comp) loop
            if Has_Tagged_Component (Etype (Comp)) then
               return True;
            end if;

            Comp := Next_Component (Typ);
         end loop;

         return False;

      else
         return False;
      end if;
   end Has_Tagged_Component;

   --------------------
   -- In_Subrange_Of --
   --------------------

   function In_Subrange_Of (T1 : Entity_Id; T2 : Entity_Id) return Boolean is
   begin
      if T1 = T2 or else Is_Subtype_Of (T1, T2) then
         return True;

      elsif not (Is_Static_Subtype (T1) and then Is_Static_Subtype (T2)) then
         return False;

      elsif Is_Discrete_Type (T1) then
         return UI_Le (Expr_Value (Type_Low_Bound (T2)),
                       Expr_Value (Type_Low_Bound (T1)))
           and then
                UI_Ge (Expr_Value (Type_High_Bound (T2)),
                       Expr_Value (Type_High_Bound (T1)));
      else
         return False;
      end if;
   end In_Subrange_Of;

   -----------------------------
   -- Index_Checks_Suppressed --
   -----------------------------

   function Index_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Index_Checks
        or else Suppress_Index_Checks (E);
   end Index_Checks_Suppressed;

   -------------------------------
   -- Inside_Preelaborated_Unit --
   -------------------------------

   function Inside_Preelaborated_Unit (N : Node_Id) return Boolean is
      Unit_Node   : constant Node_Id := Enclosing_Lib_Unit_Node (N);
      Unit_Entity :        Entity_Id;
      Unit_Kind   :        Node_Kind;

   begin
      if not Present (Unit_Node) then
         return False;
      else
         Unit_Entity := Enclosing_Lib_Unit_Entity (Unit_Node);
         Unit_Kind   := Nkind (Unit (Unit_Node));
      end if;

      return Is_Preelaborated (Unit_Entity)
        or else Is_Pure (Unit_Entity)
        or else Is_Shared_Passive (Unit_Entity)
        or else Is_Remote_Types (Unit_Entity)
        or else (Is_Remote_Call_Interface (Unit_Entity)
                  and then Unit_Kind /= N_Package_Body
                  and then Nkind (Unit_Node) /= N_Subprogram_Body);

   end Inside_Preelaborated_Unit;

   ----------------------
   -- Inside_Pure_Unit --
   ----------------------

   function Inside_Pure_Unit (N : Node_Id) return Boolean is
      Unit_Node   : constant Node_Id := Enclosing_Lib_Unit_Node (N);

   begin
      if not Present (Unit_Node) then
         return False;
      else
         return Is_Pure (Enclosing_Lib_Unit_Entity (Unit_Node));
      end if;
   end Inside_Pure_Unit;

   ---------------------------------------
   -- Inside_Remote_Call_Interface_Unit --
   ---------------------------------------

   function Inside_Remote_Call_Interface_Unit (N : Node_Id) return Boolean is
      Unit_Node   : constant Node_Id := Enclosing_Lib_Unit_Node (N);

   begin
      if not Present (Unit_Node) then
         return False;
      else
         return
           Is_Remote_Call_Interface (Enclosing_Lib_Unit_Entity (Unit_Node))
             and then Nkind (Unit_Node) /= N_Package_Body
             and then Nkind (Unit_Node) /= N_Subprogram_Body;
      end if;
   end Inside_Remote_Call_Interface_Unit;

   ---------------------------------------
   -- Inside_Subpgm_Task_Protected_Unit --
   ---------------------------------------

   function Inside_Subpgm_Task_Protected_Unit (N : Node_Id) return Boolean is
      Node : Node_Id;

   begin
      Node := N;

      while Present (Node) loop

         --  The following is to verify that N (a declaration node) is inside
         --  subprogram, generic subprogram, task unit, protected unit.
         --  Used to validate if a lib. unit is Pure. RM 10.2.1(16).

         if Nkind (Node) = N_Subprogram_Declaration
           or else Nkind (Node) = N_Generic_Subprogram_Declaration
           or else Nkind (Node) = N_Subprogram_Body
           or else Nkind (Node) = N_Task_Definition
           or else Nkind (Node) = N_Task_Body
           or else Nkind (Node) = N_Task_Type_Declaration
           or else Nkind (Node) = N_Single_Task_Declaration
           or else Nkind (Node) = N_Protected_Definition
           or else Nkind (Node) = N_Protected_Body
           or else Nkind (Node) = N_Protected_Type_Declaration
           or else Nkind (Node) = N_Single_Protected_Declaration
         then
            return True;
         end if;

         Node := Parent (Node);
      end loop;

      return False;

   end Inside_Subpgm_Task_Protected_Unit;

   ----------------------------
   -- Inside_Subprogram_Unit --
   ----------------------------

   function Inside_Subprogram_Unit (N : Node_Id) return Boolean is
      Node : Node_Id;

   begin
      Node := N;

      while Present (Node) loop
         if Nkind (Node) = N_Subprogram_Declaration
           or else Nkind (Node) = N_Generic_Subprogram_Declaration
           or else Nkind (Node) = N_Subprogram_Body
         then
            return True;
         end if;

         Node := Parent (Node);
      end loop;

      return False;

   end Inside_Subprogram_Unit;

   ---------------------
   -- Is_By_Copy_Type --
   ---------------------

   function Is_By_Copy_Type (T : Entity_Id) return Boolean is
   begin
      return
        Is_Elementary_Type (T)
          or else (Is_Private_Type (T)
                     and then Is_Elementary_Type (Underlying_Type (T)));
   end Is_By_Copy_Type;

   -------------------
   -- Is_Child_Unit --
   -------------------

   function Is_Child_Unit (Id : Entity_Id) return Boolean is
   begin
      return   (Ekind (Id) = E_Package
        or else Ekind (Id) = E_Function
        or else Ekind (Id) = E_Procedure)
        and then Present (Parent (Id))
        and then
            Nkind (Parent (Get_Declaration_Node (Id))) = N_Compilation_Unit;
   end Is_Child_Unit;

   --------------------
   -- Is_Entity_Name --
   --------------------

   function Is_Entity_Name (N : Node_Id) return Boolean is
      Kind : constant Node_Kind := Nkind (N);

   begin
      --  Identifiers and expanded names are always entity names

      return Kind = N_Identifier
        or else Kind = N_Expanded_Name

      --  Attribute references are entity names if they refer to an entity.
      --  Note that we don't do this by testing for the presence of the
      --  Entity field in the N_Attribute_Reference node, since it may not
      --  have been set yet.

        or else (Kind = N_Attribute_Reference
                  and then Is_Entity_Attribute_Name (Attribute_Name (N)));
   end Is_Entity_Name;

   --------------
   -- Is_False --
   --------------

   function Is_False (U : Uint) return Boolean is
   begin
      return UI_Is_Zero (U);
   end Is_False;

   -----------------------------
   -- Is_Library_Level_Entity --
   -----------------------------

   function Is_Library_Level_Entity (E : Entity_Id) return Boolean is
      Decl      : constant Node_Id := Get_Declaration_Node (E);
      N         : Node_Id;
      Unum      : constant Unit_Number_Type := Get_Sloc_Unit_Number (Sloc (E));
      Unit_Node : constant Node_Id := Unit (Cunit (Unum));

   begin
      if E = Cunit_Entity (Unum) then
         return True;

      elsif Nkind (Unit_Node) = N_Package_Declaration then
         N := E;

         while N /= Unit_Node loop

            if Nkind (Parent (N)) = N_Package_Specification
              and then List_Containing (N) = Private_Declarations (Parent (N))
            then
               return False;
            else
               N := Parent (N);
            end if;

         end loop;

         return True;

      else
         return False;
      end if;
   end Is_Library_Level_Entity;

   -------------
   -- Is_True --
   -------------

   function Is_True (U : Uint) return Boolean is
   begin
      return not UI_Is_Zero (U);
   end Is_True;

   -----------------
   -- Is_Variable --
   -----------------

   function Is_Variable (N : Node_Id) return Boolean is

      function Is_Variable_Prefix (N : Node_Id) return Boolean;
      --   Prefixes can involve implicit dereferences

      function Is_Variable_Prefix (N : Node_Id) return Boolean is
      begin
         return Is_Variable (N)
           or else (Is_Access_Type (Etype (N))
             and then not Is_Access_Constant (Root_Type (Etype (N))));
      end Is_Variable_Prefix;

   begin
      if Assignment_OK (N) then
         return True;

      elsif Is_Entity_Name (N) then
         declare
            K : Entity_Kind := Ekind (Entity (N));

         begin
            return K = E_Variable
              or else  K = E_Component
              or else  K = E_Out_Parameter
              or else  K = E_In_Out_Parameter
              or else  K = E_Generic_In_Out_Parameter;
         end;

      else
         case Nkind (N) is
            when N_Indexed_Component | N_Slice =>
               return Is_Variable_Prefix (Prefix (N));

            when N_Selected_Component =>
               return Is_Variable_Prefix (Prefix (N))
               and then Is_Variable (Selector_Name (N));

            when N_Expanded_Name =>
               return Is_Variable (Selector_Name (N));

            --  For an explicit dereference, we must check whether the type
            --  is ACCESS CONSTANT, since if it is, then it is not a variable.

            when N_Explicit_Dereference =>
               return Is_Access_Type (Etype (Prefix (N)))
                 and then not
                   Is_Access_Constant (Root_Type (Etype (Prefix (N))));

            --  The type conversion is the case where we do not deal with the
            --  context dependend special case of an actual parameter. Thus
            --  the type conversion is only considered a variable for the
            --  purposes of this routine if the target type is tagged.

            when N_Type_Conversion =>
               return Is_Variable (Expression (N))
                 and then Is_Tagged_Type (Etype (Subtype_Mark (N)))
                 and then Is_Tagged_Type (Etype (Expression (N)));

            --  GNAT allows an unchecked type conversion as a variable. This
            --  only affects the generation of internal expanded code, since
            --  calls to instantiations of Unchecked_Conversion are never
            --  considered variables (since they are function calls).

            when N_Unchecked_Type_Conversion =>
               return Is_Variable (Expression (N));

            when others =>  return False;
         end case;
      end if;
   end Is_Variable;

   ------------------------------
   -- Length_Checks_Suppressed --
   ------------------------------

   function Length_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Length_Checks
        or else Suppress_Length_Checks (E);
   end Length_Checks_Suppressed;

   -------------------------
   -- New_External_Entity --
   -------------------------

   function New_External_Entity
     (Kind         : Entity_Kind;
      Scope_Id     : Entity_Id;
      Sloc_Value   : Source_Ptr;
      Related_Id   : Entity_Id;
      Suffix       : Character;
      Suffix_Index : Nat := 0;
      Prefix       : Character := ' ')
      return         Entity_Id
   is
      N : constant Entity_Id :=
            Make_Defining_Identifier (Sloc_Value,
              New_External_Name
                (Chars (Related_Id), Suffix, Suffix_Index, Prefix));

   begin
      Set_Ekind          (N, Kind);
      Set_Is_Internal    (N, True);
      Append_Entity      (N, Scope_Id);
      Set_Public_Status  (N);
      Set_Current_Entity (N);
      return N;
   end New_External_Entity;

   -------------------------
   -- New_Internal_Entity --
   -------------------------

   function New_Internal_Entity
     (Kind       : Entity_Kind;
      Scope_Id   : Entity_Id;
      Sloc_Value : Source_Ptr;
      Id_Char    : Character)
      return       Entity_Id
   is
      N : constant Entity_Id :=
            Make_Defining_Identifier (Sloc_Value, New_Internal_Name (Id_Char));

   begin
      Set_Ekind          (N, Kind);
      Set_Is_Internal    (N, True);
      Append_Entity      (N, Scope_Id);
      Set_Current_Entity (N);
      return N;
   end New_Internal_Entity;

   -----------------
   -- Next_Actual --
   -----------------

   function Next_Actual (Actual_Id : Node_Id) return Node_Id is
      N  : Node_Id;

   begin
      --  If we are pointing at a positional parameter, it is a member of
      --  a node list (the list of parameters), and the next parameter
      --  is the next node on the list, unless we hit a parameter
      --  association, in which case we shift to using the chain whose
      --  head is the First_Named_Actual in the parent, and then is
      --  threaded using the Next_Named_Actual of the Parameter_Association.
      --  All this fiddling is because the original node list is in the
      --  textual call order, and what we need is the declaration order.

      if Is_List_Member (Actual_Id) then
         N := Next (Actual_Id);

         if Nkind (N) = N_Parameter_Association then
            return First_Named_Actual (Parent (Actual_Id));
         else
            return N;
         end if;

      else
         return Next_Named_Actual (Parent (Actual_Id));
      end if;
   end Next_Actual;

   -----------------------
   -- Normalize_Actuals --
   -----------------------

   --  Chain actuals according to formals of subprogram. If there are
   --  no named associations, the chain is simply the list of Parameter
   --  Associations, since the order is the same as the declaration order.
   --  If there are named associations, then the First_Named_Actual field
   --  in the N_Procedure_Call_Statement node or N_Function_Call node
   --  points to the Parameter_Association node for the parameter that
   --  comes first in declaration order. The remaining named parameters
   --  are then chained in declaration order using Next_Named_Actual.

   --  This routine also verifies that the number of actuals is compatible
   --  with the number and default values of formals, but performs no type
   --  checking (type checking is done by the caller).

   --  If the matching succeeds, Success is set to True, and the caller
   --  proceeds with type-checking. If the match is unsuccessful, then
   --  Success is set to False, and the caller attempts a different
   --  interpretation, if there is one.

   --  If the flag Report is on, the call is not overloaded, and a failure
   --  to match can be reported here, rather than in the caller.

   procedure Normalize_Actuals
     (N       : Node_Id;
      S       : Entity_Id;
      Report  : Boolean;
      Success : out Boolean)
   is
      Actuals     : constant List_Id := Parameter_Associations (N);
      Actual      : Node_Id   := Empty;
      Formal      : Entity_Id;
      Last        : Entity_Id := Empty;
      First_Named : Entity_Id := Empty;
      Found       : Boolean;

      Formals_To_Match : Integer := 0;
      Actuals_To_Match : Integer := 0;

      procedure Chain (A : Node_Id);
      --  Need some documentation on this spec ???

      procedure Chain (A : Node_Id) is
      begin
         if No (Last) then

            --  Call node points to first actual in list.

            Set_First_Named_Actual (N, Explicit_Actual_Parameter (A));

         else
            Set_Next_Named_Actual (Last, Explicit_Actual_Parameter (A));
         end if;

         Last := A;
      end Chain;

   --  Start of processing for Normalize_Actuals

   begin
      if Is_Access_Type (S) then

         --  The name in the call is a function call that returns an access
         --  to subprogram. The designated type has the list of formals.

         Formal := First_Formal (Designated_Type (S));
      else
         Formal := First_Formal (S);
      end if;

      while Present (Formal) loop
         Formals_To_Match := Formals_To_Match + 1;
         Formal := Next_Formal (Formal);
      end loop;

      --  Find if there is a named association, and verify that no positional
      --  associations appear after named ones.

      if Present (Actuals) then
         Actual := First (Actuals);
      end if;

      while Present (Actual)
        and then Nkind (Actual) /= N_Parameter_Association
      loop
         Actuals_To_Match := Actuals_To_Match + 1;
         Actual := Next (Actual);
      end loop;

      if No (Actual) and Actuals_To_Match = Formals_To_Match then

         --  Most common case: positional notation, no defaults

         Success := True;
         return;

      elsif Actuals_To_Match > Formals_To_Match then

         --  Too many actuals: will not work.

         if Report then
            Error_Msg_N ("too many arguments in call", N);
         end if;

         Success := False;
         return;
      end if;

      First_Named := Actual;

      while Present (Actual) loop
         if Nkind (Actual) /= N_Parameter_Association then
            Error_Msg_N
              ("positional parameters not allowed after named ones", Actual);
            Success := False;
            return;

         else
            Actuals_To_Match := Actuals_To_Match + 1;
         end if;

         Actual := Next (Actual);
      end loop;

      if Present (Actuals) then
         Actual := First (Actuals);
      end if;

      Formal := First_Formal (S);

      while Present (Formal) loop

         --  Match the formals in order. If the corresponding actual
         --  is positional,  nothing to do. Else scan the list of named
         --  actuals to find the one with the right name.

         if Present (Actual)
           and then Nkind (Actual) /= N_Parameter_Association
         then
            Actual := Next (Actual);
            Actuals_To_Match := Actuals_To_Match - 1;
            Formals_To_Match := Formals_To_Match - 1;

         else
            --  For named parameters, search the list of actuals to find
            --  one that matches the next formal name.

            Actual := First_Named;
            Found  := False;

            while Present (Actual) loop
               if Chars (Selector_Name (Actual)) = Chars (Formal) then
                  Found := True;
                  Chain (Actual);
                  Actuals_To_Match := Actuals_To_Match - 1;
                  Formals_To_Match := Formals_To_Match - 1;
                  exit;
               end if;

               Actual := Next (Actual);
            end loop;

            if not Found then
               if Ekind (Formal) /= E_In_Parameter
                 or else No (Default_Value (Formal))
               then
                  if Report then
                     Error_Msg_NE ("missing argument in call:&", N, Formal);
                  end if;

                  Success := False;
                  return;

               else
                  Formals_To_Match := Formals_To_Match - 1;
                  null; -- Chain_Default_Node;
               end if;
            end if;
         end if;

         Formal := Next_Formal (Formal);
      end loop;

      if  Formals_To_Match = 0 and then Actuals_To_Match = 0 then
         Success := True;
         return;

      else
         if Report then
            Error_Msg_N ("too many arguments in call", N);
         end if;

         Success := False;
         return;
      end if;
   end Normalize_Actuals;

   --------------------------------
   -- Overflow_Checks_Suppressed --
   --------------------------------

   function Overflow_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Overflow_Checks
        or else Suppress_Overflow_Checks (E);
   end Overflow_Checks_Suppressed;

   -----------------------------
   -- Range_Checks_Suppressed --
   -----------------------------

   function Range_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Range_Checks
        or else Suppress_Range_Checks (E);
   end Range_Checks_Suppressed;

   ------------------
   -- Real_Convert --
   ------------------

   --  We do the conversion to get the value of the real string by using
   --  the scanner, see Sinput for details on use of the internal source
   --  buffer for scanning internal strings.

   function Real_Convert (S : String) return Node_Id is
      Negative : Boolean;

   begin
      Source := Internal_Source_Ptr;
      Scan_Ptr := 1;

      for J in S'range loop
         Source (Source_Ptr (J)) := S (J);
      end loop;

      Source (S'Length + 1) := EOF;

      if Source (Scan_Ptr) = '-' then
         Negative := True;
         Scan_Ptr := Scan_Ptr + 1;
      else
         Negative := False;
      end if;

      Scan;

      if Negative then
         Set_Realval (Token_Node, UR_Negate (Realval (Token_Node)));
      end if;

      --  We used the scanner to construct the node, so Comes_From_Source
      --  got set True, but this literal doesn't really come from the source

      Set_Comes_From_Source (Token_Node, False);
      return Token_Node;
   end Real_Convert;

   ---------------
   -- Same_Name --
   ---------------

   function Same_Name (N1, N2 : Node_Id) return Boolean is
      K1 : constant Node_Kind := Nkind (N1);
      K2 : constant Node_Kind := Nkind (N2);

   begin
      if (K1 = N_Identifier or else K1 = N_Defining_Identifier)
        and then (K2 = N_Identifier or else K2 = N_Defining_Identifier)
      then
         return Chars (N1) = Chars (N2);

      elsif (K1 = N_Selected_Component or else K1 = N_Expanded_Name)
        and then (K2 = N_Selected_Component or else K2 = N_Expanded_Name)
      then
         return Same_Name (Selector_Name (N1), Selector_Name (N2))
           and then Same_Name (Prefix (N1), Prefix (N2));

      else
         return False;
      end if;
   end Same_Name;

   ------------------------
   -- Set_Current_Entity --
   ------------------------

   --  The given entity is to be set as the currently visible definition
   --  of its associated name (i.e. the Node_Id associated with its name).
   --  All we have to do is to get the name from the identifier, and
   --  then set the associated Node_Id to point to the given entity.

   procedure Set_Current_Entity (E : Entity_Id) is
   begin
      Set_Name_Entity_Id (Chars (E), E);
   end Set_Current_Entity;

   ---------------------------------
   -- Set_Entity_With_Style_Check --
   ---------------------------------

   procedure Set_Entity_With_Style_Check (N : Node_Id; Val : Entity_Id) is
      Val_Actual : Entity_Id;

   begin
      if Style_Check and then Nkind (N) = N_Identifier then
         Val_Actual := Val;

         --  A special situation arises for derived operations, where we want
         --  to do the check against the parent (since the Sloc of the derived
         --  operation points to the derived type declaration itself).

         while not Comes_From_Source (Val_Actual)
           and then Nkind (Val_Actual) in N_Entity
           and then (Ekind (Val_Actual) = E_Enumeration_Literal
                      or else Ekind (Val_Actual) = E_Function
                      or else Ekind (Val_Actual) = E_Generic_Function
                      or else Ekind (Val_Actual) = E_Procedure
                      or else Ekind (Val_Actual) = E_Generic_Procedure)
           and then Present (Alias (Val_Actual))
         loop
            Val_Actual := Alias (Val_Actual);
         end loop;

         Style.Check_Identifier (N, Val_Actual);
      end if;

      Set_Entity (N, Val);
   end Set_Entity_With_Style_Check;

   ------------------------
   -- Set_Name_Entity_Id --
   ------------------------

   procedure Set_Name_Entity_Id (Id : Name_Id; Val : Entity_Id) is
   begin
      Set_Name_Table_Info (Id, Int (Val));
   end Set_Name_Entity_Id;

   ---------------------
   -- Set_Next_Actual --
   ---------------------

   procedure Set_Next_Actual (Ass1_Id : Node_Id; Ass2_Id : Node_Id) is
   begin
      if Nkind (Parent (Ass1_Id)) = N_Parameter_Association then
         Set_First_Named_Actual (Parent (Ass1_Id), Ass2_Id);
      end if;
   end Set_Next_Actual;

   -----------------
   -- Set_Privals --
   -----------------

   procedure Set_Privals
      (Def : Node_Id;
       Sub : Entity_Id;
       Loc : Source_Ptr)
   is
      P       : Entity_Id;
      Priv    : Entity_Id;

   begin
      pragma Assert (Nkind (Def) = N_Protected_Definition);

      if Present (Private_Declarations (Def)) then

         P := First (Private_Declarations (Def));
         while Present (P) loop
            Priv :=
              Make_Defining_Identifier (Loc,
                New_External_Name (Chars (Defining_Identifier (P)), 'P'));
            Set_Protected_Subprogram (Defining_Identifier (P), Sub);
            Set_Prival (Defining_Identifier (P), Priv);

            P := Next (P);
         end loop;
      end if;
   end Set_Privals;

   -----------------------
   -- Set_Public_Status --
   -----------------------

   procedure Set_Public_Status (Id : Entity_Id) is
      S : constant Entity_Id := Current_Scope;

   begin
      if S = Standard_Standard
        or else (Is_Public (S)
                  and then (Ekind (S) = E_Package
                             or else Is_Record_Type (S)
                             or else Ekind (S) = E_Void))
      then
         Set_Is_Public (Id);
      end if;
   end Set_Public_Status;

   --------------------
   -- Static_Integer --
   --------------------

   function Static_Integer (N : Node_Id) return Uint is
   begin
      Analyze (N);
      Resolve (N, Any_Integer);

      if Is_Static_Expression (N) then
         return Expr_Value (N);

      elsif Etype (N) = Any_Type then
         return No_Uint;

      else
         Check_Static_Expression (N);
         return No_Uint;
      end if;
   end Static_Integer;

   -------------------------------
   -- Storage_Checks_Suppressed --
   -------------------------------

   function Storage_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Storage_Checks
        or else Suppress_Storage_Checks (E);
   end Storage_Checks_Suppressed;

   ---------------------------
   -- Tag_Checks_Suppressed --
   ---------------------------

   function Tag_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Tag_Checks
        or else Suppress_Tag_Checks (E);
   end Tag_Checks_Suppressed;

   -----------------
   -- Trace_Scope --
   -----------------

   procedure Trace_Scope (N : Node_Id; E : Entity_Id; Msg : String) is
   begin
      if Debug_Flag_W then
         for J in 0 .. Scope_Stack.Last loop
            Write_Str ("  ");
         end loop;

         Write_Str (Msg);
         Write_Name (Chars (E));
         Write_Str ("   line ");
         Write_Int (Int (Get_Line_Number (Sloc (N))));
         Write_Eol;
      end if;
   end Trace_Scope;

   -------------------
   -- Unimplemented --
   -------------------

   procedure Unimplemented (N : Node_Id; Feature : String) is
   begin
      Error_Msg_N (Feature & " not implemented yet", N);
   end Unimplemented;

   ----------------
   -- Wrong_Type --
   ----------------

   procedure Wrong_Type (Expr : Node_Id; Expected_Type : Entity_Id) is
      Found_Type : constant Entity_Id := Etype (Expr);

   begin
      --  Don't output message if either type is Any_Type, or if a message
      --  has already been posted for this node. We need to do the latter
      --  check explicitly (it is ordinarily done in Errout), because we
      --  are using ! to force the output of the error messages.

      if Expected_Type = Any_Type
        or else Found_Type = Any_Type
        or else Error_Posted (Expr)
      then
         return;
      end if;

      --  An interesting special check. If the expression is parenthesized
      --  and its type corresponds to the type of the sole component of the
      --  expected record type, or to the component type of the expected one
      --  dimensional array type, then assume we have a bad aggregate attempt.

      if Paren_Count (Expr) /= 0
        and then
          ((Is_Record_Type (Expected_Type)
             and then Present (First_Component (Expected_Type))
             and then
               Covers (Etype (First_Component (Expected_Type)), Found_Type)
             and then No (Next_Component (First_Component (Expected_Type))))

          or else
           (Is_Array_Type (Expected_Type)
             and then Number_Dimensions (Expected_Type) = 1
             and then
               Covers (Etype (Component_Type (Expected_Type)), Found_Type)))

      then
         Error_Msg_N ("positional aggregate cannot have one component", Expr);

      --  Normal case of one type found, some other type expected

      else
         Error_Msg_Sloc := Sloc (Expected_Type);
         Error_Msg_NE ("expected type& declared#!", Expr, Expected_Type);
         Error_Msg_Sloc := Sloc (Found_Type);
         Error_Msg_NE ("found type& declared#!", Expr, Found_Type);
      end if;
   end Wrong_Type;

end Sem_Util;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.131
--  date: Wed Aug 24 19:37:52 1994;  author: dewar
--  (Create_Raise_Expression): N_Raise_Constraint_Error unconditionally
--   created now that it is supported.
--  ----------------------------
--  revision 1.132
--  date: Fri Aug 26 14:42:48 1994;  author: schonber
--  (Enclosing_Lib_Unit_Entity): for a subunit, follow the chain of enclosing
--   subunits until reaching the ancestor unit.
--  ----------------------------
--  revision 1.133
--  date: Mon Aug 29 23:43:13 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
