------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             W I D E C H A R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.7 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Opt; use Opt;

with System.WCh_Con; use System.WCh_Con;
with System.WCh_JIS; use System.WCh_JIS;

package body Widechar is

   -----------------
   -- Length_Wide --
   -----------------

   function Length_Wide return Nat is
   begin
      if Wide_Character_Encoding_Method = WCEM_Hex then
         return 5;
      else
         return 2;
      end if;
   end Length_Wide;

   ---------------
   -- Scan_Wide --
   ---------------

   procedure Scan_Wide
     (S : Source_Buffer_Ptr;
      P : in out Source_Ptr;
      C : out Char_Code;
      E : out Boolean)
   is
      B1 : Natural;
      B2 : Natural;
      C1 : Character;
      C2 : Character;

   begin
      if Upper_Half_Encoding then
         C1 := S (P);
         P := P + 1;
         C2 := S (P);

         --  Second character may not be control character

         if C2 <= Character'Val (16#1F#) then
            E := True;
            return;
         else
            P := P + 1;
         end if;

         --  EUC

         if Wide_Character_Encoding_Method = WCEM_EUC then
            C := Char_Code (Wide_Character'Pos (EUC_To_JIS (C1, C2)));

         --  Shift-JIS

         elsif Wide_Character_Encoding_Method = WCEM_Shift_JIS then
            C := Char_Code (Wide_Character'Pos (Shift_JIS_To_JIS (C1, C2)));

         --  Upper (internal code = external code)

         else -- Wide_Character_Encoding_Method = WCEM_Upper
            B1 := Character'Pos (C1);
            B2 := Character'Pos (C2);
            C := Char_Code (256 * B1 + B2);
         end if;

         E := False;
         return;

      --  Only other possibility is Hex coding with ESC scanned

      else
         P := P + 1;
         C := 0;

         for J in 1 .. 4 loop
            B1 := Character'Pos (S (P));

            if B1 in Character'Pos ('0') .. Character'Pos ('9') then
               B1 := B1 - Character'Pos ('0');

            elsif B1 in Character'Pos ('A') .. Character'Pos ('F') then
               B1 := B1 - (Character'Pos ('A') - 10);

            else
               E := True;
               return;
            end if;

            C := Char_Code (Natural (C) * 16 + B1);
            P := P + 1;
         end loop;

         E := False;
         return;
      end if;

   end Scan_Wide;

   --------------
   -- Set_Wide --
   --------------

   procedure Set_Wide
     (C : Char_Code;
      S : in out String;
      P : in out Natural)
   is
      Hex : constant array (0 .. 15) of Character := "0123456789ABCDEF";
      B1  : Natural;
      B2  : Natural;
      C1  : Character;
      C2  : Character;

   begin
      B1 := Natural (C) / 256;
      B2 := Natural (C) mod 256;
      C1 := Character'Val (B1);
      C2 := Character'Val (B2);

      if Upper_Half_Encoding then

         --  EUC

         if Wide_Character_Encoding_Method = WCEM_EUC then
            JIS_To_EUC (Wide_Character'Val (C), C1, C2);

         --  Shift-JIS

         elsif Wide_Character_Encoding_Method = WCEM_Shift_JIS then
            JIS_To_Shift_JIS (Wide_Character'Val (C), C1, C2);

         --  Upper (internal code = external code)

         else
            C1 := Character'Val (B1);
            C2 := Character'Val (B2);
         end if;

         S (P + 1) := C1;
         S (P + 2) := C2;
         P := P + 2;

      --  Only other possibility is Hex encoding

      else
         S (P + 1) := ESC;
         S (P + 2) := Hex (B1 / 16);
         S (P + 3) := Hex (B2 rem 16);
         S (P + 4) := Hex (B1 / 16);
         S (P + 5) := Hex (B2 rem 16);
         P := P + 5;
      end if;

   end Set_Wide;

   ---------------
   -- Skip_Wide --
   ---------------

   procedure Skip_Wide (S : String; P : in out Natural) is
   begin
      if Upper_Half_Encoding then
         P := P + 2;

      --  Only other possibility is hex encoding

      else
         P := P + 5;
      end if;
   end Skip_Wide;

end Widechar;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.5
--  date: Mon Jul 25 19:09:15 1994;  author: dewar
--  Use System.Wide_Character_Constants to get encoding method codes
--  ----------------------------
--  revision 1.6
--  date: Mon Jul 25 23:22:57 1994;  author: dewar
--  (Length_Wide): New function
--  ----------------------------
--  revision 1.7
--  date: Wed Aug 10 14:28:36 1994;  author: dewar
--  Change name JIS_Conversions to WCh_JIS
--  Change name Wide_Character_Constants to WCh_Con
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
