------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.27 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Package containing utility procedures used throughout the expander

with Rtsfind; use Rtsfind;
with Types;   use Types;

package Exp_Util is

   Expander_Active : Boolean := False;
   --  A flag that indicates if expansion is active (True) or deactivated
   --  (False). When expansion is deactivated all calls to expander routines
   --  have no effect. Note that the initial setting of False is merely to
   --  prevent saving of an undefined value for an initial call to the
   --  Expander_Mode_Save_And_Set procedure.

   function Build_Runtime_Call (Loc : Source_Ptr; RE : RE_Id) return Node_Id;
   --  Build an N_Procedure_Call_Statement calling the given runtime entity.
   --  The call has no parameters. The first argument provides the location
   --  information for the tree and for error messages. The call node is not
   --  analyzed on return, the caller is responsible for analyzing it.

   procedure Expander_Mode_Save_And_Set (Status : Boolean);
   --  Saves the current setting of the Expander_Active flag on an internal
   --  stack and then sets the flag to the given value.

   procedure Expander_Mode_Restore;
   --  Restores the setting of the Expander_Active flag using the top entry
   --  pushed onto the stack by Expander_Mode_Save_And_Reset, popping the
   --  stack, except that if any errors have been detected, then the state
   --  of the flag is left set to False.

   function Expand_Class_Wide_Subtype
     (N                 : Node_Id;
      Class_Type        : Entity_Id;
      Expression_Entity : Entity_Id)
      return              List_Id;
   --  When a Class_Wide_Subtype value is encountered, that is to say in such
   --  cases as:
   --
   --    X: T'Class := Exp
   --    new T'Class'(Exp)
   --
   --  This function generates the list of declarations defining a record
   --  of the following form :
   --
   --    type anon is record
   --       _parent : Root_Type_Of (Typ) constrained with Exp discriminants;
   --       Extension : String (1 .. expr to match size of Exp);
   --    end record;
   --
   --  This record is compatible with any value of T'Class thanks to the
   --  first field and has the same size as Exp thanks to the second field.
   --  This record is attached to the subclass by its Equivalent_Type field.
   --  The declarations are not analyzed, the caller is responsible for
   --  analyzing them after they have been inserted into the tree.

   procedure Expand_Subtype_From_Expr
     (N             : Node_Id;
      Unc_Type      : Entity_Id;
      Subtype_Indic : Node_Id;
      Exp           : Node_Id);
   --  Build a constrained subtype from the initial value in object
   --  declarations and/or allocations when the type is indefinite (including
   --  class-wide).

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id;
   --  Find the first primitive operation of type T whose name is 'Name'.
   --  this function allows the use of a primitive operation which is not
   --  directly visible

   procedure Insert_List_Before_And_Analyze (N : Node_Id; L : List_Id);
   --  Inserts list L before node N using Atree.Insert_List_Before, and then,
   --  after this insertion is complete, analyzes all the nodes in the list.
   --  The list must be non-empty for this call.

   function Is_Unconstrained (N : Node_Id) return Boolean;
   --  Return true if N is an expression of an unconstrained type.

   function Make_Tagged_Copy
     (Loc      : Source_Ptr;
      L_Entity : Entity_Id;
      R_Entity : Entity_Id;
      Typ      : Entity_Id)
      return Node_Id;
   --  Expand the code for copying the value of R_Entity to L_Entity. The
   --  arguments are entities rather than expressions because this function is
   --  used in conjunction with prepare_multi_use_of_Exp. The value of a
   --  tagged type excludes its tag. If the type is controlled it also excludes
   --  the finalization pointers

   procedure Prepare_Multi_Use_Of_Expr
     (Exp_In        : Node_Id;
      Entity_Exp    : out Entity_Id;
      Expanded_Code : out List_Id);
   --  If the expression is an entity_Name, does nothing otherwise expand a
   --  variable (Entity_Exp) which is a pointer to the value of the
   --  expression (Exp_In). Expanded_Code contains the generated declarations
   --  which have to be inserted in the tree before any use of the
   --  expression.  This procedure is used in conjunction with
   --  New_Ref_To_Expr and give an alternative to the use of 'Evaluate_Once'
   --  when this one is not appropriate.

   function New_Ref_To_Expr
     (Entity_Exp : Entity_Id;
      Loc        : Source_Ptr)
      return       Node_Id;
   --  Return a new reference to the entity representing an Expression. (See
   --  Prepare_Multi_Use_Of_Exp). The usage of theses functions consists in
   --  calling Prepare_Multi_Use_Of_Expr first, insert the generated code in
   --  the tree (possibly in an Expression_Action) then use New_Ref_To_Expr for
   --  any appearance of Expr (including the original one).

   procedure Protect_Statements (N : Node_Id; E : Entity_Id);
   --  This function protects the handled statement sequence of node N by
   --  adding a cleanup that is a call to the procedure referenced by the
   --  entity E. If necessary (if the handled statement sequence already has
   --  an exception handler, or a cleanup), an extra block is wrapped around.

   procedure Wrap_Cleanup_Procedure (N : Node_Id);
   --  Given an N_Subprogram_Body node, this procedure adds an Abort_Defer
   --  call at the start of the statement sequence, and an Abort_Undefer call
   --  at the end of the statement sequence. All cleanup routines (i.e. those
   --  that are called from "at end" handlers) must defer abort on entry and
   --  undefer abort on exit. Note that it is assumed that the code for the
   --  procedure does not contain any return statements which would allow the
   --  flow of control to escape doing the undefer call.

end Exp_Util;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.25
--  date: Thu May 26 02:34:15 1994;  author: dewar
--  (Traceback_Store): Removed
--  Minor cleanups
--  ----------------------------
--  revision 1.26
--  date: Thu Aug 18 16:26:27 1994;  author: comar
--  Introduce Prepare_Multi_Use_Of_Expr and New_Ref_To_Expr as an alternative
--   to Evaluate_Once.
--  Change spec of Make_Tagged_Copy which now use the 2 former subprograms.
--  ----------------------------
--  revision 1.27
--  date: Thu Aug 18 20:09:27 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
