/*
 * Copyright (c) 1982, 1986, 1988, 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)tcp_subr.c	8.1 (Berkeley) 6/10/93
 * tcp_subr.c,v 1.5 1994/10/08 22:39:58 phk Exp
 */

/*
 * Changes and additions relating to SLiRP
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#include <sys/socket.h>

#ifdef HAVE_SYS_FILIO_H
#include <sys/filio.h>
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#include "h/ip.h"
#include "h/tcp.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"
#include "h/ctl.h"
#include <string.h>

/* patchable/settable parameters for tcp */
int 	tcp_mssdflt = TCP_MSS;
int 	tcp_rttdflt = TCPTV_SRTTDFLT / PR_SLOWHZ;
int	tcp_do_rfc1323 = 0;	/* Don't do rfc1323 preformance enhancments */
int	tcp_rcvspace;	/* You may want to change this */
int	tcp_sndspace;	/* Keep small if you have an error prone link */

/*
 * Tcp initialization
 */
void
tcp_init()
{
	tcp_iss = 1;		/* wrong */
	tcb.so_next = tcb.so_prev = &tcb;
	
	tcp_rcvspace = TCP_RCVSPACE;
	tcp_sndspace = TCP_SNDSPACE;
	
	/* Make sure tcp_sndspace is at *least* 2*MSS */
	if (tcp_sndspace < 2*(ifp.ifp_mtu - sizeof(struct tcpiphdr)))
		tcp_sndspace = 2*(ifp.ifp_mtu - sizeof(struct tcpiphdr));
}

/*
 * Create template to be used to send tcp packets on a connection.
 * Call after host entry created, fills
 * in a skeletal tcp/ip header, minimizing the amount of work
 * necessary when the connection is used.
 */
/* struct tcpiphdr * */
void
tcp_template(tp)
	struct tcpcb *tp;
{
	struct socket *so = tp->t_socket;
	register struct tcpiphdr *n = &tp->t_template;

#ifdef DEBUG
	debug_call(dfd,"tcp_template(tp) called ...\n");
	debug_call(dfd,"    tp = %d\n", (int)tp);
	fflush_call(dfd);
#endif
	
	n->ti_next = n->ti_prev = 0;
	n->ti_x1 = 0;
	n->ti_pr = IPPROTO_TCP;
	n->ti_len = htons(sizeof (struct tcpiphdr) - sizeof (struct ip));
	n->ti_src = so->so_faddr;
	n->ti_dst = so->so_laddr;
	n->ti_sport = so->so_fport;
	n->ti_dport = so->so_lport;
	
	n->ti_seq = 0;
	n->ti_ack = 0;
	n->ti_x2 = 0;
	n->ti_off = 5;
	n->ti_flags = 0;
	n->ti_win = 0;
	n->ti_sum = 0;
	n->ti_urp = 0;
}

/*
 * Send a single message to the TCP at address specified by
 * the given TCP/IP header.  If m == 0, then we make a copy
 * of the tcpiphdr at ti and send directly to the addressed host.
 * This is used to force keep alive messages out using the TCP
 * template for a connection tp->t_template.  If flags are given
 * then we send a message back to the TCP which originated the
 * segment ti, and discard the mbuf containing it and any other
 * attached mbufs.
 *
 * In any case the ack and sequence number of the transmitted
 * segment are as specified by the parameters.
 */
void
tcp_respond(tp, ti, m, ack, seq, flags)
	struct tcpcb *tp;
	register struct tcpiphdr *ti;
	register struct mbuf *m;
	tcp_seq ack, seq;
	int flags;
{
	register int tlen;
	int win = 0;

#ifdef DEBUG
	debug_call(dfd,"tcp_respond(tp,ti,m,ack,seq,flags) called ...\n");
	debug_call(dfd,"    tp = %d\n    ti = %d\n    m = %d\n    ack = %d\n\
    seq = %d\n    flags = %d\n", (int)tp, (int)ti, (int)m, (int)ack, (int)seq, flags);
	fflush_call(dfd);
#endif
	
	
	if (tp)
		win = sbspace(&tp->t_socket->so_rcv);
	if (m == 0) {
		if ((m = m_get()) == NULL)
			return;
#ifdef TCP_COMPAT_42
		tlen = 1;
#else
		tlen = 0;
#endif
		m->m_data += ifp.ifp_maxlinkhdr;
		*mtod(m, struct tcpiphdr *) = *ti;
		ti = mtod(m, struct tcpiphdr *);
		flags = TH_ACK;
	} else {
		/* 
		 * ti points into m so the next line is just making
		 * the mbuf point to ti
		 */
		m->m_data = (caddr_t)ti;
		
		m->m_len = sizeof (struct tcpiphdr);
		tlen = 0;
#define xchg(a,b,type) { type t; t=a; a=b; b=t; }
		xchg(ti->ti_dst.s_addr, ti->ti_src.s_addr, u_int32_t);
		xchg(ti->ti_dport, ti->ti_sport, u_int16_t);
#undef xchg
	}
	ti->ti_len = htons((u_short)(sizeof (struct tcphdr) + tlen));
	tlen += sizeof (struct tcpiphdr);
	m->m_len = tlen;

	ti->ti_next = ti->ti_prev = 0;
	ti->ti_x1 = 0;
	ti->ti_seq = htonl(seq);
	ti->ti_ack = htonl(ack);
	ti->ti_x2 = 0;
	ti->ti_off = sizeof (struct tcphdr) >> 2;
	ti->ti_flags = flags;
	if (tp)
		ti->ti_win = htons((u_int16_t) (win >> tp->rcv_scale));
	else
		ti->ti_win = htons((u_int16_t)win);
	ti->ti_urp = 0;
	ti->ti_sum = 0;
	ti->ti_sum = cksum(m, tlen);
	((struct ip *)ti)->ip_len = tlen;

	((struct ip *)ti)->ip_ttl = ip_defttl;
	
	(void) ip_output((struct socket *)0, m);
}

/*
 * Create a new TCP control block, making an
 * empty reassembly queue and hooking it to the argument
 * protocol control block.
 */
struct tcpcb *
tcp_newtcpcb(so)
	struct socket *so;
{
	register struct tcpcb *tp;
	
#ifdef DEBUG
	debug_call(dfd,"tcp_newtcpcb(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	tp = (struct tcpcb *)malloc(sizeof(*tp));
	if (tp == NULL)
		return ((struct tcpcb *)0);
	
	memset((char *) tp, 0, sizeof(struct tcpcb));
	tp->seg_next = tp->seg_prev = (struct tcpiphdr *)tp;
	tp->t_maxseg = tcp_mssdflt; /* XXX */

	tp->t_flags = tcp_do_rfc1323 ? (TF_REQ_SCALE|TF_REQ_TSTMP) : 0;
	tp->t_socket = so;
	
	/*
	 * Init srtt to TCPTV_SRTTBASE (0), so we can tell that we have no
	 * rtt estimate.  Set rttvar so that srtt + 2 * rttvar gives
	 * reasonable initial retransmit time.
	 */
	tp->t_srtt = TCPTV_SRTTBASE;
	tp->t_rttvar = tcp_rttdflt * PR_SLOWHZ << 2;
	tp->t_rttmin = TCPTV_MIN;

	TCPT_RANGESET(tp->t_rxtcur, 
	    ((TCPTV_SRTTBASE >> 2) + (TCPTV_SRTTDFLT << 2)) >> 1,
	    TCPTV_MIN, TCPTV_REXMTMAX);

	tp->snd_cwnd = TCP_MAXWIN << TCP_MAX_WINSHIFT;
	tp->snd_ssthresh = TCP_MAXWIN << TCP_MAX_WINSHIFT;
	tp->t_state = TCPS_CLOSED;
	
	so->so_tcpcb = tp;

	return (tp);
}

/*
 * Drop a TCP connection, reporting
 * the specified error.  If connection is synchronized,
 * then send a RST to peer.
 */
struct tcpcb *
tcp_drop(tp, errno)
	register struct tcpcb *tp;
	int errno;
{

#ifdef DEBUG
	debug_call(dfd,"tcp_drop(tp,errno) called ...\n");
	debug_call(dfd,"    tp = %d\n    errno = %d\n", (int)tp, errno);
	fflush_call(dfd);
#endif
	
	
	if (TCPS_HAVERCVDSYN(tp->t_state)) {
		tp->t_state = TCPS_CLOSED;
#ifdef DEBUG
		debug_misc(dfd," tcp_output  called  from tcp_drop\n");
#endif
		(void) tcp_output(tp);
		tcpstat.tcps_drops++;
	} else
		tcpstat.tcps_conndrops++;
/*	if (errno == ETIMEDOUT && tp->t_softerror)
 *		errno = tp->t_softerror;
 */
/*	so->so_error = errno; */
	return (tcp_close(tp));
}

/*
 * Close a TCP control block:
 *	discard all space held by the tcp
 *	discard internet protocol block
 *	wake up any sleepers
 */
struct tcpcb *
tcp_close(tp)
	register struct tcpcb *tp;
{
	register struct tcpiphdr *t;
	struct socket *so = tp->t_socket;
	register struct mbuf *m;

#ifdef DEBUG
	debug_call(dfd,"tcp_close(tp) called ...\n");
	fflush_call(dfd);
	debug_call(dfd,"    tp = %d\n", (int)tp);
	fflush_call(dfd);
#endif

	/* free the reassembly queue, if any */
	t = tp->seg_next;
	while (t != (struct tcpiphdr *)tp) {
		t = (struct tcpiphdr *)t->ti_next;
		m = REASS_MBUF((struct tcpiphdr *)t->ti_prev);
		remque(t->ti_prev);
		m_freem(m);
	}
	/* It's static */
/*	if (tp->t_template)
 *		(void) m_free(dtom(tp->t_template));
 */
/*	free(tp, M_PCB);  */
	free(tp);
	so->so_tcpcb = 0;
	soisfdisconnected(so);
	/* clobber input socket cache if we're closing the cached connection */
	if (so == tcp_last_so)
		tcp_last_so = &tcb;
	close(so->s);
	sbfree(&so->so_rcv);
	sbfree(&so->so_snd);
	sofree(so);
	tcpstat.tcps_closed++;
	return ((struct tcpcb *)0);
}

void
tcp_drain()
{
	/* XXX */
}

/*
 * When a source quench is received, close congestion window
 * to one segment.  We will gradually open it again as we proceed.
 */

#ifdef notdef

void
tcp_quench(i, errno)

	int errno;
{
	struct tcpcb *tp = intotcpcb(inp);

	if (tp)
		tp->snd_cwnd = tp->t_maxseg;
}

#endif /* notdef */

/*
 * TCP protocol interface to socket abstraction.
 */

/*
 * User issued close, and wish to trail through shutdown states:
 * if never received SYN, just forget it.  If got a SYN from peer,
 * but haven't sent FIN, then go to FIN_WAIT_1 state to send peer a FIN.
 * If already got a FIN from peer, then almost done; go to LAST_ACK
 * state.  In all other cases, have already sent FIN to peer (e.g.
 * after PRU_SHUTDOWN), and just have to play tedious game waiting
 * for peer to send FIN or not respond to keep-alives, etc.
 * We can let the user exit from the close as soon as the FIN is acked.
 */
void
tcp_sockclosed(tp)
	struct tcpcb *tp;
{

#ifdef DEBUG
	debug_call(dfd,"tcp_sockclosed(tp) called ...\n");
	debug_call(dfd,"    tp = %d\n", (int)tp);
	fflush_call(dfd);
#endif
	
	switch (tp->t_state) {

	case TCPS_CLOSED:
	case TCPS_LISTEN:
	case TCPS_SYN_SENT:
		tp->t_state = TCPS_CLOSED;
		tp = tcp_close(tp);
		break;

	case TCPS_SYN_RECEIVED:
	case TCPS_ESTABLISHED:
		tp->t_state = TCPS_FIN_WAIT_1;
		break;

	case TCPS_CLOSE_WAIT:
		tp->t_state = TCPS_LAST_ACK;
		break;
	}
/*	soisfdisconnecting(tp->t_socket); */
	if (tp && tp->t_state >= TCPS_FIN_WAIT_2)
		soisfdisconnected(tp->t_socket);
	if (tp)
		tcp_output(tp);
}

/* 
 * Connect to a host on the Internet
 * Called by tcp_input
 * Only do a connect, the tcp feilds will be set in tcp_input
 * return 0 if there's a result of the connect,
 * else return -1 means we're still connecting
 */
void
tcp_fconnect(so)
	struct socket *so;
{
	struct sockaddr_in addr;
	int s, ret;
	int opt = 1;
	
#ifdef DEBUG
	debug_call(dfd,"tcp_fconnect(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif

	s = socket(AF_INET,SOCK_STREAM,0);
	if (s < 0)
	   return;
	ioctl(s, FIONBIO, &opt); /* Set non-blocking */
	setsockopt(s,SOL_SOCKET,SO_REUSEADDR,(char *)&opt,sizeof(int));
	setsockopt(s,SOL_SOCKET,SO_OOBINLINE,(char *)&opt,sizeof(int));
	
	addr.sin_family = AF_INET;
	if ((ntohl(so->so_faddr.s_addr) & 0xffffff00) == ntohl(special_addr))
		addr.sin_addr.s_addr = inet_addr("127.0.0.1");
	else
		addr.sin_addr = so->so_faddr;
	addr.sin_port = so->so_fport;
	
	/* We don't care what port we get */
	ret = connect(s,(struct sockaddr *)&addr,sizeof (addr));
	
	/*
	 * If it's not in progress, it failed, so we just return 0,
	 * without clearing SS_NOFDREF
	 */
	so->s = s;
	soisfconnecting(so);
}

/*
 * Accept the socket and connect to the local-host
 * 
 * We have a problem. The correct thing to do would be
 * to first connect to the local-host, and only if the
 * connection is accepted, then do an accept() here.
 * But, a) we need to know who's trying to connect 
 * to the socket to be able to SYN the local-host, and
 * b) we are allready connected to the foregn host by
 * the time it gets to accept(), so... We simply accept
 * here and SYN the local-host.
 */ 
void
tcp_connect(inso)
	struct socket *inso;
{
	struct socket *so;
	struct sockaddr_in addr;
	int addrlen = sizeof(struct sockaddr_in);
	struct tcpcb *tp;
	int s, opt = 1;

#ifdef DEBUG
	debug_call(dfd,"tcp_connect(inso) called ...\n");
	debug_call(dfd,"    inso = %d\n", (int)inso);
	fflush_call(dfd);
#endif
	
	/*
	 * If it's an SS_ACCEPTONCE socket, no need to socreate()
	 * another socket, just use the accept() socket.
	 */
	if (inso->so_state & SS_FACCEPTONCE) {
		/* FACCEPTONCE allready have a tcpcb, so we just sbinit */
		so = inso;
		if (sbinit(&so->so_rcv, tcp_rcvspace) == NULL ||
		    sbinit(&so->so_snd, tcp_sndspace) == NULL) {
			sofree(so);
			return;
		}
	} else {
		if ((so = socreate()) == NULL) {
			/* If it failed, get rid of the pending connection */
			close(accept(inso->s,(struct sockaddr *)&addr,&addrlen));
			return;
		}
		if (tcp_attach(so) < 0) {
			free(so); /* Not sofree */
			return;
		}
		so->so_laddr = inso->so_laddr;
		so->so_lport = inso->so_lport;
	}
	
	(void) tcp_mss(sototcpcb(so), 0);

	if ((s = accept(inso->s,(struct sockaddr *)&addr,&addrlen)) < 0) {
#ifdef DEBUG
		debug_misc(dfd," errno = %d\n", errno);
#endif
		tcp_close(sototcpcb(so)); /* This will sofree() as well */
		return;
	}
	ioctl(s, FIONBIO, &opt); /* Set it non-blocking XXX */
	setsockopt(s,SOL_SOCKET,SO_REUSEADDR,(char *)&opt,sizeof(int));
	setsockopt(s,SOL_SOCKET,SO_OOBINLINE,(char *)&opt,sizeof(int));
	
	so->so_fport = addr.sin_port;
	so->so_faddr = addr.sin_addr;
	/* Translate connections from localhost to the real hostname */
	if (so->so_faddr.s_addr == 0 || so->so_faddr.s_addr == inet_addr("127.0.0.1"))
	   so->so_faddr.s_addr = our_addr;
	
	/* Close the accept() socket, set right state */
	if (inso->so_state & SS_FACCEPTONCE) {
		close(so->s); /* If we only accept once, close the accet() socket */
		so->so_state = SS_NOFDREF; /* Don't select it yet, even though we have an FD */
					   /* if it's not FACCEPTONCE, it's allready NOFDREF */
	}
	so->s = s;
	
	so->so_iptos = tcp_tos(so);
	tp = sototcpcb(so);

	tcp_template(tp);
	
	/* Compute window scaling to request.  */
/*	while (tp->request_r_scale < TCP_MAX_WINSHIFT &&
 *		(TCP_MAXWIN << tp->request_r_scale) < so->so_rcv.sb_hiwat)
 *		tp->request_r_scale++;
 */

/*	soisconnecting(so); */ /* NOFDREF used instead */
	tcpstat.tcps_connattempt++;
	
	tp->t_state = TCPS_SYN_SENT;
	tp->t_timer[TCPT_KEEP] = TCPTV_KEEP_INIT;
	tp->iss = tcp_iss; 
	tcp_iss += TCP_ISSINCR/2;
	tcp_sendseqinit(tp);
	tcp_output(tp);
}

/*
 * Attach a TCPCB to a socket.
 * Note: The caller is responsible of insque()ing the socket
 * to the relevent protocol CB head
 */
int
tcp_attach(so)
	struct socket *so;
{
	if (sbinit(&so->so_rcv, tcp_rcvspace) == NULL)
	 	return -1;
	if (sbinit(&so->so_snd, tcp_sndspace) == NULL) {
		sbfree(&so->so_rcv);
		return -1;
	}
	if ((so->so_tcpcb = tcp_newtcpcb(so)) == NULL) {
		sbfree(&so->so_rcv);
		sbfree(&so->so_snd);
		return -1;
	}
	insque(so, &tcb);

	return 0;
}

/*
 * Set the socket's type of service feild
 * 
 * Any connection that needs emulation, has IPTOS_EMU set in it's
 * tos.. make sure we remove it before puting it in the TOS feild
 * of the IP header
 */
struct tos_t {
	u_int16_t lport;
	u_int16_t fport;
	u_int8_t tos;
};

struct tos_t tcptos[] = {
	{0, 20, IPTOS_THROUGHPUT},		/* ftp data */
	{21, 21, (IPTOS_LOWDELAY|IPTOS_EMU)},	/* ftp control */
	{0, 23, IPTOS_LOWDELAY},			/* telnet */
	{0, 80, IPTOS_THROUGHPUT},			/* WWW */
	{0, 513, IPTOS_LOWDELAY},			/* rlogin */
	{0, 6667, (IPTOS_THROUGHPUT|IPTOS_EMU)},	/* IRC */
	{0, 6668, (IPTOS_THROUGHPUT|IPTOS_EMU)},	/* IRC undernet */
	{0, 0, 0}
};

/*
 * Return TOS according to the above table
 */
u_int8_t
tcp_tos(so)
	struct socket *so;
{
	int i = 0;

#ifdef DEBUG
	debug_call(dfd,"tcp_tos(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif

	while(tcptos[i].tos) {
		if ((tcptos[i].fport && (ntohs(so->so_fport) == tcptos[i].fport)) ||
		    (tcptos[i].lport && (ntohs(so->so_lport) == tcptos[i].lport)))
			return tcptos[i].tos;
		i++;
	}
	return 0;
}

int do_echo = -1;

/*
 * Emulate programs that try and connect to us
 * This includes ftp (the data connection is
 * initiated by the server) and IRC (DCC CHAT and
 * DCC SEND) for now
 * 
 * NOTE: It's possible to crash SLiRP by sending it
 * unstandard strings to emulate... if this is a problem,
 * more checks are needed here
 *
 * XXX Assumes the whole command came in one packet
 *					    
 * XXX Some ftp clients will have their TOS set to
 * LOWDELAY and so Nagel will kick in.  Because of this,
 * we'll get the first letter, followed by the rest, so
 * we simply scan for ORT instead of PORT...
 * DCC doesn't have this problem because there's other stuff
 * in the packet before the DCC command.
 * 
 * Return 1 if the mbuf m is still valid and should be 
 * sbappend()ed
 */
int
tcp_emu(so,m)
	struct socket *so;
	struct mbuf *m;
{
	u_int n1, n2, n3, n4, n5, n6;
	char buff[256];
	u_int32_t laddr;
	u_int lport;
	char *ptr;

#ifdef DEBUG
	debug_call(dfd,"tcp_emu(so,m) called ...\n");
	debug_call(dfd,"    so = %d, m = %d\n", (int)so, (int)m);
	fflush_call(dfd);
#endif
	
	/*
	 * First, check if it's a CTL address so that we
	 * can "execute" the "commands"
	 */
	if (ntohl(so->so_faddr.s_addr) == (ntohl(special_addr)|CTL_CMD)) {
		int num;
		struct sbuf *so_snd = &so->so_snd;
		struct sbuf *so_rcv = &so->so_rcv;
		
		/*
		 * If there is binary data here, we save it in so->so_m
		 */
		num = 0;
		while (num++ < m->m_len) {
			if (*mtod(m, char *) & 0x80) {
				if (so->so_m == 0)
				   so->so_m = m;
				else
				   m_free(m);
				
				return 0;
			}
		}
		
		/*
		 * Append the line
		 */
		sbappendsb(so_rcv, m);
		
		/* To avoid going over the edge of the buffer, we reset it */
		if (so_snd->sb_cc == 0)
			so_snd->sb_wptr = so_snd->sb_rptr = so_snd->sb_data;
		
		/*
		 * A bit of a hack:
		 * If the first packet we get here is 1 byte long, then it
		 * was done in telnet character mode, therefore we must echo
		 * the characters as they come.  Otherwise, we echo nothing,
		 * because in linemode, the line is allready echoed
		 */
		if (do_echo == -1) {
			if (m->m_len == 1)
				do_echo = 1;
			else
				do_echo = 0;
		}
		if (do_echo) {
			sbappendsb(so_snd, m);
			m_free(m);
			tcp_output(sototcpcb(so)); /* XXX */
		} else
			m_free(m);

		num = 0;
		while (num < so->so_rcv.sb_cc) {
			if (*(so->so_rcv.sb_rptr + num) == '\n' ||
			    *(so->so_rcv.sb_rptr + num) == '\r') {
				int n;

				*(so_rcv->sb_rptr + num) = 0;
				n = do_config(so_rcv->sb_rptr, so, PRN_SPRINTF);
				so_snd->sb_cc += n;
				so_snd->sb_wptr += n;
				/* Drop so_rcv data */
				so_rcv->sb_cc = 0;
				so_rcv->sb_wptr = so_rcv->sb_rptr = so_rcv->sb_data;
				tcp_output(sototcpcb(so)); /* Send the reply */
			}
			num++;
		}
		
		return 0;
	}
	
	switch(ntohs(so->so_fport)) {
		int x;

	 case 21: /* ftp */
ftp_emu:
		*(m->m_data+m->m_len) = 0; /* NULL terminate for strstr */
		if ((ptr = (char *)strstr(m->m_data, "ORT")) != NULL) {
#ifdef DEBUG
			debug_misc(dfd,"FTP - %s\n", ptr);
#endif
			/*
			 * Need to emulate the PORT command
			 */			
			x = sscanf(ptr, "ORT %d,%d,%d,%d,%d,%d\r\n%256[^\177]", 
				   &n1, &n2, &n3, &n4, &n5, &n6, buff);
			if (x < 6)
			   return 1;
			
			laddr = htonl((n1 << 24) | (n2 << 16) | (n3 << 8) | (n4));
			lport = htons((n5 << 8) | (n6));
			
			if ((so = solisten(0, laddr, lport, SS_FACCEPTONCE)) == NULL)
			   return 1;
			
			n6 = ntohs(so->so_fport);
			
			n5 = (n6 >> 8) & 0xff;
			n6 &= 0xff;
			
			laddr = ntohl(so->so_faddr.s_addr);
			
			n1 = ((laddr >> 24) & 0xff);
			n2 = ((laddr >> 16) & 0xff);
			n3 = ((laddr >> 8)  & 0xff);
			n4 =  (laddr & 0xff);
			
			m->m_len = ptr - m->m_data; /* Adjust length */
			m->m_len += sprintf(ptr,"ORT %d,%d,%d,%d,%d,%d\r\n%s", 
					    n1, n2, n3, n4, n5, n6, x==7?buff:"");
#ifdef DEBUG
			debug_misc(dfd,"FTP2 - %s\n", ptr);
#endif
			return 1;
		} else if ((ptr = (char *)strstr(m->m_data, "27 Entering")) != NULL) {
			/*
			 * Need to emulate the PASV response
			 */
			x = sscanf(ptr, "27 Entering Passive Mode (%d,%d,%d,%d,%d,%d)\r\n%256[^\177]",
				   &n1, &n2, &n3, &n4, &n5, &n6, buff);
			if (x < 6)
			   return 1;
			
			laddr = htonl((n1 << 24) | (n2 << 16) | (n3 << 8) | (n4));
			lport = htons((n5 << 8) | (n6));
			
			if ((so = solisten(0, laddr, lport, SS_FACCEPTONCE)) == NULL)
			   return 1;
			
			n6 = ntohs(so->so_fport);
			
			n5 = (n6 >> 8) & 0xff;
			n6 &= 0xff;
			
			laddr = ntohl(so->so_faddr.s_addr);
			
			n1 = ((laddr >> 24) & 0xff);
			n2 = ((laddr >> 16) & 0xff);
			n3 = ((laddr >> 8)  & 0xff);
			n4 =  (laddr & 0xff);
			
			m->m_len = ptr - m->m_data; /* Adjust length */
			m->m_len += sprintf(ptr,"27 Entering Passive Mode (%d,%d,%d,%d,%d,%d)\r\n%s",
					    n1, n2, n3, n4, n5, n6, x==7?buff:"");
			
			return 1;
		}
		
		return 1;
				   
	 case 6667:
	 case 6668:
		/*
		 * Need to emulate DCC CHAT and DCC SEND
		 */
		*(m->m_data+m->m_len) = 0; /* NULL terminate the string for strstr */
		if ((ptr = (char *)strstr(m->m_data, "DCC")) == NULL)
			 return 1;
		
		if (sscanf(ptr, "DCC CHAT chat %u %u", &laddr, &lport) == 2) {
			if ((so = solisten(0, htonl(laddr), htons(lport), SS_FACCEPTONCE)) == NULL)
				return 1;
			
			m->m_len = ptr - m->m_data; /* Adjust length */
			m->m_len += sprintf(ptr, "DCC CHAT chat %lu %u%c\n", 
			     ntohl(so->so_faddr.s_addr), ntohs(so->so_fport), 1);

			return 1;
		}
		if (sscanf(ptr, "DCC SEND %256s %u %u %u", buff, &laddr, &lport, &n1) == 4) {
			if ((so = solisten(0, htonl(laddr), htons(lport), SS_FACCEPTONCE)) == NULL)
				return 1;
			
			m->m_len = ptr - m->m_data; /* Adjust length */
			m->m_len += sprintf(ptr, "DCC SEND %s %lu %u %u%c\n", 
			      buff, ntohl(so->so_faddr.s_addr), ntohs(so->so_fport), n1, 1);
			return 1;
		}
		
		return 1;
	}
	
	/* Not fport, must be lport */
	switch(ntohs(so->so_lport)) {
	 case 21:
		goto ftp_emu;
	 default:
		break;
	}
	
	return 1;
}

/*
 * Do misc. config of SLiRP while its running.
 * Return 0 if this connections is to be closed, 1 otherwise,
 * return 2 if this is a command-line connection
 */
int
tcp_ctl(so)
	struct socket *so;
{
	struct sbuf *sb = &so->so_snd;
	int command;
	struct ex_list *ex_ptr;
	int do_pty;
	
#ifdef DEBUG
	debug_call(dfd,"tcp_ctl(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	/*
	 * Check if they're authorised
	 */
	if (ctl_addr && (ctl_addr == -1 || (so->so_laddr.s_addr != ctl_addr))) {
		sb->sb_cc = sprintf(sb->sb_wptr," Error: Permission denied.\n");
		sb->sb_wptr += sb->sb_cc;
		return 0;
	}
	
	command = (ntohl(so->so_faddr.s_addr) & 0xff);
	
	switch(command) {
	 default: /* Check for exec's */
		
		/*
		 * Check if it's pty_exec
		 */
		for (ex_ptr = exec_list; ex_ptr; ex_ptr = ex_ptr->ex_next) {
			if (ex_ptr->ex_fport == so->so_fport &&
			    command == ex_ptr->ex_addr) {
				do_pty = ex_ptr->ex_pty;
				goto do_exec;
			}
		}
		
		/*
		 * Nothing bound.. should never happen, except in a bizarre
		 * race condition (when user removes an "exec" while
		 * this session was connecting)
		 */
		tcp_fconnect(so);
		
		/* FALLTHROUGH */
	 case CTL_ALIAS:
		
		return 1;

do_exec:
		return fork_exec(so, ex_ptr->ex_exec, do_pty);
		
	 case CTL_CMD:
		/*
		 * This is an aweful hack
		 * We set the session IPTOS_EMU so that tcp_emu gets called,
		 * so that we can "execute" the commands given, in tcp_emu.
		 * Yuck! :)
		 */
		so->so_iptos |= IPTOS_EMU;
		sb->sb_cc = sprintf(sb->sb_wptr, " SLiRP command-line ready (type \"help\" for help).\n");
		sb->sb_wptr += sb->sb_cc;
		return 2;

	}
}
