/*
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include <termios.h>
#include "h/common.h"
#include "h/ip.h"
#include "h/tcp.h"
#include "h/sl.h"
#include "h/slcompress.h"
#include "h/main.h"
#include "h/socket.h"

#define CHECK_MSIZE { if_msize--; if (if_msize < 0) sl_outofroom(); }

/*
 * sl_init - initialise the slip data structures
 */
void
sl_init() {

	ifp.ifp_input = sl_input;
	ifp.ifp_output = if_output;
	ifp.ifp_start = if_start;
	ifp.ifp_encap = sl_encap;
	
	if (ifp.ifp_mtu == 0)
	   ifp.ifp_mtu = 552; /* XXX */
	ifp.ifp_maxlinkhdr = 40;
	if (ifp.ifp_flags == 0)
	   ifp.ifp_flags = IF_AUTOCOMP;
	if_inpkt = 0;
	esc = 0;
	if_esc = 0;
	sl_compress_init(comp_s);
	if_fastq.ifq_next = if_fastq.ifq_prev = &if_fastq;
	if_batchq.ifq_next = if_batchq.ifq_prev = &if_batchq;
	next_m = &if_batchq;
} 

void
sl_outofroom() {
	
	/* 
	 * Ran out of room. Scan up to FRAME_END then
	 * dump it.  Tell the user when she exits
	 */
	slirp_errors |= ERR_REDUCEMTU;
	while (if_n) {
		if_n--;
		if (*if_bptr++ == FRAME_END)
		   break;
	}

	m_free(if_m);
	if_inpkt = 0;
}

/* 
 * NOTE: FRAME_END means in_pkt = 0. Any other byte while in_pkt = 0
 * means we're getting a packet now.
 */
void
sl_input()
{
#ifdef ANSI
	u_char if_inbuff[2*2048+2]; /* XXX */
#else
	u_char if_inbuff[2*ifp.ifp_mtu+2];
#endif
	
#ifdef DEBUG
	debug_call(dfd,"sl_input() called ...\n");
	fflush_call(dfd);
	debug_call(dfd,"    if_inpkt = %d\n    if_n = %d\n    if_esc = %d\n    esc = %d\n", 
			if_inpkt, if_n, if_esc, esc);
	fflush_call(dfd);
#endif

	if_n = read(ttyfd, if_inbuff, 2*ifp.ifp_mtu+2);
#ifdef DEBUG
	debug_misc(dfd,"    read %d bytes\n", if_n);
#endif
	if (if_n <= 0) {
		if (if_n < 0 && errno == EINTR)
		   return;
		else
		   slirp_exit(0);
	}

	if_bptr = if_inbuff;
	
	while (if_n--) {
		if (if_esc) {
			if_esc = 0;
			
			/*
			 * If a packet starts with FRAME_ESCAPE we're in trouble 
			 * Shouldn't happen though.
			 */
			if (if_inpkt == 0)
				continue; /* while(if_n) */
			
			switch (*if_bptr) {
			 case TRANS_FRAME_ESCAPE:
				*if_mptr++ = FRAME_ESCAPE;
				break;
			 case TRANS_FRAME_END:
				*if_mptr++ = FRAME_END;
				break;
			 default: /* What to do? */
				*if_mptr++ = *if_bptr;
			}
			if_bptr++;
			CHECK_MSIZE;
			continue;
		}
			
		switch (*if_bptr) {
		 case '0':
			/* 
			 * If this was the only char read, 
			 * and it's a 0, it's an escape char 
			 */
			if (if_n == 0)
				esc++;
			else
				esc = 0; /* Otherwise reset esc */
			if (esc >= 5) /* Quit after 5 escape chars */
				slirp_exit(1);
			goto data;
		 default:
			esc = 0;
data:
			if (if_inpkt == 0) {
				/*
				 * A new packet is arriving, setup mbufs etc.
				 */
				if_inpkt = 1;

				if_m = m_get();
				if_m->m_data += ifp.ifp_maxlinkhdr; /* Allow for uncompress */
				if_mptr = mtod(if_m,u_char *);
				if_msize = M_FREEROOM(if_m);
			}
			*if_mptr++ = *if_bptr++;
			CHECK_MSIZE;
			break;
		 case FRAME_ESCAPE:
			if_esc = 1;
                        if_bptr++;
			break;
		 case FRAME_END:
			if (if_inpkt == 0) {
				if_bptr++;
				break;
			}
			/*
			 * End of packet, fix the mbuf, and dispatch
			 */
			if_m->m_len = (char *)if_mptr - (char *)if_m->m_data;
			
			sl_dispatch();
			
			if_inpkt = 0;
			if_bptr++;
			break;
		}
	}
}


void
sl_dispatch()
{
	u_char c;
	
	if ((c = (u_char)*if_m->m_data & 0xf0) != (IPVERSION << 4))  {
		if (c & 0x80)
		   c = TYPE_COMPRESSED_TCP;
		else if (c == TYPE_UNCOMPRESSED_TCP)
		   *if_m->m_data &= 0x4f; /* XXX */
		
		if (ifp.ifp_flags == IF_COMPRESS)  {
			if_m->m_len = sl_uncompress_tcp((u_char **)&if_m->m_data,
							if_m->m_len,(u_int)c,comp_s);
			if (if_m->m_len > 0)
			   ip_input(if_m);
			else
			   m_free(if_m);
		} else if((ifp.ifp_flags == IF_AUTOCOMP) &&
			  c == TYPE_UNCOMPRESSED_TCP && if_m->m_len >= 40)  {
			if_m->m_len = sl_uncompress_tcp((u_char **)&if_m->m_data,
							if_m->m_len,(u_int)c,comp_s);
			if (if_m->m_len > 0)  {
				ip_input(if_m);
				ifp.ifp_flags = IF_COMPRESS;
			} else
			   m_free(if_m);
		} else
		   m_free(if_m);
	} else
	   ip_input(if_m);
}



/*
 * Copy data from m to inbptr, applying SLIP encapsulation
 * Returns number of bytes in inbptr
 */
int
sl_encap(inbptr, m)
	char *inbptr;
	struct mbuf *m;
{
	u_char *mptr;
	int mlen;
	char *bptr = inbptr;

#ifdef DEBUG
	debug_call(dfd,"sl_encap(m) called ...\n");
	debug_call(dfd,"    inbptr = %d\n    m = %d\n", (int)inbptr, (int)m);
	fflush_call(dfd);
#endif

	mptr = mtod(m,u_char *);
	mlen = m->m_len;
	
	/*
	 * Prepend a FRAME_ESCAPE if no data is
	 * being sent, to flush any line-noise.
	 */
	if (towrite == TOWRITEMAX)
	   *bptr++ = FRAME_END;
	
	while(mlen--) {
		switch (*mptr) {
		 case FRAME_END:
			*bptr++ = FRAME_ESCAPE;
			*bptr++ = TRANS_FRAME_END;
			mptr++;
			break;
		 case FRAME_ESCAPE:
			*bptr++ = FRAME_ESCAPE;
			*bptr++ = TRANS_FRAME_ESCAPE;
			mptr++;
			break;
		 default: /* XXX What to do? */
			*bptr++ = *mptr++;
		}
	}
	*bptr++ = FRAME_END;
	
	return bptr - inbptr;
}
