/* Panel layout module for the Midnight Commander
   Copyright (C) 1995 Janne Kukonlehto

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include <string.h>
#include <stdio.h>
#include <ncurses.h>
#include "mad.h"
#include "win.h"
#include "color.h"
#include "input.h"
#include "dlg.h"
#include "widget.h"
#include "setup.h"	/* For save_setup() */
#include "dialog.h"	/* For do_refresh() */
#include "profile.h"    /* For sync_profiles() */
#include "mouse.h"
#include "main.h"
#include "subshell.h"	/* For use_subshell and resize_subshell() */

/* Needed for the extern declarations of integer parameters */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
    #include <unistd.h>
#endif
#include <signal.h>
#include "dir.h"
#include "panel.h"		/* Needed for the externs */
#include "util.h"		/* Needed for the externs */
#include "file.h"
#include "cons.saver.h"
#include "layout.h"

static char rcsid [] = "$Id: layout.c,v 1.1 1995/01/27 02:35:24 miguel Exp $";

/* If set, then we have to call the layout_change routine from main */
int layout_do_change = 0;
int nice_rotating_dash = 1;

int winch_flag = 0;
int horizontal_split = 0;
int equal_split = 1;
int first_panel_size = 0;
int menubar_visible = 0;
int output_lines = 0;
int command_prompt = 1;
int keybar_visible = 1;

/* These variables are used to avoid updating the information unless */
/* we need it */
static int old_first_panel_size;
static int old_horizontal_split;
static int old_output_lines;

static int _horizontal_split;
static int _equal_split;
static int _first_panel_size;
static int _menubar_visible;
static int _output_lines;
static int _command_prompt;
static int _keybar_visible;

static int height;

#define MINWIDTH 10
#define MINHEIGHT 5

#define BX      12
#define BY      10

#define B_2LEFT B_USER
#define B_2RIGHT B_USER + 1
#define B_PLUS B_USER + 2
#define B_MINUS B_USER + 3

static WINDOW *layout_box;
static Dlg_head *layout_dlg;

static char *s_split_direction [2] = {"Vertical", "Horizontal"};
WRadio *radio_widget;
static chtype layout_colors[4];

static struct {
    char   *text;
    int    *variable;
    WCheck *widget;
    int    hkey;
    int    hpos;
} check_options [] = {
    { "Keybar visible",   &keybar_visible, 0, 'K', -1 },
    { "command Prompt",   &command_prompt, 0, 'P', -1 },
    { "show Mini status", &show_mini_info, 0, 'M', -1 },
    { "menubar Visible",  &menubar_visible, 0, 'V',-1 },
    { "Equal split",      &equal_split, 0, 'E', -1 },
    { 0, 0, 0, 0, 0 }
};

static WButton *bleft_widget, *bright_widget;

static void _check_split (void)
{
    if (_horizontal_split){
	if (_equal_split)
	    _first_panel_size = height / 2;
	else if (_first_panel_size < MINHEIGHT)
	    _first_panel_size = MINHEIGHT;
	else if (_first_panel_size > height - MINHEIGHT)
	    _first_panel_size = height - MINHEIGHT;
    } else {
	if (_equal_split)
	    _first_panel_size = COLS / 2;
	else if (_first_panel_size < MINWIDTH)
	    _first_panel_size = MINWIDTH;
	else if (_first_panel_size > COLS - MINWIDTH)
	    _first_panel_size = COLS - MINWIDTH;
    }
}

static void update_split (void)
{
    /* To avoid setting the cursor to the wrong place */
    if ((old_first_panel_size == _first_panel_size) &&
	(old_horizontal_split == _horizontal_split)){
	return;
    }

    old_first_panel_size = _first_panel_size;
    old_horizontal_split = _horizontal_split;
    
    wattrset (layout_box, REVERSE_COLOR);
    _check_split ();
    mvwprintw (layout_box, 6, 11, "%03d", _first_panel_size);
    if (_horizontal_split)
	mvwprintw (layout_box, 6, 20, "%03d", height - _first_panel_size);
    else
	mvwprintw (layout_box, 6, 20, "%03d", COLS - _first_panel_size);
}

static int b2left_cback (int action, void *data)
{
    if (_equal_split){
	/* Turn equal split off */
	_equal_split = 0;
	check_options [4].widget->state = check_options [4].widget->state & ~C_BOOL;
	dlg_select_widget (layout_dlg, check_options [4].widget);
	dlg_select_widget (layout_dlg, bleft_widget);
    }
    _first_panel_size++;
    return 0;
}

static int b2right_cback (int action, void *data)
{
    if (_equal_split){
	/* Turn equal split off */
	_equal_split = 0;
	check_options [4].widget->state = check_options [4].widget->state & ~C_BOOL;
	dlg_select_widget (layout_dlg, check_options [4].widget);
	dlg_select_widget (layout_dlg, bright_widget);
    }
    _first_panel_size--;
    return 0;
}

static int bplus_cback (int action, void *data)
{
    _output_lines++;
    return 0;
}

static int bminus_cback (int action, void *data)
{
    if (_output_lines > 0)
	_output_lines--;
    return 0;
}

static int layout_callback (struct Dlg_head *h, int Id, int Msg)
{
    switch (Msg){
    case DLG_DRAW:
	wattrset (h->window, REVERSE_COLOR);
	wclr (layout_box);
	draw_box (h->window, 1, 2, 11, 56);
	draw_box (h->window, 2, 5, 6, 24);
	draw_box (h->window, 2, 31, 7, 24);

	wattrset (h->window, COLOR_HOT_NORMAL);
	mvwprintw (h->window, 1, 24, " Layout ");
	mvwprintw (h->window, 2, 6, " Panel split ");
	mvwprintw (h->window, 2, 32, " Other options ");
	update_split ();
	mvwaddstr (h->window, 6, 16, "--");
	if (console_flag){
	    if (old_output_lines != _output_lines){
		old_output_lines = _output_lines;
		mvwprintw (h->window, 7, 34, "%02d", _output_lines);
		mvwaddstr (h->window, 7, 38, "lines of output");
	    }
	}
	break;

    case DLG_POST_KEY:
	_equal_split = check_options [4].widget->state & C_BOOL;
	_menubar_visible = check_options [2].widget->state & C_BOOL;
	_command_prompt = check_options [1].widget->state & C_BOOL;
	_keybar_visible = check_options [0].widget->state & C_BOOL;
	if (console_flag){
	    int minimum;
	    if (_output_lines < 0)
		_output_lines = 0;
	    height = LINES - _keybar_visible - _command_prompt -
		     _menubar_visible - _output_lines;
	    minimum = MINHEIGHT * (1 + _horizontal_split);
	    if (height < minimum){
		_output_lines -= minimum - height;
		height = minimum;
	    }
	} else
	    height = LINES - _keybar_visible - _command_prompt -
		_menubar_visible - _output_lines;
	if (_horizontal_split != radio_widget->sel){
	    _horizontal_split = radio_widget->sel;
	    if (_horizontal_split)
		_first_panel_size = height / 2;
	    else
		_first_panel_size = COLS / 2;
	}
	update_split ();
	if (console_flag){
	    if (old_output_lines != _output_lines){
		old_output_lines = _output_lines;
		mvwprintw (h->window, 7, 34, "%02d", _output_lines);
	    }
	}
	break;

    case DLG_END:
	break;
    }
    return 0;
}

static void init_layout (void)
{
    int i;
#ifdef BUGGY_CURSES             /* See key.c: mi_getch /BUGGY_CURSES/ */
    touchwin (stdscr);
#endif

    layout_colors [0] = COLOR_NORMAL;
    layout_colors [1] = COLOR_FOCUS;
    layout_colors [2] = COLOR_HOT_NORMAL;
    layout_colors [3] = COLOR_HOT_FOCUS;
    layout_box = centerwin (13, 60);
    layout_dlg = dlg_new (layout_box, layout_colors, layout_callback,
			  winpos (13, 60), "[Layout]");

    add_widget (layout_dlg,
		button_new (BY, BX+26, B_CANCEL, "[ Cancel ]",'c',2, 0, 0));
    add_widget (layout_dlg,
		button_new (BY, BX+12, B_EXIT,   "[ Save ]",'s',2, 0, 0));
    add_widget (layout_dlg,
		button_new (BY, BX,    B_ENTER,  "[[ Ok ]]",'o',3, 0, 0));

    if (console_flag){
	add_widget (layout_dlg,
		    button_new (7, 36, B_MINUS, "-", '-', 0, bminus_cback, 0));
	add_widget (layout_dlg,
		    button_new (7, 33, B_PLUS, "+", '+', 0, bplus_cback, 0));
    }

#define XTRACT(i) *check_options[i].variable, check_options[i].text,                             check_options[i].hkey, check_options[i].hpos

    for (i = 0; i < 4; i ++){
	check_options [i].widget = check_new (6 - i, 33, XTRACT(i));
	add_widget (layout_dlg, check_options [i].widget);
    }
    bright_widget = button_new(6, 18, B_2RIGHT, ">", '>', 0, b2right_cback, 0);
    add_widget (layout_dlg, bright_widget);
    bleft_widget = button_new (6, 15, B_2LEFT, "<", '<', 0, b2left_cback, 0);
    add_widget (layout_dlg, bleft_widget);
    check_options [4].widget = check_new (5, 7, XTRACT(4));

    old_first_panel_size = -1;
    old_horizontal_split = -1;
    old_output_lines     = -1;
    
    _first_panel_size = first_panel_size;
    _output_lines = output_lines;
    add_widget (layout_dlg, check_options [4].widget);
    radio_widget = radio_new (3, 7, 2, s_split_direction, 1);
    add_widget (layout_dlg, radio_widget);
    radio_widget->sel = horizontal_split;

    wrefresh (layout_box);
}

void layout_change (void)
{
    destroy_panels ();
    create_panels ();
    change_view (opanel, opanel->view_type);
    change_view (cpanel, cpanel->view_type);
    layout_do_change = 0;
}

void layout_cmd (void)
{
    int result;
    int i;

    init_layout ();
    run_dlg (layout_dlg);
    result = layout_dlg->ret_value;

    if (result == B_ENTER || result == B_EXIT){
	for (i = 0; check_options [i].text; i++)
	    *check_options [i].variable = check_options [i].widget->state & C_BOOL;
	horizontal_split = radio_widget->sel;
	first_panel_size = _first_panel_size;
	output_lines = _output_lines;

	layout_do_change = 1;
    }
    if (result == B_EXIT){
	save_layout ();
	sync_profiles ();
    }

    destroy_dlg (layout_dlg);
    delwin (layout_box);
}

static void check_split (void)
{
    if (horizontal_split){
	if (equal_split)
	    first_panel_size = height / 2;
	else if (first_panel_size < MINHEIGHT)
	    first_panel_size = MINHEIGHT;
	else if (first_panel_size > height - MINHEIGHT)
	    first_panel_size = height - MINHEIGHT;
    } else {
	if (equal_split)
	    first_panel_size = COLS / 2;
	else if (first_panel_size < MINWIDTH)
	    first_panel_size = MINWIDTH;
	else if (first_panel_size > COLS - MINWIDTH)
	    first_panel_size = COLS - MINWIDTH;
    }
}

int panel_event    (Gpm_Event *event, Panel *panel);
int menu_bar_event (Gpm_Event *event, void *);
extern char *prompt;
extern Input *cmdline;
extern WINDOW *menubar_win;
extern WINDOW *output_win;
extern WINDOW *cmdline_win;
extern WINDOW *fkeys;
extern WINDOW *clean_screen;
extern WINDOW *dash_win;

void setup_event (MouseEvent **e, int x1, int y1, int x2, int y2,
		  mouse_h callback, void *data, int options)
{
    *e = push_event (x1, y1, x2, y2, callback, data, options);
}

void init_curses (void)
{
    raw ();
    noecho ();
    keypad (stdscr, TRUE);
    nodelay (stdscr, FALSE);
    init_colors ();
}

void create_panels (void)
{
    static MouseEvent *left_event = 0;
    static MouseEvent *right_event = 0;
    int start_y;

    if (console_flag){
	int minimum;
	if (output_lines < 0)
	    output_lines = 0;
	height = LINES - keybar_visible - command_prompt - menubar_visible
	         - output_lines;
	minimum = MINHEIGHT * (1 + horizontal_split);
	if (height < minimum){
	    output_lines -= minimum - height;
	    height = minimum;
	}
    } else
	height = LINES - menubar_visible - command_prompt - keybar_visible;
    check_split ();
    start_y = menubar_visible;
    clean_screen = newwin (0, 0, 0, 0);
    push_frame (0, 0, 0);
    
    if (horizontal_split){
	init_panel (&left_panel, 0, start_y, COLS, start_y+first_panel_size);
	init_panel (&right_panel, 0, start_y+first_panel_size, COLS,
		    start_y+height);

	/* Panel events */
	setup_event (&left_event, 2, 2+start_y, COLS-1,
		     start_y+first_panel_size, (mouse_h) panel_event,
		     (void *) &left_panel, event_use_frame);
	setup_event (&right_event, 2, 2+start_y+first_panel_size, COLS-1,
		     height+start_y, (mouse_h) panel_event,
		    (void *) &right_panel, event_use_frame);
    } else {
	init_panel (&left_panel, 0, start_y, first_panel_size, height+start_y);
	init_panel (&right_panel, first_panel_size, start_y, COLS,
		    height+start_y);

	/* Panel events */
	setup_event (&left_event, 2, 2+start_y, first_panel_size-2,
		     height+start_y, (mouse_h) panel_event,
		     (void *) &left_panel, event_use_frame);
	setup_event (&right_event, first_panel_size+2, 2+start_y, COLS-1,
		     height+start_y, (mouse_h) panel_event,
		    (void *) &right_panel, event_use_frame);
    }

    /* Menu bar events: */
    /* The first one is the whole line and the others are for individual */
    /* selections */
    push_event (1, 1, COLS, 1, (mouse_h) menu_bar_event, 0, event_use_frame);
    
    /* Menubar window */
    menubar_win = newwin (1, COLS, 0, 0);
    wattrset (menubar_win, SELECTED_COLOR);
    mvwprintw (menubar_win, 0, 0, "%-*.*s", COLS, COLS,
	       "   Left        File        Command     Options     Right  ");
    if (menubar_visible)
	wrefresh (menubar_win);

    /* Output window */
    if (console_flag && output_lines){
	output_win = newwin (output_lines, COLS,
			     LINES - command_prompt - keybar_visible - output_lines, 0);
	show_console_contents (output_win, LINES-output_lines-keybar_visible-1,
			       LINES-keybar_visible-1);
    } else
	output_win = NULL;

    /* Command line */
    cmdline_win = newwin (1, COLS, LINES-1-keybar_visible, 0);
    leaveok (cmdline_win, FALSE);
    if (cmdline)
	cmdline->window = cmdline_win;
    wprintw (cmdline_win, "%s", prompt);
    if (command_prompt)
	wrefresh (cmdline_win);

    /* Function key labels */
    init_labels ();

    /* The rotating dash */
    dash_win = newwin (1, 1, 0, COLS-1);
    leaveok (dash_win, TRUE);
}

static void destroy_panel (Panel *panel)
{
    delwin (panel->small_frame);
    delwin (panel->big_frame);
}

void destroy_panels (void)
{
    pop_frame ();
    delwin (dash_win);
    pop_fkey (fkeys);
    delwin (menubar_win);
    delwin (cmdline_win);
    if (output_win)
	delwin (output_win);

    destroy_panel (&left_panel);
    destroy_panel (&right_panel);
    delwin (clean_screen);
}

void flag_winch (int dummy)
{
#ifdef SIGWINCH
    winch_flag = 1;
    signal (SIGWINCH, flag_winch);
#endif
}

void change_screen_size (void)
{
#ifdef TIOCGWINSZ
    struct winsize winsz;

    destroy_panels ();
    noraw ();
    endwin ();
    winsz.ws_col = winsz.ws_row = 0;
    ioctl (STDIN_FILENO, TIOCGWINSZ, &winsz);
    if (winsz.ws_col && winsz.ws_row){
	COLS = winsz.ws_col;
	LINES = winsz.ws_row;
	#ifdef HAVE_SUBSHELL_SUPPORT
	    resize_subshell ();
	#endif
    }

    check_split ();
    initscr ();
    init_curses ();
    create_panels ();
    change_view (opanel, opanel->view_type);
    change_view (cpanel, cpanel->view_type);
#endif
    winch_flag = 0;
}

WINDOW *dash_win = NULL;
extern int verbose;

void rotate_dash (void)
{
    static char rotating_dash [] = "|/-\\";
    static int pos = 0;

    if (!nice_rotating_dash)
	return;
    
    if (!verbose)
	return;
    if (pos >= sizeof (rotating_dash)-1)
	pos = 0;
    mvwaddch (dash_win, 0, 0, rotating_dash [pos]);
    wrefresh (dash_win);
    pos++;
}

void remove_dash (void)
{
    if (!nice_rotating_dash)
	return;
    if (menubar_visible)
	wrefresh (menubar_win);
    else if (cpanel != &right_panel)
	wrefresh (opanel->win_file);
}
