/* Windows utilities.
   Copyright (C) 1994 Miguel de Icaza, Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdlib.h>	/* For free() */

#include "mad.h"
#include "win.h"
#include "color.h"
#include "mouse.h"
#include "input.h"
#include "util.h"	/* For xmalloc() */
#include "layout.h"

static char rcsid [] = "$Id: win.c,v 1.9 1995/01/27 02:36:55 miguel Exp $";

typedef void (*fnptr)(void);

typedef struct Fkey_Table_List {
    fnptr  actions[11];
    struct Fkey_Table_List *next;
    int    has_labels;
} Fkey_Table_List;

static Fkey_Table_List *fkey_table_list = NULL;

/* Clears WINDOW *w, and leaves untouched skip spaces */
void wclrn (WINDOW *w, int skip)
{
    int x, y;
    
    for (x = skip; x <= (w->_maxx-skip); x++){
	for (y = skip; y <= (w->_maxy-skip); y++){
	    mvwaddch (w, y, x, ' ');
	}
    }
}

/* Width of output is always seven characters */
void wprint_bytesize (WINDOW *w, int size, int scale)
{
    char scales[] = " kMGT";

    if (size > 0){
	while (size > 9999 && scale < sizeof (scales)){
	    size = (size + 512) / 1024;
	    scale ++;
	}
    }
    if (scale > 0)
	wprintw (w, "%4d %cB", size, scales[scale]);
    else
	wprintw (w, "%4d B ", size);
}

void set_label_text (WINDOW *fkeys, int index, char *text)
{
    if (!fkey_table_list->has_labels)
	return;
    
    /* Check if there's room for this label */
    if (index * 8 > COLS)
	return;
    /* Print label */
    wmove (fkeys, 0, 8*(index-1));
    wprintw (fkeys, "%d", index);
    wattron (fkeys, SELECTED_COLOR);
    if (COLS == index * 8)
	wprintw (fkeys, "%-5s", text);
    else
	wprintw (fkeys, "%-6s", text);
    wstandend (fkeys);
    if (keybar_visible)
	wrefresh (fkeys);
}

void nothing (void)
{
    /* Doesn't do anything */
}

int nothing_int (void)
{
    /* Doesn't do anything */
    return 0;
}

void define_label (WINDOW *fkeys, int index, char *text, void (*cback)(void))
{
    set_label_text (fkeys, index, text);
    fkey_table_list->actions[index] = cback;

    if (!fkey_table_list->has_labels)
	return;

    if (use_mouse_p && cback)
	push_event ((index-1)*8, LINES, (index*8)-1, LINES, click,
		    (void*)cback, event_absolute);
}

void define_label_quit (WINDOW *fkeys, int index, char *text, int (*cback)(void))
{
    fkey_table_list->actions[index] = (fnptr)cback;
    if (!fkey_table_list->has_labels)
	cback = nothing_int;
    
    set_label_text (fkeys, index, text);

    if (use_mouse_p && cback){
	push_event ((index-1)*8, LINES, (index*8)-1, LINES,
		    click_may_break_loop, (void *) cback, event_absolute);
    }
}

/* Return values: 0 = not a fkey, other = was a fkey */
int check_fkeys (int c)
{
    int fkey;

    if (!fkey_table_list)
	return 0;
    
    switch (c){
    case KEY_F(1):
	fkey = 1;
	break;
    case KEY_F(2):
	fkey = 2;
	break;
    case KEY_F(3):
	fkey = 3;
	break;
    case KEY_F(4):
	fkey = 4;
	break;
    case KEY_F(5):
	fkey = 5;
	break;
    case KEY_F(6):
	fkey = 6;
	break;
    case KEY_F(7):
	fkey = 7;
	break;
    case KEY_F(8):
	fkey = 8;
	break;
    case KEY_F(9):
	fkey = 9;
	break;
    case KEY_F(10):
	fkey = 10;
	break;
    default:
	return 0;
    }
    if (fkey_table_list->actions [fkey]){
	fkey_table_list->actions [fkey] ();
	return fkey;
    }
    else
	return 0;
}

WINDOW *push_fkey (int has_labels)
{
    WINDOW *w = 0;
    Fkey_Table_List *new;
    int i;

    new = (Fkey_Table_List*)xmalloc (sizeof (Fkey_Table_List),
				     "push_fkey, Fkey_Table_List");
    new->next = fkey_table_list;
    new->has_labels = has_labels;
    
    fkey_table_list = new;
    for (i = 0; i < 11; i++)
	fkey_table_list->actions [i] = NULL;

    if (has_labels){
	w = newwin (1, COLS, LINES-1, 0);
	leaveok (w, TRUE);
	push_event (0, LINES, COLS, LINES, null_event, 0, event_absolute);
    }
    return w;
}

void pop_fkey (WINDOW *w)
{
    Fkey_Table_List *old;

    old = fkey_table_list;
    fkey_table_list = fkey_table_list->next;
    free (old);

    pop_event ();
    if (w)
	delwin (w);
}

/* Return values: 0 = not a movement key, 1 = was a movement key */
int check_movement_keys (int c, int additional, int page_size,
			 movefn backfn, movefn forfn, movefn topfn,
			 movefn bottomfn)
{
    switch (c){
    case KEY_UP:
    case XCTRL ('p'):
	backfn (1);
	return 1;
    case KEY_DOWN:
    case XCTRL ('n'):
	forfn (1);
	return 1;
    case KEY_PPAGE:
    case ALT('v'):
	backfn (page_size-1);
	return 1;
    case KEY_NPAGE:
    case XCTRL('v'):
	forfn (page_size-1);
	return 1;
    case KEY_HOME:
    case KEY_A1:
	topfn (0);
	return 1;
    case KEY_END:
    case KEY_C1:
	bottomfn (0);
	return 1;
    }
    if (additional)
        switch (c){
	case 'b':  
	case XCTRL('b'):
	case XCTRL('h'):
	case KEY_BACKSPACE:
	case 0177:
	    backfn (page_size-1);
	    return 1;
	case ' ':
	    forfn (page_size-1);
	    return 1;
	case 'u':
	    backfn (page_size / 2);
	    return 1;
	case 'd':
	    forfn (page_size / 2);
	    return 1;
	case 'g':
	    topfn (0);
	    return 1;
	case 'G':
	    bottomfn (0);
	    return 1;
	}
    return 0;
}
