------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                A T R E E                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.118 $                            --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  WARNING: There is a C version of this package. Any changes to this source
--  file must be properly reflected in the C header a-atree.h (for inlined
--  bodies) and the C file a-atree.c (for remaining non-inlined bodies).

with Debug;   use Debug;
with Nlists;  use Nlists;
with Elists;  use Elists;
with Output;  use Output;
with Sinput;  use Sinput;
with Tree_IO; use Tree_IO;
with Unchecked_Conversion;

package body Atree is

   Node_Count : Nat;
   --  Count allocated nodes for Num_Nodes function

   use Unchecked_Access;
   --  We are allowed to see these from within our own body!

   use Atree_Private_Part;
   --  We are also allowed to see our private data structures!

   function E_To_N is new Unchecked_Conversion (Entity_Kind, Node_Kind);
   function N_To_E is new Unchecked_Conversion (Node_Kind, Entity_Kind);
   --  Functions used to store Entity_Kind value in Nkind field

   --  The following declarations are used to store flags 65-72 in the
   --  Nkind field of the third component of an extended (entity) node.

   type Flag_Byte is record
      Flag65 : Boolean;
      Flag66 : Boolean;
      Flag67 : Boolean;
      Flag68 : Boolean;
      Flag69 : Boolean;
      Flag70 : Boolean;
      Flag71 : Boolean;
      Flag72 : Boolean;
   end record;

   pragma Pack (Flag_Byte);
   for Flag_Byte'Size use 8;

   type Flag_Byte_Ptr is access all Flag_Byte;
   type Node_Kind_Ptr is access all Node_Kind;

   function To_Flag_Byte is new
     Unchecked_Conversion (Node_Kind, Flag_Byte);

   function To_Flag_Byte_Ptr is new
     Unchecked_Conversion (Node_Kind_Ptr, Flag_Byte_Ptr);

   --  The following declarations are used to store flags 73-96 in the
   --  Field12 field of the third component of an extended (entity) node.
   --  Note that the record contains space for 8 extra flags, but we have
   --  not made these available yet, because there seems no pressing need.
   --  need.

   type Flag_Word is record
      Flag73 : Boolean;
      Flag74 : Boolean;
      Flag75 : Boolean;
      Flag76 : Boolean;
      Flag77 : Boolean;
      Flag78 : Boolean;
      Flag79 : Boolean;
      Flag80 : Boolean;

      Flag81 : Boolean;
      Flag82 : Boolean;
      Flag83 : Boolean;
      Flag84 : Boolean;
      Flag85 : Boolean;
      Flag86 : Boolean;
      Flag87 : Boolean;
      Flag88 : Boolean;

      Flag89 : Boolean;
      Flag90 : Boolean;
      Flag91 : Boolean;
      Flag92 : Boolean;
      Flag93 : Boolean;
      Flag94 : Boolean;
      Flag95 : Boolean;
      Flag96 : Boolean;

      Flag97 : Boolean;
      Flag98 : Boolean;
      Flag99 : Boolean;

      Convention : Convention_Id;
   end record;

   pragma Pack (Flag_Word);
   for Flag_Word'Size use 32;

   type Flag_Word_Ptr is access all Flag_Word;
   type Union_Id_Ptr  is access all Union_Id;

   function To_Flag_Word is new
     Unchecked_Conversion (Union_Id, Flag_Word);

   function To_Flag_Word_Ptr is new
     Unchecked_Conversion (Union_Id_Ptr, Flag_Word_Ptr);

   --  Default value used to initialize default nodes. Note that some of the
   --  fields get overwritten, and in particular, Nkind always gets reset.

   Default_Node : Node_Record := (
      Is_Extension      => False,
      Pflag1            => False,
      Pflag2            => False,
      In_List           => False,
      Rewrite_Sub       => False,
      Rewrite_Ins       => False,
      Flag1             => False,
      Comes_From_Source => False, -- modified by Set_Comes_From_Source_Default
      Flag3             => False,
      Flag4             => False,

      Flag5             => False,
      Flag6             => False,
      Flag7             => False,
      Flag8             => False,
      Flag9             => False,
      Flag10            => False,
      Flag11            => False,
      Flag12            => False,

      Flag13            => False,
      Flag14            => False,
      Flag15            => False,
      Flag16            => False,
      Flag17            => False,
      Flag18            => False,

      Nkind             => N_Unused_At_Start,

      Sloc              => No_Location,
      Link              => Empty_List_Or_Node,
      Field1            => Empty_List_Or_Node,
      Field2            => Empty_List_Or_Node,
      Field3            => Empty_List_Or_Node,
      Field4            => Empty_List_Or_Node,
      Field5            => Empty_List_Or_Node);

   --  Default value used to initialize node extensions (i.e. the second
   --  and third components of an extended node)

   Default_Node_Extension : constant Node_Record := (
      Is_Extension      => True,
      Pflag1            => False,
      Pflag2            => False,
      In_List           => False,
      Rewrite_Sub       => False,
      Rewrite_Ins       => False,
      Flag1             => False,
      Comes_From_Source => False,
      Flag3             => False,
      Flag4             => False,

      Flag5             => False,
      Flag6             => False,
      Flag7             => False,
      Flag8             => False,
      Flag9             => False,
      Flag10            => False,
      Flag11            => False,
      Flag12            => False,

      Flag13            => False,
      Flag14            => False,
      Flag15            => False,
      Flag16            => False,
      Flag17            => False,
      Flag18            => False,

      Nkind             => E_To_N (E_Void),

      Field6            => Empty_List_Or_Node,
      Field7            => Empty_List_Or_Node,
      Field8            => Empty_List_Or_Node,
      Field9            => Empty_List_Or_Node,
      Field10           => Empty_List_Or_Node,
      Field11           => Empty_List_Or_Node,
      Field12           => Empty_List_Or_Node);

   --------------------------------------------------
   -- Implementation of Tree Substitution Routines --
   --------------------------------------------------

   --  A hash table is used to keep track of the mapping between substituted
   --  nodes and their corresponding original tree nodes. A call to the
   --  Rewrite_Substitute_Tree procedure makes an entry in this table so
   --  that Original_Tree can obtain the original tree before rewriting.

   type Hash_Entry_Id is new Nat;
   --  Id of hash table entry is its subscript in the hash table

   No_Entry : constant Hash_Entry_Id := 0;
   --  Value used to mark no chain, or end of chain

   type Hash_Entry is record
      Next     : Hash_Entry_Id;
      Old_Node : Node_Id;
      New_Node : Node_Id;
   end record;

   Num_Hash_Headers : constant := 512;
   --  Number of headers in Hash_Headers array

   Hash_Headers : array (Nat range 0 .. Num_Hash_Headers - 1) of Hash_Entry_Id;
   --  Table of hash headers, each entry points to chain of entries in the
   --  hash table whose hash value matches the subscript in the header table.
   --  The hash code is simply the Node_Id value mod Num_Hash_Headers.

   package Hash_Sub is new Table (
      Table_Component_Type => Hash_Entry,
      Table_Index_Type     => Hash_Entry_Id,
      Table_Low_Bound      => 1,
      Table_Initial        => Alloc_Hash_Sub_Initial,
      Table_Increment      => Alloc_Hash_Sub_Increment,
      Table_Name           => "Hash_Sub");

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Fix_Parent (Field : Union_Id; Old_Node, New_Node : Node_Id);
   --  This subprogram is used to fixup parent pointers that are rendered
   --  incorrect because of a node copy. Field is checked to see if it
   --  points to a node, list, or element list that has a parent that
   --  points to Old_Node. If so, the parent is reset to point to New_Node.

   -----------------
   -- Change_Node --
   -----------------

   procedure Change_Node (N : Node_Id; New_Node_Kind : Node_Kind) is
      Save_Sloc    : Source_Ptr := Sloc (N);
      Save_In_List : Boolean    := Nodes.Table (N).In_List;
      Save_Link    : Union_Id   := Nodes.Table (N).Link;

   begin
      Nodes.Table (N)         := Default_Node;
      Nodes.Table (N).Sloc    := Save_Sloc;
      Nodes.Table (N).In_List := Save_In_List;
      Nodes.Table (N).Link    := Save_Link;
      Nodes.Table (N).Nkind   := New_Node_Kind;
   end Change_Node;

   -----------------------
   -- Comes_From_Source --
   -----------------------

   function Comes_From_Source (N : Node_Id) return Boolean is
   begin
      pragma Assert (N in Nodes.First .. Nodes.Last);
      return Nodes.Table (N).Comes_From_Source;
   end Comes_From_Source;

   ----------------
   -- Convention --
   ----------------

   function Convention  (N : Node_Id) return Convention_Id is
   begin
      pragma Assert (Nkind (N) in N_Entity);
      return To_Flag_Word (Nodes.Table (N + 2).Field12).Convention;
   end Convention;

   ---------------
   -- Copy_Node --
   ---------------

   procedure Copy_Node (Source : Node_Id; Destination : Node_Id) is
      Save_In_List : constant Boolean  := Nodes.Table (Destination).In_List;
      Save_Link    : constant Union_Id := Nodes.Table (Destination).Link;

   begin
      Nodes.Table (Destination)         := Nodes.Table (Source);
      Nodes.Table (Destination).In_List := Save_In_List;
      Nodes.Table (Destination).Link    := Save_Link;

      if Has_Extension (Source) then
         pragma Assert (Has_Extension (Destination));
         Nodes.Table (Destination + 1) := Nodes.Table (Source + 1);
         Nodes.Table (Destination + 2) := Nodes.Table (Source + 2);

      else
         pragma Assert (not Has_Extension (Source));
         null;
      end if;
   end Copy_Node;

   ------------------------
   -- Copy_Original_Tree --
   ------------------------

   function Copy_Original_Tree (Source : Node_Id) return Node_Id is
      New_Id  : Node_Id;

      function Copy_Entity (E : Entity_Id) return Entity_Id;
      --  Copy Entity, copying only the Ekind and Chars fields

      function Copy_List (List : List_Id) return List_Id;
      --  Copy list

      function Possible_Copy (Field : Union_Id) return Union_Id;
      --  Given a field, returns a copy of the node or list if its parent
      --  is the current source node, and otherwise returns the input

      function Copy_Entity (E : Entity_Id) return Entity_Id is
         New_Ent : Entity_Id;

      begin
         if Nkind (E) = N_Defining_Identifier then
            New_Ent := New_Entity (N_Defining_Identifier, Sloc (E));

         elsif Nkind (E) = N_Defining_Character_Literal then
            New_Ent := New_Entity (N_Defining_Character_Literal, Sloc (E));

         elsif Nkind (E) = N_Defining_Operator_Symbol then
            New_Ent := New_Entity (N_Defining_Operator_Symbol, Sloc (E));
         end if;

         Set_Chars (New_Ent, Chars (E));
         return New_Ent;
      end Copy_Entity;

      function Copy_List (List : List_Id) return List_Id is
         NL : List_Id;
         E  : Node_Id;

      begin
         if List = No_List then
            return No_List;

         else
            NL := New_List;
            E := First (List);

            while Present (E) loop

               if Has_Extension (E) then
                  Append (Copy_Entity (E), NL);
               else
                  Append (Copy_Original_Tree (E), NL);
               end if;

               E := Next (E);
            end loop;

            return NL;
         end if;

      end Copy_List;

      function Possible_Copy (Field : Union_Id) return Union_Id is
         New_N : Union_Id;

      begin
         if Field in Node_Range then

            New_N :=  Union_Id (Copy_Original_Tree (Node_Id (Field)));
            Set_Parent (Node_Id (New_N), Node_Id (New_Id));

            return New_N;

         elsif Field in List_Range then

            New_N := Union_Id (Copy_List (List_Id (Field)));
            Set_Parent (List_Id (New_N), Node_Id (New_Id));

            return New_N;

         else
            return Field;
         end if;
      end Possible_Copy;

   --  Start of processing for Copy_Original_Tree

   begin
      if Source <= Empty_Or_Error then
         return Source;

      elsif Has_Extension (Source) then
         return Copy_Entity (Source);

      elsif Is_Rewrite_Substitution (Source) then
         return Copy_Original_Tree (Original_Node (Source));

      else
         Nodes.Increment_Last;
         New_Id := Nodes.Last;
         Nodes.Table (New_Id) := Nodes.Table (Source);
         Nodes.Table (New_Id).Link := Empty_List_Or_Node;
         Nodes.Table (New_Id).In_List := False;
         Node_Count := Node_Count + 1;

         --  Recursively copy descendents

         Set_Field1 (New_Id, Possible_Copy (Field1 (New_Id)));
         Set_Field2 (New_Id, Possible_Copy (Field2 (New_Id)));
         Set_Field3 (New_Id, Possible_Copy (Field3 (New_Id)));
         Set_Field4 (New_Id, Possible_Copy (Field4 (New_Id)));
         Set_Field5 (New_Id, Possible_Copy (Field5 (New_Id)));

         --  All done, return copied node

         return New_Id;
      end if;
   end Copy_Original_Tree;

   ------------------------
   -- Copy_Separate_Tree --
   ------------------------

   function Copy_Separate_Tree (Source : Node_Id) return Node_Id is
      New_Id  : Node_Id;

      function Copy_Entity (E : Entity_Id) return Entity_Id;
      --  Copy Entity, copying only the Ekind and Chars fields

      function Copy_List (List : List_Id) return List_Id;
      --  Copy list

      function Possible_Copy (Field : Union_Id) return Union_Id;
      --  Given a field, returns a copy of the node or list if its parent
      --  is the current source node, and otherwise returns the input

      function Copy_Entity (E : Entity_Id) return Entity_Id is
         New_Ent : Entity_Id;

      begin
         if Nkind (E) = N_Defining_Identifier then
            New_Ent := New_Entity (N_Defining_Identifier, Sloc (E));

         elsif Nkind (E) = N_Defining_Character_Literal then
            New_Ent := New_Entity (N_Defining_Character_Literal, Sloc (E));

         elsif Nkind (E) = N_Defining_Operator_Symbol then
            New_Ent := New_Entity (N_Defining_Operator_Symbol, Sloc (E));
         end if;

         Set_Chars (New_Ent, Chars (E));
         return New_Ent;
      end Copy_Entity;

      function Copy_List (List : List_Id) return List_Id is
         NL : List_Id;
         E  : Node_Id;

      begin
         if List = No_List then
            return No_List;

         else
            NL := New_List;
            E := First (List);

            while Present (E) loop

               if Has_Extension (E) then
                  Append (Copy_Entity (E), NL);
               else
                  Append (Copy_Separate_Tree (E), NL);
               end if;

               E := Next (E);
            end loop;

            return NL;
         end if;

      end Copy_List;

      function Possible_Copy (Field : Union_Id) return Union_Id is
         New_N : Union_Id;

      begin
         if Field in Node_Range then

            New_N :=  Union_Id (Copy_Separate_Tree (Node_Id (Field)));

            if Parent (Node_Id (Field)) = Source then
               Set_Parent (Node_Id (New_N), Node_Id (New_Id));
            end if;

            return New_N;

         elsif Field in List_Range then
            New_N := Union_Id (Copy_List (List_Id (Field)));

            if Parent (List_Id (Field)) = Source then
               Set_Parent (List_Id (New_N), Node_Id (New_Id));
            end if;

            return New_N;

         else
            return Field;
         end if;
      end Possible_Copy;

   --  Start of processing for Copy_Separate_Tree

   begin
      if Source <= Empty_Or_Error then
         return Source;

      elsif Has_Extension (Source) then
         return Copy_Entity (Source);

      else
         Nodes.Increment_Last;
         New_Id := Nodes.Last;
         Nodes.Table (New_Id) := Nodes.Table (Source);
         Nodes.Table (New_Id).Link := Empty_List_Or_Node;
         Nodes.Table (New_Id).In_List := False;
         Nodes.Table (New_Id).Rewrite_Sub := False;
         Node_Count := Node_Count + 1;

         --  Recursively copy descendents

         Set_Field1 (New_Id, Possible_Copy (Field1 (New_Id)));
         Set_Field2 (New_Id, Possible_Copy (Field2 (New_Id)));
         Set_Field3 (New_Id, Possible_Copy (Field3 (New_Id)));
         Set_Field4 (New_Id, Possible_Copy (Field4 (New_Id)));
         Set_Field5 (New_Id, Possible_Copy (Field5 (New_Id)));

         --  All done, return copied node

         return New_Id;
      end if;
   end Copy_Separate_Tree;

   -----------------
   -- Delete_Node --
   -----------------

   procedure Delete_Node (Node : Node_Id) is
   begin
      pragma Assert (not Nodes.Table (Node).In_List);

      if Debug_Flag_N then
         Write_Str ("Delete node ");
         Write_Int (Int (Node));
         Write_Eol;
      end if;

      Nodes.Table (Node)       := Default_Node;
      Nodes.Table (Node).Nkind := N_Unused_At_Start;
      Node_Count := Node_Count - 1;

      --  Note: for now, we are not bothering to reuse deleted nodes

   end Delete_Node;

   -----------
   -- Ekind --
   -----------

   function Ekind (N : Node_Id) return Entity_Kind is
   begin
      pragma Assert (Nkind (N) in N_Entity);
      return N_To_E (Nodes.Table (N + 1).Nkind);
   end Ekind;

   -----------------------
   -- Exchange_Entities --
   -----------------------

   procedure Exchange_Entities (E1 : Entity_Id; E2 : Entity_Id) is
      Temp_Ent : Node_Record;

   begin
      pragma Assert (Has_Extension (E1)
        and then Has_Extension (E2)
        and then not Nodes.Table (E1).In_List
        and then not Nodes.Table (E2).In_List);

      --  Exchange the contents of the two entities

      Temp_Ent := Nodes.Table (E1);
      Nodes.Table (E1) := Nodes.Table (E2);
      Nodes.Table (E2) := Temp_Ent;
      Temp_Ent := Nodes.Table (E1 + 1);
      Nodes.Table (E1 + 1) := Nodes.Table (E2 + 1);
      Nodes.Table (E2 + 1) := Temp_Ent;
      Temp_Ent := Nodes.Table (E1 + 2);
      Nodes.Table (E1 + 2) := Nodes.Table (E2 + 2);
      Nodes.Table (E2 + 2) := Temp_Ent;

      --  That exchange exchanged the parent pointers as well, which is what
      --  we want, but we need to patch up the defining identifier pointers
      --  in the parent nodes (the child pointers) to match this switch
      --  unless for Implicit types entities which have no parent

      if Present (Parent (E1)) then
         Set_Defining_Identifier (Parent (E1), E1);
      end if;

      if Present (Parent (E2)) then
         Set_Defining_Identifier (Parent (E2), E2);
      end if;
   end Exchange_Entities;

   -----------------
   -- Extend_Node --
   -----------------

   function Extend_Node (Node : Node_Id) return Entity_Id is
      Result : Entity_Id;

      procedure Debug_Extend_Node;
      --  Debug routine for debug flag N

      procedure Debug_Extend_Node is
      begin
         if Debug_Flag_N then
            Write_Str ("Extend node ");
            Write_Int (Int (Node));

            if Result = Node then
               Write_Str (" in place");
            else
               Write_Str (" copied to ");
               Write_Int (Int (Result));
            end if;
         end if;
      end Debug_Extend_Node;

      pragma Inline (Debug_Extend_Node);

   begin
      if Node /= Nodes.Last then
         Nodes.Increment_Last;
         Nodes.Table (Nodes.Last) := Nodes.Table (Node);
         Result := Nodes.Last;
      else
         Result := Node;
      end if;

      Nodes.Increment_Last;
      Nodes.Table (Nodes.Last) := Default_Node_Extension;
      Nodes.Increment_Last;
      Nodes.Table (Nodes.Last) := Default_Node_Extension;
      pragma Debug (Debug_Extend_Node);
      return Result;
   end Extend_Node;

   ----------------
   -- Fix_Parent --
   ----------------

   procedure Fix_Parent (Field : Union_Id; Old_Node, New_Node : Node_Id) is
   begin
      --  Fix parent of node that is referenced by Field. Note that we must
      --  exclude the case where the node is a member of a list, because in
      --  this case the parent is the parent of the list.

      if Field in Node_Range
        and then Present (Node_Id (Field))
        and then not Nodes.Table (Node_Id (Field)).In_List
        and then Parent (Node_Id (Field)) = Old_Node
      then
         Set_Parent (Node_Id (Field), New_Node);

      --  Fix parent of list that is referenced by Field

      elsif Field in List_Range
        and then Present (List_Id (Field))
        and then Parent (List_Id (Field)) = Old_Node
      then
         Set_Parent (List_Id (Field), New_Node);
      end if;

   end Fix_Parent;

   -------------------
   -- Has_Extension --
   -------------------

   function Has_Extension (N : Node_Id) return Boolean is
   begin
      return N < Nodes.Last and then Nodes.Table (N + 1).Is_Extension;
   end Has_Extension;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
      Dummy : Node_Id;

   begin
      Node_Count := 0;
      Nodes.Init;
      Hash_Sub.Init;

      for J in Hash_Headers'Range loop
         Hash_Headers (J) := No_Entry;
      end loop;

      --  Allocate Empty and Error nodes

      Dummy := New_Node (N_Empty, No_Location);
      Set_Name1 (Empty, No_Name);
      Dummy := New_Node (N_Error, No_Location);
      Set_Name1 (Error, Error_Name);

   end Initialize;

   --------------------------
   -- Is_Rewrite_Insertion --
   --------------------------

   function Is_Rewrite_Insertion (Node : Node_Id) return Boolean is
   begin
      return Nodes.Table (Node).Rewrite_Ins;
   end Is_Rewrite_Insertion;

   -----------------------------
   -- Is_Rewrite_Substitution --
   -----------------------------

   function Is_Rewrite_Substitution (Node : Node_Id) return Boolean is
   begin
      return Nodes.Table (Node).Rewrite_Sub;
   end Is_Rewrite_Substitution;

   ------------------
   -- Last_Node_Id --
   ------------------

   function Last_Node_Id return Node_Id is
   begin
      return Nodes.Last;
   end Last_Node_Id;

   ----------------------------
   -- Mark_Rewrite_Insertion --
   ----------------------------

   procedure Mark_Rewrite_Insertion (New_Node : Node_Id) is
   begin
      Nodes.Table (New_Node).Rewrite_Ins := True;
   end Mark_Rewrite_Insertion;

   --------------
   -- New_Copy --
   --------------

   function New_Copy (Source : Node_Id) return Node_Id is
      New_Id : Node_Id;

   begin
      if Source <= Empty_Or_Error then
         return Source;

      else
         Nodes.Increment_Last;
         New_Id := Nodes.Last;
         Nodes.Table (New_Id) := Nodes.Table (Source);
         Nodes.Table (New_Id).Link := Empty_List_Or_Node;
         Nodes.Table (New_Id).In_List := False;

         if Has_Extension (Source) then
            Nodes.Increment_Last;
            Nodes.Table (New_Id + 1) := Nodes.Table (Source + 1);
            Nodes.Increment_Last;
            Nodes.Table (New_Id + 2) := Nodes.Table (Source + 2);
         end if;

         if Is_Rewrite_Substitution (Source) then
            Nodes.Table (New_Id).Rewrite_Sub := False;
         end if;

         Node_Count := Node_Count + 1;
         return New_Id;
      end if;
   end New_Copy;

   -------------------
   -- New_Copy_Tree --
   -------------------

   function New_Copy_Tree (Source : Node_Id) return Node_Id is
   begin
      return New_Copy_With_Replacement (Source, New_Elmt_List);
   end New_Copy_Tree;

   -------------------------------
   -- New_Copy_With_Replacement --
   -------------------------------

   function New_Copy_With_Replacement (Source : Node_Id; Map : Elist_Id)
   return Node_Id is
      New_Id : Node_Id;
      O_Ityp : Node_Id;
      N_Ityp : Node_Id;

      function Possible_Copy (Field : Union_Id) return Union_Id;
      --  Given a field, returns a copy of the node or list if its parent
      --  is the current source node, and otherwise returns the input

      procedure Add_Assoc  (Old_N : Node_Id; New_N : Node_Id);
      --  Add one association

      function Assoc (N : Node_Id) return Node_Id;
      --  Replace any occurrence of a component, discriminant, or Itype
      --  in the original declaration,  with the corresponding entity in
      --  the new declaration.

      function Copy_Elist_With_Replacement (E : Elist_Id) return Elist_Id;
      --  Copy element list. Used for discriminant constraints of Itypes.

      function Copy_List_With_Replacement (List : List_Id) return List_Id;
      --  Copy list doing replacement

      function Assoc (N : Node_Id) return Node_Id is
         E : Elmt_Id := First_Elmt (Map);

      begin
         if not Has_Extension (N) then
            return N;

         elsif Ekind (N) /= E_Component
           and then Ekind (N) /= E_Discriminant
           and then Ekind (N) not in Record_Kind
           and then Ekind (N) not in Private_Kind
           and then Comes_From_Source (N)
         then
            return N;

         else
            while Present (E) loop
               if Node (E) = N then
                  return Node (Next_Elmt (E));
               else
                  E := Next_Elmt (Next_Elmt (E));
               end if;
            end loop;
         end if;

         return N;
      end Assoc;

      procedure Add_Assoc  (Old_N : Node_Id; New_N : Node_Id) is
      begin
         Append_Elmt (Old_N, Map);
         Append_Elmt (New_N, Map);
      end Add_Assoc;

      function Copy_Itype (Itype : Entity_Id) return Entity_Id;
      --  The constraints on Itypes may depend on discriminants that have
      --  to be replaced as well. Itypes have to parent node,  so they have
      --  to be treated specially. The itypes of interest are discrete
      --  subtypes, that may be constrained index types, and record subyptes.
      --  In both cases, the constraints must be copied.

      function Copy_Itype (Itype : Entity_Id) return Entity_Id is
         New_Itype : Entity_Id := New_Copy (Itype);
      begin
         if Is_Discrete_Type (Itype) then
            Set_Scalar_Range (New_Itype,
               New_Copy_With_Replacement (Scalar_Range (Itype), Map));

         elsif Has_Discriminants (Base_Type (Itype)) then
            Set_Discriminant_Constraint (New_Itype,
              Copy_Elist_With_Replacement (Discriminant_Constraint (Itype)));
         end if;

         return New_Itype;
      end Copy_Itype;

      function Copy_Elist_With_Replacement (E : Elist_Id) return Elist_Id is
         M : Elmt_Id;
         L : Elist_Id := New_Elmt_List;

      begin
         M := First_Elmt (E);

         while Present (M) loop
            Append_Elmt (New_Copy_With_Replacement (Node (M), Map), L);
            M := Next_Elmt (M);
         end loop;

         return L;
      end Copy_Elist_With_Replacement;

      function Copy_List_With_Replacement (List : List_Id) return List_Id is
         NL : List_Id;
         E  : Node_Id;

      begin
         if List = No_List then
            return No_List;

         else
            NL := New_List;
            E := First (List);

            while Present (E) loop

               if Has_Extension (E) then
                  Append (Assoc (E), NL);
               else
                  Append (New_Copy_With_Replacement (E, Map), NL);
               end if;

               E := Next (E);
            end loop;

            return NL;
         end if;

      end Copy_List_With_Replacement;

      function Possible_Copy (Field : Union_Id) return Union_Id is
         New_N : Union_Id;

      begin
         if Field in Node_Range then

            if Parent (Node_Id (Field)) = Source then
               New_N :=  Union_Id
                   (New_Copy_With_Replacement (Node_Id (Field), Map));
               Set_Parent (Node_Id (New_N), Node_Id (New_Id));
            else
               New_N :=  Union_Id (Assoc (Node_Id (Field)));
            end if;

            return New_N;

         elsif Field in List_Range then
            if Parent (List_Id (Field)) = Source then
               New_N := Union_Id
                     (Copy_List_With_Replacement (List_Id (Field)));
               Set_Parent (List_Id (New_N), Node_Id (New_Id));
               return New_N;
            else
               return Field;
            end if;

         else
            return Field;
         end if;
      end Possible_Copy;

   --  Start of processing for New_Copy_With_Replacement

   begin
      if Source <= Empty_Or_Error then
         return Source;

      elsif Has_Extension (Source) then
         return Assoc (Source);

      else
         Nodes.Increment_Last;
         New_Id := Nodes.Last;
         Nodes.Table (New_Id) := Nodes.Table (Source);
         Nodes.Table (New_Id).Link := Empty_List_Or_Node;
         Nodes.Table (New_Id).In_List := False;
         Node_Count := Node_Count + 1;

         --  Copy any attached Itypes and associate old Itype with new
         --  one, to keep consistent references in new tree.

         if Nkind (Source) in N_Has_Itypes
           and then Present (First_Itype (Source))
         then
            O_Ityp := First_Itype (Source);
            N_Ityp := Copy_Itype (O_Ityp);
            Set_First_Itype (New_Id, N_Ityp);
            Add_Assoc (O_Ityp,  N_Ityp);

            while Present (Next_Itype (N_Ityp)) loop
               O_Ityp := Next_Itype (N_Ityp);
               Set_Next_Itype (N_Ityp, Copy_Itype (O_Ityp));
               N_Ityp := Next_Itype (N_Ityp);
               Add_Assoc (O_Ityp, N_Ityp);
            end loop;
         end if;

         --  Recursively copy descendents

         Set_Field1 (New_Id, Possible_Copy (Field1 (New_Id)));
         Set_Field2 (New_Id, Possible_Copy (Field2 (New_Id)));
         Set_Field3 (New_Id, Possible_Copy (Field3 (New_Id)));
         Set_Field4 (New_Id, Possible_Copy (Field4 (New_Id)));
         Set_Field5 (New_Id, Possible_Copy (Field5 (New_Id)));

         --  If the original is marked as a rewrite subsitution or a rewrite
         --  insertion, then unmark the copy, since it is the original that
         --  was rewritten, or inserted, not the copy.

         Nodes.Table (New_Id).Rewrite_Sub := False;
         Nodes.Table (New_Id).Rewrite_Ins := False;


         --  All done, return copied node

         return New_Id;
      end if;
   end New_Copy_With_Replacement;

   ----------------
   -- New_Entity --
   ----------------

   function New_Entity
     (New_Node_Kind : Node_Kind;
      New_Sloc      : Source_Ptr)
      return          Entity_Id
   is
      procedure New_Entity_Debugging_Output;
      --  Debugging routine for debug flag N

      procedure New_Entity_Debugging_Output is
      begin
         if Debug_Flag_N then
            Write_Str ("Allocate entity, Id = ");
            Write_Int (Int (Nodes.Last));
            Write_Str ("  ");
            Write_Location (New_Sloc);
            Write_Str ("  ");
            Write_Str (Node_Kind'Image (New_Node_Kind));
            Write_Eol;
         end if;
      end New_Entity_Debugging_Output;

      pragma Inline (New_Entity_Debugging_Output);

   --  Start of processing for New_Entity

   begin
      pragma Assert (New_Node_Kind in N_Entity);

      Nodes.Increment_Last;
      Fatal_Error_Node := Nodes.Last;
      Nodes.Table (Nodes.Last)        := Default_Node;
      Nodes.Table (Nodes.Last).Nkind  := New_Node_Kind;
      Nodes.Table (Nodes.Last).Sloc   := New_Sloc;
      pragma Debug (New_Entity_Debugging_Output);

      Nodes.Increment_Last;
      Nodes.Table (Nodes.Last) := Default_Node_Extension;

      Nodes.Increment_Last;
      Nodes.Table (Nodes.Last) := Default_Node_Extension;

      Node_Count := Node_Count + 1;
      return Fatal_Error_Node;
   end New_Entity;

   --------------
   -- New_Node --
   --------------

   function New_Node
     (New_Node_Kind : Node_Kind;
      New_Sloc      : Source_Ptr)
      return          Node_Id
   is
      procedure New_Node_Debugging_Output;
      --  Debugging routine for debug flag N

      procedure New_Node_Debugging_Output is
      begin
         if Debug_Flag_N then
            Write_Str ("Allocate node, Id = ");
            Write_Int (Int (Nodes.Last));
            Write_Str ("  ");
            Write_Location (New_Sloc);
            Write_Str ("  ");
            Write_Str (Node_Kind'Image (New_Node_Kind));
            Write_Eol;
         end if;
      end New_Node_Debugging_Output;

      pragma Inline (New_Node_Debugging_Output);

   --  Start of processing for New_Node

   begin
      pragma Assert (New_Node_Kind not in N_Entity);
      Nodes.Increment_Last;
      Nodes.Table (Nodes.Last)        := Default_Node;
      Nodes.Table (Nodes.Last).Nkind  := New_Node_Kind;
      Nodes.Table (Nodes.Last).Sloc   := New_Sloc;
      pragma Debug (New_Node_Debugging_Output);
      Fatal_Error_Node := Nodes.Last;
      Node_Count := Node_Count + 1;
      return Nodes.Last;
   end New_Node;

   -----------
   -- Nkind --
   -----------

   function Nkind (N : Node_Id) return Node_Kind is
   begin
      return Nodes.Table (N).Nkind;
   end Nkind;

   --------
   -- No --
   --------

   function No (N : Node_Id) return Boolean is
   begin
      return N = Empty;
   end No;

   -------------------
   -- Nodes_Address --
   -------------------

   function Nodes_Address return System.Address is
   begin
      return Nodes.Table (First_Node_Id)'Address;
   end Nodes_Address;

   ---------------
   -- Num_Nodes --
   ---------------

   function Num_Nodes return Nat is
   begin
      return Node_Count;
   end Num_Nodes;

   -------------------
   -- Original_Node --
   -------------------

   function Original_Node (Node : Node_Id) return Node_Id is
      Index : Hash_Entry_Id;

   begin
      --  For a node inserted by Rewrite_Ins, the original tree is Empty

      if Nodes.Table (Node).Rewrite_Ins then
         return Empty;

      --  For a node rewritten by Rewrite_Substitute_Tree, search the hash
      --  table to find the original node that matches the given node.

      elsif Nodes.Table (Node).Rewrite_Sub then

         Index := Hash_Headers (Int (Node) mod Num_Hash_Headers);

         loop
            pragma Assert (Index /= No_Entry);

            if Node = Hash_Sub.Table (Index).New_Node then
               return Hash_Sub.Table (Index).Old_Node;
            else
               Index := Hash_Sub.Table (Index).Next;
            end if;
         end loop;

      --  If not rewritten, then return original node

      else
         return Node;
      end if;
   end Original_Node;

   -----------------
   -- Paren_Count --
   -----------------

   function Paren_Count (N : Node_Id) return Paren_Count_Type is
      C : Paren_Count_Type := 0;

   begin
      pragma Assert (N in Nodes.First .. Nodes.Last);

      if Nodes.Table (N).Pflag1 then
         C := C + 1;
      end if;

      if Nodes.Table (N).Pflag2 then
         C := C + 2;
      end if;

      return C;
   end Paren_Count;

   ------------
   -- Parent --
   ------------

   function Parent (N : Node_Id) return Node_Id is
   begin
      if Is_List_Member (N) then
         return Parent (List_Containing (N));
      else
         return Node_Id (Nodes.Table (N).Link);
      end if;
   end Parent;

   -------------
   -- Present --
   -------------

   function Present (N : Node_Id) return Boolean is
   begin
      return N /= Empty;
   end Present;

   -------------------
   -- Relocate_Node --
   -------------------

   function Relocate_Node (Source : Node_Id) return Node_Id is
      New_Node : constant Node_Id := New_Copy (Source);

   begin
      Fix_Parent (Field1 (Source), Source, New_Node);
      Fix_Parent (Field2 (Source), Source, New_Node);
      Fix_Parent (Field3 (Source), Source, New_Node);
      Fix_Parent (Field4 (Source), Source, New_Node);
      Fix_Parent (Field5 (Source), Source, New_Node);
      return New_Node;
   end Relocate_Node;

   -----------------------------
   -- Replace_Substitute_Tree --
   -----------------------------

   procedure Replace_Substitute_Tree (Old_Node, New_Node : Node_Id) is
      Old_Link    : constant Union_Id := Nodes.Table (Old_Node).Link;
      Old_In_List : constant Boolean  := Nodes.Table (Old_Node).In_List;
      Old_Err_Flg : constant Boolean  := Error_Posted (Old_Node);

   begin
      --  Never use Replace_Substitute_Tree on expression nodes, use
      --  Rewrite_Substitute_Tree instead, since otherwise conformance
      --  checks get messed up. We could make this an absolute rule,
      --  but that would probably be harsh, so just fix it on the fly.

      if Nkind (Old_Node) in N_Subexpr then
         Rewrite_Substitute_Tree (Old_Node, New_Node);

      --  Not a subexpression, OK to do replace

      else
         pragma Assert
           (not Has_Extension (Old_Node)
              and not Has_Extension (New_Node)
              and not Nodes.Table (New_Node).In_List);

         --  Do copy, preserving link and in list status

         Nodes.Table (Old_Node)         := Nodes.Table (New_Node);
         Nodes.Table (Old_Node).Link    := Old_Link;
         Nodes.Table (Old_Node).In_List := Old_In_List;

         Set_Error_Posted (Old_Node, Old_Err_Flg);

         --  Fix parents of substituted node, since it has changed identity

         Fix_Parent (Field1 (Old_Node), New_Node, Old_Node);
         Fix_Parent (Field2 (Old_Node), New_Node, Old_Node);
         Fix_Parent (Field3 (Old_Node), New_Node, Old_Node);
         Fix_Parent (Field4 (Old_Node), New_Node, Old_Node);
         Fix_Parent (Field5 (Old_Node), New_Node, Old_Node);

         --  Finally delete the source, since it is now copied

         Delete_Node (New_Node);
      end if;

   end Replace_Substitute_Tree;

   -----------------------------
   -- Rewrite_Substitute_Tree --
   -----------------------------

   procedure Rewrite_Substitute_Tree (Old_Node, New_Node : Node_Id) is
      Old_Link    : constant Union_Id := Nodes.Table (Old_Node).Link;
      Old_In_List : constant Boolean  := Nodes.Table (Old_Node).In_List;
      Sav_Node    : Node_Id;

   begin
      pragma Assert
        (not Has_Extension (Old_Node)
           and not Has_Extension (New_Node)
           and not Nodes.Table (New_Node).In_List);

      --  Allocate a new node, to be used to preserve the original contents
      --  of the Old_Node, for possible later retrival by Original_Node.

      Nodes.Increment_Last;
      Node_Count := Node_Count + 1;
      Sav_Node := Nodes.Last;
      Nodes.Table (Sav_Node)         := Nodes.Table (Old_Node);
      Nodes.Table (Sav_Node).In_List := False;
      Nodes.Table (Sav_Node).Link    := Union_Id (Empty);

      --  Copy substitute node into place, preserving old link and in-list flag

      Nodes.Table (Old_Node)         := Nodes.Table (New_Node);
      Nodes.Table (Old_Node).Link    := Old_Link;
      Nodes.Table (Old_Node).In_List := Old_In_List;

      Set_Error_Posted (Old_Node, Error_Posted (Sav_Node));

      --  Set flag indicating that node has been rewritten and fixup parent
      --  pointers for the substituted node, since it has changed its identity

      Nodes.Table (Old_Node).Rewrite_Sub := True;

      --  Preserve parenthesized state of original (helps -gnatdg listing)

      if Nkind (Sav_Node) in N_Subexpr
        and then Paren_Count (Sav_Node) /= 0
        and then Nkind (New_Node) in N_Subexpr
      then
         Set_Paren_Count (New_Node, 1);
      end if;

      Fix_Parent (Field1 (Old_Node), New_Node, Old_Node);
      Fix_Parent (Field2 (Old_Node), New_Node, Old_Node);
      Fix_Parent (Field3 (Old_Node), New_Node, Old_Node);
      Fix_Parent (Field4 (Old_Node), New_Node, Old_Node);
      Fix_Parent (Field5 (Old_Node), New_Node, Old_Node);

      --  Now, make an entry in the hash table. Note that we overwrite any
      --  existing entry that references Old_Node. This allows multiple steps
      --  of substitutions, and we will always get the real original.

      declare
         Hash_Code : constant Nat  := Int (Old_Node) mod Num_Hash_Headers;
         Index     : Hash_Entry_Id := Hash_Headers (Hash_Code);

      begin
         --  If the hash table is empty, make a new entry

         if Index = No_Entry then
            Hash_Sub.Increment_Last;
            Hash_Headers (Hash_Code) := Hash_Sub.Last;
            Hash_Sub.Table (Hash_Sub.Last).Next     := No_Entry;
            Hash_Sub.Table (Hash_Sub.Last).New_Node := Old_Node;
            Hash_Sub.Table (Hash_Sub.Last).Old_Node := Sav_Node;

         else
            loop
               --  If we already have an entry in the hash table for this node,
               --  it means that we already replaced it once, and the entry in
               --  the hash table should remain unchanged, indicating the node
               --  contents before the first Rewrite_Substitute_Tree call. In
               --  this case we don't need Sav_Node after all and can simply
               --  undo its allocation.

               if Hash_Sub.Table (Index).New_Node = Old_Node then
                  Nodes.Decrement_Last;
                  return;

               --  If end of hash chain, insert new entry

               elsif Hash_Sub.Table (Index).Next = No_Entry then
                  Hash_Sub.Increment_Last;
                  Hash_Sub.Table (Index).Next := Hash_Sub.Last;
                  Hash_Sub.Table (Hash_Sub.Last).Next     := No_Entry;
                  Hash_Sub.Table (Hash_Sub.Last).New_Node := Old_Node;
                  Hash_Sub.Table (Hash_Sub.Last).Old_Node := Sav_Node;
                  return;

               --  Otherwise move out on hash chain

               else
                  Index := Hash_Sub.Table (Index).Next;
               end if;
            end loop;
         end if;
      end;

   end Rewrite_Substitute_Tree;

   -----------------------------------
   -- Set_Comes_From_Source_Default --
   -----------------------------------

   procedure Set_Comes_From_Source_Default (Default : Boolean) is
   begin
      Default_Node.Comes_From_Source := Default;
   end Set_Comes_From_Source_Default;

   --------------------
   -- Set_Convention --
   --------------------

   procedure Set_Convention  (N : Node_Id; Val : Convention_Id) is
   begin
      pragma Assert (Nkind (N) in N_Entity);
      To_Flag_Word_Ptr
        (Union_Id_Ptr'
          (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Convention :=
                                                                        Val;
   end Set_Convention;

   ---------------
   -- Set_Ekind --
   ---------------

   procedure Set_Ekind (N : Node_Id; Val : Entity_Kind) is
   begin
      pragma Assert (Nkind (N) in N_Entity);
      Nodes.Table (N + 1).Nkind := E_To_N (Val);
   end Set_Ekind;

   ---------------------
   -- Set_Paren_Count --
   ---------------------

   procedure Set_Paren_Count (N : Node_Id; Val : Paren_Count_Type) is
   begin
      pragma Assert (Nkind (N) in N_Subexpr);
      Nodes.Table (N).Pflag1 := (Val mod 2 /= 0);
      Nodes.Table (N).Pflag2 := (Val >= 2);
   end Set_Paren_Count;

   ----------------
   -- Set_Parent --
   ----------------

   procedure Set_Parent (N : Node_Id; Val : Node_Id) is
   begin
      pragma Assert (not Nodes.Table (N).In_List);
      Nodes.Table (N).Link := Union_Id (Val);
   end Set_Parent;

   --------------
   -- Set_Sloc --
   --------------

   procedure Set_Sloc (N : Node_Id; Val : Source_Ptr) is
   begin
      Nodes.Table (N).Sloc := Val;
   end Set_Sloc;

   ----------
   -- Sloc --
   ----------

   function Sloc (N : Node_Id) return Source_Ptr is
   begin
      return Nodes.Table (N).Sloc;
   end Sloc;

   ---------------
   -- Tree_Read --
   ---------------

   procedure Tree_Read is
   begin
      Tree_Read_Int (Node_Count);
      Nodes.Tree_Read;
      Hash_Sub.Tree_Read;

      Tree_Read_Data
        (Hash_Headers'Address,
         Hash_Headers'Length * (Hash_Entry_Id'Size / Storage_Unit));

   end Tree_Read;

   ----------------
   -- Tree_Write --
   ----------------

   procedure Tree_Write is
   begin
      Tree_Write_Int (Node_Count);
      Nodes.Tree_Write;
      Hash_Sub.Tree_Write;

      Tree_Write_Data
        (Hash_Headers'Address,
         Hash_Headers'Length * (Hash_Entry_Id'Size / Storage_Unit));

   end Tree_Write;

   ------------------------------
   -- Unchecked Access Package --
   ------------------------------

   package body Unchecked_Access is

      function Field1 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Field1;
      end Field1;

      function Field2 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Field2;
      end Field2;

      function Field3 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Field3;
      end Field3;

      function Field4 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Field4;
      end Field4;

      function Field5 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Field5;
      end Field5;

      function Field6 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field6;
      end Field6;

      function Field7 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field7;
      end Field7;

      function Field8 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field8;
      end Field8;

      function Field9 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field9;
      end Field9;

      function Field10 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field10;
      end Field10;

      function Field11 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field11;
      end Field11;

      function Field12 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Field12;
      end Field12;

      function Field13 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field6;
      end Field13;

      function Field14 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field7;
      end Field14;

      function Field15 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field8;
      end Field15;

      function Field16 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field9;
      end Field16;

      function Field17 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field10;
      end Field17;

      function Field18 (N : Node_Id) return Union_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Field11;
      end Field18;

      function Node1 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Node_Id (Nodes.Table (N).Field1);
      end Node1;

      function Node2 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Node_Id (Nodes.Table (N).Field2);
      end Node2;

      function Node3 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Node_Id (Nodes.Table (N).Field3);
      end Node3;

      function Node4 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Node_Id (Nodes.Table (N).Field4);
      end Node4;

      function Node5 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Node_Id (Nodes.Table (N).Field5);
      end Node5;

      function Node6 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field6);
      end Node6;

      function Node7 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field7);
      end Node7;

      function Node8 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field8);
      end Node8;

      function Node9 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field9);
      end Node9;

      function Node10 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field10);
      end Node10;

      function Node11 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field11);
      end Node11;

      function Node12 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 1).Field12);
      end Node12;

      function Node13 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field6);
      end Node13;

      function Node14 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field7);
      end Node14;

      function Node15 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field8);
      end Node15;

      function Node16 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field9);
      end Node16;

      function Node17 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field10);
      end Node17;

      function Node18 (N : Node_Id) return Node_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Node_Id (Nodes.Table (N + 2).Field11);
      end Node18;

      function List1 (N : Node_Id) return List_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return List_Id (Nodes.Table (N).Field1);
      end List1;

      function List2 (N : Node_Id) return List_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return List_Id (Nodes.Table (N).Field2);
      end List2;

      function List3 (N : Node_Id) return List_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return List_Id (Nodes.Table (N).Field3);
      end List3;

      function List4 (N : Node_Id) return List_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return List_Id (Nodes.Table (N).Field4);
      end List4;

      function List5 (N : Node_Id) return List_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return List_Id (Nodes.Table (N).Field5);
      end List5;

      function Elist3 (N : Node_Id) return Elist_Id is
      begin
         return Elist_Id (Nodes.Table (N).Field3);
      end Elist3;

      function Elist6 (N : Node_Id) return Elist_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Elist_Id (Nodes.Table (N + 1).Field6);
      end Elist6;

      function Elist7 (N : Node_Id) return Elist_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Elist_Id (Nodes.Table (N + 1).Field7);
      end Elist7;

      function Elist13 (N : Node_Id) return Elist_Id is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Elist_Id (Nodes.Table (N + 2).Field6);
      end Elist13;

      function Name1 (N : Node_Id) return Name_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Name_Id (Nodes.Table (N).Field1);
      end Name1;

      function Name2 (N : Node_Id) return Name_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Name_Id (Nodes.Table (N).Field2);
      end Name2;

      function Str3 (N : Node_Id) return String_Id is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return String_Id (Nodes.Table (N).Field3);
      end Str3;

      function Char_Code2 (N : Node_Id) return Char_Code is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Char_Code (Nodes.Table (N).Field2 - Char_Code_Bias);
      end Char_Code2;

      function Uint3 (N : Node_Id) return Uint is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return From_Union (Nodes.Table (N).Field3);
      end Uint3;

      function Uint4 (N : Node_Id) return Uint is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return From_Union (Nodes.Table (N).Field4);
      end Uint4;

      function Uint8 (N : Node_Id) return Uint is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field8);
      end Uint8;

      function Uint9 (N : Node_Id) return Uint is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field9);
      end Uint9;

      function Uint11 (N : Node_Id) return Uint is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field11);
      end Uint11;

      function Uint12 (N : Node_Id) return Uint is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field12);
      end Uint12;

      function Uint15 (N : Node_Id) return Uint is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 2).Field8);
      end Uint15;

      function Ureal3 (N : Node_Id) return Ureal is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return From_Union (Nodes.Table (N).Field3);
      end Ureal3;

      function Ureal6 (N : Node_Id) return Ureal is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field6);
      end Ureal6;

      function Ureal7 (N : Node_Id) return Ureal is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return From_Union (Nodes.Table (N + 1).Field7);
      end Ureal7;

      function Flag1 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag1;
      end Flag1;

      function Flag3 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag3;
      end Flag3;

      function Flag4 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag4;
      end Flag4;

      function Flag5 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag5;
      end Flag5;

      function Flag6 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag6;
      end Flag6;

      function Flag7 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag7;
      end Flag7;

      function Flag8 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag8;
      end Flag8;

      function Flag9 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag9;
      end Flag9;

      function Flag10 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag10;
      end Flag10;

      function Flag11 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag11;
      end Flag11;

      function Flag12 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag12;
      end Flag12;

      function Flag13 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag13;
      end Flag13;

      function Flag14 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag14;
      end Flag14;

      function Flag15 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag15;
      end Flag15;

      function Flag16 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag16;
      end Flag16;

      function Flag17 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag17;
      end Flag17;

      function Flag18 (N : Node_Id) return Boolean is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         return Nodes.Table (N).Flag18;
      end Flag18;

      function Flag19 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).In_List;
      end Flag19;

      function Flag20 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Rewrite_Sub;
      end Flag20;

      function Flag21 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Rewrite_Ins;
      end Flag21;

      function Flag22 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag1;
      end Flag22;

      function Flag23 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Comes_From_Source;
      end Flag23;

      function Flag24 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag3;
      end Flag24;

      function Flag25 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag4;
      end Flag25;

      function Flag26 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag5;
      end Flag26;

      function Flag27 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag6;
      end Flag27;

      function Flag28 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag7;
      end Flag28;

      function Flag29 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag8;
      end Flag29;

      function Flag30 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag9;
      end Flag30;

      function Flag31 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag10;
      end Flag31;

      function Flag32 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag11;
      end Flag32;

      function Flag33 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag12;
      end Flag33;

      function Flag34 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag13;
      end Flag34;

      function Flag35 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag14;
      end Flag35;

      function Flag36 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag15;
      end Flag36;

      function Flag37 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag16;
      end Flag37;

      function Flag38 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag17;
      end Flag38;

      function Flag39 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Flag18;
      end Flag39;

      function Flag40 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).In_List;
      end Flag40;

      function Flag41 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Rewrite_Sub;
      end Flag41;

      function Flag42 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Rewrite_Ins;
      end Flag42;

      function Flag43 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag1;
      end Flag43;

      function Flag44 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Comes_From_Source;
      end Flag44;

      function Flag45 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag3;
      end Flag45;

      function Flag46 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag4;
      end Flag46;

      function Flag47 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag5;
      end Flag47;

      function Flag48 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag6;
      end Flag48;

      function Flag49 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag7;
      end Flag49;

      function Flag50 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag8;
      end Flag50;

      function Flag51 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag9;
      end Flag51;

      function Flag52 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag10;
      end Flag52;

      function Flag53 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag11;
      end Flag53;

      function Flag54 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag12;
      end Flag54;

      function Flag55 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag13;
      end Flag55;

      function Flag56 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag14;
      end Flag56;

      function Flag57 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag15;
      end Flag57;

      function Flag58 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag16;
      end Flag58;

      function Flag59 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag17;
      end Flag59;

      function Flag60 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Flag18;
      end Flag60;

      function Flag61 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Pflag1;
      end Flag61;

      function Flag62 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 1).Pflag2;
      end Flag62;

      function Flag63 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Pflag1;
      end Flag63;

      function Flag64 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return Nodes.Table (N + 2).Pflag2;
      end Flag64;

      function Flag65 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag65;
      end Flag65;

      function Flag66 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag66;
      end Flag66;

      function Flag67 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag67;
      end Flag67;

      function Flag68 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag68;
      end Flag68;

      function Flag69 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag69;
      end Flag69;

      function Flag70 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag70;
      end Flag70;

      function Flag71 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag71;
      end Flag71;

      function Flag72 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Byte (Nodes.Table (N + 2).Nkind).Flag72;
      end Flag72;

      function Flag73 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag73;
      end Flag73;

      function Flag74 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag74;
      end Flag74;

      function Flag75 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag75;
      end Flag75;

      function Flag76 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag76;
      end Flag76;

      function Flag77 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag77;
      end Flag77;

      function Flag78 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag78;
      end Flag78;

      function Flag79 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag79;
      end Flag79;

      function Flag80 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag80;
      end Flag80;

      function Flag81 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag81;
      end Flag81;

      function Flag82 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag82;
      end Flag82;

      function Flag83 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag83;
      end Flag83;

      function Flag84 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag84;
      end Flag84;

      function Flag85 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag85;
      end Flag85;

      function Flag86 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag86;
      end Flag86;

      function Flag87 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag87;
      end Flag87;

      function Flag88 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag88;
      end Flag88;

      function Flag89 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag89;
      end Flag89;

      function Flag90 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag90;
      end Flag90;

      function Flag91 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag91;
      end Flag91;

      function Flag92 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag92;
      end Flag92;

      function Flag93 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag93;
      end Flag93;

      function Flag94 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag94;
      end Flag94;

      function Flag95 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag95;
      end Flag95;

      function Flag96 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag96;
      end Flag96;

      function Flag97 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag97;
      end Flag97;

      function Flag98 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag98;
      end Flag98;

      function Flag99 (N : Node_Id) return Boolean is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         return To_Flag_Word (Nodes.Table (N + 2).Field12).Flag99;
      end Flag99;

      procedure Set_Nkind (N : Node_Id; Val : Node_Kind) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Nkind := Val;
      end Set_Nkind;

      procedure Set_Field1 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field1 := Val;
      end Set_Field1;

      procedure Set_Field2 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field2 := Val;
      end Set_Field2;

      procedure Set_Field3 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := Val;
      end Set_Field3;

      procedure Set_Field4 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field4 := Val;
      end Set_Field4;

      procedure Set_Field5 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field5 := Val;
      end Set_Field5;

      procedure Set_Field6 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field6 := Val;
      end Set_Field6;

      procedure Set_Field7 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field7 := Val;
      end Set_Field7;

      procedure Set_Field8 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field8 := Val;
      end Set_Field8;

      procedure Set_Field9 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field9 := Val;
      end Set_Field9;

      procedure Set_Field10 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field10 := Val;
      end Set_Field10;

      procedure Set_Field11 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field11 := Val;
      end Set_Field11;

      procedure Set_Field12 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field12 := Val;
      end Set_Field12;

      procedure Set_Field13 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field6 := Val;
      end Set_Field13;

      procedure Set_Field14 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field7 := Val;
      end Set_Field14;

      procedure Set_Field15 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field8 := Val;
      end Set_Field15;

      procedure Set_Field16 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field9 := Val;
      end Set_Field16;

      procedure Set_Field17 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field10 := Val;
      end Set_Field17;

      procedure Set_Field18 (N : Node_Id; Val : Union_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field11 := Val;
      end Set_Field18;

      procedure Set_Node1 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field1 := Union_Id (Val);
      end Set_Node1;

      procedure Set_Node2 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field2 := Union_Id (Val);
      end Set_Node2;

      procedure Set_Node3 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := Union_Id (Val);
      end Set_Node3;

      procedure Set_Node4 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field4 := Union_Id (Val);
      end Set_Node4;

      procedure Set_Node5 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field5 := Union_Id (Val);
      end Set_Node5;

      procedure Set_Node6 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field6 := Union_Id (Val);
      end Set_Node6;

      procedure Set_Node7 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field7 := Union_Id (Val);
      end Set_Node7;

      procedure Set_Node8 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field8 := Union_Id (Val);
      end Set_Node8;

      procedure Set_Node9 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field9 := Union_Id (Val);
      end Set_Node9;

      procedure Set_Node10 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field10 := Union_Id (Val);
      end Set_Node10;

      procedure Set_Node11 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field11 := Union_Id (Val);
      end Set_Node11;

      procedure Set_Node12 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field12 := Union_Id (Val);
      end Set_Node12;

      procedure Set_Node13 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field6 := Union_Id (Val);
      end Set_Node13;

      procedure Set_Node14 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field7 := Union_Id (Val);
      end Set_Node14;

      procedure Set_Node15 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field8 := Union_Id (Val);
      end Set_Node15;

      procedure Set_Node16 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field9 := Union_Id (Val);
      end Set_Node16;

      procedure Set_Node17 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field10 := Union_Id (Val);
      end Set_Node17;

      procedure Set_Node18 (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field11 := Union_Id (Val);
      end Set_Node18;

      procedure Set_List1 (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field1 := Union_Id (Val);
      end Set_List1;

      procedure Set_List2 (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field2 := Union_Id (Val);
      end Set_List2;

      procedure Set_List3 (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := Union_Id (Val);
      end Set_List3;

      procedure Set_List4 (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field4 := Union_Id (Val);
      end Set_List4;

      procedure Set_List5 (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field5 := Union_Id (Val);
      end Set_List5;

      procedure Set_Elist3 (N : Node_Id; Val : Elist_Id) is
      begin
         Nodes.Table (N).Field3 := Union_Id (Val);
      end Set_Elist3;

      procedure Set_Elist6 (N : Node_Id; Val : Elist_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field6 := Union_Id (Val);
      end Set_Elist6;

      procedure Set_Elist7 (N : Node_Id; Val : Elist_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field7 := Union_Id (Val);
      end Set_Elist7;

      procedure Set_Elist13 (N : Node_Id; Val : Elist_Id) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field6 := Union_Id (Val);
      end Set_Elist13;

      procedure Set_Name1 (N : Node_Id; Val : Name_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field1 := Union_Id (Val);
      end Set_Name1;

      procedure Set_Name2 (N : Node_Id; Val : Name_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field2 := Union_Id (Val);
      end Set_Name2;

      procedure Set_Str3 (N : Node_Id; Val : String_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := Union_Id (Val);
      end Set_Str3;

      procedure Set_Uint3 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := To_Union (Val);
      end Set_Uint3;

      procedure Set_Uint4 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field4 := To_Union (Val);
      end Set_Uint4;

      procedure Set_Uint8 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field8 := To_Union (Val);
      end Set_Uint8;

      procedure Set_Uint9 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field9 := To_Union (Val);
      end Set_Uint9;

      procedure Set_Uint11 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field11 := To_Union (Val);
      end Set_Uint11;

      procedure Set_Uint12 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field12 := To_Union (Val);
      end Set_Uint12;

      procedure Set_Uint15 (N : Node_Id; Val : Uint) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Field8 := To_Union (Val);
      end Set_Uint15;

      procedure Set_Ureal3 (N : Node_Id; Val : Ureal) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field3 := To_Union (Val);
      end Set_Ureal3;

      procedure Set_Ureal6 (N : Node_Id; Val : Ureal) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field6 := To_Union (Val);
      end Set_Ureal6;

      procedure Set_Ureal7 (N : Node_Id; Val : Ureal) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Field7 := To_Union (Val);
      end Set_Ureal7;

      procedure Set_Char_Code2 (N : Node_Id; Val : Char_Code) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Field2 := Union_Id (Val) + Char_Code_Bias;
      end Set_Char_Code2;

      procedure Set_Flag1 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag1 := Val;
      end Set_Flag1;

      procedure Set_Flag3 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag3 := Val;
      end Set_Flag3;

      procedure Set_Flag4 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag4 := Val;
      end Set_Flag4;

      procedure Set_Flag5 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag5 := Val;
      end Set_Flag5;

      procedure Set_Flag6 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag6 := Val;
      end Set_Flag6;

      procedure Set_Flag7 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag7 := Val;
      end Set_Flag7;

      procedure Set_Flag8 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag8 := Val;
      end Set_Flag8;

      procedure Set_Flag9 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag9 := Val;
      end Set_Flag9;

      procedure Set_Flag10 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag10 := Val;
      end Set_Flag10;

      procedure Set_Flag11 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag11 := Val;
      end Set_Flag11;

      procedure Set_Flag12 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag12 := Val;
      end Set_Flag12;

      procedure Set_Flag13 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag13 := Val;
      end Set_Flag13;

      procedure Set_Flag14 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag14 := Val;
      end Set_Flag14;

      procedure Set_Flag15 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag15 := Val;
      end Set_Flag15;

      procedure Set_Flag16 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag16 := Val;
      end Set_Flag16;

      procedure Set_Flag17 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag17 := Val;
      end Set_Flag17;

      procedure Set_Flag18 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         Nodes.Table (N).Flag18 := Val;
      end Set_Flag18;

      procedure Set_Flag19 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).In_List := Val;
      end Set_Flag19;

      procedure Set_Flag20 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Rewrite_Sub := Val;
      end Set_Flag20;

      procedure Set_Flag21 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Rewrite_Ins := Val;
      end Set_Flag21;

      procedure Set_Flag22 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag1 := Val;
      end Set_Flag22;

      procedure Set_Flag23 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Comes_From_Source := Val;
      end Set_Flag23;

      procedure Set_Flag24 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag3 := Val;
      end Set_Flag24;

      procedure Set_Flag25 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag4 := Val;
      end Set_Flag25;

      procedure Set_Flag26 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag5 := Val;
      end Set_Flag26;

      procedure Set_Flag27 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag6 := Val;
      end Set_Flag27;

      procedure Set_Flag28 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag7 := Val;
      end Set_Flag28;

      procedure Set_Flag29 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag8 := Val;
      end Set_Flag29;

      procedure Set_Flag30 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag9 := Val;
      end Set_Flag30;

      procedure Set_Flag31 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag10 := Val;
      end Set_Flag31;

      procedure Set_Flag32 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag11 := Val;
      end Set_Flag32;

      procedure Set_Flag33 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag12 := Val;
      end Set_Flag33;

      procedure Set_Flag34 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag13 := Val;
      end Set_Flag34;

      procedure Set_Flag35 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag14 := Val;
      end Set_Flag35;

      procedure Set_Flag36 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag15 := Val;
      end Set_Flag36;

      procedure Set_Flag37 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag16 := Val;
      end Set_Flag37;

      procedure Set_Flag38 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag17 := Val;
      end Set_Flag38;

      procedure Set_Flag39 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Flag18 := Val;
      end Set_Flag39;

      procedure Set_Flag40 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).In_List := Val;
      end Set_Flag40;

      procedure Set_Flag41 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Rewrite_Sub := Val;
      end Set_Flag41;

      procedure Set_Flag42 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Rewrite_Ins := Val;
      end Set_Flag42;

      procedure Set_Flag43 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag1 := Val;
      end Set_Flag43;

      procedure Set_Flag44 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Comes_From_Source := Val;
      end Set_Flag44;

      procedure Set_Flag45 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag3 := Val;
      end Set_Flag45;

      procedure Set_Flag46 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag4 := Val;
      end Set_Flag46;

      procedure Set_Flag47 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag5 := Val;
      end Set_Flag47;

      procedure Set_Flag48 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag6 := Val;
      end Set_Flag48;

      procedure Set_Flag49 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag7 := Val;
      end Set_Flag49;

      procedure Set_Flag50 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag8 := Val;
      end Set_Flag50;

      procedure Set_Flag51 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag9 := Val;
      end Set_Flag51;

      procedure Set_Flag52 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag10 := Val;
      end Set_Flag52;

      procedure Set_Flag53 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag11 := Val;
      end Set_Flag53;

      procedure Set_Flag54 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag12 := Val;
      end Set_Flag54;

      procedure Set_Flag55 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag13 := Val;
      end Set_Flag55;

      procedure Set_Flag56 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag14 := Val;
      end Set_Flag56;

      procedure Set_Flag57 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag15 := Val;
      end Set_Flag57;

      procedure Set_Flag58 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag16 := Val;
      end Set_Flag58;

      procedure Set_Flag59 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag17 := Val;
      end Set_Flag59;

      procedure Set_Flag60 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Flag18 := Val;
      end Set_Flag60;

      procedure Set_Flag61 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Pflag1 := Val;
      end Set_Flag61;

      procedure Set_Flag62 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 1).Pflag2 := Val;
      end Set_Flag62;

      procedure Set_Flag63 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Pflag1 := Val;
      end Set_Flag63;

      procedure Set_Flag64 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         Nodes.Table (N + 2).Pflag2 := Val;
      end Set_Flag64;

      procedure Set_Flag65 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag65 := Val;
      end Set_Flag65;

      procedure Set_Flag66 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag66 := Val;
      end Set_Flag66;

      procedure Set_Flag67 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag67 := Val;
      end Set_Flag67;

      procedure Set_Flag68 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag68 := Val;
      end Set_Flag68;

      procedure Set_Flag69 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag69 := Val;
      end Set_Flag69;

      procedure Set_Flag70 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag70 := Val;
      end Set_Flag70;

      procedure Set_Flag71 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag71 := Val;
      end Set_Flag71;

      procedure Set_Flag72 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Byte_Ptr
           (Node_Kind_Ptr'
             (Nodes.Table (N + 2).Nkind'Unrestricted_Access)).Flag72 := Val;
      end Set_Flag72;

      procedure Set_Flag73 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag73 := Val;
      end Set_Flag73;

      procedure Set_Flag74 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag74 := Val;
      end Set_Flag74;

      procedure Set_Flag75 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag75 := Val;
      end Set_Flag75;

      procedure Set_Flag76 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag76 := Val;
      end Set_Flag76;

      procedure Set_Flag77 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag77 := Val;
      end Set_Flag77;

      procedure Set_Flag78 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag78 := Val;
      end Set_Flag78;

      procedure Set_Flag79 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag79 := Val;
      end Set_Flag79;

      procedure Set_Flag80 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag80 := Val;
      end Set_Flag80;

      procedure Set_Flag81 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag81 := Val;
      end Set_Flag81;

      procedure Set_Flag82 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag82 := Val;
      end Set_Flag82;

      procedure Set_Flag83 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag83 := Val;
      end Set_Flag83;

      procedure Set_Flag84 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag84 := Val;
      end Set_Flag84;

      procedure Set_Flag85 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag85 := Val;
      end Set_Flag85;

      procedure Set_Flag86 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag86 := Val;
      end Set_Flag86;

      procedure Set_Flag87 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag87 := Val;
      end Set_Flag87;

      procedure Set_Flag88 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag88 := Val;
      end Set_Flag88;

      procedure Set_Flag89 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag89 := Val;
      end Set_Flag89;

      procedure Set_Flag90 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag90 := Val;
      end Set_Flag90;

      procedure Set_Flag91 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag91 := Val;
      end Set_Flag91;

      procedure Set_Flag92 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag92 := Val;
      end Set_Flag92;

      procedure Set_Flag93 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag93 := Val;
      end Set_Flag93;

      procedure Set_Flag94 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag94 := Val;
      end Set_Flag94;

      procedure Set_Flag95 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag95 := Val;
      end Set_Flag95;

      procedure Set_Flag96 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag96 := Val;
      end Set_Flag96;

      procedure Set_Flag97 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag97 := Val;
      end Set_Flag97;

      procedure Set_Flag98 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag98 := Val;
      end Set_Flag98;

      procedure Set_Flag99 (N : Node_Id; Val : Boolean) is
      begin
         pragma Assert (Nkind (N) in N_Entity);
         To_Flag_Word_Ptr
           (Union_Id_Ptr'
             (Nodes.Table (N + 2).Field12'Unrestricted_Access)).Flag99 := Val;
      end Set_Flag99;

      procedure Set_Node1_With_Parent (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val > Error then Set_Parent (Val, N); end if;
         Set_Node1 (N, Val);
      end Set_Node1_With_Parent;

      procedure Set_Node2_With_Parent (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val > Error then Set_Parent (Val, N); end if;
         Set_Node2 (N, Val);
      end Set_Node2_With_Parent;

      procedure Set_Node3_With_Parent (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val > Error then Set_Parent (Val, N); end if;
         Set_Node3 (N, Val);
      end Set_Node3_With_Parent;

      procedure Set_Node4_With_Parent (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val > Error then Set_Parent (Val, N); end if;
         Set_Node4 (N, Val);
      end Set_Node4_With_Parent;

      procedure Set_Node5_With_Parent (N : Node_Id; Val : Node_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val > Error then Set_Parent (Val, N); end if;
         Set_Node5 (N, Val);
      end Set_Node5_With_Parent;

      procedure Set_List1_With_Parent (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val /= No_List and then Val /= Error_List then
            Set_Parent (Val, N);
         end if;
         Set_List1 (N, Val);
      end Set_List1_With_Parent;

      procedure Set_List2_With_Parent (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val /= No_List and then Val /= Error_List then
            Set_Parent (Val, N);
         end if;
         Set_List2 (N, Val);
      end Set_List2_With_Parent;

      procedure Set_List3_With_Parent (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val /= No_List and then Val /= Error_List then
            Set_Parent (Val, N);
         end if;
         Set_List3 (N, Val);
      end Set_List3_With_Parent;

      procedure Set_List4_With_Parent (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val /= No_List and then Val /= Error_List then
            Set_Parent (Val, N);
         end if;
         Set_List4 (N, Val);
      end Set_List4_With_Parent;

      procedure Set_List5_With_Parent (N : Node_Id; Val : List_Id) is
      begin
         pragma Assert (N in Nodes.First .. Nodes.Last);
         if Val /= No_List and then Val /= Error_List then
            Set_Parent (Val, N);
         end if;
         Set_List5 (N, Val);
      end Set_List5_With_Parent;

   end Unchecked_Access;

end Atree;
