------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 9                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.190 $                            --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Expander; use Expander;
with Exp_Ch3;  use Exp_Ch3;
with Exp_Ch6;  use Exp_Ch6;
with Exp_TSS;  use Exp_TSS;
with Exp_Util; use Exp_Util;
with Freeze;   use Freeze;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch5;
with Sem_Ch6;
with Sem_Ch11; use Sem_Ch11;
with Sem_Ch13; use Sem_Ch13;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Exp_Ch9 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Array_Type (E : Entity_Id; Trec : Node_Id) return Entity_Id;
   --  Find the array type associated with an entry family in the
   --  associated record for the task type.

   function Build_Accept_Body
     (Stats : Node_Id;
      Loc   : Source_Ptr)
      return  Node_Id;

   --  Transform accept statement into a block with added exception handler.
   --  Cused both for simple accept statements and for accept alternatives in
   --  select statements.

   function Build_Corresponding_Record
     (N    : Node_Id;
      Ctyp : Node_Id;
      Loc  : Source_Ptr)
      return Node_Id;
   --  Common to tasks and protected types. Copy discriminant specifications,
   --  build record declaration.

   function Build_Entry_Count_Expression
     (Concurrent_Type : Node_Id;
      Loc             : Source_Ptr)
      return            Node_Id;
   --  Compute number of entries for concurrent object. This is a count of
   --  simple entries, followed by an expression that computes the length
   --  of the range of each entry family.

   function Build_Entry_Service_Procedure
     (Pid          : Node_Id;
      Barrier_Name : Name_Id;
      Index_Name   : Name_Id;
      Lab_Decl     : Node_Id;
      Barriers     : List_Id;
      Entry_Alts   : List_Id;
      Family_Alts  : List_Id)
      return         Node_Id;
   --  This routine constructs an entry service procedure, which services
   --  all entries waiting on open barriers of a protected object of the
   --  associated type. The barriers and entry bodies have already been
   --  expanded; Barriers is a list of assignment statements to build the
   --  barrier vector required by Protected_Entry_Call, Entry_Alts is a set
   --  of case statement alternatives associating entry index index values
   --  with the code for single entry bodies, and Family_Alts is a list of
   --  elsif clauses associating entry index ranges with the code for entry
   --  family bodies. Barrier_Name and Index_Name are the names of temporaries
   --  for the barrier vector and entry index, respectively. Lab_Decl is the
   --  declaration of the label used to jump out of entry bodies for requeue.

   function Build_Protected_Sub_Specification
     (N           : Node_Id;
      Prottyp     : Entity_Id;
      Unprotected : Boolean := False)
      return        Node_Id;
   --  Build specification for protected subprogram.

   function Build_Protected_Subprogram_Body
     (N         : Node_Id;
      Pid       : Node_Id;
      N_Op_Spec : Node_Id)
      return      Node_Id;
   --  This function is used to construct the protected version of a protected
   --  subprogram. It locks the associated protected object, then calls the
   --  unprotected version of the subprogram (for further details, see
   --  Build_Unprotected_Subprogram_Body).

   function Build_Service_Specification
     (Sloc    : Source_Ptr;
      PO_Name : Name_Id)
      return    Node_Id;
   --  Build specification for the "service entries" procedure associated
   --  with a protected type. This procedure takes the record implementing
   --  a protected object of the associated type and executes all of the
   --  entries waiting on open barriers. It returns a flag indicating
   --  whether one of the entries serviced was the one just made by the
   --  calling task (in other words, whether a call was executed immediately,
   --  without being queued).

   procedure Build_Simple_Entry_Call
     (N       : Node_Id;
      Concval : Node_Id;
      Ename   : Node_Id;
      Index   : Node_Id);

   function Build_Standard_Exception_Handlers
     (Sub  : Entity_Id;
      Pend : Entity_Id;
      Loc  : Source_Ptr)
      return List_Id;
   --  This routine constructs a list of exception handlers, one for
   --  each of the standard exceptions and one for others.  It calls
   --  the subprogram referenced by Sub in each of these handlers with
   --  the ID of the corresponding exception, and with Current_Exception
   --  for the others case.  This is an interim implementation of the
   --  mechanism for raising exceptions in other tasks (e.g. raising
   --  an exception that completes an accept statement in the caller) so
   --  that at least the standard exceptions work.  Full implementation will
   --  require Ada.Exceptions.

   function Build_Task_Proc_Specification (T : Entity_Id) return Node_Id;
   --  This routine constructs a specification for the procedure that we will
   --  build for the task body. The specification has the form
   --
   --    procedure tnameB (_Task : access tnameV);
   --
   --  where name is the character name taken from the task type entity that
   --  is passed as the argument to the procedure, and tnameV is the task
   --  value type that is associated with the task type.

   function Build_Unprotected_Subprogram_Body
     (N    : Node_Id;
      Pid  : Node_Id)
      return Node_Id;
   --  This routine constructs the unprotected version of a protected
   --  subprogram body, which is contains all of the code in the
   --  original, unexpanded body. This is the version of the protected
   --  subprogram that is called from all protected operations on the same
   --  object, including the protected version of the same subprogram.

   procedure Collect_Entry_Families
     (Loc          : Source_Ptr;
      Cdecls       : List_Id;
      Current_Node : in out Node_Id;
      Conctyp      : Entity_Id);
   --  For each entry family in a concurrent type, create an anonymous array
   --  type of the right size, and add a component to the corresponding_record.

   function Entry_Range_Expression
     (Sloc  : Source_Ptr;
      Ent   : Entity_Id;
      Ttyp  : Entity_Id)
      return  Node_Id;
   --  Returns the entry index range allocated to an entry family.
   --  Ttyp is the concurrent type.

   procedure Extract_Entry
     (N       : Node_Id;
      Concval : out Node_Id;
      Ename   : out Node_Id;
      Index   : out Node_Id);
   --  Given an entry call, returns the associated concurrent object,
   --  the entry name, and the entry family index.


   function Find_Task_Pragma (T : Node_Id; P : Name_Id) return Node_Id;
   --  Searches the task definition T for the first occurrence of the pragma
   --  whose name is given by P. The caller has ensured that the pragma is
   --  present in the task definition.

   function Concurrent_Ref (N : Node_Id) return Node_Id;
   --  Given the name of a concurrent object (task or protected object),
   --  or the name of an access to a concurrent object, this
   --  function returns an expression referencing the associated Task_Id
   --  or Protection object, respectively.
   --  Note that a special case is when the name is a reference to a task
   --  type name. This can only happen within a task body, and the meaning
   --  is to get the Task_Id for the currently executing task.

   ----------------------------------
   -- Add_Discriminal_Declarations --
   ----------------------------------

   procedure Add_Discriminal_Declarations
     (Decls : List_Id;
      Typ   : Entity_Id;
      Name  : Name_Id;
      Loc   : Source_Ptr)
   is
      D     : Entity_Id;

   begin

      if Has_Discriminants (Typ) then
         D := First_Discriminant (Typ);

         while Present (D) loop

            Prepend_To (Decls,
              Make_Object_Renaming_Declaration (Loc,
                Defining_Identifier => Discriminal (D),
                Subtype_Mark => New_Reference_To (Etype (D), Loc),
                Name =>
                  Make_Selected_Component (Loc,
                    Prefix        => Make_Identifier (Loc, Name),
                    Selector_Name => Make_Identifier (Loc, Chars (D)))));

            D := Next_Discriminant (D);
         end loop;
      end if;
   end Add_Discriminal_Declarations;

   ------------------------------
   -- Add_Private_Declarations --
   ------------------------------

   procedure Add_Private_Declarations
     (Decls : List_Id;
      Def   : Node_Id;
      Name  : Name_Id;
      Loc   : Source_Ptr)
   is
      Body_Ent : constant Entity_Id := Corresponding_Body (Parent (Def));
      P        : Node_Id;
      Pdef     : Entity_Id;

   begin
      pragma Assert (Nkind (Def) = N_Protected_Definition);

      if Present (Private_Declarations (Def)) then
         P := First (Private_Declarations (Def));

         while Present (P) loop
            if Nkind (P) = N_Component_Declaration then
               Pdef := Defining_Identifier (P);
               Prepend_To (Decls,
                 Make_Object_Renaming_Declaration (Loc,
                   Defining_Identifier => Prival (Pdef),
                   Subtype_Mark => New_Reference_To (Etype (Pdef), Loc),
                   Name =>
                     Make_Selected_Component (Loc,
                       Prefix        => Make_Identifier (Loc, Name),
                       Selector_Name => Make_Identifier (Loc, Chars (Pdef)))));
            end if;
            P := Next (P);
         end loop;
      end if;

      --  One more "prival" for the object itself.

      Prepend_To (Decls,
        Make_Object_Renaming_Declaration (Loc,
          Defining_Identifier => Object_Ref (Body_Ent),
          Subtype_Mark => New_Reference_To (RTE (RE_Protection), Loc),
          Name =>
            Make_Selected_Component (Loc,
              Prefix        => Make_Identifier (Loc, Name),
              Selector_Name => Make_Identifier (Loc, Name_uObject))));

   end Add_Private_Declarations;

   ----------------
   -- Array_Type --
   ----------------

   function Array_Type (E : Entity_Id; Trec : Node_Id) return Entity_Id is
      Arr : Entity_Id := First_Component (Trec);

   begin
      while Present (Arr) loop
         exit when Ekind (Arr) = E_Component
           and then Is_Array_Type (Etype (Arr))
           and then Chars (Arr) = Chars (E);

         Arr := Next_Component (Arr);
      end loop;

      return Arr;
   end Array_Type;

   -----------------------
   -- Build_Accept_Body --
   -----------------------

   function Build_Accept_Body
     (Stats : Node_Id;
      Loc   : Source_Ptr)
      return  Node_Id
   is
      Block : Node_Id;
      New_S : Node_Id;
      Hand  : Node_Id;
      Call  : Node_Id;

   begin
      --  Add the end of the statement sequence, Complete_Rendezvous is called.
      --  A label skipping the Complete_Rendezvous, and all other
      --  accept processing, has already been added for the expansion
      --  of requeue statements.

      Call := Build_Runtime_Call (Loc, RE_Complete_Rendezvous);
      Insert_Before (Last (Statements (Stats)), Call);
      Analyze (Call);

      --  If exception handlers are present, then append Complete_Rendezvous
      --  calls to the handlers, and construct the required outer block.

      if Present (Exception_Handlers (Stats)) then
         Hand := First (Exception_Handlers (Stats));

         while Present (Hand) loop
            Call := Build_Runtime_Call (Loc, RE_Complete_Rendezvous);
            Append (Call, Statements (Hand));
            Analyze (Call);
            Hand := Next (Hand);
         end loop;

         New_S :=
           Make_Handled_Sequence_Of_Statements (Loc,
             Statements => New_List (Block));
      else
         New_S := Stats;
      end if;

      --  At this stage we know that the new statement sequence does not
      --  have an exception handler part, so we supply one to call
      --  Exceptional_Complete_Rendezvous.

      Set_Exception_Handlers (New_S,
        Build_Standard_Exception_Handlers (
          RTE (RE_Exceptional_Complete_Rendezvous), Empty, Loc));
      Analyze_Exception_Handlers (Exception_Handlers (New_S));

      return New_S;

   end Build_Accept_Body;

   -----------------------------------
   -- Build_Activation_Chain_Entity --
   -----------------------------------

   procedure Build_Activation_Chain_Entity (N : Node_Id) is
      P     : Node_Id;
      B     : Node_Id;
      Decls : List_Id;

   begin
      --  Loop to find enclosing construct containing activation chain variable

      P := Parent (N);

      while Nkind (P) /= N_Subprogram_Body
        and then Nkind (P) /= N_Package_Declaration
        and then Nkind (P) /= N_Package_Body
        and then Nkind (P) /= N_Block_Statement
        and then Nkind (P) /= N_Task_Body
      loop
         P := Parent (P);
      end loop;

      --  If we are in a package body, the activation chain variable is
      --  allocated in the corresponding spec. First, we save the package
      --  body node because we enter the new entity in its Declarations list.

      B := P;

      if Nkind (P) = N_Package_Body then
         P := Get_Declaration_Node (Corresponding_Spec (P));
         Decls := Declarations (B);

      elsif Nkind (P) = N_Package_Declaration then
         Decls := Visible_Declarations (Specification (B));

      else
         Decls := Declarations (B);
      end if;

      --  If activation chain entity not already declared, declare it

      if No (Activation_Chain_Entity (P)) then
         Set_Activation_Chain_Entity
           (P, Make_Defining_Identifier (Sloc (P), Name_uChain));

         Prepend_To (Decls,
           Make_Object_Declaration (Sloc (P),
             Defining_Identifier => Activation_Chain_Entity (P),
             Aliased_Present => True,
             Object_Definition   =>
               New_Reference_To (RTE (RE_Activation_Chain), Sloc (P))));

         Analyze (First (Decls));
      end if;

   end Build_Activation_Chain_Entity;

   --------------------------
   -- Build_Call_With_Task --
   --------------------------

   function Build_Call_With_Task
     (N    : Node_Id;
      E    : Entity_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (N);

   begin
      return
        Make_Function_Call (Loc,
          Name => New_Reference_To (E, Loc),
          Parameter_Associations => New_List (Concurrent_Ref (N)));
   end Build_Call_With_Task;

   --------------------------------
   -- Build_Corresponding_Record --
   --------------------------------

   function Build_Corresponding_Record
    (N    : Node_Id;
     Ctyp : Entity_Id;
     Loc  : Source_Ptr)
     return Node_Id
   is
      Disc     : Entity_Id;
      Dlist    : List_Id;
      Rec_Ent  : constant Entity_Id :=
                   Make_Defining_Identifier
                     (Loc, New_External_Name (Chars (Ctyp), 'V'));
      Rec_Decl : Node_Id;
      Cdecls   : List_Id;

   begin
      Set_Corresponding_Record_Type (Ctyp, Rec_Ent);
      Set_Ekind (Rec_Ent, E_Record_Type);
      Set_Is_Concurrent_Record_Type (Rec_Ent, True);
      Set_Corresponding_Concurrent_Type (Rec_Ent, Ctyp);

      Cdecls := New_List;

      --  Make a copy of the discriminant specifications

      if Present (Discriminant_Specifications (N)) then
         Dlist := New_List;
         Disc := First (Discriminant_Specifications (N));

         while Present (Disc) loop
            Append_To (Dlist,
              Make_Discriminant_Specification (Loc,
                Defining_Identifier =>
                  New_Copy (Defining_Identifier (Disc)),
                Discriminant_Type =>
                  New_Occurrence_Of (Etype (Defining_Identifier (Disc)), Loc),
                Expression =>
                  New_Copy (Expression (Disc))));
            Disc := Next (Disc);
         end loop;

      else
         Dlist := No_List;
      end if;

      --  Now we can construct the record type declaration

      Rec_Decl :=
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Rec_Ent,
          Discriminant_Specifications => Dlist,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List =>
                Make_Component_List (Loc,
                  Component_Items => Cdecls)));

      return Rec_Decl;
   end Build_Corresponding_Record;

   ----------------------------------
   -- Build_Entry_Count_Expression --
   ----------------------------------

   function Build_Entry_Count_Expression
     (Concurrent_Type : Node_Id;
      Loc             : Source_Ptr)
      return            Node_Id
   is
      Eindx  : Nat;
      Ent    : Entity_Id;
      Ecount : Node_Id;

   begin
      Ent := First_Entity (Concurrent_Type);
      Eindx := 0;

      --  Count number of non-family entries

      while Present (Ent) loop
         if Ekind (Ent) = E_Entry then
            Eindx := Eindx + 1;
         end if;

         Ent := Next_Entity (Ent);
      end loop;

      Ecount := Make_Integer_Literal (Loc, UI_From_Int (Eindx));

      --  Loop through entry families building the addition nodes

      Ent := First_Entity (Concurrent_Type);
      while Present (Ent) loop
         if Ekind (Ent) = E_Entry_Family then
            if Ekind (Concurrent_Type) = E_Protected_Type then
               Ecount :=
                 Make_Op_Add (Loc,
                   Left_Opnd  => Ecount,
                   Right_Opnd =>
                     Make_Attribute_Reference (Loc,
                       Prefix => Make_Identifier (Loc,
                         New_External_Name (Chars (Ent), 'T')),
                       Attribute_Name => Name_Length));
            else
               Ecount :=
                 Make_Op_Add (Loc,
                   Left_Opnd  => Ecount,
                   Right_Opnd =>
                     Make_Attribute_Reference (Loc,
                       Prefix =>
                         Make_Selected_Component (Loc,
                           Prefix => Make_Identifier (Loc, Name_uInit),
                           Selector_Name =>
                             Make_Identifier (Loc, Chars (Ent))),

                       Attribute_Name => Name_Length));
            end if;
         end if;

         Ent := Next_Entity (Ent);
      end loop;
      return Ecount;
   end Build_Entry_Count_Expression;

   -----------------------------------
   -- Build_Entry_Service_Procedure --
   -----------------------------------

   function Build_Entry_Service_Procedure
     (Pid          : Node_Id;
      Barrier_Name : Name_Id;
      Index_Name   : Name_Id;
      Lab_Decl     : Node_Id;
      Barriers     : List_Id;
      Entry_Alts   : List_Id;
      Family_Alts  : List_Id)
      return         Node_Id
   is
      Loc          : constant Source_Ptr := Sloc (First (Barriers));
      Ptyp         : constant Node_Id    := Parent (Pid);
      Pdef         : constant Node_Id    := Protected_Definition (Ptyp);
      Ent          : Entity_Id;
      Param_Id     : Entity_Id;
      Cum_Name     : Name_Id;
      Op_Decls     : List_Id;
      Body_Id      : constant Entity_Id  := Corresponding_Body (Ptyp);
      Obj_Ptr      : Entity_Id;
      Others_Stats : List_Id;
      First_Family : Node_Id;
      Protnm       : constant Name_Id    := Chars (Pid);
      Serve_Stats  : List_Id;
      Pend_Exp     : Node_Id;
      Estats       : List_Id;

   begin
      Ent := First_Entity (Defining_Identifier (Ptyp));
      while Present (Ent)
        and then Ekind (Ent) /= E_Entry
        and then Ekind (Ent) /= E_Entry_Family
      loop
         Ent := Next_Entity (Ent);
      end loop;

      if Present (Ent) then
         Param_Id := Node (Last_Elmt (Accept_Address (Ent)));
      else
         Param_Id := Empty;
      end if;

      Cum_Name := New_Internal_Name ('C');

      --  Build the declaration list for the service entries procedure.

      Op_Decls := New_List (

         --  Bnn : Barrier_Vector (_object.Num_Entries);
         --  The barrier vector variable. This is a boolean array
         --  of all barrier values.

         Make_Object_Declaration (Loc,
            Defining_Identifier =>
              Make_Defining_Identifier (Loc, Barrier_Name),

            Object_Definition =>
              Make_Subtype_Indication (Loc,
                Subtype_Mark =>
                  New_Reference_To (RTE (RE_Barrier_Vector), Loc),

                Constraint =>
                  Make_Index_Or_Discriminant_Constraint (Loc,
                    Constraints => New_List (
                       Make_Range (Loc,
                         Make_Integer_Literal (Loc, Uint_1),
                         Make_Selected_Component (Loc,
                           Prefix => Make_Selected_Component (Loc,
                             Prefix => Make_Identifier (Loc, Name_uObject),
                             Selector_Name =>
                               Make_Identifier (Loc, Name_uObject)),

                           Selector_Name => Make_Identifier (Loc,
                             Chars (First_Discriminant (
                               RTE (RE_Protection)))))))))),

         --  A : System.Address;
         --  The parameter address variable. This variable contains
         --  the address of the entry call parameter record. This
         --  address is passed through the runtime from the
         --  caller of the entry.

         Make_Object_Declaration (Loc,
            Defining_Identifier => Param_Id,
            Object_Definition =>
              New_Reference_To (RTE (RE_Address), Loc)),

         --  Enn : Protected_Entry_Index;
         --  The index of the entry to be executed, as reported by the
         --  GNARL procedure Next_Entry_Call.

         Make_Object_Declaration (Loc,
            Defining_Identifier =>
              Make_Defining_Identifier (Loc, Index_Name),

            Object_Definition =>
              Make_Subtype_Indication (Loc,
                Subtype_Mark =>
                  New_Reference_To (RTE (RE_Protected_Entry_Index), Loc),

                Constraint =>
                  Make_Range_Constraint (Loc,
                    Make_Range (Loc,
                      New_Reference_To (RTE (RE_Null_Protected_Entry), Loc),
                      Make_Selected_Component (Loc,
                        Prefix => Make_Selected_Component (Loc,
                          Prefix => Make_Identifier (Loc, Name_uObject),
                          Selector_Name =>
                            Make_Identifier (Loc, Name_uObject)),

                        Selector_Name => Make_Identifier (Loc,
                          Chars (First_Discriminant (
                            RTE (RE_Protection))))))))),

         --  Pnn : Boolean;
         --  The actual Pending_Serviced parameter to Complete_Entry_Body,
         --  which is True if the call was completed without being queued.

         Make_Object_Declaration (Loc,
            Defining_Identifier =>
              Pending_Serviced (Body_Id),
            Object_Definition =>
              New_Reference_To (Standard_Boolean, Loc),
            Expression => New_Reference_To (Standard_False, Loc)),

         --  Cnn : Boolean := False;
         --  The "pending serviced" flag. This variable is used to
         --  accumulate the flags returned from the Complete_Entry_Body
         --  calls.

         Make_Object_Declaration (Loc,
            Defining_Identifier =>
              Make_Defining_Identifier (Loc, Cum_Name),
            Object_Definition =>
              New_Reference_To (Standard_Boolean, Loc),
            Expression => New_Reference_To (Standard_False, Loc)));

      Remove (Lab_Decl);
      Prepend_To (Op_Decls, Lab_Decl);

      --  <discriminant renamings>
      --  <private object renamings>
      --  Add discriminal and private renamings. These names have
      --  already been used to expand references to discriminants
      --  and private data.

      Add_Discriminal_Declarations (Op_Decls, Pid, Name_uObject, Loc);
      Add_Private_Declarations (Op_Decls, Pdef, Name_uObject, Loc);

      --  Prepend the declaration of _object. This must be first in the
      --  declaration list, since it is used by the discriminal and
      --  prival declarations.
      --
      --     type poVP is access poV;
      --     _object : poVP := poVP!O;

      Obj_Ptr := Make_Defining_Identifier (Loc,
        Chars =>
          New_External_Name
            (Chars (Corresponding_Record_Type (Pid)), 'P'));

      Prepend_To (Op_Decls,
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uObject),
          Object_Definition => New_Reference_To (Obj_Ptr, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Obj_Ptr, Loc),
              Expression => Make_Identifier (Loc, Name_uO))));

      Prepend_To (Op_Decls,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Obj_Ptr,
          Type_Definition => Make_Access_To_Object_Definition (Loc,
            Subtype_Indication =>
              New_Reference_To (Corresponding_Record_Type (Pid), Loc))));

      --  Add a case alternative for Null_Protected_Entry.
      --  Next_Entry_Call returns this when there are no entry
      --  calls eligible for execution.

      Prepend_To (Entry_Alts,
        Make_Case_Statement_Alternative (Loc,
          Discrete_Choices => New_List (
            New_Reference_To (RTE (RE_Null_Protected_Entry), Loc)),
          Statements => New_List (Make_Exit_Statement (Loc))));

      --  Add an others case alternative. This contains the if statement
      --  that processes entry family bodies, and a final catch-all
      --  for indexes which do not correspond to an entry.

      Others_Stats := New_List (
         Make_Raise_Statement (Loc,
           Name => New_Reference_To (Standard_Program_Error, Loc)));

      First_Family := Remove_Head (Family_Alts);

      if Present (First_Family) then
         Others_Stats := New_List (
            Make_If_Statement (Loc,
              Condition => Condition (First_Family),
              Then_Statements => Then_Statements (First_Family),
              Elsif_Parts => Family_Alts,
              Else_Statements => Others_Stats));
      end if;

      Append_To (Entry_Alts,
        Make_Case_Statement_Alternative (Loc,
          Discrete_Choices => New_List (Make_Others_Choice (Loc)),
          Statements => Others_Stats));

      Serve_Stats := New_List (

      --  begin
      --     Bnn (<entry index>) := <entry barrier expression>;
      --     Bnn (<entry family index range>) :=
      --         (others => <entry family barrier expression>;
      --  exception
      --  when others =>
      --     begin
      --        Broadcast_Program_Error (_object._object'Access);
      --        P := True;
      --     exception
      --     when Program_Error =>
      --       Unlock (_object._object'Access);
      --        Abort_Undefer;
      --     end;
      --  end;

      --  Assign barrier values to the barrier vector variable.

        Make_Block_Statement (Loc,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Barriers,

              Exception_Handlers => New_List (
                Make_Exception_Handler (Loc,
                  Exception_Choices =>
                    New_List (Make_Others_Choice (Loc)),

                  Statements =>  New_List (
                    Make_Block_Statement (Loc,
                      Handled_Statement_Sequence =>
                        Make_Handled_Sequence_Of_Statements (Loc,
                          Statements => New_List (

                            Make_Procedure_Call_Statement (Loc,
                              Name => New_Reference_To (
                                RTE (RE_Broadcast_Program_Error), Loc),

                              Parameter_Associations => New_List (
                                Make_Attribute_Reference (Loc,
                                  Prefix =>
                                    Make_Selected_Component (Loc,
                                      Prefix => Make_Identifier (Loc,
                                        Name_uObject),
                                      Selector_Name => Make_Identifier (Loc,
                                        Name_uObject)),
                                  Attribute_Name => Name_Access))),

                            Make_Assignment_Statement (Loc,
                              Name => Make_Identifier (Loc, Name_uP),
                              Expression =>
                                New_Reference_To (Standard_True, Loc))),

                          Exception_Handlers => New_List (
                            Make_Exception_Handler (Loc,
                              Exception_Choices => New_List (
                                New_Reference_To (
                                  Standard_Program_Error, Loc)),

                              Statements => New_List (
                                Make_Procedure_Call_Statement (Loc,
                                  Name => New_Reference_To (
                                    RTE (RE_Unlock), Loc),

                                  Parameter_Associations => New_List (
                                    Make_Attribute_Reference (Loc,
                                      Prefix =>
                                        Make_Selected_Component (Loc,
                                          Prefix => Make_Identifier (Loc,
                                            Name_uObject),
                                          Selector_Name =>
                                            Make_Identifier (Loc,
                                              Name_uObject)),
                                    Attribute_Name => Name_Access))),

                                Make_Procedure_Call_Statement (Loc,
                                  Name => New_Reference_To (
                                    RTE (RE_Abort_Undefer), Loc)),

                                Make_Raise_Statement (Loc))))))))))));

      --  Next_Entry_Call (_object, Bnn, A, Enn);

      Append_To (Serve_Stats,
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Next_Entry_Call), Loc),
          Parameter_Associations => New_List (
            Make_Attribute_Reference (Loc,
              Prefix =>
                Make_Selected_Component (Loc,
                  Prefix => Make_Identifier (Loc, Name_uObject),
                  Selector_Name =>
                    Make_Identifier (Loc, Name_uObject)),
              Attribute_Name => Name_Access),

            Make_Identifier (Loc, Barrier_Name),
            Make_Identifier (Loc, Chars (Param_Id)),
            Make_Identifier (Loc, Index_Name))));

      --  begin
      --     case Enn is
      --        when Null_Protected_Entry =>
      --           exit;
      --        when 1 =>
      --           <statement sequence for entry 1>
      --        when 2 =>
      --           <statement sequence for entry 2>
      --        ...
      --        when others => raise Program_Error;
      --     end case;
      --     <<Lmm>>
      --     Complete_Entry_Body (_object._object'Access, Pnn);
      --  exception
      --  when others =>
      --     Exceptional_Complete_Entry_Body (
      --      _object._object'Access, Pnn, Current_Exception);
      --  end;

      --  Add the case statement selecting the entry body to be
      --  executed.

      Pend_Exp := New_Reference_To (Pending_Serviced (Body_Id), Loc);
      Set_Assignment_OK (Pend_Exp);

      Estats := New_List (

         Make_Case_Statement (Loc,
           Expression => Make_Identifier (Loc, Index_Name),
           Alternatives => Entry_Alts),

         End_Of_Case (Corresponding_Body (Ptyp)),

         Make_Procedure_Call_Statement (Loc,
            Name =>
              New_Reference_To (
                RTE (RE_Complete_Entry_Body), Loc),

            Parameter_Associations => New_List (
              Make_Attribute_Reference (Loc,
                Prefix =>
                  Make_Selected_Component (Loc,
                    Prefix =>
                      Make_Identifier (Loc, Name_uObject),

                    Selector_Name =>
                      Make_Identifier (Loc, Name_uObject)),
                Attribute_Name => Name_Access),
              Pend_Exp)));

      Pend_Exp := New_Reference_To (Pending_Serviced (Body_Id), Loc);
      Set_Assignment_OK (Pend_Exp);

      Append_To (Serve_Stats,
        Make_Block_Statement (Loc,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Estats,
              Exception_Handlers =>
                Build_Standard_Exception_Handlers (
                  RTE (RE_Exceptional_Complete_Entry_Body),
                  Pending_Serviced (Body_Id),
                  Loc))));

      Append_To (Serve_Stats, End_Of_Body (Corresponding_Body (Ptyp)));

      --  Cnn := Cnn or Pnn;
      --  The calls to Complete_Entry_Body executed by the entry body
      --  case alternatives report whether the entry being completed
      --  was never queued (executed immediately). This assignment
      --  statement accumulates these flags, such that Cnn will be
      --  true if any Pnn flag is returned as true.

      Pend_Exp := New_Reference_To (Pending_Serviced (Body_Id), Loc);
      Set_Assignment_OK (Pend_Exp);

      Append_To (Serve_Stats,
        Make_Assignment_Statement (Loc,
          Name => Make_Identifier (Loc, Cum_Name),
          Expression =>
            Make_Op_Or (Loc,
              Left_Opnd => Make_Identifier (Loc, Cum_Name),
              Right_Opnd => Pend_Exp)));

      --  The barrier evaluations and case statement generated
      --  above must be repeated until Next_Entry_Call returns
      --  Null_Protected_Entry. Wrap an unconditional loop
      --  around the statements generated so far.

      Serve_Stats := New_List (
        Make_Loop_Statement (Loc, Statements => Serve_Stats));

      --  P := Cnn;
      --  Return the accumulated "pending serviced" flag to the caller.

      Append_To (Serve_Stats,
        Make_Assignment_Statement (Loc,
          Name => Make_Identifier (Loc, Name_uP),
          Expression => Make_Identifier (Loc, Cum_Name)));

      --  procedure ptypeS
      --    (O : System.Address;
      --     P : out Boolean)
      --  is
      --     ...
      --  end ptypeS;

      --  Make the service entries procedure proper.

      return Make_Subprogram_Body (Loc,
        Specification => Build_Service_Specification (Loc, Protnm),
        Declarations => Op_Decls,
        Handled_Statement_Sequence =>
          Make_Handled_Sequence_Of_Statements (Loc,
            Statements => Serve_Stats));

   end Build_Entry_Service_Procedure;

   -------------------------
   -- Build_Master_Entity --
   -------------------------

   procedure Build_Master_Entity (E : Entity_Id) is
      Loc  : constant Source_Ptr := Sloc (E);
      P    : Node_Id;
      Decl : Node_Id;

   begin
      --  Nothing to do if we already built a master entity for this scope

      if Has_Master_Entity (Scope (E)) then
         return;
      end if;

      --  Otherwise first build the master entity
      --    _Master : constant Master_Id := Current_Master;
      --  and insert it just before the current declaration

      Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uMaster),
          Constant_Present => True,
          Object_Definition => New_Reference_To (RTE (RE_Master_Id), Loc),
          Expression => New_Reference_To (RTE (RE_Current_Master), Loc));

      P := Parent (E);
      Insert_Before (P, Decl);
      Analyze (Decl);
      Set_Has_Master_Entity (Scope (E));

      --  Now mark the containing scope as a task master

      while Nkind (P) /= N_Compilation_Unit loop
         P := Parent (P);

         --  If we fall off the top, we are at the outer level, and the
         --  environment task is our effective master, so nothing to mark.

         if Nkind (P) = N_Task_Body
           or else Nkind (P) = N_Block_Statement
           or else Nkind (P) = N_Subprogram_Body
         then
            Set_Is_Task_Master (P, True);
            return;
         end if;
      end loop;
   end Build_Master_Entity;

   -------------------------------------
   -- Build_Protected_Subprogram_Body --
   -------------------------------------

   function Build_Protected_Subprogram_Body
     (N         : Node_Id;
      Pid       : Node_Id;
      N_Op_Spec : Node_Id)
      return      Node_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      Op_Spec     : Node_Id;
      Op_Def      : Entity_Id;
      Sub_Name    : Name_Id;
      P_Op_Spec   : Node_Id;
      Op_Decls    : List_Id;
      Uactuals    : List_Id;
      Pformal     : Node_Id;
      Return_Var  : Node_Id;
      Unprot_Call : Node_Id;
      Sub_Return  : Node_Id;
      Final_Sub   : Node_Id;
      Final_Stats : List_Id;
      Final_Pend  : Entity_Id;
      Final_Decls : List_Id;

   begin
      Op_Spec := Specification (N);
      Op_Def := Defining_Unit_Name (Op_Spec);

      --  Make an unprotected version of the subprogram for use
      --  within the same object, with a new name and an additional
      --  parameter representing the object.

      Sub_Name := Chars (Defining_Unit_Name (Specification (N)));

      P_Op_Spec :=
        Build_Protected_Sub_Specification (N,
          Pid, Unprotected => False);

      --  Build a list of the formal parameters of the protected
      --  version of the subprogram to use as the actual parameters
      --  of the unprotected version.

      Uactuals := New_List;
      Pformal := First (Parameter_Specifications (P_Op_Spec));

      while Present (Pformal) loop
         Append (
           Make_Identifier (Loc, Chars (Defining_Identifier (Pformal))),
           Uactuals);
         Pformal := Next (Pformal);
      end loop;

      --  Make a call to the unprotected version of the subprogram
      --  built above for use by the protected version built below.

      if Nkind (Op_Spec) = N_Function_Specification then
         Return_Var := Make_Defining_Identifier (Loc,
           New_External_Name (Name_Return, 'P'));

         Op_Decls := New_List (
           Make_Object_Declaration (Loc,
             Defining_Identifier => Return_Var,
             Object_Definition => New_Copy (Subtype_Mark (Op_Spec))));

         Unprot_Call := Make_Assignment_Statement (Loc,
           Name => New_Reference_To (Return_Var, Loc),
           Expression => Make_Function_Call (Loc,
             Name =>
               Make_Identifier (Loc,
                 Chars (Defining_Unit_Name (N_Op_Spec))),
             Parameter_Associations => Uactuals));

         Sub_Return := Make_Return_Statement (Loc,
           Expression => New_Reference_To (Return_Var, Loc));

      else
         Op_Decls := Empty_List;

         Unprot_Call := Make_Procedure_Call_Statement (Loc,
           Name =>
             Make_Identifier (Loc,
               Chars (Defining_Unit_Name (N_Op_Spec))),
           Parameter_Associations => Uactuals);

         Sub_Return := Make_Return_Statement (Loc);
      end if;

      --  Make a subprogram to perform finalization for the
      --  protected subprogram, unlocking the protected object
      --  parameter and undefering abortion.
      --  If this is a protected procedure, and the object contains
      --  entries, this also calls the entry service routine.

      Final_Stats := New_List (
        Make_Procedure_Call_Statement (Loc,
          Name =>
            New_Reference_To (
              RTE (RE_Unlock), Loc),

          Parameter_Associations => New_List (
            Make_Attribute_Reference (Loc,
              Prefix =>
                Make_Selected_Component (Loc,
                  Prefix =>
                    Make_Identifier (Loc, Name_uObject),
                  Selector_Name =>
                    Make_Identifier (Loc, Name_uObject)),
              Attribute_Name => Name_Access))),

        Make_Procedure_Call_Statement (Loc,
          Name =>
            New_Reference_To (
              RTE (RE_Abort_Undefer), Loc),
          Parameter_Associations => Empty_List));

      if Nkind (Op_Spec) = N_Procedure_Specification
        and then Has_Entries (Pid)
      then
         Final_Pend := Make_Defining_Identifier (Loc,
           New_Internal_Name ('P'));

         Final_Decls := New_List (
           Make_Object_Declaration (Loc,
              Defining_Identifier => Final_Pend,
              Object_Definition =>
                New_Reference_To (Standard_Boolean, Loc)));

         Prepend_To (Final_Stats,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (
               Defining_Unit_Name (
                 Service_Entries_Definition (Base_Type (Pid))),
               Loc),

             Parameter_Associations => New_List (
               Make_Attribute_Reference (Loc,
                 Attribute_Name => Name_Address,
                 Prefix => Make_Identifier (Loc, Name_uObject)),
               New_Reference_To (Final_Pend, Loc))));
      else
         Final_Decls := Empty_List;
      end if;

      Final_Sub :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Procedure_Specification (Loc,
              Defining_Unit_Name =>
                Make_Defining_Identifier (Loc,
                  Chars => New_External_Name (Sub_Name, 'F'))),

          Declarations => Final_Decls,

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Final_Stats));

      --  Make the protected subprogram body. This locks the protected
      --  object and calls the unprotected version of the subprogram.

      return
        Make_Subprogram_Body (Loc,
          Specification => P_Op_Spec,
          Declarations => Op_Decls,

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Make_Block_Statement (Loc,
                  Declarations => New_List (Final_Sub),
                  Handled_Statement_Sequence =>
                    Make_Handled_Sequence_Of_Statements (Loc,

                      Statements => New_List (
                        Make_Procedure_Call_Statement (Loc,
                          Name =>
                            New_Reference_To (RTE (RE_Abort_Defer), Loc),
                          Parameter_Associations => Empty_List),

                        Make_Procedure_Call_Statement (Loc,
                          Name =>
                            New_Reference_To (RTE (RE_Lock), Loc),
                          Parameter_Associations => New_List (
                            Make_Attribute_Reference (Loc,
                              Prefix =>
                                Make_Selected_Component (Loc,
                                  Prefix =>
                                    Make_Identifier (Loc, Name_uObject),
                                Selector_Name =>
                                    Make_Identifier (Loc, Name_uObject)),
                              Attribute_Name => Name_Access))),
                        Unprot_Call),

                      Identifier => New_Occurrence_Of (
                        Defining_Unit_Name (Specification (Final_Sub)),
                        Loc))),
                Sub_Return)));

   end Build_Protected_Subprogram_Body;

   -------------------------------------
   -- Build_Protected_Subprogram_Call --
   -------------------------------------

   function Build_Protected_Subprogram_Call
     (N        : Node_Id;
      Name     : Node_Id;
      Rec      : Node_Id;
      External : Boolean := True)
      return     Node_Id
   is
      Loc     : constant Source_Ptr := Sloc (N);
      Sub     : Entity_Id := Entity (Name);
      New_Sub : Node_Id;
      Params  : List_Id;
      Append  : Character;

   begin
      --  The following assumes that the protected version of the
      --  subprogram immediately follows the unprotected one in the entity
      --  chain for their common scope. There is currently no attribute to
      --  retrieve the protected version. Should there be one???

      if External then
         New_Sub := New_Occurrence_Of (
           Next_Entity (Corresponding_Unprotected (Sub)), Loc);
      else
         New_Sub :=
           New_Occurrence_Of (Corresponding_Unprotected (Sub), Loc);
      end if;

      if Present (Parameter_Associations (N)) then
         Params := New_List_Copy (Parameter_Associations (N));
      else
         Params := New_List;
      end if;

      Prepend (Rec, Params);

      if Ekind (Sub) = E_Procedure then
         return Make_Procedure_Call_Statement (Loc,
           Name => New_Sub,
           Parameter_Associations => Params);

      else
         pragma Assert (Ekind (Sub) = E_Function);
         return Make_Function_Call (Loc,
           Name => New_Sub,
           Parameter_Associations => Params);
      end if;
   end Build_Protected_Subprogram_Call;

   ---------------------------------------
   -- Build_Protected_Sub_Specification --
   ---------------------------------------

   function Build_Protected_Sub_Specification
     (N           : Node_Id;
      Prottyp     : Entity_Id;
      Unprotected : Boolean := False)
      return        Node_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      Spec        : constant Node_Id    := Specification (N);
      Ident       : constant Entity_Id  := Defining_Unit_Name (Spec);
      Nam         : constant Name_Id    := Chars (Ident);
      Formal      : Entity_Id;
      New_Plist   : List_Id;
      Append_Char : Character;
      New_Spec    : Node_Id;
      New_Param   : Node_Id;

   begin
      if Unprotected then
         Append_Char := 'N';
      else
         Append_Char := 'P';
      end if;

      New_Plist := New_List;
      Formal := First_Formal (Ident);

      while Present (Formal) loop
         New_Param :=
           Make_Parameter_Specification (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Chars (Formal)),
             In_Present => In_Present (Parent (Formal)),
             Out_Present => Out_Present (Parent (Formal)),
             Parameter_Type =>
               New_Reference_To (Etype (Formal), Loc));

         if Unprotected then
            Set_Protected_Formal (Formal, Defining_Identifier (New_Param));
         end if;

         Append (New_Param, New_Plist);
         Formal := Next_Formal (Formal);
      end loop;

      Prepend_To (New_Plist,
        Make_Parameter_Specification (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uObject),
          In_Present => True,
          Out_Present => Nkind (Spec) = N_Procedure_Specification,
          Parameter_Type =>
            New_Reference_To
              (Corresponding_Record_Type (Prottyp), Loc)));

      if Nkind (Spec) = N_Procedure_Specification then
         return
           Make_Procedure_Specification (Loc,
             Defining_Unit_Name =>
               Make_Defining_Identifier (Loc,
                 Chars => New_External_Name (Nam, Append_Char)),
             Parameter_Specifications => New_Plist);

      else
         New_Spec :=
           Make_Function_Specification (Loc,
             Defining_Unit_Name =>
               Make_Defining_Identifier (Loc,
                 Chars => New_External_Name (Nam, Append_Char)),
             Parameter_Specifications => New_Plist,
             Subtype_Mark => New_Copy (Subtype_Mark (Spec)));
         Set_Return_Present (Defining_Unit_Name (New_Spec));
         return New_Spec;
      end if;

   end Build_Protected_Sub_Specification;

   ---------------------------------
   -- Build_Service_Specification --
   ---------------------------------

   function Build_Service_Specification
     (Sloc    : Source_Ptr;
      PO_Name : Name_Id)
      return    Node_Id
   is
   begin
      return
        Make_Procedure_Specification (Sloc,
          Defining_Unit_Name =>
            Make_Defining_Identifier (Sloc,
              Chars => New_External_Name (PO_Name, 'S')),

          Parameter_Specifications => New_List (
            Make_Parameter_Specification (Sloc,
              Defining_Identifier =>
                Make_Defining_Identifier (Sloc, Name_uO),
              Parameter_Type =>
                New_Reference_To (RTE (RE_Address), Sloc)),

            Make_Parameter_Specification (Sloc,
              Defining_Identifier =>
                Make_Defining_Identifier (Sloc, Name_uP),
                Out_Present => True,
                  Parameter_Type =>
                    New_Reference_To (Standard_Boolean, Sloc))));

   end Build_Service_Specification;

   -----------------------------
   -- Build_Simple_Entry_Call --
   -----------------------------

   --  A task entry call is converted to a call to Call_Simple

   --    declare
   --       P : parms := (parm, parm, parm);
   --    begin
   --       Call_Simple (acceptor-task, entry-index, P'Address);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --    end;

   --  Here Pnn is an aggregate of the type constructed for the entry to hold
   --  the parameters, and the constructed aggregate value contains either the
   --  parameters or, in the case of non-elementary types, references to these
   --  parameters. Then the address of this aggregate is passed to the runtime
   --  routine, along with the task id value and the task entry index value.
   --  Pnn is only required if parameters are present.

   --  The assignments after the call are present only in the case of in-out
   --  or out parameters for elementary types, and are used to assign back the
   --  resulting values of such parameters.

   --  Note: the reason that we insert a block here is that in the context
   --  of selects, conditional entry calls etc. the entry call statement
   --  appears on its own, not as an element of a list.

   --  A protected entry call is converted to a Protected_Entry_Call
   --  followed by a call to the service entries procedure for the
   --  specified object (see Expand_N_Protected_Body). If the call is
   --  not serviced immediately, Wait_For_Completion is called to wait
   --  for service. Service_Entries uses Bnn to determine whether
   --  the call was cancelled and, if so, calls the appropriate
   --  entry service routine (this may not be for the original object,
   --  since the call may have been requeued).

   --  declare
   --     P   : E1_Params := (param, param, param);
   --     Pnn : Boolean;
   --     Bnn : Communications_Block;

   --  begin
   --     Abort_Defer;
   --     Lock (po._object'Access);
   --     Protected_Entry_Call (
   --       Object => po._object'Access,
   --       E => <entry index>;
   --       Uninterpreted_Data => P'Address;
   --       Mode => Simple_Call;
   --       Block => Bnn);
   --     ptypeS (po, Pnn);
   --     Unlock (po._object'Access);

   --     if not Pnn then
   --        Wait_For_Completion (Bnn);
   --        Service_Cancelled_Call (Bnn);
   --     end if;

   --     Abort_Undefer;
   --     Raise_Pending_Exception (Bnn);
   --  end;

   procedure Build_Simple_Entry_Call
     (N       : Node_Id;
      Concval : Node_Id;
      Ename   : Node_Id;
      Index   : Node_Id)
   is
   begin
      Expand_Call (N);

      --  Convert entry call to Call_Simple call

      declare
         Loc       : constant Source_Ptr := Sloc (N);
         Parms     : constant List_Id    := Parameter_Associations (N);
         Pdecl     : Node_Id;
         Xdecl     : Node_Id;
         Decls     : List_Id;
         Conctyp   : Node_Id;
         Ent       : Entity_Id;
         Ent_Acc   : Entity_Id;
         P         : Entity_Id;
         X         : Entity_Id;
         Plist     : List_Id;
         Parm1     : Node_Id;
         Parm2     : Node_Id;
         Parm3     : Node_Id;
         Call_End  : Node_Id;
         Actual    : Node_Id;
         Formal    : Node_Id;
         Stats     : List_Id;
         Pend_Name : Entity_Id;
         Comm_Name : Entity_Id;

      begin
         --  Simple entry and entry family cases merge here

         Ent     := Entity (Ename);
         Ent_Acc := Entry_Parameters_Type (Ent);
         Conctyp := Etype (Concval);

         --  If prefix is an access type, dereference to obtain the task type

         if Is_Access_Type (Conctyp) then
            Conctyp := Designated_Type (Conctyp);
         end if;

         --  Special case for protected subprogram calls.

         if Is_Protected_Type (Conctyp)
           and then Is_Subprogram (Entity (Ename))
         then
            Rewrite_Substitute_Tree (N,
              Build_Protected_Subprogram_Call
                (N, Ename, Convert_Concurrent (Concval, Conctyp)));
            Analyze (N);
            return;
         end if;

         --  First parameter is the Task_Id value from the task value or the
         --  Object from the protected object value, obtained by selecting
         --  the _Task_Id or _Object from the result of doing an unchecked
         --  conversion to convert the value to the corresponding record type.

         Parm1 := Concurrent_Ref (Concval);

         --  Second parameter is the entry index, computed by the routine
         --  provided for this purpose. The value of this expression is
         --  assigned to an intermediate variable to assure that any entry
         --  family index expressions are evaluated before the entry
         --  parameters.

         X := Make_Defining_Identifier (Loc, Name_uX);

         Xdecl :=
           Make_Object_Declaration (Loc,
             Defining_Identifier => X,
             Object_Definition =>
               New_Reference_To (RTE (RE_Task_Entry_Index), Loc),
             Expression =>
               Entry_Index_Expression (Loc, Entity (Ename), Index, Conctyp));

         Decls := New_List (Xdecl);
         Parm2 := New_Reference_To (X, Loc);

         --  The third parameter is the packaged parameters. If there are
         --  none, then it is just the null address, since nothing is passed

         if No (Parms) then
            Parm3 := New_Reference_To (RTE (RE_Null_Address), Loc);

         --  Case of parameters present, where third argument is the address
         --  of a packaged record containing the required parameter values.

         else
            --  First build a list of parameter values, which are the actual
            --  parameters in the case of elementary types and pointers to
            --  the parameters (actually references), for composite types.

            Plist := New_List;

            Actual := First_Actual (N);
            Formal := First_Formal (Ent);

            while Present (Actual) loop
               if Is_By_Copy_Type (Etype (Actual)) then
                  Append_To (Plist, New_Copy (Actual));
               else
                  Append_To (Plist,
                    Make_Reference (Loc, Prefix => New_Copy (Actual)));
               end if;

               Actual := Next_Actual (Actual);
               Formal := Next_Formal (Formal);
            end loop;

            --  Now build the declaration of parameters initialized with the
            --  aggregate containing this constructed parameter list.

            P := Make_Defining_Identifier (Loc, Name_uP);

            Pdecl :=
              Make_Object_Declaration (Loc,
                Defining_Identifier => P,
                Object_Definition =>
                  New_Reference_To (Designated_Type (Ent_Acc), Loc),
                Expression =>
                  Make_Aggregate (Loc, Expressions => Plist));

            Parm3 :=
              Make_Attribute_Reference (Loc,
                Attribute_Name => Name_Address,
                Prefix => New_Reference_To (P, Loc));

            Append (Pdecl, Decls);
         end if;

         --  Now we can create the call

         if Is_Protected_Type (Conctyp) then

            --  Change they type of the index declaration.

            Set_Object_Definition (Xdecl,
              New_Reference_To (RTE (RE_Protected_Entry_Index), Loc));

            --  Some additional declarations for protected entry calls.

            if No (Decls) then
               Decls := New_List;
            end if;

            --  Pnn : Boolean;

            Pend_Name :=
              Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

            Append_To (Decls,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Pend_Name,
                Object_Definition =>
                  New_Reference_To (Standard_Boolean, Loc)));

            --  Bnn : Communications_Block;

            Comm_Name :=
              Make_Defining_Identifier (Loc, New_Internal_Name ('B'));

            Append_To (Decls,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Comm_Name,
                Object_Definition =>
                  New_Reference_To (RTE (RE_Communication_Block), Loc)));

            --  Some additional statements for protected entry calls.

            --  Abort_Defer;

            Stats := New_List (
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Abort_Defer), Loc)));

            --  Lock (po._object'Access);

            Append_To (Stats,
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Lock), Loc),
                Parameter_Associations => New_List (
                  Make_Attribute_Reference (Loc,
                    Attribute_Name => Name_Access,
                    Prefix => Concurrent_Ref (Concval)))));

            --     Protected_Entry_Call (
            --       Object => po._object'Access,
            --       E => <entry index>;
            --       Uninterpreted_Data => P'Address;
            --       Mode => Simple_Call;
            --       Block => Bnn);

            Append_To (Stats,
              Make_Procedure_Call_Statement (Loc,
                Name =>
                  New_Reference_To (RTE (RE_Protected_Entry_Call), Loc),

                Parameter_Associations => New_List (
                  Make_Attribute_Reference (Loc,
                    Attribute_Name => Name_Access,
                    Prefix => Parm1),
                  Parm2,
                  Parm3,
                  New_Reference_To (RTE (RE_Simple_Call), Loc),
                  New_Occurrence_Of (Comm_Name, Loc))));

            --  ptypeS (po'Address, Pnn);

            Append_To (Stats,
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (
                  Defining_Unit_Name (
                    Service_Entries_Definition (Base_Type (Conctyp))),
                  Loc),

                Parameter_Associations => New_List (
                  Make_Attribute_Reference (Loc,
                    Attribute_Name => Name_Address,
                    Prefix => New_Copy_Tree (Concval)),
                  New_Occurrence_Of (Pend_Name, Loc))));

            --  Unlock (po._object'Access);

            Append_To (Stats,
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Unlock), Loc),
                Parameter_Associations => New_List (
                  Make_Attribute_Reference (Loc,
                    Attribute_Name => Name_Access,
                    Prefix => Concurrent_Ref (Concval)))));

            Append_To (Stats,

            --  if not Pnn then

              Make_If_Statement (Loc,
                Condition =>
                  Make_Op_Not (Loc,
                    Right_Opnd => New_Occurrence_Of (Pend_Name, Loc)),

                Then_Statements => New_List (

                  --  Wait_For_Completion (Bnn);

                  Make_Procedure_Call_Statement (Loc,
                    Name =>
                      New_Reference_To (RTE (RE_Wait_For_Completion), Loc),
                    Parameter_Associations => New_List (
                      New_Occurrence_Of (Comm_Name, Loc))),

                  --  Service_Cancelled_Call (Bnn);

                  Make_Procedure_Call_Statement (Loc,
                    Name => New_Reference_To (
                      RTE (RE_Service_Cancelled_Call), Loc),
                    Parameter_Associations => New_List (
                      New_Occurrence_Of (Comm_Name, Loc))))));

            --  Abort_Undefer;

            Append_To (Stats,
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Abort_Undefer), Loc)));

            --  Raise_Pending_Exception (Bnn);

            Call_End :=
              Make_Procedure_Call_Statement (Loc,
                Name =>
                  New_Reference_To (RTE (RE_Raise_Pending_Exception), Loc),
                Parameter_Associations => New_List (
                  New_Occurrence_Of (Comm_Name, Loc)));

            Append_To (Stats, Call_End);

         else
            Call_End :=
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Call_Simple), Loc),
                Parameter_Associations => New_List (Parm1, Parm2, Parm3));
            Stats := New_List (Call_End);

         end if;

         --  If there are out or in/out parameters by copy
         --  add assignment statements for the result values.

         if Present (Parms) then
            Actual := First_Actual (N);
            Formal := First_Formal (Ent);

            while Present (Actual) loop
               if Is_By_Copy_Type (Etype (Actual))
                 and then Ekind (Formal) /= E_In_Parameter
               then
                  Insert_After (Call_End,
                    Make_Assignment_Statement (Loc,
                      Name => New_Copy (Actual),
                      Expression =>
                        Make_Selected_Component (Loc,
                          Prefix => New_Reference_To (P, Loc),
                          Selector_Name =>
                            Make_Identifier (Loc, Chars (Formal)))));
               end if;

               Actual := Next_Actual (Actual);
               Formal := Next_Formal (Formal);
            end loop;
         end if;

         --  Finally, create block and analyze it

         Rewrite_Substitute_Tree (N,
           Make_Block_Statement (Loc,
             Declarations => Decls,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => Stats)));

         Analyze (N);
      end;

   end Build_Simple_Entry_Call;

   ---------------------------------------
   -- Build_Standard_Exception_Handlers --
   ---------------------------------------

   function Build_Standard_Exception_Handlers
     (Sub  : Entity_Id;
      Pend : Entity_Id;
      Loc  : Source_Ptr)
      return List_Id
   is
      type Ex_Id is
        array (S_Exceptions range S_Numeric_Error .. S_Tasking_Error) of
                                                                    Node_Id;

      Ids : constant Ex_Id := (RTE (RE_Numeric_Error_ID),
                               RTE (RE_Program_Error_ID),
                               RTE (RE_Storage_Error_ID),
                               RTE (RE_Tasking_Error_ID));
      Handlers : List_Id := New_List;
      Params   : List_Id;
      Pend_Exp : Node_Id;

   begin
      for Id in Ids'Range loop
         Params := New_List;

         if Present (Pend) then
            Append_To (Params,
              Make_Attribute_Reference (Loc,
                Prefix =>
                  Make_Selected_Component (Loc,
                    Prefix =>
                      Make_Identifier (Loc, Name_uObject),
                    Selector_Name =>
                      Make_Identifier (Loc, Name_uObject)),
                  Attribute_Name => Name_Access));

            Pend_Exp := New_Reference_To (Pend, Loc);
            Set_Assignment_OK (Pend_Exp);
            Append_To (Params, Pend_Exp);
         end if;

         Append_To (Params, New_Reference_To (Ids (Id), Loc));
         Append_To (Handlers,
           Make_Exception_Handler (Loc,
             Exception_Choices => New_List (New_Reference_To (SE (Id), Loc)),
             Statements => New_List (
               Make_Procedure_Call_Statement (Loc,
                 Name => New_Reference_To (Sub, Loc),
                 Parameter_Associations => Params))));
      end loop;

      Params := New_List;

      if Present (Pend) then
         Append_To (Params,
           Make_Attribute_Reference (Loc,
             Prefix =>
               Make_Selected_Component (Loc,
                 Prefix =>
                   Make_Identifier (Loc, Name_uObject),
                 Selector_Name =>
                   Make_Identifier (Loc, Name_uObject)),
               Attribute_Name => Name_Access));
         Pend_Exp := New_Reference_To (Pend, Loc);
         Set_Assignment_OK (Pend_Exp);
         Append_To (Params, Pend_Exp);
      end if;

      Append_To (Params, Make_Function_Call (Loc,
        Name => New_Reference_To (RTE (RE_Current_Exception), Loc)));

      Append_To (Handlers,
        Make_Exception_Handler (Loc,
          Exception_Choices => New_List (Make_Others_Choice (Loc)),
          Statements => New_List (
            Make_Procedure_Call_Statement (Loc,
              Name => New_Reference_To (Sub, Loc),
              Parameter_Associations => Params))));

      return Handlers;
   end Build_Standard_Exception_Handlers;

   --------------------------------
   -- Build_Task_Activation_Call --
   --------------------------------

   procedure Build_Task_Activation_Call (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Chain : Entity_Id;
      Call  : Node_Id;
      P     : Node_Id;

   begin
      --  Get the activation chain entity. Except in the case of a package
      --  body, this is in the node that was passed. For a package body,we
      --  have to find the corresponding package declaration node.

      if Nkind (N) = N_Package_Body then
         P := Corresponding_Spec (N);

         loop
            P := Parent (P);
            exit when Nkind (P) = N_Package_Declaration;
         end loop;

         Chain := Activation_Chain_Entity (P);

      else
         Chain := Activation_Chain_Entity (N);
      end if;

      if Present (Chain) then
         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Activate_Tasks), Loc),
             Parameter_Associations =>
               New_List (Make_Attribute_Reference (Loc,
                 Prefix => New_Occurrence_Of (Chain, Loc),
                 Attribute_Name => Name_Access)));

         if Nkind (N) = N_Package_Declaration then
            if Present (Corresponding_Body (N)) then
               null;

            elsif Present (Private_Declarations (Specification (N))) then
               Append (Call, Private_Declarations (Specification (N)));

            else
               Append (Call, Visible_Declarations (Specification (N)));
            end if;

         elsif Present (Handled_Statement_Sequence (N)) then
            Prepend (Call, Statements (Handled_Statement_Sequence (N)));

         else
            Set_Handled_Statement_Sequence (N,
               Make_Handled_Sequence_Of_Statements (Loc,
                  Statements => New_List (Call)));
         end if;

         Analyze (Call);
      end if;

   end Build_Task_Activation_Call;

   -------------------------------
   -- Build_Task_Allocate_Block --
   -------------------------------

   procedure Build_Task_Allocate_Block
     (Actions : List_Id;
      N       : Node_Id;
      Args    : List_Id)
   is
      T      : constant Entity_Id  := Entity (Expression (N));
      Init   : constant Entity_Id  := Base_Init_Proc (T);
      Loc    : constant Source_Ptr := Sloc (N);
      Blkent : Entity_Id;
      Block  : Node_Id;

   begin
      Blkent := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

      Block :=
        Make_Block_Statement (Loc,
          Identifier => New_Reference_To (Blkent, Loc),
          Declarations => New_List (

            --  _Chain  : Activation_Chain;

            Make_Object_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_uChain),
              Aliased_Present => True,
              Object_Definition   =>
                New_Reference_To (RTE (RE_Activation_Chain), Loc)),

            --  procedure _Expunge is
            --  begin
            --     Expunge_Unactivated_Tasks (_Chain);
            --  end;

            Make_Subprogram_Body (Loc,
              Specification =>
                Make_Procedure_Specification (Loc,
                  Defining_Unit_Name =>
                    Make_Defining_Identifier (Loc, Name_uExpunge)),

              Declarations => Empty_List,

              Handled_Statement_Sequence =>
                Make_Handled_Sequence_Of_Statements (Loc,
                  Statements => New_List (
                    Make_Procedure_Call_Statement (Loc,
                      Name =>
                        New_Reference_To (
                          RTE (RE_Expunge_Unactivated_Tasks), Loc),
                      Parameter_Associations => New_List (
                        Make_Identifier (Loc, Name_uChain))))))),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,

              Statements => New_List (

               --  Init (Args);

                Make_Procedure_Call_Statement (Loc,
                  Name => New_Reference_To (Init, Loc),
                  Parameter_Associations => Args),

               --  Activate_Tasks (_Chain);

                Make_Procedure_Call_Statement (Loc,
                  Name => New_Reference_To (RTE (RE_Activate_Tasks), Loc),
                  Parameter_Associations => New_List (
                    Make_Attribute_Reference (Loc,
                      Prefix => Make_Identifier (Loc, Name_uChain),
                      Attribute_Name => Name_Access)))),

              Identifier => Make_Identifier (Loc, Name_uExpunge)),

          Has_Created_Identifier => True);

      Append_To (Actions,
        Make_Implicit_Label_Declaration (Loc,
          Defining_Identifier => Blkent,
          Label => Block));

      Append_To (Actions, Block);

   end Build_Task_Allocate_Block;

   -----------------------------------
   -- Build_Task_Proc_Specification --
   -----------------------------------

   function Build_Task_Proc_Specification (T : Entity_Id) return Node_Id is
      Loc : constant Source_Ptr := Sloc (T);
      Nam : constant Name_Id    := Chars (T);
      Ent : Entity_Id;

   begin
      Ent :=
        Make_Defining_Identifier (Loc,
          Chars => New_External_Name (Nam, 'B'));
      Set_Is_Internal (Ent);

      return
        Make_Procedure_Specification (Loc,
          Defining_Unit_Name       => Ent,
          Parameter_Specifications =>
            New_List (
              Make_Parameter_Specification (Loc,
                Defining_Identifier =>
                  Make_Defining_Identifier (Loc, Name_uTask),
                Parameter_Type =>
                  Make_Access_Definition (Loc,
                    Subtype_Mark =>
                      New_Reference_To
                        (Corresponding_Record_Type (T), Loc)))));

   end Build_Task_Proc_Specification;

   ---------------------------------------
   -- Build_Unprotected_Subprogram_Body --
   ---------------------------------------

   function Build_Unprotected_Subprogram_Body
     (N    : Node_Id;
      Pid  : Node_Id)
      return Node_Id
   is
      Loc       : constant Source_Ptr := Sloc (N);
      Sub_Name  : Name_Id;
      N_Op_Spec : Node_Id;
      Op_Decls  : List_Id;

   begin
      --  Make an unprotected version of the subprogram for use
      --  within the same object, with a new name and an additional
      --  parameter representing the object.

      Op_Decls := New_List_Copy (Declarations (N));
      Sub_Name := Chars (Defining_Unit_Name (Specification (N)));

      N_Op_Spec :=
        Build_Protected_Sub_Specification
          (N, Pid, Unprotected => True);

      return
        Make_Subprogram_Body (Loc,
          Specification => N_Op_Spec,
          Declarations => Op_Decls,
          Handled_Statement_Sequence =>
            Handled_Statement_Sequence (N));

   end Build_Unprotected_Subprogram_Body;

   ----------------------------
   -- Collect_Entry_Families --
   ----------------------------

   procedure Collect_Entry_Families
     (Loc          : Source_Ptr;
      Cdecls       : List_Id;
      Current_Node : in out Node_Id;
      Conctyp      : Entity_Id)
   is
      Efam      : Entity_Id;
      Efam_Decl : Node_Id;
      Efam_Type : Entity_Id;

   begin
      Efam := First_Entity (Conctyp);

      while Present (Efam) loop

         if Ekind (Efam) = E_Entry_Family then
            Efam_Type :=
              Make_Defining_Identifier (Loc,
                Chars => New_External_Name (Chars (Efam), 'T'));

            Efam_Decl :=
              Make_Full_Type_Declaration (Loc,
                Defining_Identifier => Efam_Type,
                Type_Definition =>
                  Make_Constrained_Array_Definition (Loc,
                    Discrete_Subtype_Definitions => (New_List (
                      New_Copy (Discrete_Subtype_Definition (Parent (Efam))))),

                    Subtype_Indication =>
                      New_Reference_To (Standard_Character, Loc)));

            Insert_After (Current_Node, Efam_Decl);
            Current_Node := Efam_Decl;
            Analyze (Efam_Decl);

            Append_To (Cdecls,
              Make_Component_Declaration (Loc,
                Defining_Identifier =>
                  Make_Defining_Identifier (Loc, Chars (Efam)),
                Subtype_Indication => New_Occurrence_Of (Efam_Type, Loc)));
         end if;

         Efam := Next_Entity (Efam);
      end loop;
   end Collect_Entry_Families;

   --------------------
   -- Concurrent_Ref --
   --------------------

   --  The expression returned for a reference to a concurrent
   --  object has the form:

   --    taskV!(name)._Task_Id

   --  for a task, and

   --    objectV!(name)._Object

   --  for a protected object.

   --  For the case of an access to a concurrent object,
   --  there is an extra explicit dereference:

   --    taskV!(name.all)._Task_Id
   --    objectV!(name.all)._Object

   --  here taskV and objectV are the types for the associated records, which
   --  contain the required _Task_Id and _Object fields for tasks and
   --  protected objects, respectively.

   --  For the case of a task type name, the expression is

   --    Self;

   --  i.e. a call to the Self function which returns precisely this Task_Id

   --  For the case of a protected type name, the expression is

   --    objectR

   --  which is a renaming of the _object field of the current object
   --  object record, passed into protected operations as a parameter.


   function Concurrent_Ref (N : Node_Id) return Node_Id is
      Loc  : constant Source_Ptr := Sloc (N);
      Ntyp : constant Entity_Id  := Etype (N);
      Dtyp : Entity_Id;
      Sel  : Name_Id;

   begin
      if Is_Access_Type (Ntyp) then
         Dtyp := Designated_Type (Ntyp);

         if Is_Protected_Type (Dtyp) then
            Sel := Name_uObject;
         else
            Sel := Name_uTask_Id;
         end if;

         return
           Make_Selected_Component (Loc,
             Prefix =>
               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (
                   Corresponding_Record_Type (Dtyp), Loc),
                 Expression => Make_Explicit_Dereference (Loc, N)),
             Selector_Name => Make_Identifier (Loc, Sel));

      elsif Is_Entity_Name (N)
        and then Is_Concurrent_Type (Entity (N))
      then
         if Is_Task_Type (Entity (N)) then
            return
              Make_Function_Call (Loc,
                Name => New_Reference_To (RTE (RE_Self), Loc));
         else
            pragma Assert (Is_Protected_Type (Entity (N)));
            return
              New_Reference_To (
                Object_Ref (Corresponding_Body (Parent (Base_Type (Ntyp)))),
                Loc);
         end if;

      else
         pragma Assert (Is_Concurrent_Type (Ntyp));

         if Is_Protected_Type (Ntyp) then
            Sel := Name_uObject;
         else
            Sel := Name_uTask_Id;
         end if;

         return
           Make_Selected_Component (Loc,
             Prefix =>
               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark =>
                   New_Reference_To (Corresponding_Record_Type (Ntyp), Loc),
                 Expression => New_Copy_Tree (N)),
             Selector_Name => Make_Identifier (Loc, Sel));
      end if;
   end Concurrent_Ref;

   ------------------------
   -- Convert_Concurrent --
   ------------------------

   function Convert_Concurrent
     (N    : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (N);

   begin
      if not Is_Concurrent_Type (Typ) then
         return N;
      else
         return
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (Corresponding_Record_Type (Typ), Loc),
               Expression => New_Copy (N));
      end if;
   end Convert_Concurrent;

   ----------------------------
   -- Entry_Index_Expression --
   ----------------------------

   function Entry_Index_Expression
     (Sloc  : Source_Ptr;
      Ent   : Entity_Id;
      Index : Node_Id;
      Ttyp  : Entity_Id)
      return  Node_Id
   is
      Expr : Node_Id;
      Num  : Node_Id;
      Prev : Entity_Id;
      S    : Node_Id;
      Trec : Node_Id := Corresponding_Record_Type (Ttyp);

   begin
      --  The queues of entries and entry families appear in  textual
      --  order in the associated record. The entry index is computed as
      --  the sum of the number of queues for all entries that precede the
      --  designated one, to which is added the index expression, if this
      --  expression denotes a member of a family.

      --  The following is a place holder for the count of simple entries.

      Num := Make_Integer_Literal (Sloc, Uint_1);

      --  We construct an expression which is a series of addition
      --  operations. The first operand is the number of single entries that
      --  precede this one, the second operand is the index value relative
      --  to the start of the referenced family, and the remaining operands
      --  are the lengths of the entry families that precede this entry, i.e.
      --  the constructed expression is:

      --    number_simple_entries +
      --      (s'pos (index-value) - s'pos (family'first)) + 1 +
      --      family'length + ...

      --  where index-value is the given index value, and s is the index
      --  subtype (we have to use pos because the subtype might be an
      --  enumeration type preventing direct subtraction).
      --  Note that the task entry array is one-indexed.

      if Present (Index) then
         S := Etype (Discrete_Subtype_Definition (Declaration_Node (Ent)));

         Expr :=
           Make_Op_Add (Sloc,
             Left_Opnd  => Num,

             Right_Opnd =>
               Make_Op_Subtract (Sloc,
                 Left_Opnd =>
                   Make_Attribute_Reference (Sloc,
                     Attribute_Name => Name_Pos,
                     Prefix => New_Reference_To (S, Sloc),
                     Expressions => New_List (New_Copy (Index))),

                 Right_Opnd =>
                   Make_Attribute_Reference (Sloc,
                     Attribute_Name => Name_Pos,
                     Prefix => New_Reference_To (S, Sloc),
                     Expressions => New_List (
                       Make_Attribute_Reference (Sloc,
                         Prefix => New_Reference_To (S, Sloc),
                         Attribute_Name => Name_First)))));
      else
         Expr := Num;
      end if;

      --  Now add lengths of preceding entries and entry families.

      Prev := First_Entity (Ttyp);

      while Chars (Prev) /= Chars (Ent)
        or else (Ekind (Prev) /= Ekind (Ent))
        or else not Sem_Ch6.Type_Conformant (Ent, Prev)
      loop
         if Ekind (Prev) = E_Entry then
            Set_Intval (Num, Intval (Num) + 1);

         elsif Ekind (Prev) = E_Entry_Family then
            Expr :=
              Make_Op_Add (Sloc,
              Left_Opnd  => Expr,
              Right_Opnd =>
                Make_Attribute_Reference (Sloc,
                  Attribute_Name => Name_Length,
                  Prefix => New_Reference_To (Array_Type (Prev, Trec), Sloc)));

         --  Other components are anonymous types to be ignored.

         else
            null;
         end if;

         Prev := Next_Entity (Prev);
      end loop;

      return Expr;
   end Entry_Index_Expression;

   ----------------------------
   -- Entry_Range_Expression --
   ----------------------------

   function Entry_Range_Expression
     (Sloc  : Source_Ptr;
      Ent   : Entity_Id;
      Ttyp  : Entity_Id)
      return  Node_Id
   is
      Trec   : constant Node_Id := Corresponding_Record_Type (Ttyp);
      Right  : Node_Id;
      Left   : Node_Id;
      Scount : Node_Id;
      Fcount : Node_Id;
      Prev   : Entity_Id;

   begin
      --  The following is a place holder for the count of simple entries.

      Scount := Make_Integer_Literal (Sloc, Uint_1);
      Fcount := Make_Integer_Literal (Sloc, Uint_0);

      Prev := First_Entity (Ttyp);

      while Chars (Prev) /= Chars (Ent) loop

         if Ekind (Prev) = E_Entry then
            Set_Intval (Scount, Intval (Scount) + 1);

         elsif Ekind (Prev) = E_Entry_Family then
            Fcount :=
              Make_Op_Add (Sloc,
              Left_Opnd  => Fcount,
              Right_Opnd =>
                Make_Attribute_Reference (Sloc,
                  Attribute_Name => Name_Length,
                  Prefix => New_Reference_To (Array_Type (Prev, Trec), Sloc)));

         --  Other components are anonymous types to be ignored.

         else
            null;
         end if;

         Prev := Next_Entity (Prev);
      end loop;

      Left :=
        Make_Op_Add (Sloc,
          Left_Opnd => Scount,
          Right_Opnd => Fcount);

      Right :=
        Make_Op_Add (Sloc,
          Left_Opnd => New_Copy_Tree (Left),
          Right_Opnd =>
            Make_Op_Subtract (Sloc,
              Left_Opnd => Make_Attribute_Reference (Sloc,
                Attribute_Name => Name_Length,
                Prefix => New_Reference_To (Array_Type (Prev, Trec), Sloc)),
              Right_Opnd => Make_Integer_Literal (Sloc, Uint_1)));

      return
        Make_Range (Sloc,
          Low_Bound => Left,
          High_Bound => Right);

   end Entry_Range_Expression;

   ---------------------------
   -- Establish_Task_Master --
   ---------------------------

   procedure Establish_Task_Master (N : Node_Id) is
      Call : Node_Id;

   begin
      Call := Build_Runtime_Call (Sloc (N), RE_Enter_Master);
      Prepend_To (Declarations (N), Call);
      Analyze (Call);
   end Establish_Task_Master;

   --------------------------------
   -- Expand_Accept_Declarations --
   --------------------------------

   --  Part of the expansion of an accept statement involves the creation of
   --  a declaration that can be referenced from the statement sequence of
   --  the accept:

   --    Ann : Address;

   --  This declaration is inserted immediately before the accept statement
   --  and it is important that it be inserted before the statements of the
   --  statement sequence are analyzed. Thus it would be too late to create
   --  this declaration in the Expand_N_Accept_Statement routine, which is
   --  why there is a separate procedure to be called directly from Sem_Ch9.

   --  It is used to hold the address of the record containing the parameters
   --  (see Expand_N_Entry_Call for more details on how this record is built).
   --  References to the parameters do an unchecked conversion of this address
   --  to a pointer to the required record type, and then access the field that
   --  holds the value of the required parameter. The entity for the address
   --  variable is held as the top stack element (i.e. the last element) of the
   --  Accept_Address stack in the corresponding entry entity, and this element
   --  must be set in place  before the statements are processed.

   --  The above description applies to the case of a stand alone accept
   --  statement, i.e. one not appearing as part of a select alternative.

   --  For the case of an accept that appears as part of a select alternative
   --  of a selective accept, we must still create the declaration right away,
   --  since Ann is needed immediately, but there is an important difference:

   --    The declaration is inserted before the selective accept, not before
   --    the accept statement (which is not part of a list anyway, and so would
   --    not accommodate inserted declarations)

   --    We only need one address variable for the entire selective accept. So
   --    the Ann declaration is created only for the first accept alternative,
   --    and subsequent accept alternatives reference the same Ann variable.

   --  We can distinguish the two cases by seeing whether the accept statement
   --  is part of a list. If not, then it must be in an accept alternative.

   --  To expand the requeue statement, a label is provided at the end of
   --  the accept statement or alternative of which it is a part, so that
   --  the statement can be skipped after the requeue is complete.
   --  This label is created here rather than during the expansion of the
   --  accept statement, because it will be needed by any requeue
   --  statements within the accept, which are expanded before the
   --  accept.

   procedure Expand_Accept_Declarations (N : Node_Id; Ent : Entity_Id) is
      Loc    : constant Source_Ptr := Sloc (N);
      Ttyp   : constant Entity_Id  := Scope (Ent);
      Ann    : Entity_Id;
      Adecl  : Node_Id;
      Lab_Id : Node_Id;
      Lab    : Node_Id;
      Ldecl  : Node_Id;

   begin
      if Expander_Active then

         --  Create and declare a label to be placed at the end of the
         --  accept statement. This is used to allow requeues to skip
         --  the remainder of entry processing.

         if Present (Handled_Statement_Sequence (N)) then
            Lab_Id := Make_Identifier (Loc, New_Internal_Name ('L'));
            Set_Entity (Lab_Id,
              Make_Defining_Identifier (Loc, Chars (Lab_Id)));
            Lab := Make_Label (Loc, Lab_Id);
            Ldecl :=
              Make_Implicit_Label_Declaration (Loc,
                Defining_Identifier  => Entity (Lab_Id),
                Label => Lab);
            Append (Lab, Statements (Handled_Statement_Sequence (N)));
         end if;

         --  Case of stand alone accept statement

         if Is_List_Member (N) then
            Ann := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

            Adecl :=
              Make_Object_Declaration (Loc,
                Defining_Identifier => Ann,
                Object_Definition => New_Reference_To (RTE (RE_Address), Loc));

            Insert_Before (N, Adecl);
            Analyze (Adecl);

            if Present (Handled_Statement_Sequence (N)) then
               Insert_Before (N, Ldecl);
               Analyze (Ldecl);
            end if;

         --  Case of accept statement which is in an accept alternative

         else
            declare
               Acc_Alt : constant Node_Id := Parent (N);
               Sel_Acc : constant Node_Id := Parent (Acc_Alt);
               Alt     : Node_Id;

            begin
               pragma Assert (Nkind (Acc_Alt) = N_Accept_Alternative);
               pragma Assert (Nkind (Sel_Acc) = N_Selective_Accept);

               --  ??? Consider a single label for select statements.

               if Present (Handled_Statement_Sequence (N)) then
                  Prepend (Ldecl,
                     Statements (Handled_Statement_Sequence (N)));
                  Analyze (Ldecl);
               end if;

               --  Find first accept alternative of the selective accept. A
               --  valid selective accept must have at least one accept in it.

               Alt := First (Select_Alternatives (Sel_Acc));

               while Nkind (Alt) /= N_Accept_Alternative loop
                  Alt := Next (Alt);
               end loop;

               --  If we are the first accept statement, then we have to
               --  create the Ann variable, as for the stand alone case,
               --  except that it is inserted before the selective accept.
               --  Similarly, a label for requeue expansion must be
               --  declared.

               if N = Accept_Statement (Alt) then
                  Ann :=
                    Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

                  Adecl :=
                    Make_Object_Declaration (Loc,
                      Defining_Identifier => Ann,
                      Object_Definition =>
                        New_Reference_To (RTE (RE_Address), Loc));

                  Insert_Before (Sel_Acc, Adecl);
                  Analyze (Adecl);

               --  If we are not the first accept statement, then find the
               --  Ann variable allocated by the first accept and use it.

               else
                  Ann :=
                    Node (Last_Elmt (Accept_Address
                      (Entity (Entry_Direct_Name (Accept_Statement (Alt))))));
               end if;
            end;
         end if;

         --  Merge here with Ann either created or referenced, and Adecl
         --  pointing to the corresponding declaration. Remaining processing
         --  is the same for the two cases.

         Append_Elmt (Ann, Accept_Address (Ent));
      end if;
   end Expand_Accept_Declarations;

   ------------------------------
   -- Next_Protected_Operation --
   ------------------------------

   function Next_Protected_Operation (N : Node_Id) return Node_Id is
      Next_Op : Node_Id;

   begin
      Next_Op := Next (N);

      while Present (Next_Op)
        and then Nkind (Next_Op) /= N_Subprogram_Body
        and then Nkind (Next_Op) /= N_Entry_Body
      loop
         Next_Op := Next (Next_Op);
      end loop;

      return Next_Op;
   end Next_Protected_Operation;

   -------------------------------
   -- First_Protected_Operation --
   -------------------------------

   function First_Protected_Operation (D : List_Id) return Node_Id is
      First_Op : Node_Id;

   begin
      First_Op := First (D);
      while Present (First_Op)
        and then Nkind (First_Op) /= N_Subprogram_Body
        and then Nkind (First_Op) /= N_Entry_Body
      loop
         First_Op := Next (First_Op);
      end loop;

      return First_Op;
   end First_Protected_Operation;

   ------------------------------------
   -- Expand_Entry_Body_Declarations --
   ------------------------------------

   procedure Expand_Entry_Body_Declarations (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Index_Spec : Node_Id;

   begin
      if Expander_Active then

         --  Expand entry bodies corresponding to entry families
         --  by assigning a placeholder for the constant that will
         --  be used to expand references to the entry index parameter.

         Index_Spec :=
           Entry_Index_Specification (Entry_Body_Formal_Part (N));

         if Present (Index_Spec) then
            Set_Entry_Index_Constant (
              Defining_Identifier (Index_Spec),
              Make_Defining_Identifier (Loc, New_Internal_Name ('I')));
         end if;

      end if;
   end Expand_Entry_Body_Declarations;

   -------------------------
   -- Expand_N_Entry_Body --
   -------------------------

   procedure Expand_N_Entry_Body (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Next_Op : Node_Id;
      Dec     : Node_Id;

   begin
      --  Associate privals and discriminals with the next protected
      --  operation body to be expanded. These are used to expand
      --  references to private data objects and discriminants,
      --  respectively.

      Next_Op := Next_Protected_Operation (N);
      Dec := Parent (Current_Scope);

      if Present (Next_Op) then
         Set_Privals (Dec, Next_Op, Loc);
         Set_Discriminals (Dec, Next_Op, Loc);
      end if;

   end Expand_N_Entry_Body;

   ----------------------------------------
   -- Expand_Protected_Body_Declarations --
   ----------------------------------------

   --  Part of the expansion of a protected body involves the creation of
   --  a declaration that can be referenced from the statement sequences of
   --  the entry bodies:

   --    A : Address;

   --  This declaration is inserted in the declarations of the service
   --  entries procedure for the protected body,
   --  and it is important that it be inserted before the statements of
   --  the entry body statement sequences are analyzed.
   --  Thus it would be too late to create
   --  this declaration in the Expand_N_Protected_Body routine, which is
   --  why there is a separate procedure to be called directly from Sem_Ch9.

   --  It is used to hold the address of the record containing the parameters
   --  (see Expand_N_Entry_Call for more details on how this record is built).
   --  References to the parameters do an unchecked conversion of this address
   --  to a pointer to the required record type, and then access the field that
   --  holds the value of the required parameter. The entity for the address
   --  variable is held as the top stack element (i.e. the last element) of the
   --  Accept_Address stack in the corresponding entry entity, and this element
   --  must be set in place  before the statements are processed.

   --  No stack is needed for entry bodies, since they cannot be nested, but
   --  it is kept for consistency between protected and task entries. The
   --  stack will never contain more than one element. There is also only one
   --  such variable for a given protected body, but this is placed on the
   --  Accept_Address stack of all of the entries, again for consistency.

   --  To expand the requeue statement, a label is provided at the end of
   --  the loop in the entry service routine created by the expander (see
   --  Expand_N_Protected_Body for details), so that the statement can be
   --  skipped after the requeue is complete. This label is created during the
   --  expansion of the entry body, which will take place after the expansion
   --  of the requeue statments that it contains, so a placeholder defining
   --  identifier is associated with the task type here.

   --  Another label is provided following case statement created by the
   --  expander. This label is need for implementing return statement from
   --  entry body so that a return can be expanded as a goto to this label.
   --  This label is created during the expansion of the entry body, which will
   --  take place after the expansion of the return statements that it
   --  contains. Therefore, just like the label for expanding reqeuses, we need
   --  another placeholder for the label.

   procedure Expand_Protected_Body_Declarations
     (N : Node_Id;
      Spec_Id : Entity_Id)
   is
      Loc        : constant Source_Ptr := Sloc (N);
      Body_Id    : constant Entity_Id := Corresponding_Body (Parent (Spec_Id));

      Prec_Decl  : Node_Id;
      Prec_Def   : Node_Id;
      Obj_Decl   : Node_Id;
      P_Subtype  : Entity_Id;

      P          : Entity_Id;
      Op         : Node_Id;
      Ent        : Entity_Id;
      Lab_Id     : Node_Id;
      Lab        : Node_Id;
      Pend       : Entity_Id;
      Ent_Ref    : Entity_Id;

   begin
      if Expander_Active then
         Prec_Decl  := Parent (Corresponding_Record_Type (Spec_Id));
         Prec_Def   := Type_Definition (Prec_Decl);

         Obj_Decl := First (Component_Items (Component_List (Prec_Def)));
         while Chars (Defining_Identifier (Obj_Decl)) /= Name_uObject loop
            Obj_Decl := Next (Obj_Decl);
         end loop;

         P_Subtype  := Etype (Defining_Identifier (Obj_Decl));

         --  Set up the defining identifier used to access the
         --  Protection object within the entry service routine.
         --  This is analogous to Entry_Prival for explicit private
         --  objects.

         Ent_Ref :=
           Make_Defining_Identifier (Loc,
             Chars => New_External_Name (Chars (Body_Id), 'R'));

         --  Set the Etype to the implicit subtype of Protection created
         --  when the protected type declaration was expanded. This node will
         --  not be analyzed until it is used as the defining identifier for
         --  the renaming declaration in the entry service procedure,
         --  and it will be needed in the references expanded before
         --  that body is expanded. Since the Protection field is
         --  aliased, set the Is_Aliased flag as well.

         Set_Etype (Ent_Ref, P_Subtype);
         Set_Is_Aliased (Ent_Ref);

         Set_Entry_Object_Ref (Body_Id, Ent_Ref);

         --  Associate privals with the first subprogram or entry
         --  body to be expanded. These are used to expand references
         --  to private data objects.

         Op := First_Protected_Operation (Declarations (N));

         if Present (Op) then
            Set_Discriminals (Parent (Spec_Id), Op, Sloc (N));
            Set_Privals (Parent (Spec_Id), Op, Sloc (N));
         end if;

         --  Create the defining identifier for the address object
         --  pointing to entry parameter records. To allow this
         --  to be referenced before it is analyzed, its Etype is
         --  set here.

         P := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
         Set_Etype (P, RTE (RE_Address));

         Ent := First_Entity (Spec_Id);
         while Present (Ent) loop
            if Ekind (Ent) = E_Entry or else
              Ekind (Ent) = E_Entry_Family then
               Append_Elmt (P, Accept_Address (Ent));
            end if;

            Ent := Next_Entity (Ent);
         end loop;

         --  Create and declare a label to be placed at the end of
         --  the entry body. This is used to allow requeues to skip
         --  the remainder of entry processing.

         Lab_Id := Make_Identifier (Loc, New_Internal_Name ('L'));
         Set_Entity (Lab_Id, Make_Defining_Identifier (Loc, Chars (Lab_Id)));
         Lab := Make_Label (Loc, Lab_Id);

         Prepend_To (Declarations (N),
           Make_Implicit_Label_Declaration (Loc,
             Defining_Identifier  => Entity (Lab_Id),
             Label => Lab));

         Set_End_Of_Body (Body_Id, Lab);

         --  Create and declare a label to be placed at the end of
         --  the case statement of the expanded entry body. This is used to
         --  implement return from entry body.

         Lab_Id := Make_Identifier (Loc, New_Internal_Name ('L'));
         Set_Entity (Lab_Id, Make_Defining_Identifier (Loc, Chars (Lab_Id)));
         Lab := Make_Label (Loc, Lab_Id);
         Prepend_To (Declarations (N),
           Make_Implicit_Label_Declaration (Loc,
             Defining_Identifier  => Entity (Lab_Id),
             Label => Lab));
         Set_End_Of_Case (Body_Id, Lab);

         --  Create the defining identifier to be used in declaring the
         --  pending serviced flag variable in the entry service
         --  procedure, if any. To allow it to be used in references
         --  before its declaration is analyzed, its Etype is set here.

         Pend := Make_Defining_Identifier (Loc, New_Internal_Name ('P'));
         Set_Etype (Pend, Standard_Boolean);
         Set_Pending_Serviced (Body_Id, Pend);

      end if;
   end Expand_Protected_Body_Declarations;

   ------------------------------
   -- Expand_N_Abort_Statement --
   ------------------------------

   --  Expand abort T1, T2, .. Tn; into:
   --    Abort_Tasks (Task_List'(1 => T1.Task_Id, 2 => T2.Task_Id ...))

   procedure Expand_N_Abort_Statement (N : Node_Id) is
      Loc    : constant Source_Ptr := Sloc (N);
      Tlist  : constant List_Id    := Names (N);
      Count  : Nat;
      Aggr   : Node_Id;
      Tasknm : Node_Id;

   begin
      Aggr := Make_Aggregate (Loc, Component_Associations => New_List);
      Count := 0;

      Tasknm := First (Tlist);

      while Present (Tasknm) loop
         Count := Count + 1;
         Append_To (Component_Associations (Aggr),
           Make_Component_Association (Loc,
             Choices => New_List (
               Make_Integer_Literal (Loc, UI_From_Int (Count))),
             Expression => Concurrent_Ref (Tasknm)));
         Tasknm := Next (Tasknm);
      end loop;

      Replace_Substitute_Tree (N,
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Abort_Tasks), Loc),
          Parameter_Associations => New_List (
            Make_Qualified_Expression (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Task_List), Loc),
              Expression => Aggr))));

      Analyze (N);

   end Expand_N_Abort_Statement;

   -------------------------------
   -- Expand_N_Accept_Statement --
   -------------------------------

   --  This procedure handles expansion of accept statements that stand
   --  alone, i.e. they are not part of an accept alternative. The expansion
   --  of accept statement in accept alternatives is handled by the routines
   --  Expand_N_Accept_Alternative and Expand_N_Selective_Accept. The
   --  following description applies only to stand alone accept statements.

   --  If there is no handled statement sequence, then this is called a
   --  trivial accept, and the expansion is:

   --    Accept_Trivial (entry-index)

   --  If there is a handled statement sequence, then the expansion is:

   --    Ann : Address;
   --    {Lnn : Label}

   --    begin
   --       begin
   --          Accept_Call (entry-index, Ann);
   --          <statement sequence from N_Accept_Statement node>
   --          Complete_Rendezvous;
   --          <<Lnn>>
   --
   --       exception
   --          when ... =>
   --             <exception handler from N_Accept_Statement node>
   --             Complete_Rendezvous;
   --          when ... =>
   --             <exception handler from N_Accept_Statement node>
   --             Complete_Rendezvous;
   --          ...
   --       end;

   --    exception
   --       when others =>
   --          Exceptional_Complete_Rendezvous (Current_Exception);
   --    end;

   --  The first three declarations were already inserted ahead of the
   --  accept statement by the Expand_Accept_Declarations procedure, which
   --  was called directly from the semantics during analysis of the accept.
   --  statement, before analyzing its contained statements.

   --  The declarations from the N_Accept_Statement, as noted in Sinfo, come
   --  from possible expansion activity (the original source of course does
   --  not have any declarations associated with the accept statement, since
   --  an accept statement has no declarative part). In particular, if the
   --  expander is active, the first such declaration is the declaration of
   --  the Accept_Params_Ptr entity (see Sem_Ch9.Analyze_Accept_Statement).
   --
   --  The two blocks are merged into a single block if the inner block has
   --  no exception handlers, but otherwise two blocks are required, since
   --  exceptions might be raised in the exception handlers of the inner
   --  block, and Exceptional_Complete_Rendezvous must be called.

   procedure Expand_N_Accept_Statement (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Stats   : constant Node_Id    := Handled_Statement_Sequence (N);
      Ename   : constant Node_Id    := Entry_Direct_Name (N);
      Eindx   : constant Node_Id    := Entry_Index (N);
      Eent    : constant Entity_Id  := Entity (Ename);
      Acstack : constant Elist_Id   := Accept_Address (Eent);
      Ann     : constant Entity_Id  := Node (Last_Elmt (Acstack));
      Ttyp    : constant Entity_Id  := Etype (Scope (Eent));
      Call    : Node_Id;
      Block   : Node_Id;

   begin

      --  If accept statement is not part of a list, then its parent must be
      --  an accept alternative, and, as described above, we do not do any
      --  expansion for such accept statements at this level.

      if not Is_List_Member (N) then
         pragma Assert (Nkind (Parent (N)) = N_Accept_Alternative);
         return;

      --  Trivial accept case (no statement sequence)

      elsif No (Stats) then
         Rewrite_Substitute_Tree (N,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Accept_Trivial), Loc),
             Parameter_Associations => New_List (
               Entry_Index_Expression (Loc, Entity (Ename), Eindx, Ttyp))));

         Analyze (N);
         return;

      --  Case of statement sequence present

      else
         --  Construct the block

         Block :=
           Make_Block_Statement (Loc,
             Handled_Statement_Sequence => Build_Accept_Body (Stats, Loc));

         --  Prepend call to Accept_Call to main statement sequence

         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Accept_Call), Loc),
             Parameter_Associations => New_List (
               Entry_Index_Expression (Loc, Entity (Ename), Eindx, Ttyp),
               New_Reference_To (Ann, Loc)));

         Prepend (Call, Statements (Stats));
         Analyze (Call);

         --  Replace the accept statement by the new block

         Rewrite_Substitute_Tree (N, Block);
         Analyze (N);

         --  Last step is to unstack the Accept_Address value

         Remove_Last_Elmt (Acstack);
         return;
      end if;

   end Expand_N_Accept_Statement;

   ----------------------------------
   -- Expand_N_Asynchronous_Select --
   ----------------------------------

   --  This procedure assumes that the trigger statement is an entry
   --  call. A delay alternative should already have been expanded
   --  into an entry call to the appropriate delay object Wait entry.

   --  If the trigger is a task entry call, the select is implemented
   --  with Task_Entry_Call:

   --    declare
   --       B : Boolean;
   --       C : Boolean;
   --       P : parms := (parm, parm, parm);
   --       procedure Fn is
   --       begin
   --          Cancel_Task_Entry_Call (C);
   --       end Fn;
   --    begin
   --       Abort_Defer;
   --       Task_Entry_Call
   --         (acceptor-task,
   --          entry-index,
   --          P'Address,
   --          Asynchronous_Call,
   --          B);
   --       begin
   --          begin
   --             Abort_Undefer;
   --             abortable-part
   --          at end
   --             _clean;
   --          end;
   --       exception
   --       when Abort_Signal => null;
   --       end;
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --       if not C then
   --          triggered-statements
   --       end if;
   --    end;

   --  Note that Build_Simple_Entry_Call is used to expand the entry
   --  of the asynchronous entry call (by the
   --  Expand_N_Entry_Call_Statement procedure) as follows:

   --    declare
   --       P : parms := (parm, parm, parm);
   --    begin
   --       Call_Simple (acceptor-task, entry-index, P'Address);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --    end;

   --  so the task at hand is to convert the latter expansion into the former


   --  If the trigger is a protected entry call, the select is
   --  implemented with Protected_Entry_Call:

   --  declare
   --     P   : E1_Params := (param, param, param);
   --     Pnn : Boolean;
   --     Bnn : Communications_Block;
   --     procedure Fnn is
   --     begin
   --        if not Pnn then
   --           Cancel_Protected_Entry_Call (Bnn);
   --           Service_Cancelled_Call (Bnn);
   --        end if;
   --     end Fnn;
   --
   --  begin
   --     begin
   --        begin
   --           Abort_Defer;
   --           Lock (po._object'Access);
   --           Protected_Entry_Call (
   --             Object => po._object'Access,
   --             E => <entry index>;
   --             Uninterpreted_Data => P'Address;
   --             Mode => Asynchronous_Call;
   --             Block => Bnn);
   --           ptypeS (po, Pnn);
   --           Unlock (po._object'Access);
   --           if not Pnn then
   --              Wait_Until_Abortable (Bnn);
   --              Abort_Undefer;
   --              Raise_Pending_Exception (Bnn);
   --              abortable part
   --           else
   --              Abort_Undefer;
   --              Raise_Pending_Exception (Bnn);
   --           end if;
   --        at end
   --           Fnn;
   --        end;
   --     exception
   --     when Abort_Signal =>
   --        null;
   --     end;
   --     Raise_Pending_Exception (Bnn);
   --     if not Get_Cancelled (Bnn) then
   --        triggered statements
   --     end if;
   --  end;

   --  Build_Simple_Entry_Call is used to expand the all to a simple
   --  protected entry call:

   --  declare
   --     P   : E1_Params := (param, param, param);
   --     Pnn : Boolean;
   --     Bnn : Communications_Block;
   --  begin
   --     Abort_Defer;
   --     Lock (po._object'Access);
   --     Protected_Entry_Call (
   --       Object => po._object'Access,
   --       E => <entry index>;
   --       Uninterpreted_Data => P'Address;
   --       Mode => Simple_Call;
   --       Block => Bnn);
   --     ptypeS (po, Pnn);
   --     Unlock (po._object'Access);
   --     if not Pnn then
   --        Wait_For_Completion (Bnn);
   --        Service_Cancelled_Call (Bnn);
   --     end if;
   --     Abort_Undefer;
   --     Raise_Pending_Exception (Bnn);
   --  end;

   --  The job is to convert this to the asynchronous form.

   --  If the trigger is a delay statement, it will have been expanded
   --  into a call to one of the GNARL delay procedures. This routine
   --  will convert this into a protected entry call on a delay object
   --  and then continue processing as for a protected entry call trigger.
   --  This requires declaring a Delay_Block object and adding a pointer
   --  to this object to the parameter list of the delay procedure to form
   --  the parameter list of the entry call. This object is used by
   --  the runtime to queue the delay request.

   --  For a description of the use of P and the assignments after the
   --  call, see Expand_N_Entry_Call_Statement.

   procedure Expand_N_Asynchronous_Select (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Trig       : constant Node_Id    := Triggering_Alternative (N);
      Abrt       : constant Node_Id    := Abortable_Part (N);
      Ecall      : Node_Id    := Triggering_Statement (Trig);
      Tstats     : constant List_Id    := Statements (Trig);
      Astats     : List_Id             := Statements (Abrt);
      Concval    : Node_Id;
      Ename      : Node_Id;
      Index      : Node_Id;
      Ablk       : Node_Id;
      Hdle       : List_Id;
      Decls      : List_Id;
      Decl       : Node_Id;
      Parms      : List_Id;
      Parm       : Node_Id;
      Call       : Node_Id;
      Stmts      : List_Id;
      Stmt       : Node_Id;
      B          : Entity_Id;
      C          : Entity_Id;
      Pend       : Entity_Id;
      Comm       : Entity_Id;
      Ustat      : Node_Id;
      Rstat      : Node_Id;
      Then_Stats : List_Id;
      Else_Stats : List_Id;
      Final      : Entity_Id;
      Pdef       : Entity_Id;
      Odef       : Entity_Id;
      Dblock_Ent : Entity_Id;
      Dstat      : Node_Id;
      N_Orig     : Node_Id;
      Out_Stats  : List_Id;

   begin

      --  If a delay was used as a trigger, it will have been expanded
      --  into a procedure call. Convert it to a protected entry call
      --  on the appropriate delay object, based on the type.
      --  ??? This currently supports only Duration, Real_Time.Time,
      --      and Calendar.Time.

      if Nkind (Ecall) = N_Procedure_Call_Statement then

         --  Add a Delay_Block object to the parameter list of the
         --  delay procedure to form the parameter list of the Wait
         --  entry call.

         Dblock_Ent := Make_Defining_Identifier (Loc, New_Internal_Name ('D'));

         N_Orig := Relocate_Node (N);

         --  Wrap the asynchronous select in a block declaring the
         --  Delay_Block object. Note that this is in addition to the
         --  block that will be used to implement the select statement
         --  proper. This is because the delay trigger is expanded in two
         --  stages: once to convert the procedure call into an entry
         --  call, and once by Build_Simple_Entry_Call (below) to convert
         --  the entry call into GNARLI code for a simple entry call.

         Rewrite_Substitute_Tree (N,
           Make_Block_Statement (Loc,
             Declarations => New_List (
               Make_Object_Declaration (Loc,
                 Defining_Identifier => Dblock_Ent,
                 Aliased_Present => True,
                 Object_Definition => New_Reference_To (
                   RTE (RE_Delay_Block), Loc))),

             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => New_List)));

         --  Note that the new block must be analyzed before the select
         --  statement is added to it, otherwise this procedure would be
         --  called recursively.
         --  ??? All of these machinations are the result of the
         --      questionable context-sensitive code in
         --      Expand_N_Entry_Call statement that delays expansion of
         --      an entry call trigger. This is not being fixed now,
         --      since there is some question as to whether the delay
         --      object interface will survive.

         Analyze (N);

         Pdef := Entity (Name (Ecall));
         if Pdef = RTE (RO_CA_Delay_For) then
            Odef := RTE (RO_CA_Delay_Object);
         elsif Pdef = RTE (RO_CA_Delay_Until) then
            Odef := RTE (RO_CA_Delay_Until_Object);
         elsif Pdef = RTE (RO_RT_Delay_Until) then
            Odef := RTE (RO_RT_Delay_Until_Object);
         else
            Unimplemented (N, "This kind of trigger");
         end if;

         Append_To (Parameter_Associations (Ecall),
           Make_Attribute_Reference (Loc,
             Prefix => New_Reference_To (Dblock_Ent, Loc),
             Attribute_Name => Name_Access));

         Decl :=
           First (Visible_Declarations (Protected_Definition (
             Parent (Etype (Odef)))));
         while Nkind (Decl) /= N_Entry_Declaration loop
            Decl := Next (Decl);
         end loop;

         Rewrite_Substitute_Tree (Ecall,
           Make_Procedure_Call_Statement (Loc,
             Name => Make_Selected_Component (Loc,
               Prefix => New_Reference_To (Odef, Loc),
               Selector_Name => New_Reference_To (
                 Defining_Identifier (Decl), Loc)),
             Parameter_Associations => Parameter_Associations (Ecall)));

         Append_To (Statements (Handled_Statement_Sequence (N)), N_Orig);

         Analyze (Ecall);
      else
         N_Orig := N;
      end if;

      Extract_Entry (Ecall, Concval, Ename, Index);
      Build_Simple_Entry_Call (Ecall, Concval, Ename, Index);

      Stmts := Statements (Handled_Statement_Sequence (Ecall));
      Decls := Declarations (Ecall);

      if Is_Protected_Type (Etype (Concval)) then

         --  Get the declarations of the block expanded from the entry
         --  call.

         Decl := First (Decls);
         while Present (Decl)
           and then (Nkind (Decl) /= N_Object_Declaration
             or else Etype (Object_Definition (Decl)) /= Standard_Boolean)
         loop
            Decl := Next (Decl);
         end loop;

         pragma Assert (Present (Decl));
         Pend := Defining_Identifier (Decl);

         Decl := First (Decls);
         while Present (Decl)
           and then (Nkind (Decl) /= N_Object_Declaration
             or else Etype (Object_Definition (Decl))
               /= RTE (RE_Communication_Block))
         loop
            Decl := Next (Decl);
         end loop;

         pragma Assert (Present (Decl));
         Comm := Defining_Identifier (Decl);

         --  Make the finalization procedure.
         --  procedure Fnn is

         Final := Make_Defining_Identifier (Loc, New_Internal_Name ('F'));

         Append_To (Decls,
           Make_Subprogram_Body (Loc,
             Specification =>
               Make_Procedure_Specification (Loc, Defining_Unit_Name => Final),

             Declarations => Empty_List,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => New_List (

               --  if not Pnn then

                   Make_If_Statement (Loc,
                     Condition => Make_Op_Not (Loc,
                       Right_Opnd => New_Reference_To (Pend, Loc)),
                     Then_Statements => New_List (

                     --  Cancel_Protected_Entry_Call (Bnn);

                       Make_Procedure_Call_Statement (Loc,
                         Name => New_Reference_To (
                           RTE (RE_Cancel_Protected_Entry_Call), Loc),
                         Parameter_Associations => New_List (
                           New_Reference_To (Comm, Loc))),

                           --  Service_Cancelled_Call (Bnn);

                         Make_Procedure_Call_Statement (Loc,
                           Name => New_Reference_To (
                             RTE (RE_Service_Cancelled_Call), Loc),
                           Parameter_Associations => New_List (
                             New_Reference_To (Comm, Loc)))))))));

         --  Skip over Abort_Defer and Lock to get the call to
         --  Protected_Entry_Call. Change the mode.
         --  Protected_Entry_Call (
         --    Object => po._object'Access,
         --    E => <entry index>;
         --    Uninterpreted_Data => P'Address;
         --    Mode => Asynchronous_Call;
         --    Block => Bnn);

         Stmt := Next (Next (First (Stmts)));
         Call := Stmt;
         Parms := Parameter_Associations (Call);
         Parm := Next (Next (Next (First (Parms))));
         Rewrite_Substitute_Tree (Parm,
           New_Reference_To (RTE (RE_Asynchronous_Call), Loc));
         Analyze (Parm);

         --  Skip to the "if Pnn" statement. Rewrite it to execute
         --  the abortable part.
         --  if not Pnn then

         Stmt := Next (Next (Next (Stmt)));
         Ustat := Remove_Next (Stmt);
         Rstat := Remove_Next (Stmt);

         Then_Stats := New_List (

            --  Wait_Until_Abortable (Bnn);

            Make_Procedure_Call_Statement (Loc,
              Name =>
                New_Reference_To (RTE (RE_Wait_Until_Abortable), Loc),
              Parameter_Associations => New_List (
                New_Reference_To (Comm, Loc))),

            --  Abort_Undefer;

            Ustat,

            --  Raise_Pending_Exception (Bnn);

            Rstat);

         Append_List (Astats, Then_Stats);

         --  else

         Else_Stats := New_List (

            --  Abort_Undefer;

            Make_Procedure_Call_Statement (Loc,
              Name => New_Reference_To (RTE (RE_Abort_Undefer), Loc)),

            --  Raise_Pending_Exception (Bnn);

            Make_Procedure_Call_Statement (Loc,
              Name =>
                New_Reference_To (RTE (RE_Raise_Pending_Exception), Loc),
              Parameter_Associations => New_List (
                New_Reference_To (Comm, Loc))));

         Rewrite_Substitute_Tree (Stmt,
           Make_If_Statement (Loc,
             Condition => Condition (Stmt),
             Then_Statements => Then_Stats,
             Else_Statements => Else_Stats));

         Stmts := New_List (Make_Block_Statement (Loc,
           Handled_Statement_Sequence =>
             Make_Handled_Sequence_Of_Statements (Loc,
               Statements => Stmts,
               Identifier => New_Occurrence_Of (Final, Loc))));

         Stmts := New_List (
           Make_Block_Statement (Loc,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => Stmts,

                  --  exception

                  Exception_Handlers => New_List (
                    Make_Exception_Handler (Loc,

                  --  when Abort_Signal =>
                  --     null;

                      Exception_Choices =>
                        New_List (New_Reference_To (Stand.Abort_Signal, Loc)),
                      Statements =>  New_List (Make_Null_Statement (Loc)))))),

            --  Raise_Pending_Exception (Bnn);

            Make_Procedure_Call_Statement (Loc,
              Name =>
                New_Occurrence_Of (RTE (RE_Raise_Pending_Exception), Loc),
              Parameter_Associations => New_List (
                New_Occurrence_Of (Comm, Loc))),

            --  if not Get_Cancelled (Bnn) then
            --     triggered statements
            --  end if;

            Make_If_Statement (Loc,
              Condition => Make_Op_Not (Loc,
                Right_Opnd =>
                  Make_Function_Call (Loc,
                    Name => New_Occurrence_Of (RTE (RE_Get_Cancelled), Loc),
                    Parameter_Associations => New_List (
                      New_Occurrence_Of (Comm, Loc)))),
              Then_Statements => Tstats));

      else
      --  Asynchronous task entry call.

         if No (Decls) then
            Decls := New_List;
         end if;

         B := Make_Defining_Identifier (Loc, Name_uB);

         --  Insert declaration of B in declarations of existing block

         Prepend_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => B,
             Object_Definition => New_Reference_To (Standard_Boolean, Loc)));

         C := Make_Defining_Identifier (Loc, Name_uC);

         --  Insert declaration of C in declarations of existing block

         Prepend_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => C,
             Object_Definition => New_Reference_To (Standard_Boolean, Loc)));

         --  Insert declaration of the cleanup routine in declarations.

         Final := Make_Defining_Identifier (Loc, New_Internal_Name ('F'));

         Append_To (Decls,
           Make_Subprogram_Body (Loc,
             Specification =>
               Make_Procedure_Specification (Loc, Defining_Unit_Name => Final),

             Declarations => Empty_List,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => New_List (
                    Make_Procedure_Call_Statement (Loc,
                      Name => New_Reference_To (
                        RTE (RE_Cancel_Task_Entry_Call),
                        Loc),
                      Parameter_Associations => New_List (
                        New_Reference_To (C, Loc)))))));

         --  Remove and save the call to Call_Simple.

         Call := Remove_Head (Stmts);

         --  Create the inner block to protect the abortable part.

         Hdle :=  New_List (
           Make_Exception_Handler (Loc,
             Exception_Choices => New_List (
               New_Reference_To (Stand.Abort_Signal, Loc)),
             Statements => New_List (Make_Null_Statement (Loc))));

         Prepend_To (Astats,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Abort_Undefer), Loc),
             Parameter_Associations => Empty_List));

         Ablk :=
           Make_Block_Statement (Loc,
             Handled_Statement_Sequence =>
               Make_Handled_Sequence_Of_Statements (Loc,
                 Statements => New_List (
                   Make_Block_Statement (Loc,
                     Handled_Statement_Sequence =>
                       Make_Handled_Sequence_Of_Statements (Loc,
                         Statements => Astats,
                         Identifier => New_Reference_To (Final, Loc)))),
                 Exception_Handlers => Hdle));

         Prepend (Ablk, Stmts);

         --  Create new call statement

         Parms := Parameter_Associations (Call);
         Append_To (Parms, New_Reference_To (RTE (RE_Asynchronous_Call), Loc));
         Append_To (Parms, New_Reference_To (B, Loc));
         Call :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Task_Entry_Call), Loc),
             Parameter_Associations => Parms);

         --  Construct statement sequence for new block

         Append_To (Stmts,
           Make_If_Statement (Loc,
             Condition => Make_Op_Not (Loc, New_Reference_To (C, Loc)),
             Then_Statements => Tstats));

         Prepend_To (Stmts, Call);

         --  Protected the call against abortion

         Prepend_To (Stmts,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Abort_Defer), Loc),
             Parameter_Associations => Empty_List));

      end if;

      --  The result is the new block

      Rewrite_Substitute_Tree (N_Orig,
        Make_Block_Statement (Loc,
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc, Stmts)));

      Analyze (N_Orig);

   end Expand_N_Asynchronous_Select;

   -------------------------------------
   -- Expand_N_Conditional_Entry_Call --
   -------------------------------------

   --  The conditional entry call is converted to a call to Task_Entry_Call:

   --    declare
   --       B : Boolean;
   --       P : parms := (parm, parm, parm);

   --    begin
   --       Task_Entry_Call
   --         (acceptor-task,
   --          entry-index,
   --          P'Address,
   --          Conditional_Call,
   --          B);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --       if B then
   --          normal-statements
   --       else
   --          else-statements
   --       end if;
   --    end;

   --  For a description of the use of P and the assignments after the
   --  call, see Expand_N_Entry_Call_Statement. Note that the entry call
   --  of the conditional entry call has already been expanded (by the
   --  Expand_N_Entry_Call_Statement procedure) as follows:

   --    declare
   --       P : parms := (parm, parm, parm);
   --    begin
   --       Call_Simple (acceptor-task, entry-index, P'Address);
   --       parm := P.param;
   --       parm := P.param;
   --       ...
   --    end;

   --  so the task at hand is to convert the latter expansion into the former

   procedure Expand_N_Conditional_Entry_Call (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Alt   : constant Node_Id    := Entry_Call_Alternative (N);
      Blk   : constant Node_Id    := Entry_Call_Statement (Alt);
      Parms : List_Id;
      Call  : Node_Id;
      Stmts : List_Id;
      B     : Entity_Id;

   begin
      B := Make_Defining_Identifier (Loc, Name_uB);

      --  Insert declaration of B in declarations of existing block

      if No (Declarations (Blk)) then
         Set_Declarations (Blk, New_List);
      end if;

      Prepend_To (Declarations (Blk),
        Make_Object_Declaration (Loc,
          Defining_Identifier => B,
          Object_Definition => New_Reference_To (Standard_Boolean, Loc)));

      --  Create new call statement

      Stmts := Statements (Handled_Statement_Sequence (Blk));
      Call := Remove_Head (Stmts);
      Parms := Parameter_Associations (Call);
      Append_To (Parms, New_Reference_To (RTE (RE_Conditional_Call), Loc));
      Append_To (Parms, New_Reference_To (B, Loc));

      Call :=
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Task_Entry_Call), Loc),
          Parameter_Associations => Parms);

      --  Construct statement sequence for new block

      Append_To (Stmts,
        Make_If_Statement (Loc,
          Condition => New_Reference_To (B, Loc),
          Then_Statements => Statements (Alt),
          Else_Statements => Else_Statements (N)));

      Prepend_To (Stmts, Call);

      --  The result is the new block

      Rewrite_Substitute_Tree (N,
        Make_Block_Statement (Loc,
          Declarations => Declarations (Blk),
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc, Stmts)));

      Analyze (N);

   end Expand_N_Conditional_Entry_Call;

   ---------------------------------------
   -- Expand_N_Delay_Relative_Statement --
   ---------------------------------------

   --  Delay statement is implemented as a procedure call to Delay_For
   --  defined in Ada.Calendar.Delays in order to reduce the overhead of
   --  simple delays imposed by the use of Protected Objects.

   procedure Expand_N_Delay_Relative_Statement (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);

   begin
      Rewrite_Substitute_Tree (N,
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RO_CA_Delay_For), Loc),
          Parameter_Associations => New_List (Expression (N))));
      Analyze (N);
   end Expand_N_Delay_Relative_Statement;

   ------------------------------------
   -- Expand_N_Delay_Until_Statement --
   ------------------------------------

   --  Delay Until statement is implemented as a procedure call to
   --  Delay_Until defined in Ada.Calendar.Delays and Ada.Real_Time.Delays.

   procedure Expand_N_Delay_Until_Statement (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : Entity_Id;

   begin
      if Etype (Expression (N)) = Etype (RTE (RO_CA_Time)) then
         Typ := RTE (RO_CA_Delay_Until);
      else
         Typ := RTE (RO_RT_Delay_Until);
      end if;

      Rewrite_Substitute_Tree (N,
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (Typ, Loc),
          Parameter_Associations => New_List (Expression (N))));

      Analyze (N);
   end Expand_N_Delay_Until_Statement;

   -----------------------------------
   -- Expand_N_Entry_Call_Statement --
   -----------------------------------

   --  An entry call is expanded into GNARLI calls to implement
   --  a simple entry call (see Build_Simple_Entry_Call).

   procedure Expand_N_Entry_Call_Statement (N : Node_Id) is
      Concval : Node_Id;
      Ename   : Node_Id;
      Index   : Node_Id;

   begin
      --  If this entry call is part of an asynchronous select, don't
      --  expand it here; it will be expanded with the select statement.
      --  Don't expand timed entry calls either, as they are translated
      --  into asynchronous entry calls.

      --  ??? This whole approach is questionable; it may be better
      --  to go back to allowing the expansion to take place and then
      --  attempting to fix it up in Expand_N_Asynchronous_Select.
      --  The tricky part is figuring out whether the expanded
      --  call is on a task or protected entry.

      if Nkind (Parent (N)) /= N_Triggering_Alternative
        and then (Nkind (Parent (N)) /= N_Entry_Call_Alternative
                    or else Nkind (Parent (Parent (N))) /= N_Timed_Entry_Call)
      then
         Extract_Entry (N, Concval, Ename, Index);
         Build_Simple_Entry_Call (N, Concval, Ename, Index);
      end if;

   end Expand_N_Entry_Call_Statement;

   --------------------------------
   -- Expand_N_Entry_Declaration --
   --------------------------------

   --  If there are parameters, then first, each of the formals is marked
   --  by setting Is_Entry_Formal. Next a record type is built which is
   --  used to hold the parameter values. The name of this record type is
   --  entryP where entry is the name of the entry, with an additional
   --  corresponding access type called entryPA. The record type has matching
   --  components for each formal (the component names are the same as the
   --  formal names). For elementary types, the component type matches the
   --  formal type. For composite types, an access type is declared (with
   --  the name formalA) which designates the formal type, and the type of
   --  the component is this access type. Finally the Entry_Component of
   --  each formal is set to reference the corresponding record component.

   procedure Expand_N_Entry_Declaration (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Entry_Ent   : constant Entity_Id  := Defining_Identifier (N);
      Components  : List_Id;
      Formal      : Node_Id;
      Ftype       : Entity_Id;
      Last_Decl   : Node_Id;
      Component   : Entity_Id;
      Ctype       : Entity_Id;
      Decl        : Node_Id;
      Rec_Ent     : Entity_Id;
      Acc_Ent     : Entity_Id;

   begin
      Formal := First_Formal (Entry_Ent);
      Last_Decl := N;

      --  Most processing is done only if parameters are present

      if Present (Formal) then
         Components := New_List;

         --  Loop through formals

         while Present (Formal) loop
            Set_Is_Entry_Formal (Formal);
            Component := Make_Defining_Identifier (Loc, Chars (Formal));
            Set_Entry_Component (Formal, Component);
            Ftype := Etype (Formal);

            --  By-copy type, just append to component list

            if Is_By_Copy_Type (Ftype) then
               Ctype := Ftype;

            --  Composite type, declare new access type and then append

            else
               Ctype :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

               Decl :=
                 Make_Full_Type_Declaration (Loc,
                   Defining_Identifier => Ctype,
                   Type_Definition     =>
                     Make_Access_To_Object_Definition (Loc,
                       All_Present        => True,
                       Subtype_Indication => New_Reference_To (Ftype, Loc)));

               Insert_After (Last_Decl, Decl);
               Last_Decl := Decl;
            end if;

            Append_To (Components,
              Make_Component_Declaration (Loc,
                Defining_Identifier => Component,
                Subtype_Indication  => New_Reference_To (Ctype, Loc)));

            Formal := Next_Formal (Formal);
         end loop;

         --  Create the Entry_Parameter_Record declaration

         Rec_Ent :=
           Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

         Decl :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Rec_Ent,
             Type_Definition     =>
               Make_Record_Definition (Loc,
                 Component_List =>
                   Make_Component_List (Loc,
                     Component_Items => Components)));

         Insert_After (Last_Decl, Decl);
         Last_Decl := Decl;

         --  Construct and link in the corresponding access type

         Acc_Ent :=
           Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

         Set_Entry_Parameters_Type (Entry_Ent, Acc_Ent);

         Decl :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Acc_Ent,
             Type_Definition     =>
               Make_Access_To_Object_Definition (Loc,
                 All_Present        => True,
                 Subtype_Indication => New_Reference_To (Rec_Ent, Loc)));

         Insert_After (Last_Decl, Decl);
         Last_Decl := Decl;

      end if;

   end Expand_N_Entry_Declaration;

   -----------------------------
   -- Expand_N_Protected_Body --
   -----------------------------

   --  Protected bodies are expanded to the completion of the subprograms
   --  created for the corresponding protected type. These are a protected
   --  and unprotected version of each protected subprogram in the object,
   --  plus one procedure to service all entry calls. For example, for
   --  protected type ptype:

   --  procedure pprocN (_object : in out poV;...) is
   --     <discriminant renamings>
   --     <private object renamings>
   --  begin
   --     <sequence of statements>
   --  end pprocN;
   --  procedure pproc (_object : in out poV;...) is
   --     procedure pprocF is
   --       Pn : Boolean;
   --     begin
   --       ptypeS (_object, Pn);
   --       System.Tasking.Protected_Objects.Unlock (_object._object'Access);
   --       System.Tasking.Abortion.Undefer_Abortion;
   --     end pprocF;
   --  begin
   --     System.Tasking.Abortion.Defer_Abortion;
   --     System.Tasking.Protected_Objects.Lock (_object._object'Access);
   --     pprocN (_object;...);
   --  at end
   --     pprocF (_object._object'Access);
   --  end pproc;
   --  function pfuncN (_object : poV;...) return Return_Type is
   --     <discriminant renamings>
   --     <private object renamings>
   --  begin
   --     <sequence of statements>
   --  end pfuncN;
   --  function pfunc (_object : poV) return Return_Type is
   --     procedure pfuncF is
   --     begin
   --       System.Tasking.Protected_Objects.Unlock (_object._object'Access);
   --       System.Tasking.Abortion.Undefer_Abortion;
   --     end pprocF;
   --  begin
   --     System.Tasking.Abortion.Defer_Abortion;
   --     System.Tasking.Protected_Objects.Lock (_object._object'Access);
   --     return pfuncN (_object);
   --  at end
   --     ptypeF (_object);
   --  end pproc;
   --  ...
   --  procedure ptypeS
   --    (_object : in out poV;
   --     P : out Boolean)
   --  is
   --     <discriminant renamings>
   --     <private object renamings>
   --     Bnn : Barrier_Vector (_object.Num_Entries);
   --     A   : System.Address;
   --     Enn : Protected_Entry_Index;
   --     Pnn : Boolean;
   --     Cnn : Boolean := False;
   --  begin
   --     loop
   --        begin
   --           Bnn (<entry index>) := <entry barrier expression>;
   --           Bnn (<entry family index range>) :=
   --               (others => <entry family barrier expression>;
   --        exception
   --        when others =>
   --           begin
   --              Broadcast_Program_Error (_object._object'Access);
   --              P := True;
   --           exception
   --           when Program_Error =>
   --              Unlock (_object._object'Access);
   --              Abort_Undefer;
   --           end;
   --        end;   --
   --        Next_Entry_Call (_object._object'Access, Bnn, A, Enn);
   --        begin
   --           case Enn is
   --              when Null_Protected_Entry =>
   --                 exit;
   --              when 1 =>
   --                 <statement sequence for entry 1>
   --              when 2 =>
   --                 <statement sequence for entry 2>
   --              ...
   --              when others
   --                 if Enn in <entry family index range> then
   --                    declare
   --                       <entry index name> : Protected_Entry_Index :=
   --                          Enn - Entry_Family_Index'First;
   --                    begin
   --                       <statement sequence for entry family>
   --                    end;
   --                 elsif Enn in ...
   --                 else raise Program_Error;
   --                 end if;
   --           end case;
   --           <<Lmm>>
   --           Complete_Entry_Body (_object._object'Access, Pnn);
   --        exception
   --        when others =>
   --           Exceptional_Complete_Entry_Body (
   --            _object._object'Access, Pnn, Current_Exception);
   --        end;
   --        Cnn := Cnn or Pnn;
   --     end loop;
   --     P := Cnn;
   --  end ptypeS;

   --  The type poV is the record created for the protected type to hold
   --  the state of the protected object. The parameter P is a boolean
   --  whether one of the entries serviced was the one most recently
   --  pending on this object (using Protected_Entry_Call) by the current
   --  task. This is used to determine if the calling task has to wait
   --  for this call to be completed or not.

   --  Note that the variable P is not generated with New_Internal_Name.
   --  This is because it is used before this expansion takes place, so use
   --  of a universally available name avoids the need to store the name as
   --  an attribute. P is first used in generating a spec for the service
   --  entries procedure.

   procedure Expand_N_Protected_Body (N : Node_Id) is
      Loc          : constant Source_Ptr := Sloc (N);
      Pid          : constant Entity_Id  := Corresponding_Spec (N);
      Ptyp         : constant Node_Id    := Parent (Pid);
      Body_Id      : constant Entity_Id  := Corresponding_Body (Ptyp);
      Protnm       : constant Name_Id    := Chars (Pid);
      Pdef         : constant Node_Id    := Protected_Definition (Ptyp);
      Op_Def       : Entity_Id;
      Op_Body      : Node_Id;
      New_Op_Body  : Node_Id;
      Current_Node : Node_Id := N;
      First_Sub    : Boolean := True;
      Barriers     : List_Id := New_List;
      Entry_Alts   : List_Id := New_List;
      Ent_Formals  : Node_Id;
      Index_Exp    : Node_Id;
      Lab_Decl     : Node_Id;
      Barrier_Name : constant Name_Id := New_Internal_Name ('B');
      Index_Name   : constant Name_Id := New_Internal_Name ('E');
      Param_Id     : Entity_Id;
      Ent          : Entity_Id;
      Index_Spec   : Node_Id;
      Body_Stats   : List_Id;
      Index_Range  : Node_Id;
      Ent_Body     : Node_Id := Empty;

      Family_Stats : List_Id;
      Family_Cond  : Node_Id;
      Family_Alts  : List_Id := New_List;
      Others_Stats : List_Id;

   begin
      Ent := First_Entity (Defining_Identifier (Ptyp));
      while Present (Ent)
        and then Ekind (Ent) /= E_Entry
        and then Ekind (Ent) /= E_Entry_Family
      loop
         Ent := Next_Entity (Ent);
      end loop;

      if Present (Ent) then
         Param_Id := Node (Last_Elmt (Accept_Address (Ent)));
      else
         Param_Id := Empty;
      end if;

      Op_Body := First (Declarations (N));
      while Present (Op_Body) loop

         if Nkind (Op_Body) = N_Subprogram_Body then

            New_Op_Body :=
              Build_Unprotected_Subprogram_Body (Op_Body, Pid);

            if First_Sub then
               First_Sub := False;
               Rewrite_Substitute_Tree (N, New_Op_Body);
               Analyze (N);

            else
               Insert_After (Current_Node, New_Op_Body);
               Current_Node := New_Op_Body;
               Analyze (New_Op_Body);
            end if;

            New_Op_Body :=
               Build_Protected_Subprogram_Body (
                 Op_Body, Pid, Specification (New_Op_Body));

            Insert_After (Current_Node, New_Op_Body);
            Analyze (New_Op_Body);


         elsif Nkind (Op_Body) = N_Entry_Body then

            Ent_Body := Op_Body;
            --  Save a body for use in building the entry service procedure;
            --  any body will do.

            Ent_Formals := Entry_Body_Formal_Part (Op_Body);
            Index_Spec := Entry_Index_Specification (Ent_Formals);
            Op_Def := Defining_Identifier (Op_Body);

            Body_Stats := New_List (
               Make_Block_Statement (Loc,
                 Declarations => Declarations (Op_Body),
                 Handled_Statement_Sequence =>
                   Handled_Statement_Sequence (Op_Body)));

            Index_Exp :=
              Entry_Index_Expression (Loc,
                Op_Def, Empty, Defining_Identifier (Ptyp));

            if Present (Index_Spec) then
               Index_Range := Entry_Range_Expression (Loc,
                 Op_Def, Defining_Identifier (Ptyp));

               Append_To (Barriers,
                 Make_Assignment_Statement (Loc,
                   Name =>
                     Make_Slice (Loc,
                       Prefix => Make_Identifier (Loc, Barrier_Name),
                       Discrete_Range => Index_Range),

                   Expression =>
                     Make_Aggregate (Loc,
                       Component_Associations => New_List (
                         Make_Component_Association (Loc,
                           Choices => New_List (Make_Others_Choice (Loc)),
                           Expression => Condition (Ent_Formals))))));

               Family_Stats := New_List (
                 Make_Block_Statement (Loc,
                   Declarations => New_List (

                     Make_Object_Declaration (Loc,
                       Defining_Identifier => Entry_Index_Constant (
                         Defining_Identifier (Index_Spec)),

                       Constant_Present => True,

                       Object_Definition => New_Reference_To (
                         Etype (Defining_Identifier (Index_Spec)), Loc),

                       Expression => Make_Type_Conversion (Loc,
                         Subtype_Mark =>
                           New_Reference_To (First_Itype (Index_Spec), Loc),

                         Expression =>
                           Make_Op_Add (Loc,
                             Left_Opnd => Make_Op_Subtract (Loc,
                               Left_Opnd => Make_Identifier (Loc, Index_Name),
                               Right_Opnd => Index_Exp),

                             Right_Opnd => Make_Type_Conversion (Loc,
                               Subtype_Mark => New_Reference_To (
                                 RTE (RE_Protected_Entry_Index), Loc),

                               Expression =>
                                 Make_Attribute_Reference (Loc,
                                   Prefix => New_Reference_To (
                                     First_Itype (Index_Spec), Loc),
                                   Attribute_Name => Name_First)))))),

                  Handled_Statement_Sequence =>
                    Make_Handled_Sequence_Of_Statements (Loc,
                      Statements => Body_Stats)));

               Family_Cond :=
                 Make_In (Loc,
                   Left_Opnd => Make_Identifier (Loc, Index_Name),
                   Right_Opnd => Index_Range);

               Append_To (Family_Alts,
                 Make_Elsif_Part (Loc,
                   Condition => Family_Cond,
                   Then_Statements => Family_Stats));

            else
               Append_To (Barriers,
                 Make_Assignment_Statement (Loc,
                   Name =>
                     Make_Indexed_Component (Loc,
                       Prefix => Make_Identifier (Loc, Barrier_Name),
                       Expressions => New_List (Index_Exp)),
                   Expression => Condition (Ent_Formals)));

               Append_To (Entry_Alts,
                 Make_Case_Statement_Alternative (Loc,
                   Discrete_Choices => New_List (Index_Exp),
                   Statements => Body_Stats));
            end if;

         elsif Nkind (Op_Body) = N_Implicit_Label_Declaration then
            Lab_Decl := Op_Body;
         end if;

         Op_Body := Next (Op_Body);
      end loop;

      --  If there are any protected entries, build the service entries
      --  procedure. This is called any time that an entry may become
      --  eligible for execution, either because of a new entry call
      --  or change in the value of a barrier expression.

      if Present (Ent_Body) then
         Set_Discriminals (Ptyp, Ent_Body, Loc);
         Set_Privals (Ptyp, Ent_Body, Loc);

         New_Op_Body := Build_Entry_Service_Procedure (
           Pid,
           Barrier_Name,
           Index_Name,
           Lab_Decl,
           Barriers,
           Entry_Alts,
           Family_Alts);

         if First_Sub then
            First_Sub := False;
            Rewrite_Substitute_Tree (N, New_Op_Body);
            Analyze (N);

         else
            Insert_After (Current_Node, New_Op_Body);
            Current_Node := New_Op_Body;
            Analyze (New_Op_Body);
         end if;

      end if;

   end Expand_N_Protected_Body;

   -----------------------------------------
   -- Expand_N_Protected_Type_Declaration --
   -----------------------------------------

   --  First we create a corresponding record type declaration used to
   --  represent values of this protected type.
   --  The general form of this type declaration is

   --    type poV (discriminants) is record
   --      _Object       : aliased Protection(<entry count>);
   --      _Service      : aliased Tasking_Library.Service_Record;
   --      entry_family  : array (bounds) of Void;
   --      _Priority     : Integer   := priority_expression;
   --      <private data fields>
   --    end record;

   --  The discriminants are present only if the corresponding protected
   --  type has discriminants, and they exactly mirror the protected type
   --  discriminants. The private data fields similarly mirror the
   --  private declarations of the protected type.

   --  The Object field is always present. It contains RTS specific data
   --  used to control the protected object. It is declared as Aliased
   --  so that it can be passed as a pointer to the RTS. This allows the
   --  protected record to be referenced within RTS data structures.

   --  The Service field is present for protected objects with entries. It
   --  contains sufficient information to allow the entry service procedure
   --  for this object to be called when the object is not known till runtime.

   --  One entry_family component is present for each entry family in the
   --  task definition (see Expand_N_Task_Type_Declaration).

   --  The Priority field is present only if a Priority or Interrupt_Priority
   --  pragma appears in the protected definition. The expression captures the
   --  argument that was present in the pragma, and is used to provide
   --  the Ceiling_Priority parameter to the call to Initialize_Protection.

   --  When a protected object is declared, an instance of the protected type
   --  value record is created. The elaboration of this declaration creates
   --  the correct bounds for the entry families, and also evaluates the
   --  priority expression if needed. The initialization routine for
   --  the protected type itself then calls Initialize_Protection with
   --  appropriate parameters to initialize the value of the Task_Id field.

   --  Note: this record is passed to the subprograms created by the
   --  expansion of protected subprograms and entries. It is an in parameter
   --  to protected functions and an in out parameter to procedures and
   --  entry bodies. The Entity_Id for this created record type is placed
   --  in the Corresponding_Record_Type field of the associated protected
   --  type entity.

   --  Next we create a procedure specifications for protected subprograms
   --  and entry bodies. For each protected subprograms two subprograms are
   --  created, an unprotected and a protected version. The unprotected
   --  version is called from within other operations of the same protected
   --  object.

   --  A single subprogram is created to service all entry bodies; it has an
   --  additional boolean out parameter indicating that the previous entry
   --  call made by the current task was serviced immediately, i.e. not by
   --  proxy. The O parameter contains a pointer to a record object of the
   --  type described above. An untyped interface is used here to allow this
   --  procedure to be called in places where the type of the object to be
   --  serviced is not known. This must be done, for example, when a call
   --  that may have been requeued is cancelled; the corresponding object
   --  must be serviced, but which object that is is not known till runtime.

   --  procedure ptypeS
   --    (O : System.Address; P : out Boolean);
   --  procedure pprocN (_object : in out poV);
   --  procedure pproc (_object : in out poV);
   --  function pfuncN (_object : poV);
   --  function pfunc (_object : poV);
   --  ...

   --  Note that this must come after the record type declaration, since
   --  the specs refer to this type.

   procedure Expand_N_Protected_Type_Declaration (N : Node_Id) is
      Loc          : constant Source_Ptr := Sloc (N);
      Prottyp      : constant Entity_Id  := Defining_Identifier (N);
      Protnm       : constant Name_Id    := Chars (Prottyp);

      Pdef         : constant Node_Id    := Protected_Definition (N);
      --  This contains 2 lists; one for visible and one for private decls

      Rec_Decl     : Node_Id   := Build_Corresponding_Record (N, Prottyp, Loc);
      Rec_Ent      : Entity_Id := Defining_Identifier (Rec_Decl);
      Cdecls       : List_Id   := Component_Items (Component_List
                                    (Type_Definition (Rec_Decl)));

      Dent         : Entity_Id;
      Priv         : Node_Id;
      New_Priv     : Node_Id;
      Efam         : Entity_Id;
      Comp         : Node_Id;
      Sub          : Node_Id;
      Component    : Node_Id;
      Current_Node : Node_Id := N;
      Efam_Type    : Node_Id;
      Ecount       : constant Node_Id :=
                       Build_Entry_Count_Expression (Prottyp, Loc);

   begin
      --  Fill in the component declarations. First the _Object field.

      Component :=
        Make_Component_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uObject),
          Aliased_Present => True,

          Subtype_Indication =>
            Make_Subtype_Indication (
              Sloc => Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Protection), Loc),

              Constraint =>
                Make_Index_Or_Discriminant_Constraint (
                  Sloc => Loc,
                  Constraints => New_List (Ecount))));

      Append_To (Cdecls, Component);

      --  Next, the _Service field.

      if Has_Entries (Prottyp) then
         Component :=
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uService),
             Aliased_Present => True,
             Subtype_Indication =>
               New_Reference_To (RTE (RE_Service_Record), Loc));

         Append_To (Cdecls, Component);
      end if;

      --  Add components for entry families. For each entry family,
      --  create an anoymous type declaration with the same size, and
      --  analyze the type.

      Collect_Entry_Families (Loc, Cdecls, Current_Node, Prottyp);

      --  Add private data field components.

      if Present (Private_Declarations (Pdef)) then
         Priv := First (Private_Declarations (Pdef));

         while Present (Priv) loop
            if Nkind (Priv) = N_Component_Declaration then
               New_Priv := Make_Component_Declaration (Loc,
                 Defining_Identifier => Make_Defining_Identifier (Loc,
                   Chars => Chars (Defining_Identifier (Priv))),
                 Subtype_Indication =>
                   New_Reference_To (Etype (Defining_Identifier (Priv)), Loc),
                 Expression => Expression (Priv));

               Append_To (Cdecls, New_Priv);
            end if;
            Priv := Next (Priv);
         end loop;
      end if;

      --  Add the priority ceiling component if a priority pragma is present
      --  ??? Pure Cargo Cult; copied from Expand_N_Task_Type_Declaration.
      --      I don't know if all of this stuff has a analog in
      --      protected types or not.

      if Present (Pdef) and then Has_Priority_Pragma (Pdef) then
         Append_To (Cdecls,
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uPriority),
             Subtype_Indication => New_Reference_To (Standard_Integer, Loc),
             Expression => New_Copy (
               Expression (First (
                 Pragma_Argument_Associations (
                   Find_Task_Pragma (Pdef, Name_Priority)))))));
      end if;

      Insert_After (Current_Node, Rec_Decl);
      Current_Node := Rec_Decl;

      --  Analyze the record declaration immediately after construction,
      --  because the initialization procedure is needed for single object
      --  declarations before the next entity is analyzed (the freeze call
      --  that generates this initialization procedure is found below).

      Analyze (Rec_Decl);

      --  If protected object has any entries, build a service entries proc.

      if Has_Entries (Prottyp) then
         Sub :=
           Make_Subprogram_Declaration (Loc,
             Specification =>
               Build_Service_Specification (Loc, Protnm));

         Insert_After (Current_Node, Sub);
         Analyze (Sub);
         Current_Node := Sub;

         --  Set up discriminals and privals for use by the entry
         --  service procedure. All of the code from the entry bodies
         --  for this type will be deposited in this procedure, and will
         --  share these privals and discriminals.

         if Has_Discriminants (Prottyp) then
            Dent := First_Discriminant (Prottyp);
            while Present (Dent) loop
               Set_Entry_Discriminal (Dent,
                 Make_Defining_Identifier
                   (Loc, New_External_Name (Chars (Dent), 'D')));
               Dent := Next_Discriminant (Dent);
            end loop;
         end if;

         if Present (Private_Declarations (Pdef)) then
            Priv := First (Private_Declarations (Pdef));
            while Present (Priv) loop

               if Nkind (Priv) = N_Component_Declaration then
                  Set_Entry_Prival (Defining_Identifier (Priv),
                    Make_Defining_Identifier (Loc,
                      New_External_Name (
                        Chars (Defining_Identifier (Priv)), 'P')));
               end if;

               Priv := Next (Priv);
            end loop;
         end if;

      end if;

      --  Build two new procedure specifications for each protected
      --  subprogram; one to call from outside the object and one to
      --  call from inside.

      Comp := First (Visible_Declarations (Pdef));
      while Present (Comp) loop
         if Nkind (Comp) = N_Subprogram_Declaration then
            Sub :=
              Make_Subprogram_Declaration (Loc,
                Specification =>
                  Build_Protected_Sub_Specification
                    (Comp, Prottyp, Unprotected => True));

            Insert_After (Current_Node, Sub);
            Analyze (Sub);

            Set_Corresponding_Unprotected (
              Defining_Unit_Name (Specification (Comp)),
              Defining_Unit_Name (Specification (Sub)));

            --  Make the unprotected version of the subprogram available
            --  for expansion of interobject calls.

            Current_Node := Sub;

            Sub :=
              Make_Subprogram_Declaration (Loc,
                Specification =>
                  Build_Protected_Sub_Specification
                    (Comp, Prottyp, Unprotected => False));

            Insert_After (Current_Node, Sub);
            Analyze (Sub);
            Current_Node := Sub;
         end if;

         Comp := Next (Comp);
      end loop;

      --  Now we can freeze the corresponding record. This needs manually
      --  freezing, since it is really part of the protected type, and
      --  the protected type is frozen at this stage. We of course need
      --  the initialization procedure for this corresponding record type
      --  and we won't get it in time if we don't freeze now.

      Insert_List_After (Current_Node, Freeze_Entity (Rec_Ent, Loc));

      --  Complete the expansion of access types to the current protected
      --  type, if any were declared.
      --  ??? Pure cargo cult, imitated from Expand_N_Task_Type_Declaration.

      Expand_Previous_Access_Type (N, Prottyp);

   end Expand_N_Protected_Type_Declaration;

   --------------------------------
   -- Expand_N_Requeue_Statement --
   --------------------------------

   --  A requeue statement is expanded into one of four GNARLI operations,
   --  depending on the source and destination (task or protected object).
   --  In addition, code must be generated to jump around the remainder of
   --  processing for the original entry and, if the destination is a
   --  (different) protected object, to attempt to service it.
   --  The following illustrates the various cases:

   --  procedure ptypeS
   --    (_object : in out poV;
   --     P : out Boolean)
   --  is
   --     ...various declarations...
   --  begin
   --
   --     loop
   --        ...barrier calculation...
   --
   --        Next_Entry_Call (_object, Bnn, A, Enn);
   --        begin
   --           case Enn is
   --              when Null_Protected_Entry => exit;
   --
   --              -- Requeue from one protected entry body to an entry
   --              -- of a different protected object.
   --
   --              when 1 =>
   --                 <start of statement sequence for entry 1>
   --                 Abort_Defer;
   --                 Lock (new._object'Access);
   --                 Requeue_Protected_Entry (
   --                   _object._object'Access,
   --                   new._object'Access,
   --                   E,
   --                   Abort_Present);
   --                 newS (New, Pnn);
   --                 Unlock (new._object'Access);
   --                 Abort_Undefer;
   --                 Pnn := False;
   --                 goto Lnn;
   --                 <rest of statement sequence for entry 1>
   --
   --              --  Requeue from an entry body to a task entry.
   --
   --              when 2 =>
   --                 <start of statement sequence for entry 2>
   --                 Requeue_Protected_To_Task_Entry (
   --                   New._task_id,
   --                   E,
   --                   Abort_Present);
   --                 Pnn := False;
   --                 goto Lnn;
   --                 <rest of statement sequence for entry 2>
   --
   --              --  Requeue from an entry body to an entry of the
   --              --  same protected object.
   --
   --              when 3 =>
   --                 <start of statement sequence for entry 3>
   --                 Requeue_Protected_Entry (
   --                   _object._object'Access,
   --                   _object._object'Access,
   --                   E,
   --                   Abort_Present);
   --                 goto Lnn;
   --                 <rest of statement sequence for entry 3>
   --              ...
   --           end case;
   --           <<Lmm>>
   --           Complete_Entry_Body (_object._object'Access, Pnn);
   --        end;
   --        ...
   --        <<Lnn>> null;
   --        Cnn := Cnn or Pnn;
   --     end loop;
   --     P := Cnn;
   --  end ptypeS;

   --  Requeue of a task entry call to a task entry.
   --
   --  Accept_Call (E, Ann);
   --     <start of statement sequence for accept statement>
   --     Requeue_Task_Entry (New._task_id, E, Abort_Present);
   --     goto Lnn;
   --     <rest of statement sequence for accept statement>
   --  Complete_Rendezvous;
   --  <<Lnn>>

   --  Requeue of a task entry call to a protected entry.
   --
   --  Accept_Call (E, Ann);
   --     <start of statement sequence for accept statement>
   --     Abort_Defer;
   --     Lock (new._object'Access);
   --     Requeue_Task_To_Protected_Entry (
   --       new._object'Access,
   --       E,
   --       Abort_Present);
   --     newS (new, Pnn);
   --     Unlock (new._object'Access);
   --     Abort_Undefer;
   --     goto Lnn;
   --     <rest of statement sequence for accept statement>
   --  Complete_Rendezvous;
   --  <<Lnn>>

   --  Further details on these expansions can be found in
   --  Expand_N_Protected_Body and Expand_N_Accept_Statement.

   procedure Expand_N_Requeue_Statement (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Acc_Stat   : Node_Id;
      Concval    : Node_Id;
      Ename      : Node_Id;
      Index      : Node_Id;
      Conctyp    : Entity_Id;
      Oldent     : constant Entity_Id := Current_Scope;
      Oldtyp     : constant Entity_Id := Scope (Oldent);
      Lab_Node   : Node_Id;
      Rcall      : Node_Id;
      Abortable  : Node_Id;
      Goto_Stat  : Node_Id;
      Curr_Stat  : Node_Id;
      Self_Param : Node_Id;
      New_Param  : Node_Id;
      Params     : List_Id;
      RTS_Call   : Entity_Id;
      Pend_Exp   : Node_Id;

   begin
      if Abort_Present (N) then
         Abortable := New_Occurrence_Of (Standard_True, Loc);
      else
         Abortable := New_Occurrence_Of (Standard_False, Loc);
      end if;

      --  Set up the target object.

      Extract_Entry (N, Concval, Ename, Index);
      Conctyp := Etype (Concval);
      New_Param := Concurrent_Ref (Concval);

      --  The target entry index and abortable flag are the same for all cases.

      Params := New_List (
        Entry_Index_Expression (Loc, Entity (Ename), Index, Conctyp),
        Abortable);

      --  Figure out which GNARLI call and what additional parameters are
      --  needed.

      if Is_Task_Type (Oldtyp) then

         Unimplemented (N, "Requeue of task entry");

         if Is_Task_Type (Conctyp) then
            RTS_Call := RTE (RE_Requeue_Task_Entry);
         else
            pragma Assert (Is_Protected_Type (Conctyp));
            RTS_Call := RTE (RE_Requeue_Task_To_Protected_Entry);
            New_Param :=
              Make_Attribute_Reference (Loc,
                Prefix => New_Param,
                Attribute_Name => Name_Access);
         end if;

         Prepend (New_Param, Params);

      else
         pragma Assert (Is_Protected_Type (Oldtyp));
         Self_Param :=
           Make_Attribute_Reference (Loc,
             Prefix => Concurrent_Ref (New_Occurrence_Of (Oldtyp, Loc)),
             Attribute_Name => Name_Access);

         if Is_Task_Type (Conctyp) then

            Unimplemented (N, "Requeue to task entry");

            RTS_Call := RTE (RE_Requeue_Protected_To_Task_Entry);

         else
            pragma Assert (Is_Protected_Type (Conctyp));
            RTS_Call := RTE (RE_Requeue_Protected_Entry);
            New_Param :=
              Make_Attribute_Reference (Loc,
                Prefix => New_Param,
                Attribute_Name => Name_Access);
         end if;

         Prepend (New_Param, Params);
         Prepend (Self_Param, Params);
      end if;

      --  Create the GNARLI call.

      Rcall := Make_Procedure_Call_Statement (Loc,
        Name =>
          New_Occurrence_Of (RTS_Call, Loc),
        Parameter_Associations => Params);

      Rewrite_Substitute_Tree (N, Rcall);
      Analyze (N);


      if Is_Protected_Type (Oldtyp) then

         --  Build the goto statement to jump around the rest of the entry
         --  body, using the label attached to the body.

         Lab_Node := End_Of_Body (Corresponding_Body (Parent (Oldtyp)));

      else
         --  if the requeue is within a task, find the end label of the
         --  enclosing accept statement.

         Acc_Stat := Parent (N);
         while Nkind (Acc_Stat) /= N_Accept_Statement loop
            Acc_Stat := Parent (Acc_Stat);
         end loop;

         Lab_Node :=
            Last (Statements (Handled_Statement_Sequence (Acc_Stat)));
      end if;

      Goto_Stat :=
        Make_Goto_Statement (Loc,
          Name => New_Occurrence_Of (Entity (Identifier (Lab_Node)), Loc));

      Set_Analyzed (Goto_Stat);

      --  If the requeue is to a protected entry from another protected
      --  object or from a task, add the protection and entry service
      --  call needed for a protected entry call.

      if Is_Protected_Type (Conctyp)
        and then (Nkind (Concval) /= N_Identifier
                   or else not Is_Protected_Type (Entity (Concval)))
      then
         if Is_Task_Type (Oldtyp) then
            Curr_Stat :=
              Make_Procedure_Call_Statement (Loc,
                Name =>
                  New_Reference_To (RTE (RE_Abort_Defer), Loc),
                Parameter_Associations => Empty_List);

            Insert_Before (N, Curr_Stat);
            Analyze (Curr_Stat);
         end if;

         Curr_Stat :=
           Make_Procedure_Call_Statement (Loc,
             Name =>
               New_Reference_To (RTE (RE_Lock), Loc),
             Parameter_Associations => New_List (New_Copy_Tree (New_Param)));

         Insert_Before (N, Curr_Stat);
         Analyze (Curr_Stat);

         Insert_After (N, Goto_Stat);
         Curr_Stat :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Unlock), Loc),
             Parameter_Associations => New_List (New_Copy_Tree (New_Param)));

         Insert_After (N, Curr_Stat);
         Analyze (Curr_Stat);

         if Is_Task_Type (Oldtyp) then
            Curr_Stat :=
              Make_Procedure_Call_Statement (Loc,
                Name =>
                  New_Reference_To (RTE (RE_Abort_Undefer), Loc),
                Parameter_Associations => Empty_List);

            Insert_After (N, Curr_Stat);
            Analyze (Curr_Stat);
         end if;

         Pend_Exp :=
           New_Reference_To (
             Pending_Serviced (Corresponding_Body (Parent (Oldtyp))), Loc);

         Set_Assignment_OK (Pend_Exp);

         Curr_Stat :=
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (
               Defining_Unit_Name (
                 Service_Entries_Definition (Base_Type (Etype (Concval)))),
               Loc),

             Parameter_Associations => New_List (
               Make_Attribute_Reference (Loc,
                 Attribute_Name => Name_Address,
                 Prefix => New_Copy_Tree (Concval)),
               Pend_Exp));
         Insert_After (N, Curr_Stat);
         Analyze (Curr_Stat);

      else
         Insert_After (N, Goto_Stat);
      end if;

   end Expand_N_Requeue_Statement;

   -------------------------------
   -- Expand_N_Selective_Accept --
   -------------------------------

   procedure Expand_N_Selective_Accept (N : Node_Id) is
      Accept_Case    : List_Id;
      Accept_List    : List_Id := New_List;

      Alt            : Node_Id;
      Alts           : constant List_Id := Select_Alternatives (N);
      Alt_List       : List_Id := New_List;
      Alt_Stats      : List_Id;
      Ann            : Entity_Id := Empty;


      Block          : Node_Id;
      Decls          : List_Id := New_List;
      Stats          : List_Id := New_List;

      Body_List      : List_Id := New_List;
      Trailing_List  : List_Id := New_List;

      Choices        : List_Id;
      Else_Present   : Boolean := False;
      Terminate_Alt  : Node_Id := Empty;
      Select_Mode    : Node_Id;

      Delay_Case     : List_Id;
      Delay_Count    : Integer := 0;
      Delay_Val      : Entity_Id;
      Delay_Index    : Entity_Id;
      Delay_Min      : Entity_Id;
      Delay_Alt_List : List_Id := New_List;
      Delay_List     : List_Id := New_List;

      End_Lab        : Node_Id;
      Index          : Int := 1;
      Lab            : Node_Id;
      Loc            : constant Source_Ptr := Sloc (N);
      Num            : Int;
      Num_Accept     : Nat := 0;
      Proc           : Node_Id;
      Q              : Node_Id;
      Qnam           : Entity_Id := Make_Defining_Identifier (Loc,
                                    New_External_Name ('S', 0));
      X              : Node_Id;
      Xnam           : Entity_Id := Make_Defining_Identifier (Loc,
                                    New_External_Name ('X', 1));

      -----------------------
      -- Local subprograms --
      -----------------------

      procedure Add_Accept (Alt : Node_Id);
      --  Process a single accept statement in a select alternative. Build
      --  procedure for body of accept, and add entry to dispatch table with
      --  expression for guard, in preparation for call to run time select.

      function Make_And_Declare_Label (Num : Int) return Node_Id;
      --  Manufacture a label using Num as a serial number and declare it.
      --  The declaration is appended to Decls. The label marks the trailing
      --  statements of an accept or delay alternative.

      function Make_Select_Call (Select_Mode : Entity_Id) return Node_Id;
      --  Build call to Selective_Wait runtime routine.

      procedure Process_Delay_Alternative (Alt : Node_Id; Index : Int);
      --  Add code to compare value of delay with previous values, and
      --  generate case entry for trailing statements.

      procedure Process_Accept_Alternative
        (Alt   : Node_Id;
         Index : Int;
         Proc  : Node_Id);
      --  Add code to call corresponding procedure, and branch to
      --  trailing statements, if any.

      ----------------
      -- Add_Accept --
      ----------------

      procedure Add_Accept (Alt : Node_Id) is
         Acc_Stm   : constant Node_Id    := Accept_Statement (Alt);
         Ename     : constant Node_Id    := Entry_Direct_Name (Acc_Stm);
         Eent      : constant Entity_Id  := Entity (Ename);
         Index     : constant Node_Id    := Entry_Index (Acc_Stm);
         Null_Body : Node_Id;
         Proc_Body : Node_Id;
         Expr      : Node_Id;

      begin
         if No (Ann) then
            Ann := Node (Last_Elmt (Accept_Address (Eent)));
         end if;

         if Present (Condition (Alt)) then
            Expr := Make_Conditional_Expression (Loc,
              New_List (Condition (Alt),
                        Entry_Index_Expression
                          (Loc, Eent, Index, Scope (Eent)),
                        New_Reference_To (RTE (RE_Null_Task_Entry), Loc)));
         else
            Expr :=
              Entry_Index_Expression
                (Loc, Eent, Index, Scope (Eent));
         end if;

         if Present (Handled_Statement_Sequence (Accept_Statement (Alt))) then
            Null_Body := New_Reference_To (Standard_False, Loc);

            Proc_Body :=
              Make_Subprogram_Body (Loc,
                Specification =>
                  Make_Procedure_Specification (Loc,
                    Defining_Unit_Name =>
                      Make_Defining_Identifier (Loc,
                        New_External_Name (Chars (Ename), 'A', Num_Accept))),

               Declarations => New_List,
               Handled_Statement_Sequence =>
                 Build_Accept_Body
                   (Handled_Statement_Sequence (Accept_Statement (Alt)), Loc));

            Append (Proc_Body, Body_List);

         else
            Null_Body := New_Reference_To (Standard_True,  Loc);
         end if;

         Append_To (Accept_List,
           Make_Aggregate (Loc, Expressions => New_List (Null_Body, Expr)));

         Num_Accept := Num_Accept + 1;

      end Add_Accept;

      -------------------------------
      -- Process_Delay_Alternative --
      -------------------------------

      procedure Process_Delay_Alternative (Alt : Node_Id; Index : Int) is
         Choices   : List_Id;
         Delay_Alt : List_Id;

      begin
         --  Determine the smallest specified delay.
         --  for each delay alternative generate:

         --    if guard-expression then
         --       Delay_Val := delay-expression;
         --       if Delay_Val < Delay_Min then
         --          Delay_Min   := Delay_Val;
         --          Delay_Index := Index;
         --       end if;
         --    end if;

         --  The enclosing if-statement is omitted if there is no guard.

         if Delay_Count = 1 then
            Delay_Alt := New_List (
              Make_Assignment_Statement (Loc,
                Name => New_Reference_To (Delay_Min, Loc),
                Expression => Expression (Delay_Statement (Alt))));

         else
            Delay_Alt := New_List (
              Make_Assignment_Statement (Loc,
                Name => New_Reference_To (Delay_Val, Loc),
                Expression => Expression (Delay_Statement (Alt))));

            Append_To (Delay_Alt,
              Make_If_Statement (Loc,
                Condition =>
                  Make_Op_Lt (Loc,
                    Left_Opnd  => New_Reference_To (Delay_Val, Loc),
                    Right_Opnd => New_Reference_To (Delay_Min, Loc)),

                Then_Statements => New_List (
                  Make_Assignment_Statement (Loc,
                    Name       => New_Reference_To (Delay_Min, Loc),
                    Expression => New_Reference_To (Delay_Val, Loc)),

                  Make_Assignment_Statement (Loc,
                    Name       => New_Reference_To (Delay_Index, Loc),
                    Expression => Make_Integer_Literal (Loc,
                      Intval   => UI_From_Int (Index))))));
         end if;

         if Present (Condition (Alt)) then
            Delay_Alt := New_List (
              Make_If_Statement (Loc,
                Condition => Condition (Alt),
                Then_Statements => Delay_Alt));
         end if;

         Append_List (Delay_Alt, Delay_List);

         --  If the delay alternative has a statement part, add a
         --  choice to the case statements for delays.

         if Present (Statements (Alt)) then

            if Delay_Count = 1 then
               Append_List (Statements (Alt), Delay_Alt_List);

            else
               Choices := New_List (
                 Make_Integer_Literal (Loc,
                   Intval => UI_From_Int (Index)));

               Append_To (Delay_Alt_List,
                 Make_Case_Statement_Alternative (Loc,
                   Discrete_Choices => Choices,
                   Statements => Statements (Alt)));
            end if;

         elsif Delay_Count = 1 then

            --  If the single delay has no trailing statements, add a branch
            --  to the exit label to the selective wait.

            Delay_Alt_List := New_List (
              Make_Goto_Statement (Loc,
                Name => New_Copy (Identifier (End_Lab))));

         end if;
      end Process_Delay_Alternative;

      --------------------------------
      -- Process_Accept_Alternative --
      --------------------------------

      procedure Process_Accept_Alternative
        (Alt   : Node_Id;
         Index : Int;
         Proc  : Node_Id)
      is
         Choices   : List_Id;
         Alt_Stats : List_Id;

      begin
         Alt_Stats := No_List;

         if Present (Handled_Statement_Sequence (Accept_Statement (Alt))) then
            Choices := New_List (
              Make_Integer_Literal (Loc,
                Intval => UI_From_Int (Index)));

            Alt_Stats := New_List (
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (
                  Defining_Unit_Name (Specification (Proc)), Loc)));
         end if;

         if Statements (Alt) /= Empty_List then

            if No (Alt_Stats) then

               --  Accept with no body, followed by trailing statements.

               Choices := New_List (
                 Make_Integer_Literal (Loc,
                   Intval => UI_From_Int (Index)));

               Alt_Stats := New_List;
            end if;

            --  After the call, if any, branch to to trailing statements.
            --  We create a label for each, as well as the corresponding
            --  label declaration.

            Lab := Make_And_Declare_Label (Index);
            Append_To (Alt_Stats,
              Make_Goto_Statement (Loc,
                Name => New_Copy (Identifier (Lab))));

            Append (Lab, Trailing_List);
            Append_List (Statements (Alt), Trailing_List);
            Append_To (Trailing_List,
              Make_Goto_Statement (Loc,
                Name => New_Copy (Identifier (End_Lab))));
         end if;

         if Present (Alt_Stats) then

            --  Procedure call. and/or trailing statements.

            Append_To (Alt_List,
              Make_Case_Statement_Alternative (Loc,
                Discrete_Choices => Choices,
                Statements => Alt_Stats));
         end if;
      end Process_Accept_Alternative;

      ----------------------
      -- Make_Select_Call --
      ----------------------

      function Make_Select_Call (Select_Mode : Entity_Id) return Node_Id is
         Params : List_Id := New_List;

      begin
         Append (New_Reference_To (Qnam, Loc), Params);
         Append (Select_Mode, Params);
         Append (New_Reference_To (Ann, Loc), Params);
         Append (New_Reference_To (Xnam, Loc), Params);

         return
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Selective_Wait), Loc),
             Parameter_Associations => Params);
      end Make_Select_Call;

      -----------------------------
      --  Make_And_Declare_Label --
      -----------------------------

      function Make_And_Declare_Label (Num : Int) return Node_Id is
         Lab_Id : Node_Id;

      begin
         Lab_Id := Make_Identifier (Loc, New_External_Name ('L', Num));
         Lab := Make_Label (Loc, Lab_Id);

         Append_To (Decls,
           Make_Implicit_Label_Declaration (Loc,
             Defining_Identifier  =>
               Make_Defining_Identifier (Loc, Chars (Lab_Id)),
             Label => Lab));

         return Lab;
      end Make_And_Declare_Label;

   --  Start of processing for Expand_N_Selective_Accept

   begin
      --  First insert some declarations before the select. The first is:

      --    Ann : Address

      --  This variable holds the parameters passed to the accept body. This
      --  declaration has already been inserted by the time we get here by
      --  a call to Expand_Accept_Declarations made from the semantics when
      --  processing the first accept statement contained in the select. We
      --  can find this entity as Accept_Address (E), where E is any of the
      --  entries references by contained accept statements.

      --  The first step is to scan the list of Selective_Accept_Statements
      --  to find this entity, and also count the number of accepts, and
      --  determine if terminated, delay or else is present:

      Num := 0;

      Alt := First (Alts);
      while Present (Alt) loop

         if Nkind (Alt) = N_Accept_Alternative then
            Num := Num + 1;
            Add_Accept (Alt);

         elsif Nkind (Alt) = N_Delay_Alternative then
            Delay_Count   := Delay_Count + 1;

         elsif Nkind (Alt) = N_Terminate_Alternative then
            Terminate_Alt := Alt;
         end if;

         Alt := Next (Alt);
      end loop;

      Else_Present := Present (Else_Statements (N));

      --  At the same time (see procedure Add_Accept) we build the accept list:

      --    Qnn : Accept_List (1 .. num-select) := (
      --          (null-body, entry-index),
      --          (null-body, entry-index),
      --          ..
      --          (null_body, entry-index));

      --  In the above declaration, null-body is True if the corresponding
      --  accept has no body, and false otherwise. The entry is either the
      --  entry index expression if there is no guard, or if a guard is
      --  present, then a conditional expression of the form:

      --    (if guard then entry-index else Null_Task_Entry)

      --  If a guard is statically known to be false, the entry can simply
      --  be omitted from the accept list.

      Q :=
        Make_Object_Declaration (Loc,
          Defining_Identifier => Qnam,
          Object_Definition =>
            New_Reference_To (RTE (RE_Accept_List_Access), Loc),

          Expression =>
            Make_Allocator (Loc,
              Expression =>
                Make_Qualified_Expression (Loc,
                  Subtype_Mark =>
                    New_Reference_To (RTE (RE_Accept_List), Loc),
                  Expression =>
                    Make_Aggregate (Loc, Expressions => Accept_List))));

      Append (Q, Decls);

      --  Then we declare the variable that holds the index for the accept
      --  that will be selected for service:

      --    Xnn : Select_Index;

      X :=
        Make_Object_Declaration (Loc,
          Defining_Identifier => Xnam,
          Object_Definition =>
            New_Reference_To (RTE (RE_Select_Index), Loc));

      Append (X, Decls);

      --  After this follow  procedure declarations for each accept body.

      --    procedure Pnn is
      --    begin
      --       ...
      --    end;

      --  where the ... are statements from the corresponding procedure body.
      --  No parameters are involved, since the parameters are passed via Ann
      --  and the parameter references have already been expanded to be direct
      --  references to Ann (see Exp_Ch2.Expand_Entry_Parameter). Furthermore,
      --  any embedded tasking statements (which would normally be illegal in
      --  procedures, have been converted to calls to the tasking runtime so
      --  there is no problem in putting them into procedures.

      --  The original accept statement has been expanded into a block in
      --  the same fashion as for simple accepts (see Build_Accept_Body).

      --  Note: we don't really need to build these procedures for the case
      --  where no delay statement is present, but is is just as easy to
      --  build them unconditionally, and not significantly inefficient,
      --  since if they are short they will be inlined anyway.

      --  The procedure declarations have been assembled in Body_List.

      --  If delays are present, we must compute the required delay.
      --  We first generate the declarations:

      --    Delay_Index : Boolean := 0;
      --    Delay_Min   : Some_Time_Type.Time;
      --    Delay_Val   : Some_Time_Type.Time;

      --  Delay_Index will be set to the index of the minimum delay, i.e. the
      --   active delay that is actually chosen as the basis for the possible
      --   delay if an immediate rendez-vous is not possible.
      --   In the most common case there is a single delay statement, and this
      --   is handled specially.

      if Delay_Count > 0 then

         --  Generate the required declarations

         Delay_Val   := Make_Defining_Identifier (Loc,
                                    New_External_Name ('D', 1));
         Delay_Index := Make_Defining_Identifier (Loc,
                                    New_External_Name ('D', 2));
         Delay_Min   := Make_Defining_Identifier (Loc,
                                    New_External_Name ('D', 3));
         Append_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Delay_Val,
             Object_Definition => New_Reference_To (Standard_Duration, Loc)));

         Append_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Delay_Index,
             Object_Definition => New_Reference_To (Standard_Integer, Loc)));

         Append_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Delay_Min,
             Object_Definition => New_Reference_To (Standard_Duration, Loc)));

         Append_To (Delay_List,
           Make_Assignment_Statement (Loc,
             Name => New_Reference_To (Delay_Min, Loc),
             Expression =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Reference_To (Standard_Duration, Loc),
                 Attribute_Name => Name_Last)));
      end if;

      if Present (Terminate_Alt) then

         --  If the terminate alternative guard is False, use
         --  Simple_Mode; otherwise use Terminate_Mode.

         if Present (Condition (Terminate_Alt)) then
            Select_Mode := Make_Conditional_Expression (Loc,
              New_List (Condition (Terminate_Alt),
                        New_Reference_To (RTE (RE_Terminate_Mode), Loc),
                        New_Reference_To (RTE (RE_Simple_Mode), Loc)));
         else
            Select_Mode := New_Reference_To (RTE (RE_Terminate_Mode), Loc);
         end if;

      elsif Else_Present or Delay_Count > 0 then
         Select_Mode := New_Reference_To (RTE (RE_Else_Mode), Loc);

      else
         Select_Mode := New_Reference_To (RTE (RE_Simple_Mode), Loc);
      end if;

      Append (Make_Select_Call (Select_Mode), Stats);

      --  Now generate code to act on the result. There is an entry
      --  in this case for each accept statement with a non-null body,
      --  followed by a branch to the statements that follow the Accept.
      --  In the absence of delay alternatives, we generate:

      --    case X is
      --      when No_Rendezvous =>  --  omitted if simple mode
      --         goto Lab0;

      --      when 1 =>
      --         P1n;
      --         goto Lab1;

      --      when 2 =>
      --         P2n;
      --         goto Lab2;

      --      when others =>
      --         goto Exit;
      --    end case;
      --
      --    Lab0: Else_Statements;
      --    goto exit;

      --    Lab1:  Trailing_Statements1;
      --    goto Exit;
      --
      --    Lab2:  Trailing_Statements2;
      --    goto Exit;
      --    ...
      --    Exit:
      --

      --  Generate label for common exit.

      End_Lab := Make_And_Declare_Label (Num + 1);

      --  First entry is the default case, when no rendezvous is possible.

      Choices := New_List (
        New_Reference_To (RTE (RE_No_Rendezvous), Loc));

      if Else_Present then

         --  If no rendezvous is possible, the else part is executed.

         Lab := Make_And_Declare_Label (0);
         Alt_Stats := New_List (
           Make_Goto_Statement (Loc,
             Name => New_Copy (Identifier (Lab))));

         Append (Lab, Trailing_List);
         Append_List (Else_Statements (N), Trailing_List);
         Append_To (Trailing_List,
           Make_Goto_Statement (Loc,
             Name => New_Copy (Identifier (End_Lab))));
      else
         Alt_Stats := New_List (
           Make_Goto_Statement (Loc,
             Name => New_Copy (Identifier (End_Lab))));
      end if;

      Append_To (Alt_List,
        Make_Case_Statement_Alternative (Loc,
          Discrete_Choices => Choices,
          Statements => Alt_Stats));

      --  We make use of the fact that Accept_Index is an integer type,
      --  and generate successive literals for entries for each accept.
      --  Only those for which there is a body or trailing statements are
      --  given a case entry.

      Alt := First (Select_Alternatives (N));
      Proc := First (Body_List);

      while Present (Alt) loop

         if Nkind (Alt) = N_Accept_Alternative then
            Process_Accept_Alternative (Alt, Index, Proc);

            if Present
              (Handled_Statement_Sequence (Accept_Statement (Alt)))
            then
               Proc := Next (Proc);
            end if;

         elsif Nkind (Alt) = N_Delay_Alternative then
            Process_Delay_Alternative (Alt, Index);
         end if;

         Index := Index + 1;
         Alt := Next (Alt);
      end loop;

      --  An others choice is always added to the main case, as well
      --  as the delay case (to satisfy the compiler).

      Append_To (Alt_List,
        Make_Case_Statement_Alternative (Loc,
          Discrete_Choices =>
            New_List (Make_Others_Choice (Loc)),
          Statements       =>
            New_List (Make_Goto_Statement (Loc,
              Name => New_Copy (Identifier (End_Lab))))));

      Accept_Case := New_List (
        Make_Case_Statement (Loc,
          Expression   => New_Reference_To (Xnam, Loc),
          Alternatives => Alt_List));

      Append_List (Trailing_List, Accept_Case);
      Append (End_Lab, Accept_Case);
      Append_List (Body_List, Decls);

      --  Construct case statement for trailing statements of delay
      --  alternatives, if there are several of them.

      if Delay_Count > 1 then
         Append_To (Delay_Alt_List,
           Make_Case_Statement_Alternative (Loc,
             Discrete_Choices =>
               New_List (Make_Others_Choice (Loc)),
             Statements       =>
               New_List (Make_Null_Statement (Loc))));

         Delay_Case := New_List (
           Make_Case_Statement (Loc,
             Expression   => New_Reference_To (Delay_Index, Loc),
             Alternatives => Delay_Alt_List));
      else
         Delay_Case := Delay_Alt_List;
      end if;

      --  If there are no delay alternatives, we append the case statement
      --  to the statement list.

      if Delay_Count = 0 then
         Append_List (Accept_Case, Stats);

      else
         --  If delay alternatives are present, we generate:

         --    find minimum delay.
         --    if X = No_Rendezvous then
         --      Select
         --         Delay Delay_Min;
         --      then Abort
         --         Selective_Wait (Q, Simple_mode, P, X'Access);
         --      end select;
         --    end if;
         --
         --    if X = No_Rendezvous then
         --      case statement for delay statements.
         --    else
         --      case statement for accept alternatives.
         --    end if;

         declare
            First_Try : Node_Id;
            Asynch    : Node_Id;
            Cases     : Node_Id;

         begin
            Append_List (Delay_List, Stats);

            Asynch :=
              Make_Asynchronous_Select (Loc,
                Triggering_Alternative =>
                  Make_Triggering_Alternative (Loc,
                    Triggering_Statement =>
                       Make_Delay_Relative_Statement (Loc,
                         Expression => New_Reference_To (Delay_Min, Loc)),
                    Statements => New_List),

                Abortable_Part =>
                  Make_Abortable_Part (Loc,
                    Statements =>  New_List (
                      Make_Select_Call (
                        New_Reference_To (RTE (RE_Simple_Mode), Loc))
                 )));

            First_Try :=
              Make_If_Statement (Loc,
                Condition => Make_Op_Eq (Loc,
                  Left_Opnd  => New_Reference_To (Xnam, Loc),
                  Right_Opnd =>
                    New_Reference_To (RTE (RE_No_Rendezvous), Loc)),
                Then_Statements => New_List (Asynch));

            Append (First_Try, Stats);

            Cases :=
              Make_If_Statement (Loc,
                Condition => Make_Op_Eq (Loc,
                  Left_Opnd  => New_Reference_To (Xnam, Loc),
                  Right_Opnd =>
                    New_Reference_To (RTE (RE_No_Rendezvous), Loc)),

                Then_Statements => Delay_Case,
                Else_Statements => Accept_Case);

            Append (Cases, Stats);
         end;
      end if;

      --  Replace accept statement with appropriate block

      Block :=
        Make_Block_Statement (Loc,
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Stats));

      Rewrite_Substitute_Tree (N, Block);
      Analyze (N);

      --  Note: have to worry more about abort deferral in above code ???

      --  Final step is to unstack the Accept_Address entries for all accept
      --  statements appearing in accept alternatives in the select statement

      Alt := First (Alts);
      while Present (Alt) loop
         if Nkind (Alt) = N_Accept_Alternative then
            Remove_Last_Elmt (Accept_Address
              (Entity (Entry_Direct_Name (Accept_Statement (Alt)))));
         end if;

         Alt := Next (Alt);
      end loop;

   end Expand_N_Selective_Accept;

   --------------------------------------
   -- Expand_N_Single_Task_Declaration --
   --------------------------------------

   --  Single task declarations should never be present after semantic
   --  analysis, since we expect them to be replaced by a declaration of
   --  an anonymous task type, followed by a declaration of the task
   --  object. We include this routine to make sure that is happening!

   procedure Expand_N_Single_Task_Declaration (N : Node_Id) is
   begin
      pragma Assert (False); null;
   end Expand_N_Single_Task_Declaration;

   ------------------------
   -- Expand_N_Task_Body --
   ------------------------

   --  Given a task body

   --    task body tname is
   --       declarations
   --    begin
   --       statements
   --    end x;

   --  This expansion routine converts it into a procedure and sets the
   --  elaboration flag for the procedure to true, to represent the fact
   --  that the task body is now elaborated:

   --    procedure tnameB (_Task : access tnameV) is
   --       discriminal : dtype renames _Task.discriminant;

   --    begin
   --       System.Task_Stages.Complete_Activation;
   --       statements
   --    at end
   --       System.Task_Stages.Complete_Task;
   --    end tnameB;

   --    tnameE := True;

   --  In addition, if the task body is an activator, then a call to
   --  activate tasks is added at the start of the statements, before
   --  the call to Complete_Activation, and if in addition the task is
   --  a master then it must be established as a master. These calls are
   --  inserted and analyzed in Expand_Cleanup_Actions, when the
   --  Handled_Sequence_Of_Statements is expanded.

   --  There is one discriminal declaration line generated for each
   --  discriminant that is present to provide an easy reference point
   --  for discriminant references inside the body (see Exp_Ch2.Expand_Name).

   --  Note on relationship to GNARLI definition. In the GNARLI definition,
   --  task body procedures have a profile (Arg : System.Address). That is
   --  needed because GNARLI has to use the same access-to-subprogram type
   --  for all task types. We depend here on knowing that in GNAT, passing
   --  an address argument by value is identical to passing a a record value
   --  by access (in either case a single pointer is passed), so even though
   --  this procedure has the wrong profile. In fact it's all OK, since the
   --  callings sequence is identical.

   procedure Expand_N_Task_Body (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Ttyp : constant Entity_Id  := Corresponding_Spec (N);
      Call : Node_Id;
      Pend : Node_Id;

   begin
      Add_Discriminal_Declarations (Declarations (N), Ttyp, Name_uTask, Loc);

      Pend :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Pending_Serviced (Defining_Identifier (N)),
          Object_Definition =>
            New_Reference_To (Standard_Boolean, Loc));
      Append (Pend, Declarations (N));
      Analyze (Pend);

      --  The statement part has already been protected with an at_end and
      --  cleanup actions. The call to Complete_Activation must be placed
      --  at the head of the sequence of statements of that block. The
      --  declarations have been merged in this sequence of statements but
      --  the first real statement is accessible from the First_Real_Statement
      --  field (which was set for exactly this purpose)

      Call := Build_Runtime_Call (Loc, RE_Complete_Activation);
      Insert_Before
        (First_Real_Statement (Handled_Statement_Sequence (N)), Call);
      Analyze (Call);

      Rewrite_Substitute_Tree (N,
        Make_Subprogram_Body (Loc,
          Specification => Build_Task_Proc_Specification (Ttyp),
          Declarations  => Declarations (N),
          Handled_Statement_Sequence => Handled_Statement_Sequence (N)));

      Analyze (N);

      --  Set elaboration flag immediately after task body. If the body
      --  is a subunit, the flag is set in  the declarative part that
      --  contains the stub.

      if Nkind (Parent (N)) /= N_Subunit then
         Insert_After (N,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Identifier (Loc, New_External_Name (Chars (Ttyp), 'E')),
             Expression => New_Reference_To (Standard_True, Loc)));
      end if;

   end Expand_N_Task_Body;

   ------------------------------------
   -- Expand_N_Task_Type_Declaration --
   ------------------------------------

   --  We have several things to do. First we must create a Boolean flag used
   --  to mark if the body is elaborated yet. This variable gets set to True
   --  when the body of the task is elaborated (we can't rely on the normal
   --  ABE mechanism for the task body, since we need to pass an access to
   --  this elaboration boolean to the runtime routines).

   --    taskE : aliased Boolean := False;

   --  Next a variable is declared to hold the task stack size (either
   --  the default, which is the initial value given here, or a value that
   --  is set by a pragma Storage_Size appearing later on.

   --    taskZ : Size_Type := Unspecified_Size;

   --  Next we create a corresponding record type declaration used to represent
   --  values of this task. The general form of this type declaration is

   --    type taskV (discriminants) is record
   --      _Task_Id     : Task_Id;
   --      entry_family : array (bounds) of Void;
   --      _Priority    : Integer   := priority_expression;
   --      _Size        : Size_Type := Size_Type (size_expression);
   --    end record;

   --  The discriminants are present only if the corresponding task type has
   --  discriminants, and they exactly mirror the task type discriminants.

   --  The Id field is always present. It contains the Task_Id value, as
   --  set by the call to Create_Task. Note that although the task is
   --  limited, the task value record type is not limited, so there is no
   --  problem in passing this field as an out parameter to Create_Task.

   --  One entry_family component is present for each entry family in the
   --  task definition. The bounds correspond to the bounds of the entry
   --  family (which may depend on discriminants). The element type is
   --  void, since we only need the bounds information for determining
   --  the entry index. Note that the use of an anonymous array would
   --  normally be illegal in this context, but this is a parser check,
   --  and the semantics is quite prepared to handle such a case.

   --  The Size field is present only if a Storage_Size pragma appears in
   --  the task definition. The expression captures the argument that was
   --  present in the pragma, and is used to override the task stack size
   --  otherwise associated with the task type.

   --  The Priority field is present only if a Priority or Interrupt_Priority
   --  pragma appears in the task definition. The expression captures the
   --  argument that was present in the pragma, and is used to provide
   --  the Size parameter to the call to Create_Task.

   --  When a task is declared, an instance of the task value record is
   --  created. The elaboration of this declaration creates the correct
   --  bounds for the entry families, and also evaluates the size and
   --  priority expressions if needed. The initialization routine for
   --  the task type itself then calls Create_Task with appropriate
   --  parameters to initialize the value of the Task_Id field.

   --  Note: the address of this record is passed as the "Discriminants"
   --  parameter for Create_Task. Since Create_Task merely passes this onto
   --  the body procedure, it does not matter that it does not quite match
   --  the GNARLI model of what is being passed (the record contains more
   --  than just the discriminants, but the discriminants can be found from
   --  the record value).

   --  The Entity_Id for this created record type is placed in the
   --  Corresponding_Record_Type field of the associated task type entity.

   --  Next we create a procedure specification for the task body procedure:

   --    procedure taskB (_Task : access taskV);

   --  Note that this must come after the record type declaration, since
   --  the spec refers to this type. It turns out that the initialization
   --  procedure for the value type references the task body spec, but that's
   --  fine, since it won't be generated till the freeze point for the type,
   --  which is certainly after the task body spec declaration.

   --  Finally, we set the task index value field of the entry attribute in
   --  the case of a simple entry.

   procedure Expand_N_Task_Type_Declaration (N : Node_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Tasktyp   : constant Entity_Id  := Etype (Defining_Identifier (N));
      Tasknm    : constant Name_Id    := Chars (Tasktyp);
      Taskdef   : constant Node_Id    := Task_Definition (N);
      Proc_Spec : Node_Id;

      Rec_Decl  : Node_Id   := Build_Corresponding_Record (N, Tasktyp, Loc);
      Rec_Ent   : Entity_Id := Defining_Identifier (Rec_Decl);
      Cdecls    : List_Id   := Component_Items (Component_List
                                 (Type_Definition (Rec_Decl)));

      Efam      : Entity_Id;
      Elab_Decl : Node_Id;
      Size_Decl : Node_Id;
      Body_Decl : Node_Id;

   begin
      --  First create the elaboration variable

      Elab_Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, New_External_Name (Tasknm, 'E')),
          Aliased_Present      => True,
          Object_Definition    => New_Reference_To (Standard_Boolean, Loc),
          Expression           => New_Reference_To (Standard_False, Loc));
      Insert_After (N, Elab_Decl);

      --  Next create the declaration of the size variable (tasknmZ)

      Size_Decl :=
        Make_Object_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, New_External_Name (Tasknm, 'Z')),
          Object_Definition => New_Reference_To (RTE (RE_Size_Type), Loc),
          Expression => New_Reference_To (RTE (RE_Unspecified_Size), Loc));

      Insert_After (Elab_Decl, Size_Decl);

      --  Next build the rest of the corresponding record declaration.
      --  This is done last, since the corresponding record initialization
      --  procedure will reference the previously created entities.

      --  Fill in the component declarations. First the _Task_Id field:

      Append_To (Cdecls,
        Make_Component_Declaration (Loc,
          Defining_Identifier =>
            Make_Defining_Identifier (Loc, Name_uTask_Id),
          Subtype_Indication => New_Reference_To (RTE (RE_Task_ID), Loc)));

      --  Add components for entry families

      Collect_Entry_Families (Loc, Cdecls, Size_Decl, Tasktyp);

      --  Add the priority component if a priority pragma is present

      if Present (Taskdef) and then Has_Priority_Pragma (Taskdef) then
         Append_To (Cdecls,
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uPriority),
             Subtype_Indication => New_Reference_To (Standard_Integer, Loc),
             Expression => New_Copy (
               Expression (First (
                 Pragma_Argument_Associations (
                   Find_Task_Pragma (Taskdef, Name_Priority)))))));
      end if;

      --  Add the task_size component if a priority pragma is present

      if Present (Taskdef)
        and then Has_Storage_Size_Pragma (Taskdef)
      then
         Append_To (Cdecls,
           Make_Component_Declaration (Loc,
             Defining_Identifier =>
               Make_Defining_Identifier (Loc, Name_uSize),

             Subtype_Indication => New_Reference_To (RTE (RE_Size_Type), Loc),

             Expression =>
               Make_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (RTE (RE_Size_Type), Loc),
                 Expression => Relocate_Node (
                   Expression (First (
                     Pragma_Argument_Associations (
                       Find_Task_Pragma (Taskdef, Name_Storage_Size))))))));
      end if;

      Insert_After (Size_Decl, Rec_Decl);

      --  Analyze the record declaration immediately after construction,
      --  because the initialization procedure is needed for single task
      --  declarations before the next entity is analyzed.

      Analyze (Rec_Decl);

      --  Create the declaration of the task body procedure

      Proc_Spec := Build_Task_Proc_Specification (Tasktyp);
      Body_Decl :=
        Make_Subprogram_Declaration (Loc,
          Specification => Proc_Spec);

      Insert_After (Rec_Decl, Body_Decl);

      --  Now we can freeze the corresponding record. This needs manually
      --  freezing, since it is really part of the task type, and the task
      --  type is frozen at this stage. We of course need the initialization
      --  procedure for this corresponding record type and we won't get it
      --  in time if we don't freeze now.

      Insert_List_After (Body_Decl, Freeze_Entity (Rec_Ent, Loc));

      --  Complete the expansion of access types to the current task
      --  type, if any were declared.

      Expand_Previous_Access_Type (N, Tasktyp);

   end Expand_N_Task_Type_Declaration;

   -------------------------------
   -- Expand_N_Timed_Entry_Call --
   -------------------------------

   --  The timed entry call:

   --     select
   --        T.E;
   --        S1;
   --     or
   --        Delay D;
   --        S2;
   --     end select;

   --  is expanded into an asynchronous select:

   --    declare
   --       Timed_Out : Boolean := True;
   --    begin
   --       select
   --          T.E;
   --          Timed_Out := False;
   --          S1;
   --       then abort
   --          Delay D;
   --          S2;
   --       end select;
   --       if Timed_Out then
   --          S2;
   --       end if;
   --    end

   procedure Expand_N_Timed_Entry_Call (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      E_Call  : constant Node_Id    := Entry_Call_Statement
                                           (Entry_Call_Alternative (N));
      E_Stats : constant List_Id    := Statements (Entry_Call_Alternative (N));
      D_Stat  : constant Node_Id    := Delay_Statement (Delay_Alternative (N));
      Stats   : constant List_Id    := Statements (Delay_Alternative (N));

      Asynch  : Node_Id;
      Blk     : Node_Id;
      T_Out   : Entity_Id := Make_Defining_Identifier
                                            (Loc, New_Internal_Name ('T'));

   begin
      Asynch :=
         Make_Asynchronous_Select (Loc,
           Triggering_Alternative =>
             Make_Triggering_Alternative (Loc,
               Triggering_Statement => E_Call,
               Statements => New_List (
                 Make_Assignment_Statement (Loc,
                   Name => New_Reference_To (T_Out, Loc),
                   Expression => New_Reference_To (Standard_False, Loc)))),

           Abortable_Part =>
             Make_Abortable_Part (Loc,
               Statements =>  New_List (D_Stat)));

      Append_List (E_Stats, Statements (Triggering_Alternative (Asynch)));

      Blk :=
        Make_Block_Statement (Loc,
          Declarations => New_List (
              Make_Object_Declaration (Loc,
                Defining_Identifier => T_Out,
                Object_Definition => New_Reference_To (Standard_Boolean, Loc),
                Expression => New_Reference_To (Standard_True, Loc))),

           Handled_Statement_Sequence =>
             Make_Handled_Sequence_Of_Statements (Loc,
             Statements => New_List (
                 Asynch,
                 Make_If_Statement (Loc,
                   Condition => New_Reference_To (T_Out, Loc),
                   Then_Statements => Stats))));

      Rewrite_Substitute_Tree (N, Blk);
      Analyze (N);

   end Expand_N_Timed_Entry_Call;

   -----------------------------------
   -- Expand_Task_Body_Declarations --
   -----------------------------------

   procedure Expand_Task_Body_Declarations (N : Node_Id) is
      Pend : Entity_Id;

   begin
      --  Set up a pending serviced flag. This is returned by entry service
      --  procedures. This flag is actually a dummy; it is used in expanding
      --  requeue statements on protected entries. Since the requeue statement
      --  is expanded before the task, this defining identifier is set up
      --  beforehand and used to define the flag when the body is expanded.
      --  This is the same mechanism used by protected bodies, where the flag
      --  is not a dummy. To allow it to be used in referenced before its
      --  declaration is analyzed, its Etype is set here.

      Pend := Make_Defining_Identifier (Sloc (N), New_Internal_Name ('P'));
      Set_Etype (Pend, Standard_Boolean);
      Set_Pending_Serviced (Defining_Identifier (N), Pend);

   end Expand_Task_Body_Declarations;

   -------------------
   -- Extract_Entry --
   -------------------

   procedure Extract_Entry
     (N       : Node_Id;
      Concval : out Node_Id;
      Ename   : out Node_Id;
      Index   : out Node_Id)
   is
      Nam : constant Node_Id := Name (N);

   begin
      --  For a simple entry, the name is a selected component, with the
      --  prefix being the task value, and the selector being the entry.

      if Nkind (Nam) = N_Selected_Component then
         Concval := Prefix (Nam);
         Ename   := Selector_Name (Nam);
         Index   := Empty;

         --  For a member of an entry family, the name is an indexed
         --  component where the prefix is a selected component,
         --  whose prefix in turn is the task value, and whose
         --  selector is the entry family. The single expression in
         --  the expressions list of the indexed component is the
         --  subscript for the family.

      else
         pragma Assert (Nkind (Nam) = N_Indexed_Component);
         Concval := Prefix (Prefix (Nam));
         Ename   := Selector_Name (Prefix (Nam));
         Index   := First (Expressions (Nam));
      end if;

   end Extract_Entry;

   ----------------------
   -- Find_Task_Pragma --
   ----------------------

   function Find_Task_Pragma (T : Node_Id; P : Name_Id) return Node_Id is
      N : Node_Id;

   begin
      N := First (Visible_Declarations (T));

      while Present (N) loop
         if Nkind (N) = N_Pragma and then Chars (N) = P then
            return N;
         else
            N := Next (N);
         end if;
      end loop;

      N := First (Private_Declarations (T));

      while Present (N) loop
         if Nkind (N) = N_Pragma and then Chars (N) = P then
            return N;
         else
            N := Next (N);
         end if;
      end loop;

      pragma Assert (False);
   end Find_Task_Pragma;

   -------------------------------------
   -- Make_Initialize_Protection_Call --
   -------------------------------------

   function Make_Initialize_Protection_Call
     (Protect_Rec : Entity_Id)
      return        Node_Id
   is
      Loc    : constant Source_Ptr := Sloc (Protect_Rec);
      Pdef   : Node_Id;
      Pdec   : Node_Id;
      Ptyp   : Node_Id;
      Pnam   : Name_Id;
      Args   : List_Id;

   begin
      Ptyp := Corresponding_Concurrent_Type (Protect_Rec);
      Pnam := Chars (Ptyp);

      --  Get protected declaration. In the case of a task type declaration,
      --  this is simply the parent of the protected type entity.
      --  In the single protected object
      --  declaration, this parent will be the implicit type, and we can find
      --  the corresponding single protected object declaration by
      --  searching forward in the declaration list in the tree.
      --  ??? I am not sure that the test for N_Single_Protected_Declaration
      --      is needed here. Nodes of this type should have been removed
      --      during semantic analysis.

      Pdec := Parent (Ptyp);

      while Nkind (Pdec) /= N_Protected_Type_Declaration
        and then Nkind (Pdec) /= N_Single_Protected_Declaration
      loop
         Pdec := Next (Pdec);
      end loop;

      --  Now we can find the object definition from this declaration

      Pdef := Protected_Definition (Pdec);

      --  Build the parameter list for the call. Note that _Init is the name
      --  of the formal for the object to be initialized, which is the task
      --  value record itself.

      Args := New_List;

      --  Object parameter. This is a pointer to the object of type
      --  Protection used by the GNARL to control the protected object.

      Append_To (Args,
        Make_Attribute_Reference (Loc,
          Prefix =>
            Make_Selected_Component (Loc,
              Prefix => Make_Identifier (Loc, Name_uInit),
              Selector_Name => Make_Identifier (Loc, Name_uObject)),
          Attribute_Name => Name_Access));

      --  Priority parameter. Set to Unspecified_Priority unless there is a
      --  priority pragma, in which case we take the value from the pragma.

      if Present (Pdef)
        and then Has_Priority_Pragma (Pdef)
      then
         Append_To (Args,
           Make_Selected_Component (Loc,
             Prefix => Make_Identifier (Loc, Name_uInit),
             Selector_Name => Make_Identifier (Loc, Name_uPriority)));

      else
         Append_To (Args,
           New_Reference_To (RTE (RE_Unspecified_Priority), Loc));
      end if;

      --  Service parameter. This parameter allows the entry service
      --  procedure to be called when the object effected cannot
      --  be known until runtime. The entry service field is only
      --  present if the protected object has entries; if it does not,
      --  a null address is passed.

      if Has_Entries (Ptyp) then
         Append_To (Args,
           Make_Attribute_Reference (Loc,
             Prefix =>
               Make_Selected_Component (Loc,
                 Prefix => Make_Identifier (Loc, Name_uInit),
                 Selector_Name => Make_Identifier (Loc, Name_uService)),

             Attribute_Name => Name_Address));

      else
         Append_To (Args, New_Reference_To (RTE (RE_Null_Address), Loc));
      end if;

      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Initialize_Protection), Loc),
          Parameter_Associations => Args);

   end Make_Initialize_Protection_Call;

   ---------------------------
   -- Make_Task_Create_Call --
   ---------------------------

   function Make_Task_Create_Call (Task_Rec : Entity_Id) return Node_Id is
      Loc    : constant Source_Ptr := Sloc (Task_Rec);
      Tdef   : Node_Id;
      Tdec   : Node_Id;
      Ttyp   : Node_Id;
      Tnam   : Name_Id;
      Args   : List_Id;
      Ecount : Node_Id;

   begin
      Ttyp := Corresponding_Concurrent_Type (Task_Rec);
      Tnam := Chars (Ttyp);

      --  Get task declaration. In the case of a task type declaration, this
      --  is simply the parent of the task type entity. In the single task
      --  declaration, this parent will be the implicit type, and we can find
      --  the corresponding single task declaration by searching forward in
      --  the declaration list in the tree.
      --  ??? I am not sure that the test for N_Single_Task_Declaration
      --      is needed here. Nodes of this type should have been removed
      --      during semantic analysis.

      Tdec := Parent (Ttyp);

      while Nkind (Tdec) /= N_Task_Type_Declaration
        and then Nkind (Tdec) /= N_Single_Task_Declaration
      loop
         Tdec := Next (Tdec);
      end loop;

      --  Now we can find the task definition from this declaration

      Tdef := Task_Definition (Tdec);

      --  Build the parameter list for the call. Note that _Init is the name
      --  of the formal for the object to be initialized, which is the task
      --  value record itself.

      Args := New_List;

      --  Size parameter. If no Storage_Size pragma is present, then
      --  the size is taken from the taskZ variable for the type, which
      --  is either Unspecified_Size, or has been reset by the use of
      --  a Storage_Size attribute definition clause. If a pragma is
      --  present, then the size is taken from the _Size field of the
      --  task value record, which was set from the pragma value.

      if Present (Tdef)
        and then Has_Storage_Size_Pragma (Tdef)
      then
         Append_To (Args,
           Make_Selected_Component (Loc,
             Prefix => Make_Identifier (Loc, Name_uInit),
             Selector_Name => Make_Identifier (Loc, Name_uSize)));

      else
         Append_To (Args,
           Make_Identifier (Loc, New_External_Name (Tnam, 'Z')));
      end if;

      --  Priority parameter. Set to Unspecified_Priority unless there is a
      --  priority pragma, in which case we take the value from the pragma.

      if Present (Tdef)
        and then Has_Priority_Pragma (Tdef)
      then
         Append_To (Args,
           Make_Selected_Component (Loc,
             Prefix => Make_Identifier (Loc, Name_uInit),
             Selector_Name => Make_Identifier (Loc, Name_uPriority)));

      else
         Append_To (Args,
           New_Reference_To (RTE (RE_Unspecified_Priority), Loc));
      end if;

      --  Number of entries. This is an expression of the form:
      --
      --    n + _Init.a'Length + _Init.a'B'Length + ...
      --
      --  where a,b... are the entry family names for the task definition

      Ecount := Build_Entry_Count_Expression (Ttyp, Loc);
      Append_To (Args, Ecount);

      --  Master parameter. This is a reference to the _Master parameter of
      --  the initialization procedure.

      Append_To (Args, Make_Identifier (Loc, Name_uMaster));

      --  State parameter. This is a pointer to the task body procedure. We get
      --  the required value by taking the address of the task body procedure,
      --  and then converting it (with an unchecked conversion) to the type
      --  required by the task kernel. See description of Expand_Task_Body
      --  for further details.

      Append_To (Args,
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark =>
            New_Reference_To (RTE (RE_Task_Procedure_Access), Loc),

          Expression =>
            Make_Attribute_Reference (Loc,
              Prefix =>
                Make_Identifier (Loc, New_External_Name (Tnam, 'B')),
              Attribute_Name => Name_Address)));

      --  Discriminants parameter. This is just the address of the task
      --  value record itself (which contains the discriminant values

      Append_To (Args,
        Make_Attribute_Reference (Loc,
          Prefix => Make_Identifier (Loc, Name_uInit),
          Attribute_Name => Name_Address));

      --  Elaborated parameter. This is an access to the elaboration Boolean

      Append_To (Args,
        Make_Attribute_Reference (Loc,
          Prefix => Make_Identifier (Loc, New_External_Name (Tnam, 'E')),
          Attribute_Name => Name_Access));

      --  Chain parameter. This is a reference to the _Chain parameter of
      --  the initialization procedure.

      Append_To (Args, Make_Identifier (Loc, Name_uChain));

      --  Created_Task parameter. This is the _Task_Id field of the task
      --  record value

      Append_To (Args,
        Make_Selected_Component (Loc,
          Prefix => Make_Identifier (Loc, Name_uInit),
          Selector_Name => Make_Identifier (Loc, Name_uTask_Id)));

      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Create_Task), Loc),
          Parameter_Associations => Args);

   end Make_Task_Create_Call;

   ----------------------
   -- Set_Discriminals --
   ----------------------

   procedure Set_Discriminals
     (Dec : Node_Id;
      Op  : Node_Id;
      Loc : Source_Ptr)
   is
      D       : Entity_Id;
      Pdef    : Entity_Id;
      D_Minal : Entity_Id;

   begin
      pragma Assert (Nkind (Dec) = N_Protected_Type_Declaration);
      Pdef := Defining_Identifier (Dec);

      if Has_Discriminants (Pdef) then
         D := First_Discriminant (Pdef);

         while Present (D) loop
            if Nkind (Op) = N_Entry_Body then
               D_Minal := Entry_Discriminal (D);

            else
               D_Minal :=
                 Make_Defining_Identifier (Loc,
                   Chars => New_External_Name (Chars (D), 'D'));
            end if;

            Set_Ekind (D_Minal, E_Constant);
            Set_Etype (D_Minal, Etype (D));
            Set_Discriminal (D, D_Minal);

            D := Next_Discriminant (D);
         end loop;
      end if;
   end Set_Discriminals;

   -----------------
   -- Set_Privals --
   -----------------

   procedure Set_Privals
      (Dec : Node_Id;
       Op : Node_Id;
       Loc : Source_Ptr)
   is
      P         : Entity_Id;
      Priv      : Entity_Id;
      Def       : Node_Id;
      Body_Ent  : Entity_Id;

      Prec_Decl : constant Node_Id :=
                    Parent (Corresponding_Record_Type
                             (Defining_Identifier (Dec)));

      Prec_Def  : constant Entity_Id := Type_Definition (Prec_Decl);

      Obj_Decl  : constant Node_Id :=
                    First (Component_Items (Component_List (Prec_Def)));

      P_Subtype : constant Entity_Id :=
                    Etype (Defining_Identifier (Obj_Decl));

   begin
      pragma Assert (Nkind (Dec) = N_Protected_Type_Declaration);
      pragma Assert
        (Nkind (Op) = N_Subprogram_Body or else Nkind (Op) = N_Entry_Body);
      Def := Protected_Definition (Dec);

      if Present (Private_Declarations (Def)) then

         P := First (Private_Declarations (Def));
         while Present (P) loop
            if Nkind (P) = N_Component_Declaration then
               if Nkind (Op) = N_Entry_Body then
                  Priv := Entry_Prival (Defining_Identifier (P));

               else
                  Priv :=
                    Make_Defining_Identifier (Loc,
                      New_External_Name
                        (Chars (Defining_Identifier (P)), 'P'));
               end if;

               Set_Ekind (Priv, E_Variable);
               Set_Etype (Priv, Etype (Defining_Identifier (P)));
               Set_Protected_Operation (Defining_Identifier (P), Op);
               Set_Prival (Defining_Identifier (P), Priv);
            end if;

            P := Next (P);
         end loop;
      end if;

      --  There is one more implicit private declaration: the object
      --  itself. A "prival" for this is attached to the protected
      --  body defining identifier.

      Body_Ent := Corresponding_Body (Dec);

      if Nkind (Op) = N_Entry_Body then
         Priv := Entry_Object_Ref (Body_Ent);

      else
         Priv :=
           Make_Defining_Identifier (Loc,
             New_External_Name (Chars (Body_Ent), 'R'));

         --  Set the Etype to the implicit subtype of Protection created when
         --  the protected type declaration was expanded. This node will not
         --  be analyzed until it is used as the defining identifier for the
         --  renaming declaration in the protected operation body, and it will
         --  be needed in the references expanded before that body is expanded.
         --  Since the Protection field is aliased, set Is_Aliased as well.

         Set_Etype (Priv, P_Subtype);
         Set_Is_Aliased (Priv);

      end if;

      Set_Object_Ref (Body_Ent, Priv);

   end Set_Privals;

end Exp_Ch9;
