------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                  S C N                                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.75 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Csets;    use Csets;
with Debug;    use Debug;
with Errout;   use Errout;
with Features; use Features;
with Lib;      use Lib;
with Namet;    use Namet;
with Opt;      use Opt;
with Output;   use Output;
with Scans;    use Scans;
with Sinput;   use Sinput;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Style;
with Widechar; use Widechar;

with System.Parameters;
with System.WCh_Con; use System.WCh_Con;

package body Scn is

   use Ascii;
   --  Make control characters visible

   Used_As_Identifier : array (Token_Type) of Boolean;
   --  Flags set True if a given keyword is used as an identifier (used to
   --  make sure that we only post an error message for incorrect use of a
   --  keyword as an identifier once for a given keyword).

   Max_Allowed_Line_Length : Nat := System.Parameters.Max_Line_Length;
   --  Maximum allowed line length (maybe reset by Style.Set_Max_Line_Length)

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Double_Char_Token (C : Character) return Boolean;
   --  This function is used for double character tokens like := or <>. It
   --  checks if the character following Source (Scan_Ptr) is C, and if so
   --  bumps Scan_Ptr past the pair of characters and returns True. A space
   --  between the two characters is also recognized with an appropriate
   --  error message being issued. If C is not present, False is returned.
   --  Note that Double_Char_Token can only be used for tokens defined in
   --  the Ada syntax.

   procedure Error_Illegal_Character;
   --  Give illegal character error, Scan_Ptr points to character. On return,
   --  Scan_Ptr is bumped past the illegal character.

   procedure Error_Illegal_Wide_Character;
   --  Give illegal character in wide character escape sequence message. On
   --  return, Scan_Ptr is bumped past the illegal character, which may still
   --  leave us pointing to junk, not much we can do if the escape sequence
   --  is messed up!

   procedure Error_Long_Line;
   --  Signal error of excessively long line

   procedure Error_No_Double_Underline;
   --  Signal error of double underline character

   procedure Nlit;
   --  This is the procedure for scanning out numeric literals

   function Set_Start_Column return Column_Number;
   --  This routine is called with Scan_Ptr pointing to the first character
   --  of a line. On exit, Scan_Ptr is advanced to the first non-blank
   --  character of this line (or to the terminating format effector if the
   --  line contains no non-blank characters), and the returned result is the
   --  column number of this non-blank character (zero origin), which is the
   --  value to be stored in the Start_Column scan variable.

   procedure Slit;
   --  This is the procedure for scanning out string literals

   --------------
   -- Subunits --
   --------------

   --  For some reason, these must come early in the file or we run into an
   --  infinite loop in GNAT, to be looked at some time ???

   procedure Nlit is separate;

   procedure Slit is separate;

   ----------------------------
   -- Determine_Token_Casing --
   ----------------------------

   function Determine_Token_Casing return Casing_Type is

      All_Lower : Boolean := True;
      --  Set False if upper case letter found

      All_Upper : Boolean := True;
      --  Set False if lower case letter found

      Mixed : Boolean := True;
      --  Set False if exception to mixed case rule found (lower case letter
      --  at start or after underline, or upper case letter elsewhere).

      Decisive : Boolean := False;
      --  Set True if at least one instance of letter not after underline

      After_Und : Boolean := True;
      --  True at start of string, and after an underline character

   begin
      for S in Token_Ptr .. Scan_Ptr - 1 loop
         if Source (S) = '_' or else Source (S) = '.' then
            After_Und := True;

         elsif Is_Lower_Case_Letter (Source (S)) then
            All_Upper := False;

            if not After_Und then
               Decisive := True;
            else
               After_Und := False;
               Mixed := False;
            end if;

         elsif Is_Upper_Case_Letter (Source (S)) then
            All_Lower := False;

            if not After_Und then
               Decisive := True;
               Mixed := False;
            else
               After_Und := False;
            end if;
         end if;
      end loop;

      --  Now we can figure out the result from the flags we set in that loop

      if All_Lower then
         return All_Lower_Case;

      elsif not Decisive then
         return Unknown;

      elsif All_Upper then
         return All_Upper_Case;

      elsif Mixed then
         return Mixed_Case;

      else
         return Unknown;
      end if;
   end Determine_Token_Casing;

   -----------------------
   -- Double_Char_Token --
   -----------------------

   function Double_Char_Token (C : Character) return Boolean is
   begin
      if Source (Scan_Ptr + 1) = C then
         Scan_Ptr := Scan_Ptr + 2;
         return True;

      elsif Source (Scan_Ptr + 1) = ' '
        and then Source (Scan_Ptr + 2) = C
      then
         Scan_Ptr := Scan_Ptr + 1;
         Error_Msg_S ("no space allowed here");
         Scan_Ptr := Scan_Ptr + 2;
         return True;

      else
         return False;
      end if;
   end Double_Char_Token;

   -----------------------------
   -- Error_Illegal_Character --
   -----------------------------

   procedure Error_Illegal_Character is
   begin
      Error_Msg_S ("illegal character");
      Scan_Ptr := Scan_Ptr + 1;
   end Error_Illegal_Character;

   ----------------------------------
   -- Error_Illegal_Wide_Character --
   ----------------------------------

   procedure Error_Illegal_Wide_Character is
   begin
      Error_Msg_S ("illegal character in wide character escape sequence");
      Scan_Ptr := Scan_Ptr + 1;
   end Error_Illegal_Wide_Character;

   ---------------------
   -- Error_Long_Line --
   ---------------------

   procedure Error_Long_Line is
   begin
      Error_Msg ("this line is too long", Current_Line_Start);
   end Error_Long_Line;

   -------------------------------
   -- Error_No_Double_Underline --
   -------------------------------

   procedure Error_No_Double_Underline is
   begin
      Error_Msg_S ("two consecutive underlines not permitted");
   end Error_No_Double_Underline;

   ------------------------
   -- Initialize_Scanner --
   ------------------------

   procedure Initialize_Scanner (Unit : Unit_Number_Type) is
   begin
      --  Set up Token_Type values in Names Table entries for reserved keywords
      --  We use the Pos value of the Token_Type value. Note we are relying on
      --  the fact that Token_Type'Val (0) is not a reserved word!

      Set_Name_Table_Byte (Name_Abort,      Token_Type'Pos (Tok_Abort));
      Set_Name_Table_Byte (Name_Abs,        Token_Type'Pos (Tok_Abs));
      Set_Name_Table_Byte (Name_Abstract,   Token_Type'Pos (Tok_Abstract));
      Set_Name_Table_Byte (Name_Accept,     Token_Type'Pos (Tok_Accept));
      Set_Name_Table_Byte (Name_Access,     Token_Type'Pos (Tok_Access));
      Set_Name_Table_Byte (Name_And,        Token_Type'Pos (Tok_And));
      Set_Name_Table_Byte (Name_Aliased,    Token_Type'Pos (Tok_Aliased));
      Set_Name_Table_Byte (Name_All,        Token_Type'Pos (Tok_All));
      Set_Name_Table_Byte (Name_Array,      Token_Type'Pos (Tok_Array));
      Set_Name_Table_Byte (Name_At,         Token_Type'Pos (Tok_At));
      Set_Name_Table_Byte (Name_Begin,      Token_Type'Pos (Tok_Begin));
      Set_Name_Table_Byte (Name_Body,       Token_Type'Pos (Tok_Body));
      Set_Name_Table_Byte (Name_Case,       Token_Type'Pos (Tok_Case));
      Set_Name_Table_Byte (Name_Constant,   Token_Type'Pos (Tok_Constant));
      Set_Name_Table_Byte (Name_Declare,    Token_Type'Pos (Tok_Declare));
      Set_Name_Table_Byte (Name_Delay,      Token_Type'Pos (Tok_Delay));
      Set_Name_Table_Byte (Name_Delta,      Token_Type'Pos (Tok_Delta));
      Set_Name_Table_Byte (Name_Digits,     Token_Type'Pos (Tok_Digits));
      Set_Name_Table_Byte (Name_Do,         Token_Type'Pos (Tok_Do));
      Set_Name_Table_Byte (Name_Else,       Token_Type'Pos (Tok_Else));
      Set_Name_Table_Byte (Name_Elsif,      Token_Type'Pos (Tok_Elsif));
      Set_Name_Table_Byte (Name_End,        Token_Type'Pos (Tok_End));
      Set_Name_Table_Byte (Name_Entry,      Token_Type'Pos (Tok_Entry));
      Set_Name_Table_Byte (Name_Exception,  Token_Type'Pos (Tok_Exception));
      Set_Name_Table_Byte (Name_Exit,       Token_Type'Pos (Tok_Exit));
      Set_Name_Table_Byte (Name_For,        Token_Type'Pos (Tok_For));
      Set_Name_Table_Byte (Name_Function,   Token_Type'Pos (Tok_Function));
      Set_Name_Table_Byte (Name_Generic,    Token_Type'Pos (Tok_Generic));
      Set_Name_Table_Byte (Name_Goto,       Token_Type'Pos (Tok_Goto));
      Set_Name_Table_Byte (Name_If,         Token_Type'Pos (Tok_If));
      Set_Name_Table_Byte (Name_In,         Token_Type'Pos (Tok_In));
      Set_Name_Table_Byte (Name_Is,         Token_Type'Pos (Tok_Is));
      Set_Name_Table_Byte (Name_Limited,    Token_Type'Pos (Tok_Limited));
      Set_Name_Table_Byte (Name_Loop,       Token_Type'Pos (Tok_Loop));
      Set_Name_Table_Byte (Name_Mod,        Token_Type'Pos (Tok_Mod));
      Set_Name_Table_Byte (Name_New,        Token_Type'Pos (Tok_New));
      Set_Name_Table_Byte (Name_Not,        Token_Type'Pos (Tok_Not));
      Set_Name_Table_Byte (Name_Null,       Token_Type'Pos (Tok_Null));
      Set_Name_Table_Byte (Name_Of,         Token_Type'Pos (Tok_Of));
      Set_Name_Table_Byte (Name_Or,         Token_Type'Pos (Tok_Or));
      Set_Name_Table_Byte (Name_Others,     Token_Type'Pos (Tok_Others));
      Set_Name_Table_Byte (Name_Out,        Token_Type'Pos (Tok_Out));
      Set_Name_Table_Byte (Name_Package,    Token_Type'Pos (Tok_Package));
      Set_Name_Table_Byte (Name_Pragma,     Token_Type'Pos (Tok_Pragma));
      Set_Name_Table_Byte (Name_Private,    Token_Type'Pos (Tok_Private));
      Set_Name_Table_Byte (Name_Procedure,  Token_Type'Pos (Tok_Procedure));
      Set_Name_Table_Byte (Name_Protected,  Token_Type'Pos (Tok_Protected));
      Set_Name_Table_Byte (Name_Raise,      Token_Type'Pos (Tok_Raise));
      Set_Name_Table_Byte (Name_Range,      Token_Type'Pos (Tok_Range));
      Set_Name_Table_Byte (Name_Record,     Token_Type'Pos (Tok_Record));
      Set_Name_Table_Byte (Name_Rem,        Token_Type'Pos (Tok_Rem));
      Set_Name_Table_Byte (Name_Renames,    Token_Type'Pos (Tok_Renames));
      Set_Name_Table_Byte (Name_Requeue,    Token_Type'Pos (Tok_Requeue));
      Set_Name_Table_Byte (Name_Return,     Token_Type'Pos (Tok_Return));
      Set_Name_Table_Byte (Name_Reverse,    Token_Type'Pos (Tok_Reverse));
      Set_Name_Table_Byte (Name_Select,     Token_Type'Pos (Tok_Select));
      Set_Name_Table_Byte (Name_Separate,   Token_Type'Pos (Tok_Separate));
      Set_Name_Table_Byte (Name_Subtype,    Token_Type'Pos (Tok_Subtype));
      Set_Name_Table_Byte (Name_Tagged,     Token_Type'Pos (Tok_Tagged));
      Set_Name_Table_Byte (Name_Task,       Token_Type'Pos (Tok_Task));
      Set_Name_Table_Byte (Name_Terminate,  Token_Type'Pos (Tok_Terminate));
      Set_Name_Table_Byte (Name_Then,       Token_Type'Pos (Tok_Then));
      Set_Name_Table_Byte (Name_Type,       Token_Type'Pos (Tok_Type));
      Set_Name_Table_Byte (Name_Until,      Token_Type'Pos (Tok_Until));
      Set_Name_Table_Byte (Name_Use,        Token_Type'Pos (Tok_Use));
      Set_Name_Table_Byte (Name_When,       Token_Type'Pos (Tok_When));
      Set_Name_Table_Byte (Name_While,      Token_Type'Pos (Tok_While));
      Set_Name_Table_Byte (Name_With,       Token_Type'Pos (Tok_With));
      Set_Name_Table_Byte (Name_Xor,        Token_Type'Pos (Tok_Xor));

      --  Initialize scan control variables

      Current_Source_File       := Source_Index (Unit);
      Source                    := Source_Text (Current_Source_File);
      Current_Source_Unit       := Unit;
      Scan_Ptr                  := Source'First;
      Token                     := No_Token;
      Token_Ptr                 := Source'First;
      Current_Line_Start        := Source'First;
      Token_Node                := Empty;
      Token_Name                := No_Name;
      Start_Column              := Set_Start_Column;
      First_Non_Blank_Location  := Scan_Ptr;

      if Style_Check then
         Style.Set_Max_Line_Length (Max_Allowed_Line_Length);
      end if;

      --  Set default for Comes_From_Source. All nodes built now until we
      --  reenter the analyzer will have Comes_From_Source set to True

      Set_Comes_From_Source_Default (True);

      --  Scan initial token (note this initializes Prev_Token, Prev_Token_Ptr)

      Scan;

      --  Clear flags for reserved words used as indentifiers

      for I in Token_Type loop
         Used_As_Identifier (I) := False;
      end loop;

   end Initialize_Scanner;

   ----------
   -- Scan --
   ----------

   procedure Scan is
   begin
      Prev_Token := Token;
      Prev_Token_Ptr := Token_Ptr;
      Token_Name := Error_Name;

      --  The following loop runs more than once only if a format effector
      --  (tab, vertical tab, form  feed, line feed, carriage return) is
      --  encountered and skipped, or some error situation, such as an
      --  illegal character, is encountered.

      loop
         --  Skip past blanks, loop is opened up for speed

         while Source (Scan_Ptr) = ' ' loop

            if Source (Scan_Ptr + 1) /= ' ' then
               Scan_Ptr := Scan_Ptr + 1;
               exit;
            end if;

            if Source (Scan_Ptr + 2) /= ' ' then
               Scan_Ptr := Scan_Ptr + 2;
               exit;
            end if;

            if Source (Scan_Ptr + 3) /= ' ' then
               Scan_Ptr := Scan_Ptr + 3;
               exit;
            end if;

            if Source (Scan_Ptr + 4) /= ' ' then
               Scan_Ptr := Scan_Ptr + 4;
               exit;
            end if;

            if Source (Scan_Ptr + 5) /= ' ' then
               Scan_Ptr := Scan_Ptr + 5;
               exit;
            end if;

            if Source (Scan_Ptr + 6) /= ' ' then
               Scan_Ptr := Scan_Ptr + 6;
               exit;
            end if;

            if Source (Scan_Ptr + 7) /= ' ' then
               Scan_Ptr := Scan_Ptr + 7;
               exit;
            end if;

            Scan_Ptr := Scan_Ptr + 8;
         end loop;

         --  We are now at a non-blank character, which is the first character
         --  of the token we will scan, and hence the value of Token_Ptr.

         Token_Ptr := Scan_Ptr;

         --  Here begins the main case statement which transfers control on
         --  the basis of the non-blank character we have encountered.

         case Source (Scan_Ptr) is

         --  Line terminator characters

         when CR | LF | FF | VT => Line_Terminator_Case : begin
            if Int (Scan_Ptr) - Int (Current_Line_Start)
                           > Max_Allowed_Line_Length
            then
               Error_Long_Line;
            end if;

            if Style_Check then Style.Check_Line_Terminator; end if;

            declare
               Physical : Boolean;

            begin
               Skip_Line_Terminators (Scan_Ptr, Physical);

               --  If we are at start of physical line, update scan pointers
               --  to reflect the start of the new line.

               Current_Line_Start       := Scan_Ptr;
               Start_Column             := Set_Start_Column;
               First_Non_Blank_Location := Scan_Ptr;
            end;
         end Line_Terminator_Case;

         --  Horizontal tab, just skip past it

         when HT =>
            if Style_Check then Style.Check_HT; end if;
            Scan_Ptr := Scan_Ptr + 1;

         --  End of file character

         when EOF =>
            Token := Tok_EOF;
            return;

         --  Ampersand

         when '&' =>
            if Source (Scan_Ptr + 1) = '&' then
               Error_Msg_S ("'&'& should be `AND THEN`");
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_And;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Ampersand;
               return;
            end if;

         --  Asterisk (can be multiplication operator or double asterisk
         --  which is the exponentiation compound delimtier).

         when '*' =>
            if Source (Scan_Ptr + 1) = '*' then
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_Double_Asterisk;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Asterisk;
               return;
            end if;

         --  Colon, which can either be an isolated colon, or part of an
         --  assignment compound delimiter.

         when ':' =>
            if Double_Char_Token ('=') then
               Token := Tok_Colon_Equal;
               if Style_Check then Style.Check_Colon_Equal; end if;
               return;

            elsif Source (Scan_Ptr + 1) = '-'
              and then Source (Scan_Ptr + 2) /= '-'
            then
               Token := Tok_Colon_Equal;
               Error_Msg (":- should be :=", Scan_Ptr);
               Scan_Ptr := Scan_Ptr + 2;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Colon;
               if Style_Check then Style.Check_Colon; end if;
               return;
            end if;

         --  Left parenthesis

         when '(' =>
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Left_Paren;
            if Style_Check then Style.Check_Left_Paren; end if;
            return;

         --  Left bracket or left brace, treated as left paren

         when '[' | '{' =>
            Error_Msg_S ("illegal character, replaced by ""(""");
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Left_Paren;
            return;

         --  Comma

         when ',' =>
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Comma;
            if Style_Check then Style.Check_Comma; end if;
            return;

         --  Dot, which is either an isolated period, or part of a double
         --  dot compound delimiter sequence. We also check for the case of
         --  a digit following the period, to give a better error message.

         when '.' =>
            if Source (Scan_Ptr + 1) = '.' then
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_Dot_Dot;
               if Style_Check then Style.Check_Dot_Dot; end if;
               return;

            elsif Source (Scan_Ptr + 1) in '0' .. '9' then
               Error_Msg_S ("numeric literal cannot start with point");
               Scan_Ptr := Scan_Ptr + 1;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Dot;
               return;
            end if;

         --  Equal, which can either be an equality operator, or part of the
         --  arrow (=>) compound delimiter.

         when '=' =>
            if Double_Char_Token ('>') then
               Token := Tok_Arrow;
               if Style_Check then Style.Check_Arrow; end if;
               return;

            elsif Source (Scan_Ptr + 1) = '=' then
               Error_Msg_S ("== should be =");
               Scan_Ptr := Scan_Ptr + 1;
            end if;

            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Equal;
            return;

         --  Greater than, which can be a greater than operator, greater than
         --  or equal operator, or first character of a right label bracket.

         when '>' =>
            if Double_Char_Token ('=') then
               Token := Tok_Greater_Equal;
               return;

            elsif Double_Char_Token ('>') then
               Token := Tok_Greater_Greater;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Greater;
               return;
            end if;

         --  Less than, which can be a less than operator, less than or equal
         --  operator, or the first character of a left label bracket, or the
         --  first character of a box (<>) compound delimiter.

         when '<' =>
            if Double_Char_Token ('=') then
               Token := Tok_Less_Equal;
               return;

            elsif Double_Char_Token ('>') then
               Token := Tok_Box;
               if Style_Check then Style.Check_Box; end if;
               return;

            elsif Double_Char_Token ('<') then
               Token := Tok_Less_Less;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Less;
               return;
            end if;

         --  Minus, which is either a subtraction operator, or the first
         --  character of double minus starting a comment

         when '-' => Minus_Case : begin
            if Source (Scan_Ptr + 1) = '>' then
               Error_Msg_S ("-> should be =>");
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_Arrow;
               return;

            elsif Source (Scan_Ptr + 1) /= '-' then
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Minus;
               return;

            --  Comment

            else -- Source (Scan_Ptr + 1) = '-' then
               if Style_Check then Style.Check_Comment; end if;
               Scan_Ptr := Scan_Ptr + 2;

               --  Loop to scan comment (this loop runs more than once only if
               --  a horizontal tab or other non-graphic character is scanned)

               loop
                  --  Scan to non graphic character (opened up for speed)

                  loop
                     exit when Source (Scan_Ptr) not in Graphic_Character;
                     Scan_Ptr := Scan_Ptr + 1;
                     exit when Source (Scan_Ptr) not in Graphic_Character;
                     Scan_Ptr := Scan_Ptr + 1;
                     exit when Source (Scan_Ptr) not in Graphic_Character;
                     Scan_Ptr := Scan_Ptr + 1;
                     exit when Source (Scan_Ptr) not in Graphic_Character;
                     Scan_Ptr := Scan_Ptr + 1;
                     exit when Source (Scan_Ptr) not in Graphic_Character;
                     Scan_Ptr := Scan_Ptr + 1;
                  end loop;

                  --  Keep going if horizontal tab

                  if Source (Scan_Ptr) = HT then
                     if Style_Check then Style.Check_HT; end if;
                     Scan_Ptr := Scan_Ptr + 1;

                  --  Terminate scan of comment if line terminator or EOF

                  elsif Source (Scan_Ptr) in Line_Terminator
                     or else Source (Scan_Ptr) = EOF
                  then
                     if Int (Scan_Ptr) - Int (Current_Line_Start) >
                                           Max_Allowed_Line_Length
                     then
                        Error_Long_Line;
                     end if;

                     exit;

                  --  Terminate scan of comment if end of file encountered
                  --  (embedded EOF character or real last character in file)

                  elsif Source (Scan_Ptr) = EOF then
                     exit;

                  --  Keep going if character in 80-FF range. These characters
                  --  are allowed in comments according to the approved AI.
                  --  Also allow ESC, which just got added to the AI (June 93)

                  elsif Source (Scan_Ptr) in Upper_Half_Character
                    or else Source (Scan_Ptr) = ESC
                  then
                     Scan_Ptr := Scan_Ptr + 1;

                  --  Otherwise we have an illegal comment character

                  else
                     Error_Illegal_Character;
                  end if;

               end loop;

               --  Note that we do NOT execute a return here, instead we fall
               --  through to reexecute the scan loop to look for a token.

            end if;
         end Minus_Case;

         --  Double quote or percent starting a string constant

         when '"' | '%' =>
            Slit;
            return;

         --  Apostrophe. This can either be the start of a character literal,
         --  or an isolated apostrophe used in a qualified expression or an
         --  attribute. We treat it as a character literal if it does not
         --  follow a right parenthesis, identifier or literal. This means
         --  that we correctly treat constructs like:

         --    A := CHARACTER'('A');

         --  which appears to be illegal according to 2.2(2) (since the rule
         --  there would seem to require separators to avoid the confusion
         --  with the character literal), but all compilers accept the above
         --  statement, and there are at least six ACVC tests that use this
         --  type of lexical sequence, expecting it to be legal, so in fact
         --  all compilers must accept this and we must too!

         when ''' => Char_Literal_Case : declare
            Code : Char_Code;
            Err  : Boolean;

         begin
            Scan_Ptr := Scan_Ptr + 1;

            --  Here is where we make the test to distinguish the cases. Treat
            --  as apostrophe if previous token is an identifier, right paren
            --  or the reserved word "all" (latter case as in A.all'Address)
            --  Also treat it as apostrophe after a literal (wrong anyway, but
            --  that's probably the better choice).

            if Prev_Token = Tok_Identifier
               or else Prev_Token = Tok_Right_Paren
               or else Prev_Token = Tok_All
               or else Prev_Token in Token_Class_Literal
            then
               Token := Tok_Apostrophe;
               return;

            --  Otherwise the apostrophe starts a character literal

            else
               --  Case of wide character literal

               if (Source (Scan_Ptr) = ESC
                    and then Wide_Character_Encoding_Method /= WCEM_None)
                 or else (Upper_Half_Encoding
                            and then Source (Scan_Ptr) in Upper_Half_Character)

               then
                  Note_Feature (Wide_Characters_And_Strings, Scan_Ptr);
                  Scan_Wide (Source, Scan_Ptr, Code, Err);

                  if Err then
                     Error_Illegal_Wide_Character;
                  end if;

                  if Source (Scan_Ptr) /= ''' then
                     Error_Msg_S ("missing apostrophe");
                  else
                     Scan_Ptr := Scan_Ptr + 1;
                  end if;

               --  If we do not find a closing quote in the expected place then
               --  assume that we have a misguided attempt at a string literal.

               elsif Source (Scan_Ptr + 1) /= ''' then
                  Scan_Ptr := Scan_Ptr - 1;
                  Error_Msg_S
                    ("strings are delimited by double quote character");
                  Scn.Slit;
                  return;

               --  Otherwise we have a (non-wide) character literal

               else
                  if Source (Scan_Ptr) not in Graphic_Character then
                     if Source (Scan_Ptr) in Upper_Half_Character then
                        Note_Feature (Latin_1, Scan_Ptr);

                        if Ada_83 then
                           Error_Illegal_Character;
                        end if;

                     else
                        Error_Illegal_Character;
                     end if;
                  end if;

                  Code := Get_Char_Code (Source (Scan_Ptr));
                  Scan_Ptr := Scan_Ptr + 2;
               end if;

               --  Fall through here with Scan_Ptr updated past the closing
               --  quote, and Code set to the Char_Code value for the literal

               Token := Tok_Char_Literal;
               Token_Node := New_Node (N_Character_Literal, Token_Ptr);
               Set_Char_Literal_Value (Token_Node, Code);
               Set_Character_Literal_Name (Code);
               Token_Name := Name_Find;
               Set_Chars (Token_Node, Token_Name);
               return;
            end if;
         end Char_Literal_Case;

         --  Right parenthesis

         when ')' =>
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Right_Paren;
            if Style_Check then Style.Check_Right_Paren; end if;
            return;

         --  Right bracket or right brace, treated as right paren

         when ']' | '}' =>
            Error_Msg_S ("illegal character, replaced by "")""");
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Right_Paren;
            return;

         --  Slash (can be division operator or first character of not equal)

         when '/' =>
            if Double_Char_Token ('=') then
               Token := Tok_Not_Equal;
               return;
            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Slash;
               return;
            end if;

         --  Semicolon

         when ';' =>
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Semicolon;
            if Style_Check then Style.Check_Semicolon; end if;
            return;

         --  Vertical bar

         when '|' => Vertical_Bar_Case : begin

            --  Special check for || to give nice message

            if Source (Scan_Ptr + 1) = '|' then
               Error_Msg_S ("""||"" should be `OR ELSE`");
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_Or;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Vertical_Bar;
               if Style_Check then Style.Check_Vertical_Bar; end if;
               return;
            end if;
         end Vertical_Bar_Case;

         --  Exclamation, replacement character for vertical bar

         when '!' => Exclamation_Case : begin
            if Source (Scan_Ptr + 1) = '=' then
               Error_Msg_S ("'!= should be /=");
               Scan_Ptr := Scan_Ptr + 2;
               Token := Tok_Not_Equal;
               return;

            else
               Scan_Ptr := Scan_Ptr + 1;
               Token := Tok_Vertical_Bar;
               return;
            end if;

         end Exclamation_Case;

         --  Plus

         when '+' => Plus_Case : begin
            Scan_Ptr := Scan_Ptr + 1;
            Token := Tok_Plus;
            return;
         end Plus_Case;

         --  Digits starting a numeric constant

         when '0' .. '9' =>
            Nlit;

            if Identifier_Char (Source (Scan_Ptr)) then
               Error_Msg_S
                 ("delimiter required between literal and identifier");
            end if;

            return;

         --  Keyword scan routines. These are separated by initial lower case
         --  letter and for each letter, the appropriate keywords are tested
         --  directly. If a keyword is scanned, then control is passed to
         --  Scan_Keyword_n, otherwise control passes to Scan_Identifier_n,
         --  where n is a digit which indicates the number of characters,
         --  starting at Scan_Ptr, which are known to be lower case letters.

         --  The assumption is that keywords are most likely to be spelled
         --  in lower case letters, and that identifiers are likely to be
         --  spelled with an initial upper case letter. If an identifier
         --  starts with a lower case letter, then everything is fine since
         --  control goes to Scan_Identifier_n as described above. If a
         --  keyword is spelled in upper case, then things are also OK, but
         --  the mechanism for recognizing the keywords is quite different
         --  and depends on the use of the Name table (see Scan_Identifier).

         --  Keywords starting with A

         when 'a' => Lower_Case_A : begin

            --  ABORT

            if Source (Scan_Ptr + 1) = 'b' then
               if Source (Scan_Ptr + 2) = 'o' then
                  if Source (Scan_Ptr + 3) = 'r' then
                     if Source (Scan_Ptr + 4) = 't' then
                        Token := Tok_Abort;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  ABSTRACT

               elsif Source (Scan_Ptr + 2) = 's' then
                  if Ada_95 and then Source (Scan_Ptr + 3) = 't' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 'a' then
                           if Source (Scan_Ptr + 6) = 'c' then
                              if Source (Scan_Ptr + 7) = 't' then
                                 Token := Tok_Abstract;
                                 goto Scan_Keyword_8;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;

            --  ABS

                  else
                     Token := Tok_Abs;
                     goto Scan_Keyword_3;
                  end if;
               else goto Scan_Identifier_2; end if;

            --  ACCEPT

            elsif Source (Scan_Ptr + 1) = 'c' then
               if Source (Scan_Ptr + 2) = 'c' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'p' then
                        if Source (Scan_Ptr + 5) = 't' then
                           Token := Tok_Accept;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;

            --  ACCESS

                     elsif Source (Scan_Ptr + 4) = 's' then
                        if Source (Scan_Ptr + 5) = 's' then
                           Token := Tok_Access;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  ALL

            elsif Source (Scan_Ptr + 1) = 'l' then
               if Source (Scan_Ptr + 2) = 'l' then
                  Token := Tok_All;
                  goto Scan_Keyword_3;

            --  ALIASED (95 only)

               elsif Ada_95 and then Source (Scan_Ptr + 2) = 'i' then
                  if Source (Scan_Ptr + 3) = 'a' then
                     if Source (Scan_Ptr + 4) = 's' then
                        if Source (Scan_Ptr + 5) = 'e' then
                           if Source (Scan_Ptr + 6) = 'd' then
                              Token := Tok_Aliased;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  AND

            elsif Source (Scan_Ptr + 1) = 'n' then
               if Source (Scan_Ptr + 2) = 'd' then
                  Token := Tok_And;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;

            --  ARRAY

            elsif Source (Scan_Ptr + 1) = 'r' then
               if Source (Scan_Ptr + 2) = 'r' then
                  if Source (Scan_Ptr + 3) = 'a' then
                     if Source (Scan_Ptr + 4) = 'y' then
                        Token := Tok_Array;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  AT

            elsif Source (Scan_Ptr + 1) = 't' then
               Token := Tok_At;
               goto Scan_Keyword_2;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_A;

         --  Keywords starting with B

         when 'b' => Lower_Case_B : begin

            --  BEGIN

            if Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'g' then
                  if Source (Scan_Ptr + 3) = 'i' then
                     if Source (Scan_Ptr + 4) = 'n' then
                        Token := Tok_Begin;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  BODY

            elsif Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'd' then
                  if Source (Scan_Ptr + 3) = 'y' then
                     Token := Tok_Body;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_B;

         --  Keywords starting with C

         when 'c' => Lower_Case_C : begin

            --  CASE

            if Source (Scan_Ptr + 1) = 'a' then
               if Source (Scan_Ptr + 2) = 's' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     Token := Tok_Case;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  CONSTANT

            elsif Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'n' then
                  if Source (Scan_Ptr + 3) = 's' then
                     if Source (Scan_Ptr + 4) = 't' then
                        if Source (Scan_Ptr + 5) = 'a' then
                           if Source (Scan_Ptr + 6) = 'n' then
                              if Source (Scan_Ptr + 7) = 't' then
                                 Token := Tok_Constant;
                                 goto Scan_Keyword_8;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_C;

         --  Keywords starting with D

         when 'd' => Lower_Case_D : begin

            --  DECLARE

            if Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'c' then
                  if Source (Scan_Ptr + 3) = 'l' then
                     if Source (Scan_Ptr + 4) = 'a' then
                        if Source (Scan_Ptr + 5) = 'r' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Declare;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  DELAY

               elsif Source (Scan_Ptr + 2) = 'l' then
                  if Source (Scan_Ptr + 3) = 'a' then
                     if Source (Scan_Ptr + 4) = 'y' then
                        Token := Tok_Delay;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;

            --  DELTA

                  elsif Source (Scan_Ptr + 3) = 't' then
                     if Source (Scan_Ptr + 4) = 'a' then
                        Token := Tok_Delta;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  DIGITS

            elsif Source (Scan_Ptr + 1) = 'i' then
               if Source (Scan_Ptr + 2) = 'g' then
                  if Source (Scan_Ptr + 3) = 'i' then
                     if Source (Scan_Ptr + 4) = 't' then
                        if Source (Scan_Ptr + 5) = 's' then
                           Token := Tok_Digits;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  DO

            elsif Source (Scan_Ptr + 1) = 'o' then
               Token := Tok_Do;
               goto Scan_Keyword_2;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_D;

         --  Keywords starting with E

         when 'e' => Lower_Case_E : begin

            --  ELSE

            if Source (Scan_Ptr + 1) = 'l' then
               if Source (Scan_Ptr + 2) = 's' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     Token := Tok_Else;
                     goto Scan_Keyword_4;

            --  ELSIF

                  elsif Source (Scan_Ptr + 3) = 'i' then
                     if Source (Scan_Ptr + 4) = 'f' then
                        Token := Tok_Elsif;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  END

            elsif Source (Scan_Ptr + 1) = 'n' then
               if Source (Scan_Ptr + 2) = 'd' then
                  Token := Tok_End;
                  goto Scan_Keyword_3;

            --  ENTRY

               elsif Source (Scan_Ptr + 2) = 't' then
                  if Source (Scan_Ptr + 3) = 'r' then
                     if Source (Scan_Ptr + 4) = 'y' then
                        Token := Tok_Entry;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  EXCEPTION

            elsif Source (Scan_Ptr + 1) = 'x' then
               if Source (Scan_Ptr + 2) = 'c' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'p' then
                        if Source (Scan_Ptr + 5) = 't' then
                           if Source (Scan_Ptr + 6) = 'i' then
                              if Source (Scan_Ptr + 7) = 'o' then
                                 if Source (Scan_Ptr + 8) = 'n' then
                                    Token := Tok_Exception;
                                    goto Scan_Keyword_9;
                                 else goto Scan_Identifier_8; end if;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  EXIT

               elsif Source (Scan_Ptr + 2) = 'i' then
                  if Source (Scan_Ptr + 3) = 't' then
                     Token := Tok_Exit;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_E;

         --  Keywords starting with F

         when 'f' => Lower_Case_F : begin

            --  FOR

            if Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'r' then
                  Token := Tok_For;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;

            --  FUNCTION

            elsif Source (Scan_Ptr + 1) = 'u' then
               if Source (Scan_Ptr + 2) = 'n' then
                  if Source (Scan_Ptr + 3) = 'c' then
                     if Source (Scan_Ptr + 4) = 't' then
                        if Source (Scan_Ptr + 5) = 'i' then
                           if Source (Scan_Ptr + 6) = 'o' then
                              if Source (Scan_Ptr + 7) = 'n' then
                                 Token := Tok_Function;
                                 goto Scan_Keyword_8;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_F;

         --  Keywords starting with G

         when 'g' => Lower_Case_G : begin

            --  GENERIC

            if Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'n' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 'i' then
                           if Source (Scan_Ptr + 6) = 'c' then
                              Token := Tok_Generic;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  GOTO

            elsif Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 't' then
                  if Source (Scan_Ptr + 3) = 'o' then
                     Token := Tok_Goto;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_G;

         --  Keywords starting with I

         when 'i' => Lower_Case_I : begin

            --  IF

            if Source (Scan_Ptr + 1) = 'f' then
               Token := Tok_If;
               goto Scan_Keyword_2;

            --  IN

            elsif Source (Scan_Ptr + 1) = 'n' then
               Token := Tok_In;
               goto Scan_Keyword_2;

            --  IS

            elsif Source (Scan_Ptr + 1) = 's' then
               Token := Tok_Is;
               goto Scan_Keyword_2;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_I;

         --  Keywords starting with L

         when 'l' => Lower_Case_L : begin

            --  LIMITED

            if Source (Scan_Ptr + 1) = 'i' then
               if Source (Scan_Ptr + 2) = 'm' then
                  if Source (Scan_Ptr + 3) = 'i' then
                     if Source (Scan_Ptr + 4) = 't' then
                        if Source (Scan_Ptr + 5) = 'e' then
                           if Source (Scan_Ptr + 6) = 'd' then
                              Token := Tok_Limited;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  LOOP

            elsif Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'o' then
                  if Source (Scan_Ptr + 3) = 'p' then
                     Token := Tok_Loop;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_L;

         --  Keywords starting with M

         when 'm' => Lower_Case_M : begin

            --  MOD

            if Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'd' then
                  Token := Tok_Mod;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_M;

         --  Keywords starting with N

         when 'n' => Lower_Case_N : begin

            --  NEW

            if Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'w' then
                  Token := Tok_New;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;

            --  NOT

            elsif Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 't' then
                  Token := Tok_Not;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;

            --  NULL

            elsif Source (Scan_Ptr + 1) = 'u' then
               if Source (Scan_Ptr + 2) = 'l' then
                  if Source (Scan_Ptr + 3) = 'l' then
                     Token := Tok_Null;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_N;

         --  Keywords starting with O

         when 'o' => Lower_Case_O : begin

            --  OF

            if Source (Scan_Ptr + 1) = 'f' then
               Token := Tok_Of;
               goto Scan_Keyword_2;

            --  OR

            elsif Source (Scan_Ptr + 1) = 'r' then
               Token := Tok_Or;
               goto Scan_Keyword_2;

            --  OTHERS

            elsif Source (Scan_Ptr + 1) = 't' then
               if Source (Scan_Ptr + 2) = 'h' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 's' then
                           Token := Tok_Others;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  OUT

            elsif Source (Scan_Ptr + 1) = 'u' then
               if Source (Scan_Ptr + 2) = 't' then
                  Token := Tok_Out;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_O;

         --  Keywords starting with P

         when 'p' => Lower_Case_P : begin

            --  PACKAGE

            if Source (Scan_Ptr + 1) = 'a' then
               if Source (Scan_Ptr + 2) = 'c' then
                  if Source (Scan_Ptr + 3) = 'k' then
                     if Source (Scan_Ptr + 4) = 'a' then
                        if Source (Scan_Ptr + 5) = 'g' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Package;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  PRAGMA

            elsif Source (Scan_Ptr + 1) = 'r' then
               if Source (Scan_Ptr + 2) = 'a' then
                  if Source (Scan_Ptr + 3) = 'g' then
                     if Source (Scan_Ptr + 4) = 'm' then
                        if Source (Scan_Ptr + 5) = 'a' then
                           Token := Tok_Pragma;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  PRIVATE

               elsif Source (Scan_Ptr + 2) = 'i' then
                  if Source (Scan_Ptr + 3) = 'v' then
                     if Source (Scan_Ptr + 4) = 'a' then
                        if Source (Scan_Ptr + 5) = 't' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Private;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  PROCEDURE

               elsif Source (Scan_Ptr + 2) = 'o' then
                  if Source (Scan_Ptr + 3) = 'c' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        if Source (Scan_Ptr + 5) = 'd' then
                           if Source (Scan_Ptr + 6) = 'u' then
                              if Source (Scan_Ptr + 7) = 'r' then
                                 if Source (Scan_Ptr + 8) = 'e' then
                                    Token := Tok_Procedure;
                                    goto Scan_Keyword_9;
                                 else goto Scan_Identifier_8; end if;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;

            --  PROTECTED (95 only)

                  elsif Ada_95 and then Source (Scan_Ptr + 3) = 't' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        if Source (Scan_Ptr + 5) = 'c' then
                           if Source (Scan_Ptr + 6) = 't' then
                              if Source (Scan_Ptr + 7) = 'e' then
                                 if Source (Scan_Ptr + 8) = 'd' then
                                    Token := Tok_Protected;
                                    goto Scan_Keyword_9;
                                 else goto Scan_Identifier_8; end if;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_P;

         --  Keywords starting with R

         when 'r' => Lower_Case_R : begin

            --  RAISE

            if Source (Scan_Ptr + 1) = 'a' then
               if Source (Scan_Ptr + 2) = 'i' then
                  if Source (Scan_Ptr + 3) = 's' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        Token := Tok_Raise;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  RANGE

               elsif Source (Scan_Ptr + 2) = 'n' then
                  if Source (Scan_Ptr + 3) = 'g' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        Token := Tok_Range;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  RECORD

            elsif Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'c' then
                  if Source (Scan_Ptr + 3) = 'o' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 'd' then
                           Token := Tok_Record;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  REM

               elsif Source (Scan_Ptr + 2) = 'm' then
                  Token := Tok_Rem;
                  goto Scan_Keyword_3;

            --  RENAMES

               elsif Source (Scan_Ptr + 2) = 'n' then
                  if Source (Scan_Ptr + 3) = 'a' then
                     if Source (Scan_Ptr + 4) = 'm' then
                        if Source (Scan_Ptr + 5) = 'e' then
                           if Source (Scan_Ptr + 6) = 's' then
                              Token := Tok_Renames;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  REQUEUE (95 only)

               elsif Ada_95 and then Source (Scan_Ptr + 2) = 'q' then
                  if Source (Scan_Ptr + 3) = 'u' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        if Source (Scan_Ptr + 5) = 'u' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Requeue;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  RETURN

               elsif Source (Scan_Ptr + 2) = 't' then
                  if Source (Scan_Ptr + 3) = 'u' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 'n' then
                           Token := Tok_Return;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  REVERSE

               elsif Source (Scan_Ptr + 2) = 'v' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 's' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Reverse;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_R;

         --  Keywords starting with S

         when 's' => Lower_Case_S : begin

            --  SELECT

            if Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'l' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     if Source (Scan_Ptr + 4) = 'c' then
                        if Source (Scan_Ptr + 5) = 't' then
                           Token := Tok_Select;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  SEPARATE

               elsif Source (Scan_Ptr + 2) = 'p' then
                  if Source (Scan_Ptr + 3) = 'a' then
                     if Source (Scan_Ptr + 4) = 'r' then
                        if Source (Scan_Ptr + 5) = 'a' then
                           if Source (Scan_Ptr + 6) = 't' then
                              if Source (Scan_Ptr + 7) = 'e' then
                                 Token := Tok_Separate;
                                 goto Scan_Keyword_8;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  SUBTYPE

            elsif Source (Scan_Ptr + 1) = 'u' then
               if Source (Scan_Ptr + 2) = 'b' then
                  if Source (Scan_Ptr + 3) = 't' then
                     if Source (Scan_Ptr + 4) = 'y' then
                        if Source (Scan_Ptr + 5) = 'p' then
                           if Source (Scan_Ptr + 6) = 'e' then
                              Token := Tok_Subtype;
                              goto Scan_Keyword_7;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_S;

         --  Keywords starting with T

         when 't' => Lower_Case_T : begin

            --  TAGGED (95 only)

            if Source (Scan_Ptr + 1) = 'a' then
               if Ada_95 and then Source (Scan_Ptr + 2) = 'g' then
                  if Source (Scan_Ptr + 3) = 'g' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        if Source (Scan_Ptr + 5) = 'd' then
                           Token := Tok_Tagged;
                           goto Scan_Keyword_6;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;

            --  TASK

               elsif Source (Scan_Ptr + 2) = 's' then
                  if Source (Scan_Ptr + 3) = 'k' then
                     Token := Tok_Task;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  TERMINATE

            elsif Source (Scan_Ptr + 1) = 'e' then
               if Source (Scan_Ptr + 2) = 'r' then
                  if Source (Scan_Ptr + 3) = 'm' then
                     if Source (Scan_Ptr + 4) = 'i' then
                        if Source (Scan_Ptr + 5) = 'n' then
                           if Source (Scan_Ptr + 6) = 'a' then
                              if Source (Scan_Ptr + 7) = 't' then
                                 if Source (Scan_Ptr + 8) = 'e' then
                                    Token := Tok_Terminate;
                                    goto Scan_Keyword_9;
                                 else goto Scan_Identifier_8; end if;
                              else goto Scan_Identifier_7; end if;
                           else goto Scan_Identifier_6; end if;
                        else goto Scan_Identifier_5; end if;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  THEN

            elsif Source (Scan_Ptr + 1) = 'h' then
               if Source (Scan_Ptr + 2) = 'e' then
                  if Source (Scan_Ptr + 3) = 'n' then
                     Token := Tok_Then;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  TYPE

            elsif Source (Scan_Ptr + 1) = 'y' then
               if Source (Scan_Ptr + 2) = 'p' then
                  if Source (Scan_Ptr + 3) = 'e' then
                     Token := Tok_Type;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_T;

         --  Keywords starting with U

         when 'u' => Lower_Case_U : begin

            --  UNTIL (95 only)

            if Ada_95 and then Source (Scan_Ptr + 1) = 'n' then
               if Source (Scan_Ptr + 2) = 't' then
                  if Source (Scan_Ptr + 3) = 'i' then
                     if Source (Scan_Ptr + 4) = 'l' then
                        Token := Tok_Until;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  USE

            elsif Source (Scan_Ptr + 1) = 's' then
               if Source (Scan_Ptr + 2) = 'e' then
                  Token := Tok_Use;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_U;

         --  Keywords starting with W

         when 'w' => Lower_Case_W : begin

            --  WHEN

            if Source (Scan_Ptr + 1) = 'h' then
               if Source (Scan_Ptr + 2) = 'e' then
                  if Source (Scan_Ptr + 3) = 'n' then
                     Token := Tok_When;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;

            --  WHILE

               elsif Source (Scan_Ptr + 2) = 'i' then
                  if Source (Scan_Ptr + 3) = 'l' then
                     if Source (Scan_Ptr + 4) = 'e' then
                        Token := Tok_While;
                        goto Scan_Keyword_5;
                     else goto Scan_Identifier_4; end if;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;

            --  WITH

            elsif Source (Scan_Ptr + 1) = 'i' then
               if Source (Scan_Ptr + 2) = 't' then
                  if Source (Scan_Ptr + 3) = 'h' then
                     Token := Tok_With;
                     goto Scan_Keyword_4;
                  else goto Scan_Identifier_3; end if;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_W;

         --  Keywords starting with X

         when 'x' => Lower_Case_X : begin

            --  XOR

            if Source (Scan_Ptr + 1) = 'o' then
               if Source (Scan_Ptr + 2) = 'r' then
                  Token := Tok_Xor;
                  goto Scan_Keyword_3;
               else goto Scan_Identifier_2; end if;
            else goto Scan_Identifier_1; end if;

         end Lower_Case_X;

         --  Lower case letters which cannot start a keyword

         when 'h' | 'j' | 'k' | 'q' | 'v' | 'y' | 'z' =>
            goto Scan_Identifier_1;

         --  Upper case letters. We assume for the moment that these are
         --  identifiers. We may turn out to have a keyword after all, but
         --  we will find that out later on in the Scan_Identifier routine

         when 'A' .. 'Z' => Upper_Case_Letters : begin
            Name_Len := 1;
            Name_Buffer (1) :=
              Character'Val (Character'Pos (Source (Scan_Ptr)) + 32);
            Scan_Ptr := Scan_Ptr + 1;
            goto Scan_Identifier;
         end Upper_Case_Letters;

         --  Underline character

         when '_' =>
            Error_Msg_S ("identifier cannot start with underline");
            goto Scan_Identifier_1;

         --  Space (not possible, because we scanned past blanks)

         when ' ' =>
            pragma Assert (False); null;

         --  Characters in top half of ASCII 8-bit chart

         when Upper_Half_Character =>

            --  If wide character, then illegal, not allowed outside literal

            if Upper_Half_Encoding then
               Error_Illegal_Character;

            --  Otherwise we have OK Latin-1 character

            else
               Note_Feature (Latin_1, Scan_Ptr);

               --  Upper half characters may possibly be identifier letters
               --  but can never be digits, so Identifier_Character can be
               --  used to test for a valid start of identifier character.

               if Identifier_Char (Source (Scan_Ptr)) then
                  Name_Len := 0;
                  goto Scan_Identifier;
               else
                  Error_Illegal_Character;
               end if;
            end if;

         when ESC =>
            --  ESC character, possible start of identifier if wide characters
            --  are allowed in identifiers, which we can tell by looking at
            --  the Identifier_Char flag for ESC.

            if Identifier_Char (ESC) then
               goto Scan_Identifier;
            else
               Error_Illegal_Wide_Character;
            end if;

         --  Invalid control characters

         when NUL | SOH | STX | ETX | EOT | ENQ | ACK | BEL | BS  | SO  |
              SI  | DLE | DC1 | DC2 | DC3 | DC4 | NAK | SYN | ETB | CAN |
              EM  | FS  | GS  | RS  | US  | DEL
         => Error_Illegal_Character;

         --  Invalid graphic characters

         when '#' | '$' | '?' | '@' | '`' | '\' | '^' | '~' =>
            Error_Illegal_Character;

         --  End switch on non-blank character

         end case;

      --  End loop past format effectors. The exit from this loop is by
      --  executing a return statement following completion of token scan
      --  (control never falls out of this loop to the code which follows)

      end loop;

      --  Keyword and Identifier scanning routines

      --  Scan_Keyword_9 is passed control after scanning a nine character
      --  keyword consisting entirely of lower case letters. Token is already
      --  set to the required value, but Scan_Ptr is not yet adjusted, it
      --  still points to the first of the nine characters. The one check
      --  that needs to be made is that no identifier character follows
      --  (e.g. a case like protectedx, where the initial nine characters
      --  look like the keyword protected).

      <<Scan_Keyword_9>>
         if not Identifier_Char (Source (Scan_Ptr + 9)) then
            Scan_Ptr := Scan_Ptr + 9;
            return;

         else
            Name_Buffer (1) := Source (Scan_Ptr);
            Name_Buffer (2) := Source (Scan_Ptr + 1);
            Name_Buffer (3) := Source (Scan_Ptr + 2);
            Name_Buffer (4) := Source (Scan_Ptr + 3);
            Name_Buffer (5) := Source (Scan_Ptr + 4);
            Name_Buffer (6) := Source (Scan_Ptr + 5);
            Name_Buffer (7) := Source (Scan_Ptr + 6);
            Name_Buffer (8) := Source (Scan_Ptr + 7);
            Name_Buffer (9) := Source (Scan_Ptr + 8);

            Name_Len := 9;
            Scan_Ptr := Scan_Ptr + 9;
            goto Scan_Identifier;
         end if;

      --  Scan_Keyword_8 is similar, except that in the case where we do not
      --  have a keyword, we can use the Scan_Identifier_8 routine to handle
      --  the merge into the normal identifier scanning circuit.

      <<Scan_Keyword_8>>
         if not Identifier_Char (Source (Scan_Ptr + 8)) then
            Scan_Ptr := Scan_Ptr + 8;
            return;
         end if;

      --  Scan_Identifier_8 is passed control after scanning 8 lower case
      --  letters that turn out not to be a keyword after all, e.g. if the
      --  character sequence protectejunk is given, then control will be
      --  passed here with Scan_Ptr pointing to the P. The eight characters
      --  are known to be lower case letters because the expanded keyword
      --  tests only recognize lower case letters.

      <<Scan_Identifier_8>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Buffer (4) := Source (Scan_Ptr + 3);
         Name_Buffer (5) := Source (Scan_Ptr + 4);
         Name_Buffer (6) := Source (Scan_Ptr + 5);
         Name_Buffer (7) := Source (Scan_Ptr + 6);
         Name_Buffer (8) := Source (Scan_Ptr + 7);
         Name_Len := 8;
         Scan_Ptr := Scan_Ptr + 8;
         goto Scan_Identifier;

      --  Similar processing for the 7-character cases

      <<Scan_Keyword_7>>
         if not Identifier_Char (Source (Scan_Ptr + 7)) then
            Scan_Ptr := Scan_Ptr + 7;
            return;
         end if;

      <<Scan_Identifier_7>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Buffer (4) := Source (Scan_Ptr + 3);
         Name_Buffer (5) := Source (Scan_Ptr + 4);
         Name_Buffer (6) := Source (Scan_Ptr + 5);
         Name_Buffer (7) := Source (Scan_Ptr + 6);
         Name_Len := 7;
         Scan_Ptr := Scan_Ptr + 7;
         goto Scan_Identifier;

      --  Similar processing for the 6-character cases

      <<Scan_Keyword_6>>
         if not Identifier_Char (Source (Scan_Ptr + 6)) then
            Scan_Ptr := Scan_Ptr + 6;
            return;
         end if;

      <<Scan_Identifier_6>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Buffer (4) := Source (Scan_Ptr + 3);
         Name_Buffer (5) := Source (Scan_Ptr + 4);
         Name_Buffer (6) := Source (Scan_Ptr + 5);
         Name_Len := 6;
         Scan_Ptr := Scan_Ptr + 6;
         goto Scan_Identifier;

      --  Similar processing for the 5-character cases

      <<Scan_Keyword_5>>
         if not Identifier_Char (Source (Scan_Ptr + 5)) then
            Scan_Ptr := Scan_Ptr + 5;
            return;
         end if;

      <<Scan_Identifier_5>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Buffer (4) := Source (Scan_Ptr + 3);
         Name_Buffer (5) := Source (Scan_Ptr + 4);
         Name_Len := 5;
         Scan_Ptr := Scan_Ptr + 5;
         goto Scan_Identifier;

      --  Similar processing for the 4-character cases

      <<Scan_Keyword_4>>
         if not Identifier_Char (Source (Scan_Ptr + 4)) then
            Scan_Ptr := Scan_Ptr + 4;
            return;
         end if;

      <<Scan_Identifier_4>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Buffer (4) := Source (Scan_Ptr + 3);
         Name_Len := 4;
         Scan_Ptr := Scan_Ptr + 4;
         goto Scan_Identifier;

      --  Similar processing for the 3-character cases

      <<Scan_Keyword_3>>
         if not Identifier_Char (Source (Scan_Ptr + 3)) then
            Scan_Ptr := Scan_Ptr + 3;
            return;
         end if;

      <<Scan_Identifier_3>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Buffer (3) := Source (Scan_Ptr + 2);
         Name_Len := 3;
         Scan_Ptr := Scan_Ptr + 3;
         goto Scan_Identifier;

      --  Similar processing for the 2-character cases

      <<Scan_Keyword_2>>
         if not Identifier_Char (Source (Scan_Ptr + 2)) then
            Scan_Ptr := Scan_Ptr + 2;
            return;
         end if;

      <<Scan_Identifier_2>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Buffer (2) := Source (Scan_Ptr + 1);
         Name_Len := 2;
         Scan_Ptr := Scan_Ptr + 2;
         goto Scan_Identifier;

      --  Similar processing for the 1-character cases (note that there
      --  are no one character reserved words, so we don't need an entry
      --  for Scan_Keyword_1).

      <<Scan_Identifier_1>>
         Name_Buffer (1) := Source (Scan_Ptr);
         Name_Len := 1;
         Scan_Ptr := Scan_Ptr + 1;

      --  Here we scan out an identifier, Scan_Ptr has been updated past an
      --  initial sequence of zero or more identifier characters that has
      --  already been stored in Name_Buffer with Name_Len set to indicate
      --  the number of stored characters. An underline character can appear
      --  immediately (if this is an error, the caller has already checked
      --  for this error and given an error message).

      <<Scan_Identifier>>

         --  This loop scans as fast as possible past lower half letters
         --  and digits, which we expect to be the most common characters.

         loop
            if Source (Scan_Ptr) in 'a' .. 'z'
              or else Source (Scan_Ptr) in '0' .. '9'
            then
               Name_Buffer (Name_Len + 1) := Source (Scan_Ptr);

            elsif Source (Scan_Ptr) in 'A' .. 'Z' then
               Name_Buffer (Name_Len + 1) :=
                 Character'Val (Character'Pos (Source (Scan_Ptr)) + 32);
            else
               exit;
            end if;

            --  Open out the loop a couple of times for speed

            if Source (Scan_Ptr + 1) in 'a' .. 'z'
              or else Source (Scan_Ptr + 1) in '0' .. '9'
            then
               Name_Buffer (Name_Len + 2) := Source (Scan_Ptr + 1);

            elsif Source (Scan_Ptr + 1) in 'A' .. 'Z' then
               Name_Buffer (Name_Len + 2) :=
                 Character'Val (Character'Pos (Source (Scan_Ptr + 1)) + 32);
            else
               Scan_Ptr := Scan_Ptr + 1;
               Name_Len := Name_Len + 1;
               exit;
            end if;

            if Source (Scan_Ptr + 2) in 'a' .. 'z'
              or else Source (Scan_Ptr + 2) in '0' .. '9'
            then
               Name_Buffer (Name_Len + 3) := Source (Scan_Ptr + 2);

            elsif Source (Scan_Ptr + 2) in 'A' .. 'Z' then
               Name_Buffer (Name_Len + 3) :=
                 Character'Val (Character'Pos (Source (Scan_Ptr + 2)) + 32);
            else
               Scan_Ptr := Scan_Ptr + 2;
               Name_Len := Name_Len + 2;
               exit;
            end if;

            if Source (Scan_Ptr + 3) in 'a' .. 'z'
              or else Source (Scan_Ptr + 3) in '0' .. '9'
            then
               Name_Buffer (Name_Len + 4) := Source (Scan_Ptr + 3);

            elsif Source (Scan_Ptr + 3) in 'A' .. 'Z' then
               Name_Buffer (Name_Len + 4) :=
                 Character'Val (Character'Pos (Source (Scan_Ptr + 3)) + 32);
            else
               Scan_Ptr := Scan_Ptr + 3;
               Name_Len := Name_Len + 3;
               exit;
            end if;

            Scan_Ptr := Scan_Ptr + 4;
            Name_Len := Name_Len + 4;

         end loop;

         --  If we fall through, then we have encountered either an underline
         --  character, or an extended identifier character (i.e. one from the
         --  upper half), or a wide character, or an identifier terminator.
         --  The initial test speeds us up in the most common case where we
         --  have an identifier terminator. Note that ESC is an identifier
         --  character only if a wide character encoding method is active.

         if Identifier_Char (Source (Scan_Ptr)) then

            --  Case of underline, check for error cases of double underline,
            --  and for a trailing underline character

            if Source (Scan_Ptr) = '_' then
               Name_Len := Name_Len + 1;
               Name_Buffer (Name_Len) := '_';

               if Identifier_Char (Source (Scan_Ptr + 1)) then
                  Scan_Ptr := Scan_Ptr + 1;

                  if Source (Scan_Ptr) = '_' then
                     Error_No_Double_Underline;
                  end if;

               else
                  Error_Msg_S ("identifier cannot end with underline");
                  Scan_Ptr := Scan_Ptr + 1;
               end if;

            --  We know we have either an ESC or an upper half character.
            --  First test for wide character case.

            elsif (Source (Scan_Ptr) = ESC
                    and then Wide_Character_Encoding_Method /= WCEM_None)
              or else Upper_Half_Encoding
            then
               if Identifier_Character_Set /= 'w' then
                  Error_Msg_S ("wide character not allowed in identifier");
               end if;

               --  Scan out the wide character and insert the appropriate
               --  encdoing into the name table entry for the identifier.

               declare
                  Code : Char_Code;
                  Err  : Boolean;

               begin
                  Note_Feature (Wide_Characters_And_Strings, Scan_Ptr);
                  Scan_Wide (Source, Scan_Ptr, Code, Err);

                  if Err then
                     Error_Illegal_Wide_Character;
                  else
                     Store_Encoded_Character (Code);
                  end if;
               end;

            --  Case of an extended character from the upper half. We insert
            --  the appropriate encoding of the character, folded to lower
            --  case (see Namet for details of the encoding).

            else
               Note_Feature (Latin_1, Scan_Ptr);
               Store_Encoded_Character
                 (Get_Char_Code (Fold_Lower (Source (Scan_Ptr))));
               Scan_Ptr := Scan_Ptr + 1;
            end if;

            --  In all above cases, keep scanning identifier characters

            goto Scan_Identifier;
         end if;

         --  Scan of identifier is complete. The identifier is stored in
         --  Name_Buffer, and Scan_Ptr points past the last character.

         Token_Name := Name_Find;

         --  Here is where we check if it was a keyword after all (happens
         --  when a keyword is spelled with an initial upper case letter)


         if Get_Name_Table_Byte (Token_Name) /= 0
           and then (Ada_95 or else Token_Name not in Ada_95_Reserved_Words)
         then
            Token := Token_Type'Val (Get_Name_Table_Byte (Token_Name));

            --  Deal with possible style check for non-lower case keyword,
            --  but we don't treat ACCESS, DELTA, DIGITS, RANGE as keywords
            --  for this purpose if they appear as attribute designators.

            if Style_Check
              and then (Prev_Token /= Tok_Apostrophe
                          or else
                            (Token /= Tok_Access
                               and then Token /= Tok_Delta
                               and then Token /= Tok_Digits
                               and then Token /= Tok_Range))
            then
               Style.Non_Lower_Case_Keyword;
            end if;

            --  We must reset Token_Name since this is not an identifier
            --  and if we leave Token_Name set, the parser gets confused
            --  because it thinks it is dealing with an identifier instead
            --  of the corresponding keyword.

            Token_Name := No_Name;
            return;

         --  It is an identifier after all

         else
            Token_Node := New_Node (N_Identifier, Token_Ptr);
            Set_Chars (Token_Node, Token_Name);
            Token := Tok_Identifier;
            return;
         end if;
   end Scan;

   ---------------------
   -- Scan_First_Char --
   ---------------------

   function Scan_First_Char return Source_Ptr is
      Ptr : Source_Ptr := Current_Line_Start;

   begin
      loop
         if Source (Ptr) = ' ' then
            Ptr := Ptr + 1;

         elsif Source (Ptr) = HT then
            if Style_Check then Style.Check_HT; end if;
            Ptr := Ptr + 1;

         else
            return Ptr;
         end if;
      end loop;
   end Scan_First_Char;

   ------------------------------
   -- Scan_Reserved_Identifier --
   ------------------------------

   procedure Scan_Reserved_Identifier (Force_Msg : Boolean) is
      Token_Chars : constant String := Token_Type'Image (Token);

   begin
      --  We have in Token_Chars the image of the Token name, i.e. Tok_xxx.
      --  This code extracts the xxx and makes an identifier out of it.

      Name_Len := 0;

      for J in 5 .. Token_Chars'Length loop
         Name_Len := Name_Len + 1;
         Name_Buffer (Name_Len) := Fold_Lower (Token_Chars (J));
      end loop;

      Token_Name := Name_Find;

      if not Used_As_Identifier (Token) or else Force_Msg then
         Error_Msg_Name_1 := Token_Name;
         Error_Msg_SC ("reserved word* cannot be used as identifier!");
         Used_As_Identifier (Token) := True;
      end if;

      Token := Tok_Identifier;
      Token_Node := New_Node (N_Identifier, Token_Ptr);
      Set_Chars (Token_Node, Token_Name);
   end Scan_Reserved_Identifier;

   ----------------------
   -- Set_Start_Column --
   ----------------------

   --  Note: it seems at first glance a little expensive to compute this value
   --  for every source line (since it is certainly not used for all source
   --  lines). On the other hand, it doesn't take much more work to skip past
   --  the initial white space on the line counting the columns than it would
   --  to scan past the white space using the standard scanning circuits.

   function Set_Start_Column return Column_Number is
      Start_Column : Column_Number := 0;

   begin
      --  Outer loop scans past horizontal tab characters

      Tabs_Loop : loop

         --  Inner loop scans past blanks as fast as possible, bumping Scan_Ptr
         --  past the blanks and adjusting Start_Column to account for them.

         Blanks_Loop : loop
            if Source (Scan_Ptr) = ' ' then
               if Source (Scan_Ptr + 1) = ' ' then
                  if Source (Scan_Ptr + 2) = ' ' then
                     if Source (Scan_Ptr + 3) = ' ' then
                        if Source (Scan_Ptr + 4) = ' ' then
                           if Source (Scan_Ptr + 5) = ' ' then
                              if Source (Scan_Ptr + 6) = ' ' then
                                 Scan_Ptr := Scan_Ptr + 7;
                                 Start_Column := Start_Column + 7;
                              else
                                 Scan_Ptr := Scan_Ptr + 6;
                                 Start_Column := Start_Column + 6;
                                 exit Blanks_Loop;
                              end if;
                           else
                              Scan_Ptr := Scan_Ptr + 5;
                              Start_Column := Start_Column + 5;
                              exit Blanks_Loop;
                           end if;
                        else
                           Scan_Ptr := Scan_Ptr + 4;
                           Start_Column := Start_Column + 4;
                           exit Blanks_Loop;
                        end if;
                     else
                        Scan_Ptr := Scan_Ptr + 3;
                        Start_Column := Start_Column + 3;
                        exit Blanks_Loop;
                     end if;
                  else
                     Scan_Ptr := Scan_Ptr + 2;
                     Start_Column := Start_Column + 2;
                     exit Blanks_Loop;
                  end if;
               else
                  Scan_Ptr := Scan_Ptr + 1;
                  Start_Column := Start_Column + 1;
                  exit Blanks_Loop;
               end if;
            else
               exit Blanks_Loop;
            end if;
         end loop Blanks_Loop;

         --  Outer loop keeps going only if a horizontal tab follows

         if Source (Scan_Ptr) = HT then
            if Style_Check then Style.Check_HT; end if;
            Scan_Ptr := Scan_Ptr + 1;
            Start_Column := (Start_Column / 8) * 8 + 8;
         else
            exit Tabs_Loop;
         end if;

      end loop Tabs_Loop;

      return Start_Column;
   end Set_Start_Column;

end Scn;
