------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ A T T R                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.13 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Attribute handling is isolated in a separate package to ease the addition
--  of implementation defined attributes. Logically this processing belongs
--  in chapter 4. See Sem_Ch4 for a description of the relation of the
--  Analyze and Resolve routines for expression components.

--  This spec also documents all GNAT implementation defined pragmas

with Snames; use Snames;
with Types;  use Types;

package Sem_Attr is

   type Attribute_Class_Array is array (Attribute_Id) of Boolean;
   --  Type used to build attribute classification flag arrays

   -----------------------------------------
   -- Implementation Dependent Attributes --
   -----------------------------------------

   --  This section describes the implementation dependent attributes
   --  provided in GNAT, as well as constructing an array of flags
   --  indicating which attributes these are.

   Attribute_Impl_Def : Attribute_Class_Array := Attribute_Class_Array'(

      ------------------
      -- Abort_Signal --
      ------------------

      Attribute_Abort_Signal => True,
      --  Standard'Abort_Signal (Standard is the only allowed prefix)
      --  provides the entity for the special exception used to signal
      --  task abort or asynchronous transfer of control. Normally this
      --  attribute should only be used in the tasking runtime (it is
      --  highly peculiar, and completely outside the normal semantics
      --  of Ada, for a user program to intercept the abort exception).

      ------------------
      -- Address_Size --
      ------------------

      Attribute_Address_Size => True,
      --  Standard'Address_Size (Standard is the only allowed prefix) is
      --  a static constant giving the number of bits in an Address. It
      --  is used primarily for constructing the definition of Memory_Size
      --  in package Standard, but may be freely used in user programs.

      -----------------------
      -- Default_Bit_Order --
      -----------------------

      Attribute_Default_Bit_Order => True,
      --  Standard'Default_Bit_Order (Standard is the only permissible prefix),
      --  provides the value System.Default_Bit_Order as a Pos value (0 for
      --  High_Order_First, 1 for Low_Order_First). This is used to construct
      --  the definition of Default_Bit_Order in package System.

      ---------------
      -- Elab_Body --
      ---------------

      Attribute_Elab_Body => True,
      --  This attribute can only be applied to a program unit name. It
      --  returns the entity for the corresponding elaboration procedure
      --  for elaborating the body of the referenced unit. This is used
      --  in the main generated elaboration procedure by the binder, and
      --  is not normally used in any other context, but there may be
      --  specialized situations in which it is useful to be able to
      --  call this elaboration procedure from Ada code, e.g. if it
      --  is necessary to do selective reelaboration to fix some error.

      ---------------
      -- Elab_Spec --
      ---------------

      Attribute_Elab_Spec => True,
      --  This attribute can only be applied to a program unit name. It
      --  returns the entity for the corresponding elaboration procedure
      --  for elaborating the spec of the referenced unit. This is used
      --  in the main generated elaboration procedure by the binder, and
      --  is not normally used in any other context, but there may be
      --  specialized situations in which it is useful to be able to
      --  call this elaboration procedure from Ada code, e.g. if it
      --  is necessary to do selective reelaboration to fix some error.

      --------------
      -- Enum_Rep --
      --------------

      Attribute_Enum_Rep => True,
      --  For every enumeration subtype S, S'Enum_Rep denotes a function
      --  with the following specification:
      --
      --    function S (Arg : S'Base) return universal_integer;
      --
      --  The function returns the representation value for the given
      --  enumeration value. This will be equal to the 'Pos value in the
      --  absence of an enumeration representation clause. This is a static
      --  attribute (i.e. the result is static if the argument is static).

      -----------------
      -- Fixed_Value --
      -----------------

      Attribute_Fixed_Value => True,
      --  For every fixed-point type S, S'Fixed_Value denotes a function
      --  with the following specification:
      --
      --    function Fixed_Value (Arg : universal_integer) return S;
      --
      --  The value returned is the fixed-point value V such that
      --
      --    V = Arg * S'Small
      --
      --  The effect is thus equivalent to first converting the argument
      --  to the integer type used to represent S, and then doing an
      --  unchecked conversion to the fixed-point type. This attribute is
      --  primarily intended for use in implementation of the input-output
      --  functions for fixed-point values.

      ---------
      -- Img --
      ---------

      Attribute_Img => True,
      --  The 'Img attribute differs from 'Image in two respects. First, it
      --  may be applied to objects as well as types, in which case it is
      --  applies to the subtype of the object. Second, for numeric types
      --  there is no leading space in the non-negative case. The Img
      --  attribute is intended primarily to ease the writing of debugging
      --  statements, but is not restricted to this use.

      -------------------
      -- Integer_Value --
      -------------------

      Attribute_Integer_Value => True,
      --  For every integer type S, S'Integer_Value denotes a function
      --  with the following specification:
      --
      --    function Integer_Value (Arg : universal_fixed) return S;
      --
      --  The value returned is the integer value V, such that
      --
      --    Arg = V * fixed-type'Small
      --
      --  The effect is thus equivalent to first doing an unchecked convert
      --  from the fixed-point type to its corresponding implementation type,
      --  and then converting the result to the target integer type. This
      --  attribute is primarily intended for use in implementation of the
      --  standard input-output functions for fixed-point values.

      ----------------------------
      -- Max_Interrupt_Priority --
      ----------------------------

      Attribute_Max_Interrupt_Priority => True,
      --  Standard'Max_Interrupt_Priority (Standard is the only permissible
      --  prefix), provides the value System.Max_Interrupt_Priority, and is
      --  intended primarily for constructing this definition in package
      --  System (see note above in Default_Bit_Order description}.

      ------------------
      -- Max_Priority --
      ------------------

      Attribute_Max_Priority => True,
      --  Standard'Max_Priority (Standard is the only permissible prefix)
      --  provides the value System.Max_Priority, and is intended primarily
      --  for constructing this definition in package System (see note above
      --  in Default_Bit_Order description).

      -----------------------
      -- Maximum_Alignment --
      -----------------------

      Attribute_Maximum_Alignment => True,
      --  Standard'Maximum_Alignment (Standard is the only permissible prefix)
      --  provides the maximum useful alignment value for the target. This
      --  is a static value that can be used to specify the alignment for an
      --  object, guaranteeing that it is properly aligned in all cases. The
      --  time this is useful is when an external object is imported and its
      --  alignment requirements are unknown.

      -------------------------
      -- Passed_By_Reference --
      -------------------------

      Attribute_Passed_By_Reference => True,
      --  T'Passed_By_Reference for any subtype T returns a boolean value
      --  that is true if the type is normally passed by reference and
      --  false if the type is normally passed by copy in calls. For scalar
      --  types, the result is always False and is static. For non-scalar
      --  types, the result is non-static (since it is computed by Gigi).

      ------------------
      -- Range_Length --
      ------------------

      Attribute_Range_Length => True,
      --  T'Range_Length for any discrete type T yields the number of
      --  values represented by the subtype (zero for a null range). The
      --  result is static for static subtypes. Note that Range_Length
      --  applied to the index subtype of a one dimensional array always
      --  gives the same result as Range applied to the array itself.

      ------------------
      -- Storage_Unit --
      ------------------

      Attribute_Storage_Unit => True,
      --  Standard'Storage_Unit (Standard is the only permissible prefix)
      --  provides the value System.Storage_Unit, and is intended primarily
      --  for constructing this definition in package System (see note above
      --  in Default_Bit_Order description).

      ----------
      -- Tick --
      ----------

      Attribute_Tick => True,
      --  Standard'Tick (Standard is the only permissible prefix) provides
      --  the value System.Tick, and is intended primarily for constructing
      --  this definition in package System (see note above in description
      --  of Default_Bit_Order).

      ------------------------------
      -- Universal_Literal_String --
      ------------------------------

      Attribute_Universal_Literal_String => True,
      --  The prefix of 'Universal_Literal_String must be a named number.
      --  The static result is the string consisting of the characters of
      --  the number as defined in the original source. This allows the
      --  user program to access the actual text of named numbers without
      --  intermediate conversions and without the need to enclose the
      --  strings in quotes (which would preclude their use as numbers).
      --  This is used internally for the construction of values of the
      --  floating-point attributes from the file ttypef.ads, but may
      --  also be used by user programs.

      -------------------------
      -- Unrestricted_Access --
      -------------------------

      Attribute_Unrestricted_Access => True,
      --  The Unrestricted_Access attribute is similar to Access except that
      --  all accessibility and aliased view checks are omitted. This is very
      --  much a user-beware attribute. Basically its status is very similar
      --  to Address, for which it is a desirable replacement where the value
      --  desired is an access type. In other words, its effect is identical
      --  to first taking 'Address and then doing an unchecked conversion to
      --  a desired access type. Note that in GNAT, but not necessarily in
      --  other implementations, the use of static chains for inner level
      --  subprograms means that Unrestricted_Access applied to a subprogram
      --  yields a value that can be called as long as the subprogram is in
      --  scope (normal Ada 95 accessibility rules restrict this usage).

      ---------------
      -- Word_Size --
      ---------------

      Attribute_Word_Size => True,
      --  Standard'Word_Size (Standard is the only permissible prefix)
      --  provides the value System.Word_Size, and is intended primarily
      --  for constructing this definition in package System (see note above
      --  in Default_Bit_Order description).


      others                             => False);

   -----------------
   -- Subprograms --
   -----------------

   procedure Analyze_Attribute (N : Node_Id);
   --  Performs bottom up semantic analysis of an attribute. Note that the
   --  parser has already checked that type returning attributes appear only
   --  in appropriate contexts (i.e. in subtype marks, or as prefixes for
   --  other attributes).

   procedure Eval_Attribute (N : Node_Id);
   --  Performs compile time evaluation of attributes where possible, leaving
   --  the Is_Static_Expression/Raises_Constraint_Error flags appropriately
   --  set, and replacing the node with a literal node if the value can be
   --  computed at compile time. All static attribute references are folded,
   --  as well as a number of cases of non-static attributes that can always
   --  be computed at compile time (e.g. floating-point model attributes that
   --  are applied to non-static subtypes). Of course in such cases, the
   --  Is_Static_Expression flag will not be set on the resulting literal.

   procedure Resolve_Attribute (N : Node_Id; Typ : Entity_Id);
   --  Performs type resolution of attribute. If the attribute yields
   --  a universal value, mark its type as that of the context. On
   --  the other hand, if the context itself is universal (as in
   --  T'Val (T'Pos (X)), mark the type as being the largest type of
   --  that class that can be used at run-time. This is correct since
   --  either the value gets folded (in which case it doesn't matter
   --  what type of the class we give if, since the folding uses universal
   --  arithmetic anyway) or it doesn't get folded (in which case it is
   --  going to be dealt with at runtime, and the largest type is right).

end Sem_Attr;
