/*
 *      Copyright (C) 1993-1995 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 /home/cvs/zftape/ftape-ctl.c,v
  claus
  *
 1.3
 1995/11/16 15:07:23
 Exp
 *
 *      This file contains the non-read/write ftape functions
 *      for the QIC-40/80 floppy-tape driver for Linux.
 */

static char RCSid[] =
"ftape-ctl.c,v 1.3 1995/11/16 15:07:23 claus Exp";

  
#include "ftape.h"
#include <linux/errno.h>
#include <linux/mm.h>
#include <asm/segment.h>
#include <linux/string.h>

#include "ftape-eof.h"
#include "ftape-io.h"
#include "ftape-ctl.h"
#include "ftape-write.h"
#include "ftape-read.h"
#include "ftape-rw.h"
#include "qic117.h"
#include "ftape-bsm.h"
#include "qic80-vtbl.h"
#include "qic80-compress.h"
#include "ftape-ioctl.h"


/*      Global vars.
 */
int segments_per_track = 102;
int segments_per_head = 1020;
int segments_per_cylinder = 4;
int tracks_per_tape = 20;
int ftape_failure = 1;
int ftape_seg_pos = 0;
int first_data_segment = -1;
int ftape_last_data_segment = -1;
int ftape_state = idle;         /* use buffer_state_enum */
history_record history;
int write_protected;
int ftape_offline = 0;
int no_tape = 1;
int formatted = 0;
int ftape_data_rate = 0;
int going_offline = 0;
int read_only = 0;
int ftape_block_size = DEFAULT_BLOCK_SIZE;
int ftape_resid = 0;


/*      Local vars.
 */
static int ftape_last_error = 0;
static const vendor_struct vendors[] = QIC117_VENDORS;
static const wakeup_method methods[] = WAKEUP_METHODS;
static int init_drive_needed = 1;


typedef int (mt_fun)( int *argptr );
typedef int (*mt_funp)( int *argptr );
typedef struct
{
  mt_funp function;
  unsigned offline         : 1; /* operation permitted if offline or no_tape */
  unsigned write_protected : 1; /* operation permitted if write-protected    */
  unsigned not_formatted   : 1; /* operation permitted if tape not formatted */
  unsigned raw_mode        : 1; /* permitted if ftape_qic80_mode == 0        */
  unsigned need_idle_state : 1; /* need to call def_idle_state               */
} fun_entry;

static mt_fun mt_dummy, mt_reset, mt_fsr, mt_bsr, mt_rew, mt_offl, mt_nop,
              mt_weof, mt_erase, mt_ras2, mt_setblk, mt_setdensity,
              mt_seek, mt_tell, mt_reten, mt_eom, mt_fsf, mt_bsf,
              mt_fsfm, mt_bsfm, mt_setdrvbuffer;

static fun_entry mt_funs[]=
{ 
  {mt_reset       , 1, 1, 1, 1, 0 }, /* MT_RESET  0 */
  {mt_fsf         , 0, 1, 0, 0, 1 }, /* MT_FSF      */
  {mt_bsf         , 0, 1, 0, 0, 1 }, /* MT_BSF      */
  {mt_fsr         , 0, 1, 0, 1, 1 }, /* MT_FSR      */
  {mt_bsr         , 0, 1, 0, 1, 1 }, /* MT_BSR      */
  {mt_weof        , 0, 0, 0, 0, 1 }, /* MT_WEOF   5 */
  {mt_rew         , 0, 1, 1, 1, 0 }, /* MT_REW      */
  {mt_offl        , 0, 1, 1, 1, 0 }, /* MT_OFFL     */
  {mt_nop         , 1, 1, 1, 1, 0 }, /* MT_NOP      */
  {mt_reten       , 0, 1, 1, 1, 0 }, /* MT_RETEN    */
  {mt_bsfm        , 0, 1, 0, 0, 1 }, /* MT_BSFM  10 */
  {mt_fsfm        , 0, 1, 0, 0, 1 }, /* MT_FSFM     */
  {mt_eom         , 0, 1, 0, 0, 1 }, /* MT_EOM      */
  {mt_erase       , 0, 0, 0, 1, 1 }, /* MT_ERASE    */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /* MT_RAS1     */
  {mt_ras2        , 0, 0, 0, 1, 0 }, /* MT_RAS2 ftape_fix */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /* MT_RAS3     */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /*      ?      */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /*      ?      */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /*      ?      */
  {mt_setblk      , 1, 1, 1, 1, 1 }, /* MT_SETBLK 20*/
  {mt_setdensity  , 1, 1, 1, 1, 0 }, /* MT_SETDENSITY  set trace-level */
  {mt_seek        , 0, 1, 0, 1, 1 }, /* MT_SEEK     */
  {mt_tell        , 0, 1, 0, 1, 1 }, /* MT_TELL     */
  {mt_setdrvbuffer, 1, 1, 1, 1, 0 }, /* MT_SETDRVBUFFER:
                                        * change number of dma-buffers 
                                        */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /* MT_FSS   25 */
  {mt_dummy       , 1, 1, 1, 1, 0 }, /* MT_BSS      */
  {mt_dummy       , 1, 1, 1, 1, 0 }  /* MT_WSM      */
};  

#define NR_MT_CMDS NR_ITEMS(mt_funs)



static int
ftape_not_operational( int status)
{
  /* return true if status indicates tape can not be used.
   */
  return ( (status ^ QIC_STATUS_CARTRIDGE_PRESENT) &
          (QIC_STATUS_ERROR |
           QIC_STATUS_CARTRIDGE_PRESENT |
           QIC_STATUS_NEW_CARTRIDGE) );
}

int
ftape_seek_to_eot( void)
{
  TRACE_FUN( 8, "ftape_seek_to_eot");
  int result;
  int status;

  result = ftape_ready_wait( timeout.pause, &status);
  while ((status & QIC_STATUS_AT_EOT) == 0) {
    if (result < 0) {
      TRACE( 1, "failed");
      TRACE_EXIT;
      return result;
    }
    if (ftape_not_operational( status)) {
      TRACE_EXIT;
      return -EIO;
    }
    result = ftape_command_wait( QIC_PHYSICAL_FORWARD,
                                timeout.rewind, &status);
  }
  TRACE_EXIT;
  return 0;
}

int
ftape_seek_to_bot( void)
{
  TRACE_FUN( 8, "ftape_seek_to_bot");
  int result;
  int status;

  result = ftape_ready_wait( timeout.pause, &status);
  while ((status & QIC_STATUS_AT_BOT) == 0) {
    if (result < 0) {
      TRACE( 1, "failed");
      TRACE_EXIT;
      return result;
    }
    if (ftape_not_operational( status)) {
      TRACE_EXIT;
      return -EIO;
    }
    result = ftape_command_wait( QIC_PHYSICAL_REVERSE,
                                timeout.rewind, &status);
  }
  TRACE_EXIT;
  return 0;
}

void
ftape_reset_position( void)
{
  if ( ftape_qic80_mode ) {
    /*
     *  we need to keep track of the volume table and compression map
     */
    if ( ftape_compression_map_location != 0 ) {
      ftape_seg_pos = ftape_first_user_segment;
    } else {
      ftape_seg_pos = first_data_segment + 1;
    }
  } else {
    ftape_seg_pos = first_data_segment;
  }
  ftape_seg_data_pos =
  ftape_uncmpr_pos   = 0;
  if ( first_data_segment != -1 ) {
    /*
     *  we need to keep track of the volume table and compression map
     */
    ftape_data_pos = ftape_calc_data_pos( ftape_seg_pos, ftape_seg_data_pos );
  } else {
    ftape_data_pos     = 0;
  }
  just_before_eof = 0;
  ftape_reset_cmpr_locals();
}

int
ftape_new_cartridge( void)
{
  location.track = -1;          /* force seek on first access */
  first_data_segment = -1;      /* unknown */
  ftape_first_user_segment = -1;
  ftape_resid = 0;
  ftape_io_state = io_idle;
  ftape_block_size = DEFAULT_BLOCK_SIZE;
  ftape_zap_read_buffers();
  ftape_zap_write_buffers();
  ftape_reset_position();
  qic80_init_vtbl();
  return 0;
}

int
ftape_abort_operation( void)
{
  TRACE_FUN( 5, "ftape_abort_operation");
  int result = 0;
  int i;
  int status;

  if (runner_status == running) {
    TRACE( 5, "aborting runner, waiting");
    runner_status = do_abort;
    /* set timeout so that the tape will run to logical EOT
     * if we missed the last sector and there are no queue pulses.
     */
    result = ftape_dumb_stop();
    if (result == 0) {
      runner_status = idle;
    }
  }
  if (runner_status != idle) {
    if (runner_status == do_abort) {
      TRACE( 5, "forcing runner abort");
    }
    TRACE( 5, "stopping tape");
    result = ftape_command_wait( QIC_STOP_TAPE, timeout.stop, &status);
    location.known = 0;
    runner_status = idle;
  }
  for (i = 0; i < ftape_num_buffers; ++i) {
    buffer[ i].status = waiting;
  }
  head = tail = 0;
  TRACE_EXIT;
  return result;
}

int
lookup_vendor_id( int vendor_id)
{
  int i = 0;

  while ( vendors[ i].vendor_id != vendor_id) {
    if (++i >= NR_ITEMS( vendors)) {
      return -1;
    }
  }
  return i;
}

void
ftape_detach_drive( void)
{
  TRACE_FUN( 8, "ftape_detach_drive");

  TRACE( 5, "disabling tape drive and fdc");
  ftape_put_drive_to_sleep( drive_type);
  fdc_catch_stray_interrupts( 1);   /* one always comes */
  fdc_disable();
  fdc_release_irq_and_dma();
  TRACE_EXIT;
}

static void
clear_history( void)
{
  history.used = 0;
  history.id_am_errors =
  history.id_crc_errors =
  history.data_am_errors =
  history.data_crc_errors =
  history.overrun_errors =
  history.no_data_errors =
  history.retries =
  history.crc_errors =
  history.crc_failures =
  history.ecc_failures =
  history.corrected =
  history.defects =
  history.rewinds =
  history.cmpr_time =
  history.wr_uncompressed =
  history.wr_compressed =
  history.rd_uncompressed =
  history.rd_compressed = 0;
}

int
ftape_activate_drive( vendor_struct* drive_type)
{
  TRACE_FUN( 5, "ftape_activate_drive");
  int result = 0;

  /* If we already know the drive type, wake it up.
   * Else try to find out what kind of drive is attached.
   */
  if (drive_type->wake_up != unknown_wake_up) {
    TRACE( 5, "enabling tape drive and fdc");
    result = ftape_wakeup_drive( drive_type->wake_up);
    if (result < 0) {
      TRACE( 1, "known wakeup method failed");
    }
  } else {
#ifndef NO_TRACE_AT_ALL
    int old_tracing = tracing;
#endif
    wake_up_types method;

    /*  Try to awaken the drive using all known methods.
     *  Lower tracing for a while.
     */       
#ifndef NO_TRACE_AT_ALL
    if (tracing <= 4) {
      tracing = 0;
    }                  
#endif
    for (method = no_wake_up; method < NR_ITEMS( methods); ++method) {
      drive_type->wake_up = method;
      /*  Test setup for dual drive configuration in dodo.
       *  /dev/rft2 uses mountain wakeup only -> Archive QIC-80
       *  /dev/rft3 uses colorado wakeup only -> Jumbo QIC-40
       *  Other systems will use the normal scheme.
       */
      if (!DODO || (FTAPE_UNIT < 2) ||
          (FTAPE_UNIT == 2 && method == wake_up_mountain) ||
          (FTAPE_UNIT == 3 && method == wake_up_colorado)) {
        result = ftape_wakeup_drive( drive_type->wake_up);
      } else {
        result = -EIO;
      }
      if (result >= 0) {
#ifndef NO_TRACE_AT_ALL
        int tracing = old_tracing; /* fool TRACE */
#endif
        TRACEx1( 2, "drive wakeup method: %s",
                methods[ drive_type->wake_up].name);
        break;
      }
    } 
#ifndef NO_TRACE_AT_ALL
    tracing = old_tracing;
#endif
    if (method >= NR_ITEMS( methods)) {
      /* no response at all, cannot open this drive */
      drive_type->wake_up = unknown_wake_up;
      TRACE( 1, "no tape drive found !");
#ifndef NO_TRACE_AT_ALL
      tracing = old_tracing;
#endif
      result = -ENODEV;
    }
  }
  TRACE_EXIT;
  return result;
}

int
ftape_get_drive_status( int* new_tape, int* no_tape, int* wp_tape)
{
  TRACE_FUN( 5, "ftape_get_drive_status");
  int result;
  int status;

  *no_tape =
  *wp_tape = 0;
  /*    Tape drive is activated now.
   *    First clear error status if present.
   */
  do {
    result = ftape_ready_wait( timeout.reset, &status);
    if (result < 0) {
      if (result == -ETIME) {
        TRACE( 1, "ftape_ready_wait timeout");
      } else if (result == -EINTR) {
        TRACE( 1, "ftape_ready_wait aborted");
      } else {
        TRACE( 1, "ftape_ready_wait failed");
      }
      result = -EIO;
      break;
    }
    /*  Clear error condition (drive is ready !)
     */
    if (status & QIC_STATUS_ERROR) {
      int error;
      int command;

      TRACE( 1, "error status set");
      result = ftape_report_error( &error, &command, 1);
      if (result < 0) {
        TRACEi( 1, "report_error_code failed:", result);
        ftape_reset_drive();    /* hope it's working next time */
        init_drive_needed = 1;
        result = -EIO;
        break;
      } else if (error != 0) {
        TRACEi( 4, "error code   :", error);
        TRACEi( 4, "error command:", command);
      }
    }
    if (status & QIC_STATUS_NEW_CARTRIDGE) {
      int error;
      int command;
#ifndef NO_TRACE_AT_ALL
      int old_tracing = tracing;

      /*  Undocumented feature: Must clear (not present!) error
       *  here or we'll fail later.
       */
      tracing = 0;
      ftape_report_error( &error, &command, 1);
      tracing = old_tracing;
#else
      ftape_report_error( &error, &command, 1);
#endif
      TRACE( 3, "status: new cartridge");
      *new_tape = 1;
    }
  } while (status & QIC_STATUS_ERROR);

  *no_tape = !(status & QIC_STATUS_CARTRIDGE_PRESENT);
  *wp_tape = (status & QIC_STATUS_WRITE_PROTECT);
  if (*no_tape) {
    TRACE( 1, "no cartridge present");
  } else {
    if (*wp_tape) {
      TRACE( 2, "Write protected cartridge");
    }
  }
  TRACE_EXIT;
  return result;
}

void
ftape_log_vendor_id( void)
{
  TRACE_FUN( 5, "ftape_log_vendor_id");
  int vendor_index;

  ftape_report_vendor_id( &drive_type.vendor_id);
  vendor_index = lookup_vendor_id( drive_type.vendor_id);
  if (drive_type.vendor_id == UNKNOWN_VENDOR && 
      drive_type.wake_up == wake_up_colorado) {
    vendor_index = 0;
    drive_type.vendor_id = 0; /* hack to get rid of all this mail */
  }
  if (vendor_index < 0) {
    /* Unknown vendor id, first time opening device.
     * The drive_type remains set to type found at wakeup time, this
     * will probably keep the driver operating for this new vendor.
     */
    TRACE( -1,   "============ unknown vendor id ===========");
    TRACE( -1,   "A new, yet unsupported tape drive is found");
    TRACE( -1,   "Please report the following values:");
    TRACEx1( -1, "   Vendor id     : 0x%05x", drive_type.vendor_id);
    TRACEx1( -1, "   Wakeup method : %s", methods[ drive_type.wake_up].name);
    TRACE( -1,   "And a description of your tape drive to:");
    TRACE( -1,   "Bas Laarhoven <bas@vimec.nl>");
    TRACE( -1,   "==========================================");
    drive_type.speed = 500;     /* deci-ips: very safe value */
  } else {
    drive_type.name = vendors[ vendor_index].name;
    drive_type.speed = vendors[ vendor_index].speed;
    TRACEx1( 3, "tape drive type: %s", drive_type.name);
    /* scan all methods for this vendor_id in table */
    while (drive_type.wake_up != vendors[ vendor_index].wake_up) {
      if (vendor_index < NR_ITEMS( vendors) - 1 &&
          vendors[ vendor_index + 1].vendor_id == drive_type.vendor_id) {
        ++vendor_index;
      } else {
        break;
      }
    }
    if (drive_type.wake_up != vendors[ vendor_index].wake_up) {
      TRACE( -1, "==========================================");
      TRACE( -1, "wakeup type mismatch:");
      TRACEx2( -1, "found: %s, expected: %s",
              methods[ drive_type.wake_up].name,
              methods[ vendors[ vendor_index].wake_up].name);
      TRACE( -1, "please report this to <bas@vimec.nl>");
      TRACE( -1, "==========================================");
    }
  }
  TRACE_EXIT;
}

void
ftape_calc_timeouts( void)
{
  TRACE_FUN( 8, "ftape_calc_timeouts");
  int speed;                    /* deci-ips ! */
  int length;

  /*                           tape transport speed
   *  data rate:        QIC-40   QIC-80   QIC-3010 QIC-3020
   *
   *    250 Kbps        25 ips     n/a      n/a      n/a
   *    500 Kbps        50 ips   34 ips   22.6 ips   n/a
   *      1 Mbps          n/a    68 ips   45.2 ips 22.6 ips
   *      2 Mbps          n/a      n/a      n/a    45.2 ips
   *
   *  fast tape transport speed is at least 68 ips.
   */
  switch (qic_std) {
  case QIC_TAPE_QIC40:
    speed = (ftape_data_rate == 3) ? 250 : 500;
    break;
  case QIC_TAPE_QIC80:
    speed = (ftape_data_rate == 2) ? 340 : 680;
    break;
  case QIC_TAPE_QIC3010:
    speed = (ftape_data_rate == 2) ? 226 : 452;
    break;
  case QIC_TAPE_QIC3020:
    speed = (ftape_data_rate == 1) ? 226 : 452;
    break;
  default:
    TRACE( -1, "Unknown qic_std (bug) ?");
    speed = 500;
    break;
  }
  if (tape_len <= 0) {
    /*  Handle unknown length tapes as 1100 ft ones (worst case)
     */
    TRACE( 1, "Unknown tape length, using worst case timing values!");
    length = 1100;
  } else {
    length = tape_len;
  }
  if (drive_type.speed == 0) {
    unsigned long t0;
    int dt;

    ftape_seek_to_bot();
    t0 = jiffies;
    ftape_seek_to_eot();
    ftape_seek_to_bot();
    dt = (int) ((jiffies - t0) * MSPT);
    drive_type.speed = (2 * 12 * length * 1000) / dt;
    TRACE( -1, "==========================================");
    TRACEx1( -1, "drive : %s", drive_type.name);
    TRACEx2( -1, "delta time = %d, length = %d", dt, length);
    TRACEx1( -1, "has max tape speed of %d ips", drive_type.speed);
    TRACE( -1, "please report this to <bas@vimec.nl>");
    TRACE( -1, "==========================================");
  }
  /*  time to go from bot to eot at normal speed (data rate):
   *  time = (1+delta) * length (ft) * 12 (inch/ft) / speed (ips)
   *  delta = 10 % for seek speed, 20 % for rewind speed.
   */
  timeout.seek = (length * 132 * SECOND) / speed;
  timeout.rewind = (length * 144 * SECOND) / (10 * drive_type.speed);
  timeout.reset = 20 * SECOND + timeout.rewind;
  TRACEx2( 4, "speed = %d, length = %d", speed, length);
  TRACEx1( 4, "seek timeout: %d sec", (timeout.seek + 500) / 1000);
  TRACEx1( 4, "rewind timeout: %d sec", (timeout.rewind + 500) / 1000);
  TRACE_EXIT;
}

int
ftape_init_drive( int* formatted)
{
  TRACE_FUN( 5, "ftape_init_drive");
  int result = 0;
  int status;

  result = ftape_report_raw_drive_status( &status);
  if (result >= 0 && (status & QIC_STATUS_CARTRIDGE_PRESENT)) {
    if (!(status & QIC_STATUS_AT_BOT)) {
      /*  Antique drives will get here after a soft reset,
       *  modern ones only if the driver is loaded when the
       *  tape wasn't rewound properly.
       */
      ftape_seek_to_bot();
    }
    if (!(status & QIC_STATUS_REFERENCED)) {
      TRACE( 5, "starting seek_load_point");
      result = ftape_command_wait( QIC_SEEK_LOAD_POINT,
                                  timeout.reset, &status);
      if (result < 0) {
        TRACE( 1, "seek_load_point failed (command)");
      }
    }
  }
  if (result >= 0) {
    int rate;

    *formatted = (status & QIC_STATUS_REFERENCED);
    if (!*formatted) {
      TRACE( 1, "Warning: tape is not formatted !");
    }
    /*  Select highest rate supported by both fdc and drive.
     *  Start with highest rate supported by the fdc.
     */
    for (rate = (fdc.type >= i82077) ? 1 : 2 ; rate < 4 ; ++rate) {
      result = ftape_set_data_rate( rate);
      if (result >= 0) {
        ftape_calc_timeouts();
        break;
      }
    }
    if (result < 0) {
      result = -EIO;
    }
  }
  if (result >= 0) {
    /* Tape should be at bot if new cartridge ! */
    ftape_new_cartridge();
  }
  init_drive_needed = 0;
  TRACE_EXIT;
  return result;
}


static int
def_idle_state( void )
{ 
TRACE_FUN( 5, "def_idle_state");
int result= 0;

  if ( first_data_segment == -1)
  {
    TRACEi(5,"first_data_segment: ",first_data_segment);
    ftape_deblock_segment = -1;
    result = read_header_segment( deblock_buffer );
    TRACEi(5,"first_data_segment: ",first_data_segment);
    ftape_deblock_segment= -1;
  } else {
    result = ftape_flush_buffers();
    ftape_zap_write_buffers();
    TRACE( 5, "calling ftape_abort_operation");
    result = ftape_abort_operation();
    if (result < 0) {
      TRACE( 1, "ftape_abort_operation failed");
      result = -EIO;
    }
    /* clear remaining read buffers */
    ftape_zap_read_buffers();
  }
  ftape_state= idle;
  ftape_io_state = io_idle;
  TRACE_EXIT;
  return result;
}

/******************************************************************************
 *
 *  functions for the MTIOCTOP commands
 *
 *****************************************************************************/

static int 
mt_dummy( int *dummy )
{
TRACE_FUN( 8, "mt_dummy");

  TRACE_EXIT;
  return -ENOSYS;
}

static int 
mt_reset( int *dummy )
{        
TRACE_FUN( 8, "mt_reset");
int result;

  (void)ftape_seek_to_bot();
  result = ftape_reset_drive();
  init_drive_needed = 1;
  ftape_offline = 0;
  ftape_new_cartridge();
  TRACE_EXIT;
  return result ;
}

static int 
mt_fsf( int *arg )
{
TRACE_FUN( 8, "mt_fsf");
int result;

  result = qic80_skip_files( ftape_seg_pos, *arg );
  just_before_eof = 0;
  TRACE_EXIT;
  return result;
}

static int 
mt_bsf( int *arg )
{
TRACE_FUN( 8, "mt_bsf");
int result;

  result = qic80_skip_files( ftape_seg_pos, -*arg );
  just_before_eof = 0;
  TRACE_EXIT;
  return result;
}

static int
seek_block( int data_offset, int block_increment )
{ 
TRACE_FUN( 8, "seek_block");
int result     = 0;
int act_block  = 0;
int start_seg, end_seg;
int vol_size;                                  
unsigned block_size;
int new_block_pos = -1;

  start_seg = qic80_get_volume_boundaries( ftape_seg_pos, &end_seg );
  if ( start_seg == 0 || end_seg == 0 ) {
    result = -EIO;
  }
  if ( result >= 0 ) {
    /*
     *  user doesn't need to worry about blocksize if ftape_use_compression
     */
    block_size = ftape_use_compression ?
                 qic80_get_volume_block_size( start_seg ) :
                 ftape_block_size;
    new_block_pos = data_offset/block_size + block_increment;
    if ( new_block_pos < 0 ) {
      new_block_pos = 0;
      result = -EINVAL;
    }
    vol_size = qic80_get_volume_size( start_seg );
    if ( ftape_use_compression ) {
      result = ftape_compress_seek( new_block_pos, start_seg, end_seg, 
                                                              block_size );
    } else {
      ftape_uncmpr_pos = block_size * new_block_pos;
      if ( vol_size < ftape_uncmpr_pos ) {
        ftape_uncmpr_pos      = vol_size;
        result = -EIO;
      }
      ftape_data_pos   =   ftape_calc_data_pos ( start_seg, 0 )
                         + ftape_uncmpr_pos;
      ftape_seg_pos= ftape_calc_seg_byte_coord( &ftape_seg_data_pos,
                                                 ftape_data_pos );
    }
    if ( vol_size <= ftape_uncmpr_pos ) {
      just_before_eof = 1;
    } else {
      just_before_eof = 0;
    }
    TRACEi( 3, "new_seg_pos:", ftape_seg_pos);
    TRACEi( 3, "new_data_pos:", ftape_data_pos);
    TRACEi( 3, "new_uncmpr_pos:", ftape_uncmpr_pos);
    TRACEi( 3, "new_seg_data_pos:", ftape_seg_data_pos);
    act_block = ftape_uncmpr_pos / block_size;
    TRACEi( 3,"block_size: ", block_size );
  }                  
  if ( new_block_pos >= 0 ) {
    ftape_resid = new_block_pos - act_block;
  } else {
    ftape_resid = block_increment;
  }
  if ( ftape_resid < 0 ) {
    ftape_resid = -ftape_resid;
  }
  TRACE_EXIT;
  return(result);
}     

static int 
mt_fsr( int *arg )
{ 
TRACE_FUN( 8, "mt_fsr");
int result;

  result = seek_block( ftape_uncmpr_pos, *arg );
  TRACE_EXIT;
  return result;
}

static int 
mt_bsr( int *arg )
{   
TRACE_FUN( 8, "mt_bsr");
int result;

  result = seek_block( ftape_uncmpr_pos,  -*arg );
  TRACE_EXIT;
  return result;
}

static int 
mt_weof( int *arg )
{
TRACE_FUN( 8, "mt_weof");
int result;

  result = qic80_weof( *arg );

  TRACE_EXIT;
  return result;
}

static int 
mt_rew( int *dummy )
{          
TRACE_FUN( 8, "mt_rew");
int result = 0;

  if( first_data_segment != -1 ) {
    (void)def_idle_state();
  }
  ftape_update_header_segments( NULL, 1);
  result = ftape_seek_to_bot();
  ftape_reset_position();
  ftape_zap_read_buffers();
  ftape_zap_write_buffers();
  TRACE_EXIT;
  return result;
}

static int 
mt_offl( int *dummy )
{
TRACE_FUN( 8, "mt_offl");
int result;

  result = mt_rew( NULL );
  going_offline= 1;
  TRACE_EXIT;
  return result;
}

static int 
mt_nop( int *dummy )
{
TRACE_FUN( 8, "mt_nop");
/*  should we set tape status?
 *
 */
/*  if ( !ftape_offline && !no_tape ) (void)def_idle_state( ); */
  TRACE_EXIT;
  return 0; 
}

static int 
mt_reten( int *dummy )
{  
TRACE_FUN( 8, "mt_reten");
int result;

  if( first_data_segment != -1 ) {
    (void)def_idle_state();
  }
  result = ftape_seek_to_eot();
  if (result >= 0) {
    result= ftape_seek_to_bot();
  }
  TRACE_EXIT;
  return( result );
}

static int 
fsfbsfm( int arg )
{ 
TRACE_FUN( 8, "fsfbsfm");
int result = 0;    
int size;
int block_pos;

  /*
   *  What to do? This should seek to the next file-mark and position BEFORE.
   *  That is, a next write would just extend the current file. Well. Let's
   *  just seek to the end of the current file, if count == 1. If count > 1,
   *  then do a "mt_fsf( count - 1 )", and then seek to the end of that file.
   *  If count == 0, do nothing
   *
   */
  if ( arg != 0 ) {
    result = qic80_skip_files( ftape_seg_pos, arg - 1 );
    if ( result >= 0 ) {
      size = qic80_get_volume_size( ftape_seg_pos );
      if ( ftape_use_compression ) {
        block_pos =  size / qic80_get_volume_block_size( ftape_seg_pos );
      } else {
        block_pos =  size / ftape_block_size;
      }              
      block_pos++; /* maybe one too low if volume-size not block aligned */
      (void)mt_seek ( &block_pos );
      if ( ftape_uncmpr_pos != size ) {
        /*
         *   we didn't managed to go there
         */
        TRACEx2(-1,"Error: wanted position at %d, only got to %d", size, ftape_uncmpr_pos );
        ftape_resid = 1;
        result = -EIO;
      } else {
        /*
         *  allow next write to append data
         */
        ftape_resid = 0;
        just_before_eof = 1;
      }
    } else {
      just_before_eof = 0;
      if ( arg > 0 ) {
        ftape_resid++;
      } else {
        ftape_resid--;
      }
    }
  }
  TRACE_EXIT;
  return result;
}   

static int 
mt_bsfm( int *arg )
{
TRACE_FUN( 8, "mt_bsfm");
int result;

  result = fsfbsfm( -*arg );
  TRACE_EXIT;
  return result;
}

static int 
mt_fsfm( int *arg )
{
TRACE_FUN( 8, "mt_fsfm");
int result;

  result = fsfbsfm( *arg );
  TRACE_EXIT;
  return result;
}

static int 
mt_eom( int *dummy )
{              
TRACE_FUN( 8, "mt_eom");

  qic80_skip_to_eom();
  TRACE_EXIT;
  return 0;
}

static int 
mt_erase( int *dummy )
{
TRACE_FUN( 8, "mt_erase");
int result;

  result = ftape_erase();
#ifndef NO_TRACE_AT_ALL
  if ( result < 0 ) {
    TRACE(-1,"Error: ftape_erase failed.");
  }   
#endif
  TRACE_EXIT;
  return result;
}

static int 
mt_ras2( int *dummy ) {
TRACE_FUN( 8, "mt_ras2");
int result;

  result = ftape_fix();
  TRACE_EXIT;
  return result;
} 

static int 
mt_setblk( int *new_size )
{
TRACE_FUN( 8, "mt_setblk");
int result= 0;

  if( (*new_size <= 0) || (*new_size > MAX_BLOCK_SIZE ) ) {
#ifndef NO_TRACE_AT_ALL
      TRACEx3(-1,"Error: desired block_size (%d) should be >= %d and <= %d bytes.\n",
                  *new_size, 1, MAX_BLOCK_SIZE);
#endif
    result= -EINVAL;
  } else {
    ftape_block_size = *new_size;
  }
  TRACE_EXIT;
  return( result );
} 

static int 
mt_setdensity( int *arg )
{
TRACE_FUN( 8, "mt_setdensity");

#ifndef NO_TRACE_AT_ALL
  tracing = *arg;
  TRACEx1( 2, "tracing set to %d", tracing);
  TRACE_EXIT;
  return 0;
#else
  TRACE_EXIT;
  return -ENOSYS;
#endif
}          

static int 
mt_seek( int *new_block_pos )
{ 
TRACE_FUN( 8, "mt_seek");
int result= 0;        

  result = seek_block( 0, *new_block_pos );

  TRACE_EXIT;
  return result;
}

/*
 *  return the block position from start of volume
 */
static int
mt_tell( int *arg )
{
TRACE_FUN( 8, "mt_tell");
int block_size;

  if ( ftape_use_compression ) {
    block_size = qic80_get_volume_block_size ( ftape_seg_pos );
    *arg = (ftape_uncmpr_pos + block_size - 1) / block_size;
  } else {
    *arg = (ftape_uncmpr_pos + ftape_block_size - 1) / ftape_block_size;
  }        
  TRACE_EXIT;
  return 0;
}

/*
 *  change the amount of dma-able memory consumed by the driver.
 *  only usable if compiled with DYN_ALLOC
 *
 */
static int
mt_setdrvbuffer( int *bytes )
{          
TRACE_FUN( 8, "mt_setdrvbuffer");
int result= 0;

#ifdef DYN_ALLOC
  if ( first_data_segment != -1 && !(ftape_offline || no_tape) ) {
    /*
     *  better flush buffers before doing anything to the dma-buffers.
     */
    (void)def_idle_state();
  }
  if ( !suser() ) {
    /* 
     *  Only root is
     *  permitted to change the amount of dma-memory. Currently a maximum
     *  of MAX_DMA_BUFFERS * 128k = 6 * 128k = 768k of memory can be consumed
     *  by the driver.
     */
    result = -EACCES;
  } else if ( *bytes % BUFF_SIZE != 0 ) {
    TRACEx2(-1,"Error: new drive buffering (%d) is not a multiple of %d",
                                        *bytes,                BUFF_SIZE);
    result= -EINVAL;
  } else if ( (*bytes = *bytes / BUFF_SIZE ) < 2 ) {
    TRACEx3(-1,"Error: number of buffers (%d) must be greater %d"
               " and smaller %d", *bytes, 2, MAX_DMA_BUFFERS );
    result= -EINVAL;
  } else {
    result= ftape_change_dma_size( *bytes );
    /*
     *  install the new addresses to the buffer struct
     */
    ftape_zap_read_buffers();
  }
#else 
  result = -ENOSYS;
#endif
  TRACE_EXIT;
  return result;
}


/*      OPEN routine called by kernel-interface code
 */
int
_ftape_open( void)
{
  TRACE_FUN( 5, "_ftape_open");
  int result;
  static int new_tape = 1;

  result = fdc_init();
  if (result >= 0) {
    result = ftape_activate_drive( &drive_type);
    if (result < 0) {
      fdc_disable();
      fdc_release_irq_and_dma();

    } else {
      result = ftape_get_drive_status( &new_tape, &no_tape, &write_protected);
      if (result < 0) {
        ftape_detach_drive();
      } else {
        if (drive_type.vendor_id == UNKNOWN_VENDOR) {
          ftape_log_vendor_id();
        }
        ftape_set_flags( ftape_unit );
        if (no_tape) {
          ftape_offline = 1;
        } else if (new_tape) {
            ftape_offline = 0;
          init_drive_needed = 1;
          read_only = 0;        /* enable writes again */
        }
        if (!ftape_offline && init_drive_needed) {
          ftape_offline = 0;
          result = ftape_init_drive( &formatted);
          if (result >= 0) {
            new_tape = 0;
          } else {
            ftape_detach_drive();
          }
        }
        if (result >= 0) {
          clear_history();
        }
      }
    }
  }
  TRACE_EXIT;
  return result;
}

/*      RELEASE routine called by kernel-interface code
 */
int
_ftape_close( void)
{
  TRACE_FUN( 5, "_ftape_close");
  int result = 0;
  int last_segment = 0;

  if (!ftape_offline) {
    result = ftape_flush_buffers();
    last_segment = ftape_seg_pos - 1;
    if (!(ftape_unit & FTAPE_NO_REWIND)) {
      if (result >= 0) {
        result = ftape_update_header_segments( NULL, 1);
        if (result < 0) {
          TRACE( 1, "error: update of header segments failed");
        }   
      } else {
        TRACE( 1, "error: unable to update header segments");
      }
    }
    ftape_abort_operation();
    if (!(ftape_unit & FTAPE_NO_REWIND)) {
      if (!no_tape) {
        TRACE( 5, "rewinding tape");
        result = ftape_seek_to_bot();
      }
      ftape_reset_position();
      ftape_zap_read_buffers();
      ftape_zap_write_buffers();
    }
  } 
  ftape_detach_drive();
  fdc_uninit();
  if (history.used) {
    TRACE( 3,  "== Non-fatal errors this run: ==");
    TRACE( 3,  "fdc isr statistics:");
    TRACEi( 3, " id_am_errors     :", history.id_am_errors);
    TRACEi( 3, " id_crc_errors    :", history.id_crc_errors);
    TRACEi( 3, " data_am_errors   :", history.data_am_errors);
    TRACEi( 3, " data_crc_errors  :", history.data_crc_errors);
    TRACEi( 3, " overrun_errors   :", history.overrun_errors);
    TRACEi( 3, " no_data_errors   :", history.no_data_errors);
    TRACEi( 3, " retries          :", history.retries);
    if (history.used & 1) {
      TRACE( 3,  "ecc statistics:");
      TRACEi( 3, " crc_errors       :", history.crc_errors);
      TRACEi( 3, " crc_failures     :", history.crc_failures);
      TRACEi( 3, " ecc_failures     :", history.ecc_failures);
      TRACEi( 3, " sectors corrected:", history.corrected);
    }
    TRACEx2( 3, "media defects     : %d%s", history.defects,
            history.defects ? " !!!" : "");
    TRACEi( 3, "repositions       :", history.rewinds);
    TRACEi( 3, "last segment      :", last_segment);
    if ( history.wr_uncompressed != 0 )
    {
      TRACE(  3, "");
      TRACE(  3, "compression statistics (writes):");
      TRACEi( 3, "before compression:", history.wr_uncompressed );
      TRACEi( 3, "after  compression:", history.wr_compressed );
      TRACEx1( 3,"compr./uncmpr.    : %d %% ",
                    (history.wr_compressed * 100)
                   / history.wr_uncompressed   );
    }
    if ( history.rd_uncompressed != 0 )
    {
      TRACE(  3, "");
      TRACE(  3, "compression statistics (reads):");
      TRACEi( 3, "before decompression:", history.rd_compressed );
      TRACEi( 3, "after  decompression:", history.rd_uncompressed );
      TRACEx1( 3,"compr./uncmpr.      : %d %% ",
                 ( history.rd_compressed * 100 )
                   / history.rd_uncompressed );
    }
  }
  if (going_offline) {
    going_offline = 0;
    ftape_offline = 1;
  }
  TRACE_EXIT;
  return result;
}

/*      IOCTL routine called by kernel-interface code
 */
int
_ftape_ioctl( unsigned int command, void * arg)
{
  TRACE_FUN( 5, "_ftape_ioctl");
  int result = EINVAL;
  union { struct mtop mtop;
          struct mtget mtget;
          struct mtpos mtpos;
          struct mtqic80info qic80info;
	  struct mtblksz mtblksz;
  } krnl_arg;
  int arg_size = (command & IOCSIZE_MASK) >> IOCSIZE_SHIFT;
  fun_entry *mt_fun_entry;
  int error;

  /* This check will only catch arguments that are too large !
   */
  if ((command & IOC_INOUT) && arg_size > sizeof( krnl_arg)) {
    TRACEi( 1, "bad argument size:", arg_size);
    TRACE_EXIT;
    return -EINVAL;
  }
  if (command & IOC_IN) {
    error = verify_area( VERIFY_READ, arg, arg_size);
    if (error) {
      TRACE_EXIT;
      return error;
    }
    memcpy_fromfs( &krnl_arg.mtop, arg, arg_size);
  }
  TRACEx1( 5, "called with ioctl command: 0x%08x", command);
  switch (command) {
    /* cpio compatibility
     * mtrasx and mtreset are mt extension by Hennus Bergman
     * mtseek and mttell are mt extension by eddy olk
     */
  case MTIOCTOP:
    TRACEx1( 5, "calling MTIOCTOP command: 0x%08x", krnl_arg.mtop.mt_op);
    if ( arg_size != sizeof( krnl_arg.mtop ) )
    {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }
    if( krnl_arg.mtop.mt_op >= NR_MT_CMDS ) {
      result= -EINVAL;
    } else {
      mt_fun_entry= &mt_funs[ krnl_arg.mtop.mt_op ];
      if ( ( (ftape_offline || no_tape) && !mt_fun_entry->offline) ||
           (!formatted && !mt_fun_entry->not_formatted)
         )
      {                                                      
#ifndef NO_TRACE_AT_ALL
        if ( ftape_offline ) {
          TRACE(-1,"Error: drive is offline");
        } else if ( no_tape ) {
          TRACE(-1,"Error: no tape present");
        } else {
          TRACE(-1,"Error: tape is not formatted" );
        }
#endif
        result = -EPERM;
      } else if (write_protected && !mt_fun_entry->write_protected) {
        TRACE(-1,"Error: drive is write-protected");
        result = -EROFS;
      } else {
        if (    mt_fun_entry->need_idle_state
             && !(ftape_offline || no_tape || !formatted) ) {
          result= def_idle_state();
        }
        if( result >= 0 ) {
          if ( !ftape_qic80_mode && !mt_fun_entry->raw_mode) {
            TRACE(-1,"Error: Drive needs to be in QIC-80 compatibility mode"
                     "for this command");
            result = -EPERM;
          } else {
            result= (mt_fun_entry->function)(&krnl_arg.mtop.mt_count);
          }
        }
      }
    }
    break;
  case MTIOCGET:
    if ( arg_size != sizeof( krnl_arg.mtget ) ) {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }
    krnl_arg.mtget.mt_type  = drive_type.vendor_id + 0x800000;
    if ( ftape_report_raw_drive_status( (int *)&krnl_arg.mtget.mt_dsreg ) < 0 ) {
      /* 
       * status register 
       */
      krnl_arg.mtget.mt_dsreg = 0;
    }
    krnl_arg.mtget.mt_erreg = ftape_last_error; /* error register */
    krnl_arg.mtget.mt_resid = ftape_resid;
    krnl_arg.mtget.mt_gstat = (no_tape       ? GMT_DR_OPEN(-1L) : 0)|
                              (ftape_offline ? 0 : GMT_ONLINE(-1L) );
    if ( !(ftape_offline || no_tape) ) {
      krnl_arg.mtget.mt_gstat |= (write_protected || ftape_old_ftape) ? 
                                 GMT_WR_PROT(-1L) : 0;
      if( first_data_segment != -1 ) {
        if ( ftape_use_compression ) {
          if ( ftape_data_pos >
              (ftape_total_data_amount - (ftape_block_size + 8) ) ) {
            krnl_arg.mtget.mt_gstat |= GMT_EOT(-1L);
          }  
        } else if ( ftape_data_pos >= ftape_total_data_amount ) {
          krnl_arg.mtget.mt_gstat |= GMT_EOT(-1L);
        }
        krnl_arg.mtget.mt_blkno=  ftape_uncmpr_pos
                                / qic80_get_volume_block_size( ftape_seg_pos );
        if ( qic80_check_for_eom( ftape_seg_pos ) ) {
          krnl_arg.mtget.mt_gstat |= GMT_EOD(-1L);
        }
        krnl_arg.mtget.mt_gstat |= ( just_before_eof ?  GMT_EOF(-1L) : 0);
        krnl_arg.mtget.mt_fileno = qic80_find_file_no( ftape_seg_pos );
        if ( krnl_arg.mtget.mt_fileno == 0 && ftape_uncmpr_pos == 0 ) {
          krnl_arg.mtget.mt_gstat |= GMT_BOT(-1L);
        }
      } else {
        krnl_arg.mtget.mt_fileno =
        krnl_arg.mtget.mt_blkno  = -1;
        if ( krnl_arg.mtget.mt_dsreg & QIC_STATUS_AT_BOT ) {
          krnl_arg.mtget.mt_gstat |= GMT_BOT(-1L);
        }
      }
    } else {
      krnl_arg.mtget.mt_fileno = 
      krnl_arg.mtget.mt_blkno  = -1;
    }
    result= 0;
    break;
  case MTIOCPOS:
    TRACE( 5, "Mag tape ioctl command: MTIOCPOS");
    if ( arg_size != sizeof( krnl_arg.mtpos ) )
    {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }
    result= mt_tell( (int *)&krnl_arg.mtpos.mt_blkno );
    break;
  case MTIOC_ZFTAPE_GETBLKSZ:
    TRACE( 5, "Mag tape ioctl command: MTIOC_ZFTAPE_GETBLKSZ");
    if ( arg_size != sizeof( krnl_arg.mtblksz ) )
    {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }
    krnl_arg.mtblksz.mt_blksz = qic80_get_volume_block_size( ftape_seg_pos );
    result = 0;
    break;

  case MTIOCGETQIC80INFO:
    TRACE( 5, "Mag tape ioctl command: MTIOCGETQIC80INFO");
    if ( ftape_qic80_mode ) {
      TRACE_EXIT;
      return -EPERM;
    } 
    if ( arg_size != sizeof( krnl_arg.qic80info ) ) {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }
    if ( krnl_arg.qic80info.bad_sector_map != NULL ) {
      error = verify_area( VERIFY_WRITE, krnl_arg.qic80info.bad_sector_map, BAD_SECTOR_MAP_SIZE );
      if (error) {
        TRACEx1(-1,"Error: Failed to verify pointer %08lx for writing", (unsigned long)krnl_arg.qic80info.bad_sector_map );
        TRACE_EXIT;
        return error;
      }
      memcpy_tofs( krnl_arg.qic80info.bad_sector_map, bad_sector_map, BAD_SECTOR_MAP_SIZE );
    }      
    if ( krnl_arg.qic80info.failed_sector_log != NULL ) {
      error = verify_area( VERIFY_WRITE, krnl_arg.qic80info.failed_sector_log, EOF_MAP_SIZE );
      if (error) {
        TRACEx1(-1,"Error: Failed to verify pointer %08lx for writing", (unsigned long)krnl_arg.qic80info.failed_sector_log );
        TRACE_EXIT;
        return error;
      }
      memcpy_tofs( krnl_arg.qic80info.failed_sector_log, eof_map, EOF_MAP_SIZE );
    }      
    krnl_arg.qic80info.nr_of_failed_sectors = nr_of_eof_marks;
    memcpy( krnl_arg.qic80info.tape_label, ftape_label, 44 );
    krnl_arg.qic80info.compression_map_location = ftape_compression_map_location;
    krnl_arg.qic80info.first_data_segment = first_data_segment;
    krnl_arg.qic80info.last_data_segment = ftape_last_data_segment;
    break;
  case MTIOCSETQIC80INFO:
    TRACE( 5, "Mag tape ioctl command: MTIOCSETQIC80INFO");
    if ( ftape_qic80_mode ) {
      TRACE_EXIT;
      return -EPERM;
    } 
    if ( arg_size != sizeof( krnl_arg.qic80info ) ) {
      TRACEi( 1, "bad argument size:", arg_size);
      TRACE_EXIT;
      return -EINVAL;
    }                                           
    if ( krnl_arg.qic80info.bad_sector_map != NULL ) {
      error = verify_area( VERIFY_READ, krnl_arg.qic80info.bad_sector_map, BAD_SECTOR_MAP_SIZE );
      if (error) {
        TRACEx1(-1,"Error: Failed to verify pointer %08lx for writing", (unsigned long)krnl_arg.qic80info.bad_sector_map );
        TRACE_EXIT;
        return error;
      }
      memcpy_fromfs( bad_sector_map, krnl_arg.qic80info.bad_sector_map, BAD_SECTOR_MAP_SIZE );
      bad_sector_map_changed = 1;
    }                              
    if ( krnl_arg.qic80info.failed_sector_log != NULL ) {
      error = verify_area( VERIFY_READ, krnl_arg.qic80info.failed_sector_log, EOF_MAP_SIZE );
      if (error) {
        TRACEx1(-1,"Error: Failed to verify pointer %08lx for writing", (unsigned long)krnl_arg.qic80info.failed_sector_log );
        TRACE_EXIT;
        return error;
      }
      memcpy_fromfs( eof_map, krnl_arg.qic80info.failed_sector_log, EOF_MAP_SIZE );
      failed_sector_log_changed = 1;
      nr_of_eof_marks = krnl_arg.qic80info.nr_of_failed_sectors;
    }  
    memcpy( ftape_label, krnl_arg.qic80info.tape_label, 44 );
    ftape_compression_map_location = krnl_arg.qic80info.compression_map_location;
    break;
  case MTIOCQIC80WRHEADER:
    TRACE( 5, "Mag tape ioctl command: MTIOCQIC80WRHEADER");
    if ( ftape_qic80_mode ) {
      TRACE_EXIT;
      return -EPERM;
    } 
    result = def_idle_state( );
    if (result >= 0) {               
      /*
       *  force data written to tape
       *
       */
      bad_sector_map_changed          =
      failed_sector_log_changed       =
      ftape_label_changed             =
      ftape_compression_map_changed   = 1;
      result = ftape_update_header_segments( NULL, 1 );
      if (result < 0) {
        TRACE( 1, "error: update of header segments failed");
      }   
    } else {
      TRACE( 1, "error: unable to update header segments");
    }
    ftape_abort_operation();
    break;
  case MTIOCQIC80RDHEADER:
    TRACE( 5, "Mag tape ioctl command: MTIOCQIC80RDHEADER");
    if ( ftape_qic80_mode ) {
      TRACE_EXIT;
      return -EPERM;
    } 
    if ( first_data_segment != -1 ) {
      (void)def_idle_state();
    }
    ftape_new_cartridge();
    result = def_idle_state();
    break;
  
  default:
    result = -EINVAL;
    break;
  }
  if (command & IOC_OUT) {
    error = verify_area( VERIFY_WRITE, arg, arg_size);
    if (error) {
      TRACE_EXIT;
      return error;
    }
    memcpy_tofs( arg, &krnl_arg, arg_size);
  }
  TRACE_EXIT;
  return result;
}

void
ftape_init_driver( void)
{
  drive_type.vendor_id = UNKNOWN_VENDOR;
  drive_type.speed = 0;
  drive_type.wake_up = unknown_wake_up;
  drive_type.name = "Unkown";

  timeout.seek = 650 * SECOND;
  timeout.reset = 670 * SECOND;
  timeout.rewind = 650 * SECOND;
  timeout.head_seek = 15 * SECOND;
  timeout.stop = 5 * SECOND;
  timeout.pause = 16 * SECOND;

  qic_std = -1;
  tape_len = -1;
  current_command = 0;
  current_cylinder = -1;

  segments_per_track = 102;
  segments_per_head = 1020;
  segments_per_cylinder = 4;
  tracks_per_tape = 20;
  ftape_failure = 1;
  ftape_seg_pos = 0;
  first_data_segment = -1;
  ftape_state = idle;
  no_tape = 1;
  formatted = 0;
  ftape_data_rate = 0;
  going_offline = 0;
  read_only = 0;

  init_drive_needed = 1;
  header_segment_1 = -1;
  header_segment_2 = -1;
  used_header_segment = -1;
  location.track = -1;
  location.known = 0;
  tape_running = 0;
  might_be_off_track = 1;

  ftape_new_cartridge();        /* init some tape related variables */
  ftape_init_bsm();
}

