/*
 *      Copyright (C) 1993-1995 Bas Laarhoven.
 *
 * Considerable changes by Claus-Justus Heine, 1994, 1995
 * implemented compression and random block access
 *

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 /home/cvs/zftape/ftape-read.c,v
 claus
 *
 1.3
 1995/11/16 22:31:47
 Exp
 *
 *      This file contains the reading code
 *      for the QIC-117 floppy-tape driver for Linux.
 */

static char RCSid[] =
"ftape-read.c,v 1.3 1995/11/16 22:31:47 claus Exp";


#include "ftape.h"
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <asm/segment.h>

#include "ftape-read.h"
#include "qic117.h"
#include "ftape-io.h"
#include "ftape-ctl.h"
#include "ftape-rw.h"
#include "ftape-write.h"
#include "ftape-eof.h"
#include "ecc.h"
#include "ftape-bsm.h"
#include "ftape-dynmem.h"
#include "qic80-compress.h"
#include "qic80-vtbl.h"

/*      Global vars.
 */
int just_before_eof = 0;

/*      Local vars.
 */
static int buf_len_rd = 0;

void
ftape_zap_read_buffers( void)
{
  int i;

  for (i = 0; i < ftape_num_buffers; ++i) {
#ifdef DYN_ALLOC
    ftape_init_tape_buffer();
#else
    buffer[ i].address = ((byte*) tape_buffer) + i * BUFF_SIZE;
#endif
    buffer[ i].status = waiting;
    buffer[ i].bytes = 0;
    buffer[ i].skip = 0;
    buffer[ i].retry = 0;
  }
  buf_len_rd = 0;
  ftape_state = idle;
}

static unsigned long
convert_sector_map( buffer_struct* buff)
{
  TRACE_FUN( 8, "convert_sector_map");
  int i = 0;
  unsigned long bad_map = get_bad_sector_entry( buff->segment_id);
  unsigned long src_map = buff->soft_error_map | buff->hard_error_map;
  unsigned long dst_map = 0;

  if (bad_map || src_map) {
    TRACEx1( 5, "bad_map = 0x%08lx", bad_map);
    TRACEx1( 5, "src_map = 0x%08lx", src_map);
  }
  while (bad_map) {
    while ((bad_map & 1) == 0) {
      if (src_map & 1) {
        dst_map |= (1 << i);
      }
      src_map >>= 1;
      bad_map >>= 1;
      ++i;
    }
    /* (bad_map & 1) == 1 */
    src_map >>= 1;
    bad_map >>= 1;
  }
  if (src_map) {
    dst_map |= (src_map << i);
  }
  if (dst_map) {
    TRACEx1( 5, "dst_map = 0x%08lx", dst_map);
  }
  TRACE_EXIT;
  return dst_map;
}

static int
correct_and_copy_fraction( unsigned int tail, byte* destination, int start, int size)
{
  TRACE_FUN( 8, "correct_and_copy");
  struct memory_segment mseg;
  int result;
  BAD_SECTOR read_bad;

  mseg.read_bad = convert_sector_map( &buffer[ tail]);
  mseg.marked_bad = 0;          /* not used... */
  mseg.blocks = buffer[ tail].bytes / SECTOR_SIZE;
  mseg.data = buffer[ tail].address;
  /*    If there are no data sectors we can skip this segment.
   */
  if (mseg.blocks <= 3) {
    TRACE( 4, "empty segment");
    TRACE_EXIT;
    return 0;
  }
  read_bad = mseg.read_bad;
  history.crc_errors += count_ones( read_bad);
  result = ecc_correct_data( &mseg);
  if (read_bad != 0 || mseg.corrected != 0) {
    TRACElx( 4, "crc error map:", read_bad);
    TRACElx( 4, "corrected map:", mseg.corrected);
    history.corrected += count_ones( mseg.corrected);
  }
  if (result == ECC_CORRECTED || result == ECC_OK) {
    if (result == ECC_CORRECTED) {
      TRACEi( 3, "ecc corrected segment:", buffer[ tail].segment_id);
    }
    if( (start+size) > ((mseg.blocks - 3) * SECTOR_SIZE) ) {
      size = (mseg.blocks - 3) * SECTOR_SIZE  - start;
    } else if ( size < 0 ) {
      size= 0;
    }
    if( start < 0 ) start= 0;
    if( size > 0 ) {
      memcpy( destination + start, mseg.data + start, size );
    }
    if ((read_bad ^ mseg.corrected) & mseg.corrected) {
      /* sectors corrected without crc errors set */
      history.crc_failures++;
    }
    TRACE_EXIT;
    return size;
  } else {
    TRACEi( 1, "ecc failure on segment", buffer[ tail].segment_id);
    history.ecc_failures++;
    TRACE_EXIT;
    return -EAGAIN;             /* should retry */
  }
  TRACE_EXIT;
  return 0;
}

/*      Read given segment into buffer at address.
 */
int
read_segment_fraction( unsigned segment_id, byte* address, int read_ahead,
                       int start, int  size)
{
  TRACE_FUN( 5, "read_segment");
  int read_done= 0;
  int result = 0;
  int bytes_read = 0;
  int retry = 0;
  
  TRACEi( 5, "segment_id =", segment_id);
  if (ftape_state != reading) {
    if (ftape_state == writing) {
      ftape_flush_buffers();    /* flush write buffer */
      TRACE( 5, "calling ftape_abort_operation");
      result = ftape_abort_operation();
      if (result < 0) {
        TRACE( 1, "ftape_abort_operation failed");
        TRACE_EXIT;
        return -EIO;
      }
    } else {
      /* clear remaining read buffers */
      ftape_zap_read_buffers();
    }
    ftape_state = reading;
  }
  if (segment_id >= segments_per_track * tracks_per_tape) {
    TRACE( 5, "reading past end of tape");
    TRACE_EXIT;
    return -ENOSPC;
  }
  for (;;) {
    /*    Search all full buffers for the first matching the wanted segment.
     *    Clear other buffers on the fly.
     */
    while (!read_done && buffer[ tail].status == done) {
      if (buffer[ tail].segment_id == segment_id) {

        /*        If out buffer is already full, return its contents.
         */
        if (buffer[ tail].deleted) {
          TRACEi( 5, "found segment in cache :", segment_id);
          TRACE_EXIT;
          /*  Return a value that read_header_segment understands.
           *  As this should only occur when searching for the header
           *  segments it shouldn't be misinterpreted elsewhere.
           */
          return 0;
        }
        TRACEi( 5, "found segment in cache :", segment_id);
        result = correct_and_copy_fraction( tail, address, start, size);
        TRACEi( 5, "segment contains (bytes) :", result);
        if (result < 0) {
          if (result != -EAGAIN) {
            TRACE_EXIT;
            return result;
          }
          /* keep read_done == 0, will trigger ftape_abort_operation
           * because reading wrong segment.
           */
          TRACE( 1, "ecc failed, retry");
          ++retry;
        } else {
          read_done = 1;
        }
        bytes_read = result;
      } else {
        TRACEi( 5, "zapping segment in cache :", buffer[ tail].segment_id);
      }
      buffer[ tail].status = waiting;
      next_buffer( &tail);
    }
    if (!read_done && buffer[ tail].status == reading) {
      if (buffer[ tail].segment_id == segment_id) {
        int result = wait_segment( reading);
        if (result < 0) {
          if (result == -EINTR) {
            TRACE_EXIT;
            return result;
          }
          TRACE( 1, "wait_segment failed while reading");
          ftape_abort_operation();
        }
      } else {
        /*        We're reading the wrong segment, stop runner.
         */
        ftape_abort_operation();
      }
    }
    /*    if just passed the last segment on a track, wait for BOT or EOT mark.
     */
    if (runner_status == logical_eot) {
      int status;
      result = ftape_ready_wait( timeout.seek, &status);
      if (result < 0) {
        TRACE( 1, "ftape_ready_wait waiting for eot/bot failed");
      }
      if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT) == 0) {
        TRACE( 1, "eot/bot not reached");
      }
      runner_status = end_of_tape;
    }
    /*    should runner stop ?
     */
    if (runner_status == aborting || runner_status == buffer_overrun ||
        runner_status == end_of_tape) {
      if (runner_status != end_of_tape &&
          !(runner_status == aborting && !tape_running)) {
        ftape_dumb_stop();
      }
      if (runner_status == aborting) {
        if (buffer[ head].status == reading || buffer[ head].status == error) {
          if (buffer[ head].status == error) {
            history.defects += count_ones( buffer[ head].hard_error_map);
          }
          buffer[ head].status = waiting;
        }
      }
      runner_status = idle;       /* aborted ? */
    }
    /*    If segment to read is empty, do not start runner for it,
     *    but wait for next read call.
     */
    if (get_bad_sector_entry( segment_id) == EMPTY_SEGMENT) {
      bytes_read = 0;           /* flag empty segment */
      read_done = 1;
    }
    /*  Allow escape from this loop on signal !
     */
    if (current->signal & _DONT_BLOCK) {
      TRACE( 2, "interrupted by non-blockable signal");
      TRACE_EXIT;
      return -EINTR;
    }
    /*    If we got a segment: quit, or else retry up to limit.
     */
    if (read_done) {
      break;
    }
    if (retry > RETRIES_ON_ECC_ERROR) {
      history.defects++;
      TRACE( 1, "too many retries on ecc failure");
      TRACE_EXIT;
      return -ENODATA;
    }
    /*    Now at least one buffer is empty !
     *    Restart runner & tape if needed.
     */
    TRACEx3( 8, "head: %d, tail: %d, runner_status: %d",
            head, tail, runner_status);
    TRACEx2( 8, "buffer[].status, [head]: %d, [tail]: %d",
            buffer[ head].status, buffer[ tail].status);
    if (buffer[ tail].status == waiting) {
      setup_new_segment( &buffer[ head], segment_id, -1);
      if (!read_ahead) {
        buffer[ head].next_segment = 0; /* disable read-ahead */
      }
      calc_next_cluster( &buffer[ head]);
      if (runner_status == idle) {
        result = ftape_start_tape( segment_id,
                                  buffer[ head].sector_offset);
        if (result < 0) {
          TRACEx1( 1, "Error: segment %d unreachable", segment_id);
          TRACE_EXIT;
          return result;
        }
        runner_status = running;
      }
      buffer[ head].status = reading;
      setup_fdc_and_dma( &buffer[ head], FDC_READ);
    }
  }
  if (read_done) {
    TRACE_EXIT;
    return bytes_read;
  } else {
    TRACE( 1, "too many retries");
    TRACE_EXIT;
    return -EIO;
  }
}

int
read_header_segment( byte* address)
{
  TRACE_FUN( 5, "read_header_segment");
  int i;
  int result;
  int header_segment = -1;
  unsigned int max_floppy_side;
  unsigned int max_floppy_track;
  unsigned int max_floppy_sector;
  int first_failed = 0;
  int status;
  int new_tape_len;

  result = ftape_report_drive_status( &status);
  if (result < 0) {
    TRACE( 1, "error: error_status or report failure");
    TRACE_EXIT;
    return -EIO;
  }
  TRACE( 5, "reading...");
  ftape_last_data_segment = 68;   /* will allow us to read the header ! */
  /*  We're looking for the first header segment.
   *  A header segment cannot contain bad sectors, therefor at the
   *  tape start, segments with bad sectors are (according to QIC-40/80)
   *  written with deleted data marks and must be skipped.
   */
  used_header_segment = -1;
  result = 0;
  for (header_segment = 0;
       header_segment < ftape_last_data_segment && result == 0;
       ++header_segment) {
    /*  Set no read-ahead, the isr will force read-ahead whenever
     *  it encounters deleted data !
     */
    result = read_segment( header_segment, address, 0);
    if (result < 0 && !first_failed) {
      TRACE( 1, "header segment damaged, trying backup");
      first_failed = 1;
      result = 0;               /* force read of next (backup) segment */
    }
  }
  if (result < 0 || header_segment >= ftape_last_data_segment) {
    TRACE( 1, "no readable header segment found");
    TRACE_EXIT;
    return -EIO;
  }
  result = ftape_abort_operation();
  if (result < 0) {
    TRACE( 1, "ftape_abort_operation failed");
    TRACE_EXIT;
    return -EIO;
  }
  if (GET4( address, 0) != 0xaa55aa55) {
    TRACE( 1, "wrong signature in header segment");
    TRACE_EXIT;
    return -EIO;
  }
  header_segment_1 = GET2( address, 6);
  header_segment_2 = GET2( address, 8);
  TRACEx2( 2, "header segments are %d and %d",
          header_segment_1, header_segment_2);
  used_header_segment = (first_failed) ? header_segment_2 : header_segment_1;

  /*    Verify tape parameters...
   *    QIC-40/80 spec:                 tape_parameters:
   *
   *    segments-per-track              segments_per_track
   *    tracks-per-cartridge            tracks_per_tape
   *    max-floppy-side                 (segments_per_track *
   *                                    tracks_per_tape - 1) /
   *                                    segments_per_head
   *    max-floppy-track                segments_per_head /
   *                                    segments_per_cylinder - 1
   *    max-floppy-sector               segments_per_cylinder *
   *                                    SECTORS_PER_SEGMENT
   */
  format_code = (format_type) *(address + 4);
  segments_per_track = GET2( address, 24);
  tracks_per_tape = *(address + 26);
  max_floppy_side = *(address + 27);
  max_floppy_track = *(address + 28);
  max_floppy_sector = *(address + 29);
  TRACEx6( 4, "(fmt/spt/tpc/fhm/ftm/fsm) = %d/%d/%d/%d/%d/%d",
          format_code, segments_per_track, tracks_per_tape,
          max_floppy_side, max_floppy_track, max_floppy_sector);
  new_tape_len = tape_len;
  switch (format_code) {
    case fmt_425ft:
      new_tape_len = 425;
      break;
    case fmt_normal:
      if (tape_len == 0) {        /* otherwise 307 ft */
        new_tape_len = 205;
      }
      break;
    case fmt_1100ft:
      new_tape_len = 1100;
      break;
    case fmt_wide: {
      int segments_per_1000_inch = 1; /* non-zero default for switch */
      switch (qic_std) {
        case QIC_TAPE_QIC40:
          segments_per_1000_inch = 332;
          break;
        case QIC_TAPE_QIC80:
          segments_per_1000_inch = 488;
          break;
        case QIC_TAPE_QIC3010:
          segments_per_1000_inch = 730;
          break;
        case QIC_TAPE_QIC3020:
          segments_per_1000_inch = 1430;
          break;
      }
      new_tape_len = (1000 * segments_per_track +
                     (segments_per_1000_inch - 1)) / segments_per_1000_inch;
      break;
    }
    default:
      TRACE( 1, "unknown tape format, please report !");
      TRACE_EXIT;
      return -EIO;
  }
  if (new_tape_len != tape_len) {
    tape_len = new_tape_len;
    TRACEx1( 1, "calculated tape length is %d ft", tape_len);
    ftape_calc_timeouts();
  }
  if (segments_per_track == 0 && tracks_per_tape == 0 &&
      max_floppy_side == 0 && max_floppy_track == 0 &&
      max_floppy_sector == 0) {
    /*  QIC-40 Rev E and earlier has no values in the header.
     */
    segments_per_track = 68; 
    tracks_per_tape = 20;
    max_floppy_side = 1;
    max_floppy_track = 169;
    max_floppy_sector = 128;
  }
  /*  This test will compensate for the wrong parameter on tapes
   *  formatted by Conner software.
   */
  if (segments_per_track == 150 &&
      tracks_per_tape == 28 &&
      max_floppy_side == 7 &&
      max_floppy_track == 149 &&
      max_floppy_sector == 128) {
    TRACE( -1, "the famous CONNER bug: max_floppy_side off by one !");
    max_floppy_side = 6;
  }
  /*  This test will compensate for the wrong parameter on tapes
   *  formatted by Colorado Windows software.
   */
  if (segments_per_track == 150 &&
      tracks_per_tape == 28 &&
      max_floppy_side == 6 &&
      max_floppy_track == 150 &&
      max_floppy_sector == 128) {
    TRACE( -1, "the famous Colorado bug: max_floppy_track off by one !");
    max_floppy_track = 149;
  }
  segments_per_head = ((max_floppy_sector / SECTORS_PER_SEGMENT) *
                       (max_floppy_track + 1));
  /*
   *    Verify drive_configuration with tape parameters
   */
  if ( segments_per_head == 0 || segments_per_cylinder == 0 ||
      ((segments_per_track * tracks_per_tape - 1) / segments_per_head
       != max_floppy_side) ||
      (segments_per_head / segments_per_cylinder - 1 != max_floppy_track) ||
      (segments_per_cylinder * SECTORS_PER_SEGMENT != max_floppy_sector)
#ifdef TESTING
      || (format_code == 4 && (max_floppy_track != 254 || max_floppy_sector != 128))
#endif
      ) {
    TRACE( 1, "Tape parameters inconsistency, please report");
    TRACE_EXIT;
    return -EIO;
  }
  first_data_segment = GET2( address, 10); /* first data segment */
  TRACEi( 4, "first data segment:", first_data_segment);
  ftape_last_data_segment = GET2( address, 12);
  TRACEi( 4, "last data segment:", ftape_last_data_segment);
  extract_bad_sector_map( address);
  /* Copy the failed sector log into our local buffer.
   */
  strncpy(ftape_label,&address[30],44);
  result = ftape_validate_label( ftape_label );
  extract_file_marks( address);
  if ( result ) {
    qic80_fake_volume_headers  ( eof_map, nr_of_eof_marks );
    ftape_old_ftape = 1;  /* don't write to old ftape-cartridges */
  } else {
    ftape_old_ftape = 0;
    for ( i= 0; i < nr_of_eof_marks; i++ ) {
      if ( valid_segment_no( eof_map[i].mark.segment) ) {
    TRACEx2( 4, "moved entry %d from failed sector log (%d)",
            i, eof_map[i].mark.segment );
        put_bad_sector_entry( eof_map[i].mark.segment, EMPTY_SEGMENT );
      }
    }
    ftape_compression_map_location = GET2( address, 78);
    if ( used_header_segment + 1 < first_data_segment ) {
      /*
       *   just for the sake of streaming the tape
       */
      (void)read_segment( used_header_segment + 1, address, 1);
    }
    result = read_segment( first_data_segment, address, 1);
    if( result < 0 ) {
      TRACEi( 4, "Couldn't read volume table segment at", first_data_segment );
    } else {
      qic80_extract_volume_headers( address );
      result= ftape_extract_compression_map( address, ftape_compression_map_location );
      if ( result < 0 ) {
        TRACEi(1,"Error extracting compression map,", result);
        ftape_compression_map_location = 0;
        result = 0; /* don't worry */
      }
    }
  }
  ftape_calc_data_amount();
  ftape_set_flags( ftape_unit ); /* sets raw mode and compression flag */
  ftape_reset_position();
  TRACE_EXIT;
  return result;
}

/*
 * out:
 *
 * int *read_cnt: the number of bytes we removed from the deblock_buffer
 *                (result)
 * int *to_do   : the remaining size of the read-request. Is changed.
 *
 * int:
 *
 * char *buff      : buff is the address of the upper part of the user buffer,
 *                   that hasn't been filled with data yet.
 * int buf_pos_read: copy of buf_pos_rd
 * int buf_len_read: copy of buf_len_rd
 * char *deblock_buffer: ftape_deblock_buffer
 *
 * returns the amount of data actually copied to the user-buffer
 *
 * *to_do MUST NOT SHRINK except to indicate an EOT. In this case *to_do has to
 * set to 0
 * We cannot return -ENOSPC, because we return the amount of data actually
 * copied to the user-buffer
 *
 */
int
ftape_simple_read ( int *read_cnt,
                    int *to_do,
                    char *buff,
                    int buf_pos_read,
                    int buf_len_read,
                    char *deblock_buffer,
                    unsigned short dummy )
{
TRACE_FUN( 5, "ftape_simple_read");

  *read_cnt = (buf_len_rd < *to_do) ? buf_len_rd : *to_do;
  memcpy_tofs( buff, deblock_buffer + buf_pos_read, *read_cnt);
  TRACEi( 4, "nr bytes just read:", *read_cnt);
  if( ftape_total_data_amount - (ftape_data_pos + *read_cnt) <= 0 ) {
    TRACEx1( 4, "clipped request from %d to 0.", *to_do);
    *to_do = 0;
  }
  TRACE_EXIT;
  return *read_cnt;
}

int
_ftape_read( char* buff, int req_len)
{
  TRACE_FUN( 5, "_ftape_read");
  int result = 0;
  int cnt;
  int to_do = req_len;
  int bytes_read = 0;
  static int this_segs_size = 0;
  static int read_compressed = 0;
  int first_call= 0;
  long remaining;
  int block_size = 1;

  ftape_resid = req_len;

  if ( ftape_offline || !formatted || no_tape) {
    TRACEx3( -1, "offline = %d, formatted = %d, no_tape = %d",
            ftape_offline, formatted, no_tape);
    result = -EIO;
  } else {
    history.used |= 1;
    if ( first_data_segment == -1) {
      result = read_header_segment( deblock_buffer );
    }
  }
  if ( result >= 0 ) {
    if ( ! just_before_eof ) {
      /*
       *  no need for further checks if ftape_just_before_eof, user won't get 
       *  anything. 
       *  NOTE: the qic80_**** functions also handle the case when the driver is
       *  in raw mode, i.e. when called via the *rft* devices
       */
      if ( qic80_check_for_eom( ftape_seg_pos) != 0 ) {
        result = 0;
      } else {
        block_size = qic80_get_volume_block_size ( ftape_seg_pos );
        if ( (req_len % block_size) != 0 ) {
          result= -EINVAL;
        }
      }
    }
    if ( result >= 0 ) {
      if ( (result = verify_area( VERIFY_WRITE, buff, req_len )) != 0 ) {
        TRACE( -1, "verify_area failed");
        result = -EIO;
      }
    }
  }
  if (result < 0) {
    TRACE_EXIT;
    return result;
  }   
  /*  As GNU tar doesn't accept partial read counts when the multiple
   *  volume flag is set, we make sure to return the requested amount
   *  of data. Except, of course, at the end of the tape or file mark.
   */
  if ( just_before_eof ) {
    /*
     *   don't exceed EOF
     */
    remaining = -to_do;
  } else {
    (void)qic80_check_for_eof( ftape_seg_pos, ftape_uncmpr_pos + to_do, 
                               &remaining );
  }
  if ( remaining < 0 ) {
    TRACEx2( 4, "clipped request from %d to %d.", to_do, (int)(to_do + remaining) );
    to_do += remaining;
  }
  if ( ftape_io_state != io_reading ) {
    ftape_io_state= io_reading;
    TRACE(4,"first call");
    first_call= 1; /* first call after writing or repositioning */
    read_compressed = ftape_use_compression && qic80_get_volume_cmpr_flag( ftape_seg_pos );
  }
  while (to_do > 0) { /* don't return with a partial count ! */
    /*
     * buf_len_rd == 0 means that we need to read a new segment.
     * in that case we first check ftape_seg_pos and sector_pos
     * to be consistent and then compute this_segs_size
     */
    if ( buf_len_rd == 0 ) {
      /*
       * there is the vague possibility that ftape_seg_pos points to
       * an empty segment. In that case ftape_seg_data_pos should be 0
       * In all other cases ftape_seg_data_pos should point to a non-bad
       * sector. We check it out.
       */
      do {
        this_segs_size = ftape_get_seg_size( ftape_seg_pos );
        if ( this_segs_size == 0 ) {
          if ( ftape_seg_data_pos != 0 ) {
            TRACEx2(-1, "Error: Empty segment %d and ftape_seg_data_pos > 0 (%d).\n",
                         ftape_seg_pos,ftape_seg_data_pos);
            result = -EIO;
          } else {
            ftape_seg_pos++;
          }
        } else if ( ftape_seg_data_pos >= this_segs_size ) { 
          /* 
           * this should never happen? But we correct it gracefully. 
           */
          ftape_seg_pos++;
          ftape_seg_data_pos =
          this_segs_size     = 0;
          TRACEx3(0, "Warning: segment %d: ftape_seg_data_pos %d"
                     "beyond number of good sectors %d\n.",
                      ftape_seg_pos, ftape_seg_data_pos,
                      this_segs_size/SECTOR_SIZE );
        }
#if 1   
        if ( ftape_seg_pos > ftape_last_data_segment ) {
          TRACEx1(-1,"Error: Segment %d out of range!", ftape_seg_pos);
          result = -EIO; /* we return -EIO and not -ENOSPC becaus this should
                          * not happen. We have already checked for EOT
                          */
        }
#endif                      
        if ( result < 0 ) {
          ftape_resid -= bytes_read;
          TRACE_EXIT;
          return result;
        }
      } while ( this_segs_size == 0 );
    }
    /*
     *  after the checks above we are sure that ftape_seg_pos doesn't point
     * to an empty segment and that ftape_seg_data_pos is inside a good
     * sector inside this segment, that is,
     * 0 <= ftape_seg_data_pos < this_segs_size
     * note that ftape_data_pos remained unchanged.
     */
    if ( buf_len_rd == 0  ) {
      /* 
       * note: we store the segment id of the segment that is inside the
       * deblock buffer. This spares a lot of read_segment()s when we use
       * small block-sizes. The block-size may be 1kb (SECTOR_SIZE). In this
       * case a MTFSR 28 maybe still inside the same segment.
       */
      if ( ftape_seg_pos != ftape_deblock_segment ) {
        result = read_segment( ftape_seg_pos, deblock_buffer,  1);
        if (result != this_segs_size ) {
          TRACEx1( 4, "read_segment result: %d", result);
          if ( result >= 0 ) {
            TRACEx3(-1,"Segment %d should contain %d bytes, "
                       "but we got only %d", 
                        ftape_seg_pos, this_segs_size, result );
            result = -EIO;
          }
          ftape_deblock_segment = -1;
          ftape_resid -= bytes_read;
          TRACE_EXIT;
          return result;
        } else {
          ftape_deblock_segment= ftape_seg_pos;
        }
      } else {
        /*        
         *  If out buffer is already full, return its contents.
         */
        result = this_segs_size;
      }
#if 0
/*
 *  hack to debug the creation of the compression info
 */
      qic80_decode_compression_header( deblock_buffer, ftape_seg_pos );
#endif
      buf_len_rd = result;
      /*
       *  Take as much as we can use
       *  This handles fractional read_segment()'s
       *  Note: the former variable buf_pos_rd is replaced by ftape_seg_data_pos, which 
       *  is also used during write request
       */
      if ( first_call ) {
        TRACEi( 4, "buf_len_rd: ", buf_len_rd );
        buf_len_rd -= ftape_seg_data_pos;
        first_call  = 0;
      };
    }
    /*  Allow escape from this loop on signal !
     */
   if (current->signal & _DONT_BLOCK) {
    TRACE( 2, "interrupted by non-blockable signal");
      ftape_resid -= bytes_read;
      TRACE_EXIT;
      return -EINTR;
    }
    if ( !read_compressed ) {
      /*   
       *  covers old ftape and non-zftape DOS :( archives.
       *  This reads the entire archive and performs no 
       *  translation of QIC-80 archives. That must be done
       *  by a program in user-space.
       *
       *  Also, we fake a single entry volume table if in raw mode 
       *  (i.e. qic80_mode == 0).
       */
      result= ftape_simple_read ( &cnt,
                                  &to_do, 
                                  buff,
                                  ftape_seg_data_pos, 
                                  buf_len_rd,
                                  deblock_buffer,
                                  block_size );
      ftape_uncmpr_pos += cnt;
    } else {                                  
      /*
       *  write an archive that is ** minimal ** QIC-80 compatible.
       *  no file-headers, just a volume-table that protects our
       *  archive against DOS :( programs.
       */
      result= ftape_compress_read ( &cnt,
                                    &to_do,
                                    buff,
                                    ftape_seg_data_pos,
                                    buf_len_rd,
                                    deblock_buffer,
                                    block_size );
    }
    if (result < 0 ) {
      TRACEi(-1,"ftape_read_fun() failed with result ",result);
      ftape_resid -= bytes_read;
      TRACE_EXIT;
      return result;
    }
    TRACEi( 4, "bytes just read                  : ", result );
    TRACEi( 4, "bytes moved out of deblock-buffer: ", cnt );

    bytes_read += result; /* what we got so far       */
    buff       += result; /* index in user-buffer     */
    to_do      -= result; /* what's left from req_len */

    ftape_data_pos     += cnt;
    ftape_seg_data_pos += cnt;
    buf_len_rd         -= cnt;          /* remaining bytes in buffer */
    /*
     *  processed entire segment, setup for reading new one
     */
    if( ftape_seg_data_pos >= this_segs_size ) {
      ftape_seg_pos++;
      ftape_seg_data_pos= 0;
    }
  } /* while (to_do > 0) */

  if ( remaining <= 0 ) {
    /* 
     * nothing left 
     */
    TRACE( 5, "maybe partial count because of eof mark");
    if (bytes_read == 0) {
      /*
       *  the user has read in the eof-mark
       */
      ftape_zap_read_buffers();
      just_before_eof = 0;
      /*
       *  now skip to next file
       *  if the last file ended on segment boundary, we
       *  needn't do anything (handled by advance_to_next_setment)
       */
      ftape_advance_to_next_segment();
      read_compressed = ftape_use_compression && qic80_get_volume_cmpr_flag( ftape_seg_pos );
      ftape_uncmpr_pos = 0;
      ftape_reset_cmpr_locals(); /* sure is sure */
    } else {
      /*
       *  don't skip to the next file. Maybe someone wants to
       *  append data. Just mark that we are at EOF.
       */
      just_before_eof = 1;
    }
  }
  ftape_resid -= result; /* for MTSTATUS       */
  TRACE_EXIT;
  return bytes_read;
}

