------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S I N P U T . L                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.5 $                              --
--                                                                          --
--     Copyright (C) 1992,1993,1994,1995 Free Software Foundation, Inc.     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

--  This child package contains the routines used to actually load a source
--  file and create entries in the source file table. It also contains the
--  routines to create virtual entries for instantiations. This is separated
--  off into a child package to avoid a dependence of Sinput on Osint which
--  would cause trouble in the tree read/write routines.

with Types; use Types;

package Sinput.L is

   -------------------------------------
   -- Handling Generic Instantiations --
   -------------------------------------

   --  As described in Sem_Ch12, a generic instantiation involves making a
   --  copy of the tree of the generic template. The source locations in
   --  this tree directly reference the source of the template. However it
   --  is also possible to find the location of the instantiation.

   --  This is achieved as follows. When an instantiation occurs, a new entry
   --  is made in the source file table. This entry points to the same source
   --  text, i.e. the file that contains the instantiation, but has a distinct
   --  set of Source_Ptr index values. The separate range of Sloc values avoids
   --  confusion, and means that the Sloc values can still be used to uniquely
   --  identify the source file table entry. It is possible for both entries
   --  to point to the same text, because of the virtual origin pointers used
   --  in the source table.

   --  The Instantiation field of this source file index entry, usually set
   --  to No_Source_File, instead contains the Sloc of the instantiation. In
   --  the case of nested instantiations, this Sloc may itself refer to an
   --  instantiation, so the complete chain can be traced.

   --  Two routines are used to build these special entries in the source
   --  file table. Create_Instantiation_Source is first called to build
   --  the virtual source table entry for the instantiation, and then the
   --  Sloc values in the copy are adjusted using Adjust_Instantiation_Sloc.

   type Sloc_Adjustment is private;
   --  Type returned by Create_Instantiation_Source for use in subsequent
   --  calls to Adjust_Instantiation_Sloc.

   -----------------
   -- Subprograms --
   -----------------

   procedure Adjust_Instantiation_Sloc (N : Node_Id; A : Sloc_Adjustment);
   --  The instantiation tree is created by copying the tree of the generic
   --  template (including the original Sloc values), and then applying
   --  Adjust_Instantiation_Sloc to each copied node to adjust the Sloc
   --  to reference the source entry for the instantiation.

   procedure Create_Instantiation_Source
     (X   : Source_File_Index;
      Loc : Source_Ptr;
      A   : out Sloc_Adjustment);
   --  This procedure creates the source table entry for an instantiation.
   --  X is the index of the source file containing the generic template,
   --  and A is set to an adjustment factor to be used in subsequent calls
   --  to Adjust_Instantiation_Sloc. Loc is the SLoc for the instantiation.

   function Load_Source_File (N : File_Name_Type) return Source_File_Index;
   --  Given a source file name, returns the index of the corresponding entry
   --  in the the source file table. If the file is not currently loaded, then
   --  this is the call that causes the source file to be read and an entry
   --  made in the table. A new entry in the table has the file name and time
   --  stamp entries set and the Casing entries set to Unknown. Version is set
   --  to all blanks, and the lines table is initialized but only the first
   --  entry is set (and Last_Line is set to 1). If the given source file
   --  cannot be opened, then the value returned is No_Source_File.

private
   type Sloc_Adjustment is new Int;
   --  This is simply the Int value to be added to the template Sloc value
   --  to obtain the corresponding Sloc for the instantiation reference.

end Sinput.L;
