/*  LAST EDIT: Thu Oct  5 15:43:43 1995 by Thorsten Kukuk (jkuku1)  */
/*
    YPS-0.2, NIS-Server for Linux
    Copyright (C) 1994  Tobias Reber

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Modified for use with ypserv by Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "system.h"

#include <rpc/rpc.h>
#ifdef HAVE_RPC_CLNT_SOC_H
#include <rpc/clnt_soc.h>
#endif
#include "yp.h"
#include <rpcsvc/ypclnt.h>
#include <netdb.h>
#include <fcntl.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <memory.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <gdbm.h>
#include <unistd.h>
#if defined(HAVE_GETOPT_H)
#include <getopt.h>
#elif !defined(HAVE_GETOPT) || defined(sun)
#include <compat/getopt.h>
#endif


#ifndef HAVE_GETOPT
#include <compat/getopt.c>
#endif

#ifdef __FreeBSD__
#define __yp_master yp_master
#else
extern int __yp_master( char *DomainName, char *MapName, char **OutMaster);
#endif
extern int __yp_all( char *, char *, struct ypall_callback *);
extern int _yp_bind(struct sockaddr_in *, char *);
extern int _yp_clear(char *);

#ifndef YPMAPDIR
#define YPMAPDIR "/var/yp"
#endif

static char SourceHost[MAXHOSTNAMELEN], *TargetDomain=NULL, *SourceDomain=NULL;
static struct in_addr IpAddress;
static int Force=0, NoClear=0, TaskId=0, ProgramNumber=0,
	PortNumber=0, logflag=0;
static GDBM_FILE dbm;

static void Perror(char *fmt, ...)
{
   va_list ap;
   va_start(ap, fmt);
   if(logflag)
     vsyslog(LOG_NOTICE, fmt, ap);
   else
     vfprintf(stderr,fmt,ap);
   va_end(ap);
}

static inline char *
ypxfr_err_string(enum ypxfrstat y) 
{
  switch(y) 
    {
    case YPXFR_SUCC:	    return "Success";
    case YPXFR_AGE:	    return "Master's version not newer";
    case YPXFR_NOMAP:	    return "Can't find server for map";
    case YPXFR_NODOM:	    return "Domain not supported";
    case YPXFR_RSRC:	    return "Local resource alloc failure";
    case YPXFR_RPC:	    return "RPC failure talking to server";
    case YPXFR_MADDR:	    return "Can't get master address";
    case YPXFR_YPERR:	    return "YP server/map db error";
    case YPXFR_BADARGS:     return "Request arguments bad";
    case YPXFR_DBM:	    return "Local dbm operation failed";
    case YPXFR_FILE:	    return "Local file I/O operation failed";
    case YPXFR_SKEW:	    return "Map version skew during transfer";
    case YPXFR_CLEAR:	    return "Can't send \"Clear\" req to local ypserv";
    case YPXFR_FORCE:	    return "No local order number in map  use -f flag.";
    case YPXFR_XFRERR:    return "ypxfr error";
    case YPXFR_REFUSED:   return "Transfer request refused by ypserv";
    }
  return "Unknown Error, should not happen";
}

static int 
ypxfr_foreach(int status, char *key, int keylen, char *val, int vallen,
	      int *data)
{
  datum outKey, outData;
  
  if (status==YP_NOMORE)
    return 0;
  if (status!=YP_TRUE) 
    {
      int s=ypprot_err(status);
      Perror("%s\n", yperr_string(s));
      return 1;
    }
	
  outKey.dptr=keylen?key:""; outKey.dsize=keylen;
  outData.dptr=vallen?val:""; outData.dsize=vallen;
  gdbm_store(dbm,outKey, outData, GDBM_REPLACE);
  
  return 0;
}

static enum ypxfrstat ypxfr(char *mapName) 
{
  int localOrderNum=0;
  int masterOrderNum=0;
  char *masterName;
  struct sockaddr_in localHost;
  struct sockaddr_in masterHost;
  struct ypall_callback callback;
  char dbName[1024];
  char dbName2[1024];
  int y, masterSock;
  CLIENT *masterClient;
  static struct timeval tv = { 25, 0 };
  struct ypreq_nokey req;
  struct ypresp_order resp;
  datum outKey, outData;
  char orderNum[12];
    
  memset(&localHost, '\0', sizeof localHost);
  localHost.sin_family=AF_INET;
  localHost.sin_addr.s_addr=htonl(INADDR_LOOPBACK);

  if (SourceHost[0] != '\0') 
    {
      if ((y=__yp_master(SourceDomain, mapName, &masterName)))
	return YPXFR_MADDR;
      strcpy(SourceHost,masterName);
    }

  memset(&masterHost, '\0', sizeof masterHost);
  masterHost.sin_family=AF_INET;
  {
    struct hostent *h=gethostbyname(SourceHost);
    if (!h) 
      {
	return YPXFR_MADDR;
      }
    memcpy(&masterHost.sin_addr, h->h_addr, sizeof masterHost.sin_addr);
  }

  if ((y=_yp_bind(&masterHost, SourceDomain))) return YPXFR_RPC;

  masterSock=RPC_ANYSOCK;
  masterClient=clnttcp_create(&masterHost, YPPROG, YPVERS, &masterSock, 0, 0);
  if (masterClient==NULL) 
    {
      clnt_pcreateerror("YPXFR");
      return YPXFR_RPC;
    }

  req.domain=SourceDomain;
  req.map=mapName;
  y=clnt_call(masterClient, YPPROC_ORDER, xdr_ypreq_nokey,
	      (void *) &req, xdr_ypresp_order, (void *) &resp, tv);
  if (y!=RPC_SUCCESS) 
    {
      clnt_perror(masterClient, "masterOrderNum");
      masterOrderNum=0x7fffffff;
    } 
  else 
    {
      masterOrderNum=resp.ordernum;
    }
  xdr_free(xdr_ypresp_order, (char *)&resp);

  if (!Force) 
    {
      datum inKey, inVal;
      sprintf(dbName, "%s/%s/%s", YPMAPDIR, TargetDomain, mapName);
      if ((dbm=gdbm_open(dbName,0,GDBM_READER, 0600, NULL))==NULL)
	{
	  Perror("Cannot open %s - ignored.\n", dbName);
	  localOrderNum=0;
	} 
      else 
	{
	  inKey.dptr="YP_LAST_MODIFIED"; inKey.dsize=strlen(inKey.dptr);
	  inVal=gdbm_fetch(dbm,inKey);
	  if (inVal.dptr) 
	    {
	      int i;
	      char *d=inVal.dptr;
	      for (i=0; i<inVal.dsize; i++, d++) 
		{
		  if (!isdigit(*d)) 
		    {
		      gdbm_close(dbm);
		      return YPXFR_SKEW;
		    }
		}
	      localOrderNum=atoi(inVal.dptr);
	    }
	  gdbm_close(dbm);
	}
      if (localOrderNum>=masterOrderNum) return YPXFR_AGE;
    }
  
  sprintf(dbName, "%s/%s/%s~", YPMAPDIR, TargetDomain, mapName);
  if ((dbm=gdbm_open(dbName,0,GDBM_NEWDB,0600,NULL))==NULL) 
    {
      Perror("Cannot open %s\n", dbName);
      return YPXFR_DBM;
    }
  
  outKey.dptr="YP_MASTER_NAME"; outKey.dsize=strlen(outKey.dptr);
  outData.dptr=SourceHost; outData.dsize=strlen(outData.dptr);
  gdbm_store(dbm,outKey, outData, GDBM_REPLACE);
  sprintf(orderNum, "%d", masterOrderNum);
  outKey.dptr="YP_LAST_MODIFIED"; outKey.dsize=strlen(outKey.dptr);
  outData.dptr=orderNum; outData.dsize=strlen(outData.dptr);
  gdbm_store(dbm, outKey, outData, GDBM_REPLACE);

  callback.foreach=ypxfr_foreach;
  callback.data=NULL;
  y=__yp_all(SourceDomain, mapName, &callback);
  
  gdbm_close(dbm);
  sprintf(dbName, "%s/%s/%s~", YPMAPDIR, TargetDomain, mapName);
  sprintf(dbName2, "%s/%s/%s", YPMAPDIR, TargetDomain, mapName);
  unlink(dbName2);
  rename(dbName, dbName2);
  
  if (!NoClear) 
    {
      memset(&localHost, '\0', sizeof localHost);
      localHost.sin_family=AF_INET;
      localHost.sin_addr.s_addr=htonl(INADDR_LOOPBACK);
      if (_yp_bind(&localHost, TargetDomain) ||
	  _yp_clear(TargetDomain)) 
	return YPXFR_CLEAR;
    }
  return y==0?YPXFR_SUCC:YPXFR_YPERR;
}

void usage()
{
  fprintf(stderr,"usage: ypxfr [-f] [-c] [-d target domain] [-h source host]\n");
  fprintf(stderr,"             [-s source domain] [-C taskid program-number ipaddr port] mapname\n");
}

int main (int argc, char **argv)
{
  static char res;

  if (!isatty(2)) 
    {
      openlog("ypxfr", LOG_PID, LOG_DAEMON);
      logflag = 1;
    }
  else
    logflag = 0;

  if (argc < 2)
    {
      usage();
      exit(1);
    }

  memset(SourceHost, '\0', sizeof(SourceHost));

  while(1) 
    {
      int c=getopt(argc, argv, "fcd:h:s:C:S");
      if (c==EOF) break;
      switch (c) 
	{
	case 'f':
	  Force++;
	  break;
	case 'c':
	  NoClear++;
	  break;
	case 'd':
	  TargetDomain=optarg;
	  break;
	case 'h':
	  strcpy(SourceHost,optarg);
	  break;
	case 's':
	  SourceDomain=optarg;
	  break;
	case 'C':
	  TaskId=atoi(optarg);
	  ProgramNumber=atoi(argv[optind++]);
	  IpAddress.s_addr=inet_addr(argv[optind++]);
	  PortNumber=atoi(argv[optind++]);
	  break;
	}
    }
  argc-=optind;
  argv+=optind;

  if (!TargetDomain) 
    {
      yp_get_default_domain(&TargetDomain);
    }
  if (!SourceDomain) 
    {
      SourceDomain=TargetDomain;
    }

  for (; *argv; argv++) 
    {
      enum ypxfrstat y;
      if ((y=ypxfr(*argv))!=YPXFR_SUCC) 
	{
	  Perror("%s\n", ypxfr_err_string(y));
	}
      if (TaskId) 
	{
	  struct sockaddr_in addr;
	  struct timeval wait;
	  CLIENT *clnt;
	  int s;
	  ypresp_xfr resp;
	  static struct timeval tv={0,0};
	  
	  memset(&addr, '\0', sizeof addr);
	  addr.sin_addr=IpAddress;
	  addr.sin_port=htons(PortNumber);
	  addr.sin_family=AF_INET;
	  wait.tv_sec=25; wait.tv_usec=0;
	  s=RPC_ANYSOCK;

	  clnt=clntudp_create(&addr, ProgramNumber, 1, wait, &s);
	  if (!clnt) 
	    {
	      clnt_pcreateerror("ypxfr_callback");
	      continue;
	    }

	  resp.transid=TaskId;
	  resp.xfrstat=y;

	  switch (clnt_call(clnt, 1, xdr_ypresp_xfr, (void *) &resp,
			    xdr_void, &res, tv)) 
	    {
	    case RPC_SUCCESS:
	    case RPC_TIMEDOUT:
	      break;
	    default:
	      clnt_perror(clnt, "ypxfr_callback");
	    }
	  
	  clnt_destroy(clnt);
	}
    }
  return 0;
}
